#include <iostream>
#include <iomanip>
#include <fstream>
#include <cmath>
#include "bob.h"

#include "helpers.h"

#include "xlink_management.h"
#include "xlink_entry.h"
#include "macros.h"
#include "probabilities.h"
#include <gsl/gsl_integration.h>
#include "lookup_table.h"
#define SKIN 8.0

XlinkManagement::XlinkManagement(system_parameters *parameters,
                                 system_properties *properties,
                                 const char *crosslink_file,
                                 const char *crosslink_config) {
    Init(parameters, properties, crosslink_file, crosslink_config);
}

XlinkManagement::XlinkManagement() {
    attachment_model_ = 0;
    n_types_ = 0;
    n_bonds_ = 0;
}

// Destroy the XlinkManagement class
XlinkManagement::~XlinkManagement() {
    // Just to be safe, at the end, destroy the shared lock
    #ifdef ENABLE_OPENMP
    omp_destroy_lock(steplock_);
    #endif
}

void XlinkManagement::Init(system_parameters *parameters,
                           system_properties *properties,
                           const char *crosslink_file,
                           const char *crosslink_config) {
    if (crosslink_file == NULL) {
        std::cerr << "Warning: No crosslink file provided" << std::endl;
        attachment_model_ = 0;
        return;
    }

    //ParseParams(crosslink_file);
    std::cout << "reading crosslink structure from " << crosslink_file << std::endl;
    YAML::Node node = YAML::LoadFile(crosslink_file);
    Init(parameters, properties, &node);
}

void XlinkManagement::Init(system_parameters* parameters,
                           system_properties* properties,
                           YAML::Node *pnode) {
    parameters_ = parameters;
    properties_ = properties;

    // Set up the shared lock needed for the Doubly bound stepping of xlinks
    #ifdef ENABLE_OPENMP
    steplock_ = new omp_lock_t;
    omp_init_lock(steplock_);
    #endif

    n_bonds_ = properties->bonds.n_bonds;
    n_dim_ = parameters->n_dim;

    node_ = *pnode;
   
    ParseParams();

    {
        double max_length = 0.0;
        if (parameters->dynamic_instability_flag) {
            max_length = (parameters->n_periodic == 0) ? properties->unit_cell.h[0][0] :
                0.5 * properties->unit_cell.h[0][0];
        }
        CalcCutoff(properties->bonds.length, max_length);
    }

    if (attachment_model_ == 1)
        parameters->nl_twoway_flag = 0;
    else if (attachment_model_ == 2)
        parameters->nl_twoway_flag = 1;

    BuildTables();

    InitPositions(parameters, properties);

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        int i_site = properties->bonds.bond_site_1[i_bond];
        int j_site = properties->bonds.bond_site_1[i_bond];

        for (int i = 0; i < parameters->n_dim; ++i) {
                r_init_[i_site][i] = properties->bonds.r_bond[i_bond][i] -
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

                r_init_[j_site][i] = properties->bonds.r_bond[i_bond][i] +
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
        }
    }
}

void XlinkManagement::InitPositions(system_parameters *parameters,
                                    system_properties *properties) {

    // If we have explicit two-stage xlinks populate xlink list with unbound xlinks
    if (!implicit_0_ && attachment_model_ == 2) {
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            stage_0_xlinks_[i_type].reserve(n_tot_[i_type]);
            double r_cutoff = r_cutoff_0_1_[i_type];
            double r_cutoff2 = SQR(SKIN + r_cutoff);

            for (int i_link = 0; i_link < n_tot_[i_type]; ++i_link) {
                XlinkEntry new_link(this);
                new_link.SetActive();
                new_link.type_ = i_type;

                double r_mag2;
                do {
                    r_mag2 = 0.0;
                    for (int i = 0; i < parameters->n_dim; ++i) {
                        new_link.r_cross_[i] =
                            (gsl_rng_uniform(properties->rng.r) - 0.5) *
                            properties->unit_cell.h[0][0];
                        r_mag2 += SQR(new_link.r_cross_[i]);
                    }
                } while (r_mag2 > SQR(0.5*properties->unit_cell.h[0][0]));

                new_link.dr_[0] = new_link.dr_[1] = new_link.dr_[2] = 0.0;

                new_link.UpdateNeighbs(parameters->n_dim,
                                       parameters->n_periodic,
                                       properties->bonds.n_bonds,
                                       properties->unit_cell.h,
                                       properties->bonds.r_bond,
                                       properties->bonds.s_bond,
                                       properties->bonds.u_bond,
                                       properties->bonds.length,
                                       r_cutoff2);

                PushStage0(&new_link);
            }
        }
    }
}

void XlinkManagement::Init(system_parameters *parameters,
                           system_properties *properties,
                           int n_types, int attachment_model,
                           FILE **f_crosslink) {
    n_bonds_ = properties->bonds.n_bonds;
    n_types_ = n_types;
    attachment_model_ = attachment_model;

    Allocate();

    ReadConfig(parameters, properties, f_crosslink);
}

void XlinkManagement::CalcCutoff(double *length, double max_length) {
    if (attachment_model_ == 0) {
        for (int i_type = 0; i_type < n_types_; ++i_type)
            r_cutoff_1_2_[i_type]= 0.0;
        return;
    }
    std::cout << "Calculating Xlink 1->2 cutoff...\n";
    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
        max_length = MAX(length[i_bond], max_length);
    }

    const double eps = 1E-3; // Tolerance for number of expected crosslinks between pairs
    const double temp = 1.0;
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        double eps_eff = 0.0;
        if (attachment_model_ == 1)
            eps_eff = MAX(eps_eff_[i_type], eps_eff);
        else if (attachment_model_ == 2) {
            // TODO Check this out for correctness
            eps_eff = MAX(eps_eff_2_[i_type][0] + eps_eff_2_[i_type][1], eps_eff);
        }
        double rc_0 = 0.0; //FIXME State what this is 
        if (force_dependent_) {
            double epseffprime = eps_eff * exp(0.5*k_stretch_[i_type]*
                                               xc_[i_type]*xc_[i_type]);
            //std::cout << "epseffprime: " << epseffprime << std::endl;
            rc_0 = sqrt(2.0 / (k_stretch_[i_type]) * temp *
                        log(epseffprime * max_length / eps *
                        sqrt(2.0 * temp / k_stretch_[i_type])));
            rc_0 += xc_[i_type]; // add on the characteristic length
        } else {
            rc_0 = sqrt(2.0 / ( (1-barrier_weight_[i_type]) * k_stretch_[i_type]) *
                        temp *
                        log(eps_eff * max_length / eps * sqrt(2.0 * temp / k_stretch_[i_type])));
        }
        r_cutoff_1_2_[i_type] = rc_0 + r_equil_[i_type];
        parameters_->r_cutoff = MAX(parameters_->r_cutoff, r_cutoff_1_2_[i_type]);

        //std::cout << "Max length: " << max_length << std::endl;
        //std::cout << "eps_eff: " << eps_eff << std::endl;
        //std::cout << "requil: " << r_equil_[i_type] << std::endl;
        //std::cout << "alpha: " << 0.5*k_stretch_[i_type]*(1-barrier_weight_[i_type]) << std::endl;
        //std::cout << "barrier weight: " << barrier_weight_[i_type] << std::endl;
        //std::cout << "spring constant: " << k_stretch_[i_type] << std::endl;

        std::cout << "   Xlink 1->2 cutoff[" << i_type << "]: "
                  << r_cutoff_1_2_[i_type] << "\n";
    }
    std::cout << std::endl;

    for (int i_type = 0; i_type < n_types_; ++i_type)
        r_cutoff_0_1_[i_type] = 1.0;

}

//void XlinkManagement::ParseParams(const char *crosslink_file) {
void XlinkManagement::ParseParams() {
    /* Count and set number of crosslink types */
    n_types_ = (int)node_["crosslink"].size();
    std::cout << "Number of unique crosslink types: " << n_types_ << std::endl;

    /* Allocate all arrays within crosslink structure */
    Allocate();

    /* Parse any parameters that are shared among all crosslink models */
    ParseShared(node_);

    // Check to see if we're writing posit files or not
    if (node_["write_posit"]) {
        write_posit_ = node_["write_posit"].as<bool>();
    }

    if (force_dependent_) {
        std::cout << "Force Dependent Unbinding Enabled!\n";
    }

    if (!node_["attachment-model"].as<std::string>().compare("one-stage")) {
        attachment_model_ = 1;

        this->ParseOneStage(node_);
        if (force_dependent_) {
            std::cout << "ERROR: Force Dependent Kinetics not written for single stage xlinks, exiting!\n";
            exit(1);
        }
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            std::cout << "Crosslink " << i_type << ":" << std::endl;
            std::cout << "   eps_eff = " << eps_eff_[i_type] << std::endl;
            std::cout << "   k_stretch = " << k_stretch_[i_type] << std::endl;
            std::cout << "   r_equil = " << r_equil_[i_type] << std::endl;
            std::cout << "   velocity = " << velocity_[i_type][0] << std::endl;
            std::cout << "   on_rate = " << on_rate_[i_type] << std::endl;
            std::cout << "   f_stall = " << f_stall_[i_type][0] << std::endl;
            std::cout << "   polar_affinity = " << polar_affinity_[i_type] << std::endl;
            std::cout << "   pa_off " << pa_off_[i_type] << std::endl;
            std::cout << "   barrier_weight = " << barrier_weight_[i_type] << std::endl;
            std::cout << "   stall_type = " << stall_type_[i_type] << std::endl;
            std::cout << "   l_stabilize = " << l_stabilize_[i_type] << std::endl;
            std::cout << "   f_stabilize_vg = " << f_stabilize_vg_[i_type] << std::endl;
            std::cout << "   f_stabilize_vs = " << f_stabilize_vs_[i_type] << std::endl;
            std::cout << "   f_stabilize_fc = " << f_stabilize_fc_[i_type] << std::endl;
            std::cout << "   f_stabilize_fr = " << f_stabilize_fr_[i_type] << std::endl;
            std::cout << std::endl;
        }
    }
    else if (!node_["attachment-model"].as<std::string>().compare("two-stage") ||
             !node_["attachment-model"].as<std::string>().compare("two-stage-explicit") ) {
        attachment_model_ = 2;
        if (!node_["attachment-model"].as<std::string>().compare("two-stage-explicit"))
            implicit_0_ = false;
        else
            implicit_0_ = true;

        ParseTwoStage(node_);
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            std::cout << "Crosslink " << i_type << ":" << std::endl;
            std::cout << "   eps_eff_1 = [" << eps_eff_1_[i_type][0] << " " <<
                eps_eff_1_[i_type][1] << "]" << std::endl;
            std::cout << "   eps_eff_2 = [" << eps_eff_2_[i_type][0] << " " <<
                eps_eff_2_[i_type][1] << "]" << std::endl;
            std::cout << "   k_stretch = " << k_stretch_[i_type] << std::endl;
            std::cout << "   r_equil (input) = " << r_equil_[i_type] << std::endl;
            r_equil_[i_type] += 1.0;
            std::cout << "   r_equil (sim adjusted) = " << r_equil_[i_type] << std::endl;
            if (bi_directional_[i_type] == 0) {
                std::cout << "   velocity = [" << velocity_[i_type][0] << " " <<
                    velocity_[i_type][1] << "]" << std::endl;
                std::cout << "   velocity_ap_scale = [" << velocity_ap_scale_[i_type][0] << " " <<
                    velocity_ap_scale_[i_type][1] << "]" << std::endl;
                std::cout << "   velocity_p_scale = [" << velocity_p_scale_[i_type][0] << " " <<
                    velocity_p_scale_[i_type][1] << "]" << std::endl;
                if (arrest_time_[i_type] > 0.0) {
                    std::cout << "   arrest = " << arrest_[i_type] << std::endl;
                    std::cout << "   arrest_time = " << arrest_time_[i_type] << std::endl;
                    std::cout << "   arrest_vel = [" << arrest_vel_[i_type][0] << " " <<
                        arrest_vel_[i_type][1] << "]" << std::endl;
                    std::cout << "   arrest_vel_original = [" << arrest_vel_original_[i_type][0] << " " <<
                        arrest_vel_original_[i_type][1] << "]" << std::endl;
                }
            } else if (bi_directional_[i_type] == 1) {
                std::cout << "   velocity_plus = [" << svelocity_plus_[i_type][0] << " " <<
                    svelocity_plus_[i_type][1] << "]\n";
                std::cout << "   velocity_minus = [" << svelocity_minus_[i_type][0] << " " <<
                    svelocity_minus_[i_type][1] << "]\n";
                std::cout << "   frequency_plus = [" << sfreq_plus_[i_type][0] << " " <<
                    sfreq_plus_[i_type][1] << "]\n";
                std::cout << "   frequency_minus = [" << sfreq_minus_[i_type][0] << " " <<
                    sfreq_minus_[i_type][1] << "]\n";
                std::cout << "   frequency_plus_parallel = [" << sfreq_plus_p_[i_type][0] << " " <<
                    sfreq_plus_p_[i_type][1] << "]\n";
                std::cout << "   frequency_minus_parallel = [" << sfreq_minus_p_[i_type][0] << " " <<
                    sfreq_minus_p_[i_type][1] << "]\n";
                std::cout << "   frequency_plus_antiparallel = [" << sfreq_plus_ap_[i_type][0] << " " <<
                    sfreq_plus_ap_[i_type][1] << "]\n";
                std::cout << "   frequency_minus_antiparallel = [" << sfreq_minus_ap_[i_type][0] << " " <<
                    sfreq_minus_ap_[i_type][1] << "]\n";
            } else {
                std::cerr << "Something went wrong on the velocity for crosslink type " << i_type << std::endl;
                exit(1);
            }
            std::cout << "   on_rate_1 = [" << on_rate_1_[i_type][0] << " " <<
                on_rate_1_[i_type][1] << "]" << std::endl;
            std::cout << "   on_rate_2 = [" << on_rate_2_[i_type][0] << " " <<
                on_rate_2_[i_type][1] << "]" << std::endl;
            std::cout << "   f_stall = [" << f_stall_[i_type][0] << " " <<
                f_stall_[i_type][1] << "]" << std::endl;
            std::cout << "   diffusion_bound = [" << diffusion_bound_[i_type][0] << " " <<
                diffusion_bound_[i_type][1] << "]" << std::endl;
            std::cout << "   diffusion_bound_2 = [" << diffusion_bound_2_[i_type][0] << " " <<
                diffusion_bound_2_[i_type][1] << "]" << std::endl;
            std::cout << "   diffusion_free = " << diffusion_free_[i_type] << std::endl;
            std::cout << "   end_pause = [" << end_pause_[i_type][0] << " " <<
                end_pause_[i_type][1] << "]\n";
            std::cout << "   polar_affinity = " << polar_affinity_[i_type] << std::endl;
            std::cout << "   pa_off = " << pa_off_[i_type] << std::endl;
            if (force_dependent_) {
                std::cout << "   xc = " << xc_[i_type] << std::endl;
            } else {
                std::cout << "   barrier_weight = " << barrier_weight_[i_type] << std::endl;
            }
            std::cout << "   stall_type = " << stall_type_[i_type] << std::endl;
            std::cout << "   l_stabilize = " << l_stabilize_[i_type] << std::endl;
            std::cout << "   f_stabilize_vg = " << f_stabilize_vg_[i_type] << std::endl;
            std::cout << "   f_stabilize_vs = " << f_stabilize_vs_[i_type] << std::endl;
            std::cout << "   f_stabilize_fc = " << f_stabilize_fc_[i_type] << std::endl;
            std::cout << "   f_stabilize_fr = " << f_stabilize_fr_[i_type] << std::endl;

            std::cout << "   color = [" << color_[i_type][0] << " " <<
                color_[i_type][1] << " " << color_[i_type][2] << " " << color_[i_type][3] << "]" << std::endl;

            std::cout << "   n_tot = " << n_tot_[i_type] << std::endl;
            std::cout << std::endl;
        }
    }
    else {
        std::cerr << "Invalid attachment model: " << node_["attachment-model"].as<std::string>() << std::endl;
        exit(1);
    }
}

void XlinkManagement::ReadConfig(system_parameters* parameters,
                                 system_properties* properties,
                                 FILE **f_crosslink) {
    std::cout << "XlinkManagement::ReadConfig Deprecated, do not use!\n";
    exit(1);
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond)
            n_on_bond_2_[i_type][i_bond] = 0;

        /* Deactivate all old crosslinks */
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
            stage_2_xlinks_[i_type][i_bond].clear();
        }
        int n_activelinks;
        int result = fread(&n_activelinks, sizeof(int), 1, f_crosslink[i_type]);
        if (result != 0) {
            for (int i_link = 0; i_link < n_activelinks; ++i_link) {
                int bond_1, bond_2;
                double lambda, mu;
                fread(&(bond_1), sizeof(int), 1, f_crosslink[i_type]);
                fread(&(bond_2), sizeof(int), 1, f_crosslink[i_type]);
                fread(&(lambda), sizeof(float), 1, f_crosslink[i_type]);
                fread(&(mu), sizeof(float), 1, f_crosslink[i_type]);
                XlinkEntry new_xlink(this,
                                     parameters->n_dim,
                                     parameters->n_periodic,
                                     properties->unit_cell.h,
                                     properties->bonds.r_bond, properties->bonds.s_bond,
                                     properties->bonds.u_bond, properties->bonds.length,
                                     i_type, bond_1, bond_2, lambda, mu,
                                     0); // FIXME Needs to read direction state, hardcode for now

                PushStage2(&new_xlink);
            }
        }
        else {
            n_bound_2_[i_type] = 0;
        }
    }
}

void XlinkManagement::Allocate() {
    stall_type_ = (int *) allocate_1d_array(n_types_, sizeof(int));
    end_pause_ = (int **) allocate_2d_array(n_types_, 2, sizeof(int));
    bi_directional_ = (int*) allocate_1d_array(n_types_, sizeof(int));
    k_stretch_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    r_equil_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    velocity_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    arrest_ = (bool *) allocate_1d_array(n_types_, sizeof(bool));
    arrest_time_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    arrest_vel_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    arrest_vel_original_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    diffusion_bound_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    diffusion_bound_2_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    diffusion_free_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    polar_affinity_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    pa_off_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    barrier_weight_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    xc_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    on_rate_ = (double *) allocate_1d_array(n_types_, sizeof(double));
    f_stall_ = (double **) allocate_2d_array(n_types_, 2, sizeof(double));
    n_on_bond_2_ = (int **) allocate_2d_array(n_types_, n_bonds_, sizeof(int));
    n_free_ = (int*) allocate_1d_array(n_types_, sizeof(int));
    n_bound_1_ = (int**) allocate_2d_array(n_types_, 2, sizeof(int));
    n_bound_2_ = (int*) allocate_1d_array(n_types_, sizeof(int));
    n_tot_ = (int*) allocate_1d_array(n_types_, sizeof(int));
    n_exp_bond_ = (double **) allocate_2d_array(n_types_, n_bonds_, sizeof(double));
    n_exp_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    n_exp_0_1_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    eps_eff_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    eps_eff_1_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    eps_eff_2_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    on_rate_1_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    on_rate_2_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    f_stabilize_vg_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    f_stabilize_vs_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    f_stabilize_fc_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    f_stabilize_fr_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    l_stabilize_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    n_exp_lookup_ = (LookupTable*) allocate_1d_array(n_types_, sizeof(LookupTable));
    velocity_p_scale_  = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    velocity_ap_scale_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    velocity_switch_costheta_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    svelocity_plus_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    svelocity_minus_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_plus_  = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_minus_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_plus_p_  = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_minus_p_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_plus_ap_  = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    sfreq_minus_ap_ = (double**) allocate_2d_array(n_types_, 2, sizeof(double));
    color_ = (float**) allocate_2d_array(n_types_, 4, sizeof(float));
    r_init_ = (double **) allocate_2d_array(2 * n_bonds_, n_dim_, sizeof(double));
    r_cutoff_0_1_ = (double*) allocate_1d_array(n_types_, sizeof(double));
    r_cutoff_1_2_ = (double*) allocate_1d_array(n_types_, sizeof(double));

    /* Set default values that are not actually 0 (velocity scales) */
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        velocity_p_scale_[i_type][0] = velocity_p_scale_[i_type][1] =
            velocity_ap_scale_[i_type][0] = velocity_ap_scale_[i_type][1] = 1.0;
        arrest_[i_type] = false;
    }

    /* Initialize xlink arrays. */
    stage_0_xlinks_ = new std::vector<XlinkEntry>[n_types_];
    stage_1_xlinks_ = new std::vector<XlinkEntry>*[n_types_];
    for (int i = 0; i < n_types_; ++i) {
        stage_1_xlinks_[i] = new std::vector<XlinkEntry>[n_bonds_];
    }
    stage_2_xlinks_ = new std::vector<XlinkEntry>*[n_types_];
    for (int i = 0; i < n_types_; ++i) {
        stage_2_xlinks_[i] = new std::vector<XlinkEntry>[n_bonds_];
    }
    one_stage_probability_ = new std::vector<nl_entry>*[n_types_];
    for (int i = 0; i < n_types_; ++i) {
        one_stage_probability_[i] = new std::vector<nl_entry>[n_bonds_];
    }
}

void XlinkManagement::ParseShared(YAML::Node &node) {
    // Check if we are force or energy depenedent
    if (node["force_dependent"]) {
        force_dependent_ = node["force_dependent"].as<bool>();
    }

    // Check what our name for crosslinks is
    if (node["filename_base"]) {
        xlink_filename_ = node["filename_base"].as<std::string>();
    } else {
        xlink_filename_ = "crosslinks";
    }
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        k_stretch_[i_type] = node["crosslink"][i_type]["spring_constant"].as<double>();
        r_equil_[i_type] = node["crosslink"][i_type]["equilibrium_length"].as<double>();
        // Define constants relating to polar affinity or whatever we are calling it
        polar_affinity_[i_type] = node["crosslink"][i_type]["polar_affinity"].as<double>();
        if (YAML::Node pnode = node["crosslink"][i_type]["pa_exp_split"]){
            /* The exponent applied to polar affinity when acting on kon; 
            koff polar affinity contribution is raised the power of (pa_exp_split-1)*/
            double pa_exp_split = pnode.as<double>();
            if (pa_exp_split == 1.0) pa_off_[i_type] = 1.0;
            else pa_off_[i_type] = pow(polar_affinity_[i_type], pa_exp_split-1.0);
        } else pa_off_[i_type] = 1.0;
        // Only use if xlink are force dependent when attaching instead of energy dependent
        if (force_dependent_) {
            xc_[i_type] = node["crosslink"][i_type]["characteristic_length"].as<double>();
        } else {
            barrier_weight_[i_type] = node["crosslink"][i_type]["barrier_weight"].as<double>();
        }
        if (node["common_stabilization_scaling_vg"]) {
            f_stabilize_vg_[i_type] = node["common_stabilization_scaling_vg"].as<double>();
            f_stabilize_vs_[i_type] = node["common_stabilization_scaling_vs"].as<double>();
            f_stabilize_fc_[i_type] = node["common_stabilization_scaling_fc"].as<double>();
            f_stabilize_fr_[i_type] = node["common_stabilization_scaling_fr"].as<double>();
            l_stabilize_[i_type]    = node["common_stabilization_length"].as<double>();
        } else {
            if (node["crosslink"][i_type]["stabilization_scaling"]) {
                f_stabilize_vg_[i_type] = node["crosslink"][i_type]["stabilization_scaling"].as<double>();
                f_stabilize_vs_[i_type] = node["crosslink"][i_type]["stabilization_scaling"].as<double>();
                f_stabilize_fc_[i_type] = node["crosslink"][i_type]["stabilization_scaling"].as<double>();
                f_stabilize_fr_[i_type] = node["crosslink"][i_type]["stabilization_scaling"].as<double>();
            } else {
                f_stabilize_vg_[i_type] = node["crosslink"][i_type]["stabilization_scaling_vg"].as<double>();
                f_stabilize_vs_[i_type] = node["crosslink"][i_type]["stabilization_scaling_vs"].as<double>();
                f_stabilize_fc_[i_type] = node["crosslink"][i_type]["stabilization_scaling_fc"].as<double>();
                f_stabilize_fr_[i_type] = node["crosslink"][i_type]["stabilization_scaling_fr"].as<double>();
            }
            l_stabilize_[i_type] = node["crosslink"][i_type]["stabilization_length"].as<double>();
        }

        if(node["crosslink"][i_type]["color"].IsSequence()) {
            for (int i = 0; i < 4; ++i)
                color_[i_type][i] = node["crosslink"][i_type]["color"][i].as<float>();
        }
        else {
            color_[i_type][0] = 1.0; color_[i_type][1] = 0.0; color_[i_type][2] = 0.5; color_[i_type][3] = 0.0;
        }
        {
            std::string stall_str = node["crosslink"][i_type]["stall_type"].as<std::string>();
            if (!stall_str.compare("none"))
                stall_type_[i_type] = 0;
            else if (!stall_str.compare("parallel"))
                stall_type_[i_type] = 1;
            else if (!stall_str.compare("absolute"))
                stall_type_[i_type] = 2;
            else {
                std::cout << "Wrong stall_type " << stall_str << " declared." << std::endl;
                exit(1);
            }
        }
        n_tot_[i_type] = node["crosslink"][i_type]["reservoir"].as<int>();

        if (node["crosslink"][i_type]["bi_directional"]) {
            if (node["crosslink"][i_type]["bi_directional"].as<bool>()) {
                bi_directional_[i_type] = true;
            }
        }

        if (node["crosslink"][i_type]["arrest_time"]) {
            arrest_time_[i_type] = node["crosslink"][i_type]["arrest_time"].as<double>();
            arrest_vel_[i_type][0] = arrest_vel_[i_type][1] = node["crosslink"][i_type]["arrest_vel"].as<double>();
            arrest_[i_type] = true;
        }
    }
}

void XlinkManagement::ParseOneStage(YAML::Node &node) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        eps_eff_[i_type] = node["crosslink"][i_type]["concentration"].as<double>();
        on_rate_[i_type] = node["crosslink"][i_type]["on_rate"].as<double>();
        velocity_[i_type][0] = velocity_[i_type][1] =
            node["crosslink"][i_type]["velocity"].as<double>();
        f_stall_[i_type][0] = f_stall_[i_type][1] =
            node["crosslink"][i_type]["stall_force"].as<double>();

        switch (node["crosslink"][i_type]["diffusion_free"].Type()) {
        case YAML::NodeType::Scalar:
            diffusion_free_[i_type] =
                node["crosslink"][i_type]["diffusion_free"].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["diffusion_bound"].Type()) {
        case YAML::NodeType::Scalar:
            diffusion_bound_[i_type][0] = diffusion_bound_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            diffusion_bound_[i_type][0] =
                node["crosslink"][i_type]["diffusion_bound"][0].as<double>();
            diffusion_bound_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound"][1].as<double>();
            break;
        }
        diffusion_bound_2_[i_type][0] = diffusion_bound_[i_type][0];
        diffusion_bound_2_[i_type][1] = diffusion_bound_[i_type][1];

        switch (node["crosslink"][i_type]["end_pause"].Type()) {
        case YAML::NodeType::Scalar:
            end_pause_[i_type][0] = end_pause_[i_type][1] =
                node["crosslink"][i_type]["end_pause"].as<int>();
            break;
        case YAML::NodeType::Sequence:
            end_pause_[i_type][0] =
                node["crosslink"][i_type]["end_pause"][0].as<int>();
            end_pause_[i_type][1] =
                node["crosslink"][i_type]["end_pause"][1].as<int>();
            break;
        }

        // XXX FIXME turn off diffusion for now
        //diffusion_free_[i_type] = 0.0;
        //diffusion_bound_[i_type][0] = diffusion_bound_[i_type][1] = 0.0;
        //diffusion_bound_2_[i_type][0] = diffusion_bound_2_[i_type][1] = 0.0;
    
    }
}

void XlinkManagement::ParseTwoStage(YAML::Node &node) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        switch (node["crosslink"][i_type]["concentration_1"].Type()) {
        case YAML::NodeType::Scalar:
            eps_eff_1_[i_type][0] = eps_eff_1_[i_type][1] =
                0.5 * node["crosslink"][i_type]["concentration_1"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            eps_eff_1_[i_type][0] = node["crosslink"][i_type]["concentration_1"][0].as<double>();
            eps_eff_1_[i_type][1] = node["crosslink"][i_type]["concentration_1"][1].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["concentration_2"].Type()) {
        case YAML::NodeType::Scalar:
            eps_eff_2_[i_type][0] = eps_eff_2_[i_type][1] =
                node["crosslink"][i_type]["concentration_2"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            eps_eff_2_[i_type][0] = node["crosslink"][i_type]["concentration_2"][0].as<double>();
            eps_eff_2_[i_type][1] = node["crosslink"][i_type]["concentration_2"][1].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["on_rate_1"].Type()) {
        case YAML::NodeType::Scalar:
            on_rate_1_[i_type][0] = on_rate_1_[i_type][1] =
                node["crosslink"][i_type]["on_rate_1"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            on_rate_1_[i_type][0] = node["crosslink"][i_type]["on_rate_1"][0].as<double>();
            on_rate_1_[i_type][1] = node["crosslink"][i_type]["on_rate_1"][1].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["on_rate_2"].Type()) {
        case YAML::NodeType::Scalar:
            on_rate_2_[i_type][0] = on_rate_2_[i_type][1] =
                node["crosslink"][i_type]["on_rate_2"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            on_rate_2_[i_type][0] = node["crosslink"][i_type]["on_rate_2"][0].as<double>();
            on_rate_2_[i_type][1] = node["crosslink"][i_type]["on_rate_2"][1].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["stall_force"].Type()) {
        case YAML::NodeType::Scalar:
            f_stall_[i_type][0] = f_stall_[i_type][1] =
                node["crosslink"][i_type]["stall_force"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            f_stall_[i_type][0] = node["crosslink"][i_type]["stall_force"][0].as<double>();
            f_stall_[i_type][1] = node["crosslink"][i_type]["stall_force"][1].as<double>();
            break;
        }

        switch (node["crosslink"][i_type]["diffusion_bound"].Type()) {
        case YAML::NodeType::Scalar:
            diffusion_bound_[i_type][0] = diffusion_bound_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            diffusion_bound_[i_type][0] =
                node["crosslink"][i_type]["diffusion_bound"][0].as<double>();
            diffusion_bound_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound"][1].as<double>();
            break;
        }

        diffusion_bound_2_[i_type][0] = diffusion_bound_[i_type][0];
        diffusion_bound_2_[i_type][1] = diffusion_bound_[i_type][1];

        switch (node["crosslink"][i_type]["diffusion_bound_2"].Type()) {
        case YAML::NodeType::Scalar:
            diffusion_bound_2_[i_type][0] = diffusion_bound_2_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound_2"].as<double>();
            break;
        case YAML::NodeType::Sequence:
            diffusion_bound_2_[i_type][0] =
                node["crosslink"][i_type]["diffusion_bound_2"][0].as<double>();
            diffusion_bound_2_[i_type][1] =
                node["crosslink"][i_type]["diffusion_bound_2"][1].as<double>();
            break;
        }


        switch (node["crosslink"][i_type]["end_pause"].Type()) {
        case YAML::NodeType::Scalar:
            end_pause_[i_type][0] = end_pause_[i_type][1] =
                node["crosslink"][i_type]["end_pause"].as<int>();
            break;
        case YAML::NodeType::Sequence:
            end_pause_[i_type][0] =
                node["crosslink"][i_type]["end_pause"][0].as<int>();
            end_pause_[i_type][1] =
                node["crosslink"][i_type]["end_pause"][1].as<int>();
            break;
        }

        switch (node["crosslink"][i_type]["diffusion_free"].Type()) {
        case YAML::NodeType::Scalar:
            diffusion_free_[i_type] =
                node["crosslink"][i_type]["diffusion_free"].as<double>();
            break;
        }

        // If the motor is not tagged ab bi-directional stochastic switching
        if (bi_directional_[i_type] == 0) {
            switch (node["crosslink"][i_type]["velocity"].Type()) {
            case YAML::NodeType::Scalar:
                velocity_[i_type][0] = velocity_[i_type][1] =
                    node["crosslink"][i_type]["velocity"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                velocity_[i_type][0] = node["crosslink"][i_type]["velocity"][0].as<double>();
                velocity_[i_type][1] = node["crosslink"][i_type]["velocity"][1].as<double>();
                break;
            }

            if (arrest_time_[i_type] > 0.0) {
                arrest_vel_original_[i_type][0] = velocity_[i_type][0];
                arrest_vel_original_[i_type][1] = velocity_[i_type][1];
            }

            switch (node["crosslink"][i_type]["velocity_antipolar_scale"].Type()) {
            case YAML::NodeType::Scalar:
                velocity_ap_scale_[i_type][0] = velocity_ap_scale_[i_type][1] =
                    node["crosslink"][i_type]["velocity_antipolar_scale"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                velocity_ap_scale_[i_type][0] =
                    node["crosslink"][i_type]["velocity_antipolar_scale"][0].as<double>();
                velocity_ap_scale_[i_type][1] =
                    node["crosslink"][i_type]["velocity_antipolar_scale"][1].as<double>();
                break;
            }

            switch (node["crosslink"][i_type]["velocity_polar_scale"].Type()) {
            case YAML::NodeType::Scalar:
                velocity_p_scale_[i_type][0] = velocity_p_scale_[i_type][1] =
                    node["crosslink"][i_type]["velocity_polar_scale"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                velocity_p_scale_[i_type][0] =
                    node["crosslink"][i_type]["velocity_polar_scale"][0].as<double>();
                velocity_p_scale_[i_type][1] =
                    node["crosslink"][i_type]["velocity_polar_scale"][1].as<double>();
                break;
            }


            switch (node["crosslink"][i_type]["velocity_switch_costheta"].Type()) {
            case YAML::NodeType::Scalar:
                velocity_switch_costheta_[i_type][0] = velocity_switch_costheta_[i_type][1] =
                    node["crosslink"][i_type]["velocity_switch_costheta"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                velocity_switch_costheta_[i_type][0] =
                    node["crosslink"][i_type]["velocity_switch_costheta"][0].as<double>();
                velocity_switch_costheta_[i_type][1] =
                    node["crosslink"][i_type]["velocity_switch_costheta"][1].as<double>();
                break;
            }
        } else {
            // The motor undergoes stochastic bi-directionaltiy
            // Plus end directed velocity
            switch (node["crosslink"][i_type]["velocity_plus"].Type()) {
            case YAML::NodeType::Scalar:
                svelocity_plus_[i_type][0] = svelocity_plus_[i_type][1] =
                    node["crosslink"][i_type]["velocity_plus"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                svelocity_plus_[i_type][0] = node["crosslink"][i_type]["velocity_plus"][0].as<double>();
                svelocity_plus_[i_type][1] = node["crosslink"][i_type]["velocity_plus"][1].as<double>();
                break;
            }

            // Minus end directed velocity
            switch (node["crosslink"][i_type]["velocity_minus"].Type()) {
            case YAML::NodeType::Scalar:
                svelocity_minus_[i_type][0] = svelocity_minus_[i_type][1] =
                    node["crosslink"][i_type]["velocity_minus"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                svelocity_minus_[i_type][0] = node["crosslink"][i_type]["velocity_minus"][0].as<double>();
                svelocity_minus_[i_type][1] = node["crosslink"][i_type]["velocity_minus"][1].as<double>();
                break;
            }

            // Plus end directed transition frequency
            switch (node["crosslink"][i_type]["frequency_plus"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_plus_[i_type][0] = sfreq_plus_[i_type][1] =
                    node["crosslink"][i_type]["frequency_plus"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_plus_[i_type][0] = node["crosslink"][i_type]["frequency_plus"][0].as<double>();
                sfreq_plus_[i_type][1] = node["crosslink"][i_type]["frequency_plus"][1].as<double>();
                break;
            }

            // Plus end directed transition frequency in the parallel case
            switch (node["crosslink"][i_type]["frequency_plus_parallel"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_plus_p_[i_type][0] = sfreq_plus_p_[i_type][1] =
                    node["crosslink"][i_type]["frequency_plus_parallel"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_plus_p_[i_type][0] = node["crosslink"][i_type]["frequency_plus_parallel"][0].as<double>();
                sfreq_plus_p_[i_type][1] = node["crosslink"][i_type]["frequency_plus_parallel"][1].as<double>();
                break;
            }

            // Plus end directed transition frequency in the antiparallel case
            switch (node["crosslink"][i_type]["frequency_plus_antiparallel"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_plus_ap_[i_type][0] = sfreq_plus_ap_[i_type][1] =
                    node["crosslink"][i_type]["frequency_plus_antiparallel"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_plus_ap_[i_type][0] = node["crosslink"][i_type]["frequency_plus_antiparallel"][0].as<double>();
                sfreq_plus_ap_[i_type][1] = node["crosslink"][i_type]["frequency_plus_antiparallel"][1].as<double>();
                break;
            }

            // Minus end directed transition frequency
            switch (node["crosslink"][i_type]["frequency_minus"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_minus_[i_type][0] = sfreq_minus_[i_type][1] =
                    node["crosslink"][i_type]["frequency_minus"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_minus_[i_type][0] = node["crosslink"][i_type]["frequency_minus"][0].as<double>();
                sfreq_minus_[i_type][1] = node["crosslink"][i_type]["frequency_minus"][1].as<double>();
                break;
            }

            // Minus end directed transition frequency in the parallel case
            switch (node["crosslink"][i_type]["frequency_minus_parallel"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_minus_p_[i_type][0] = sfreq_minus_p_[i_type][1] =
                    node["crosslink"][i_type]["frequency_minus_parallel"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_minus_p_[i_type][0] = node["crosslink"][i_type]["frequency_minus_parallel"][0].as<double>();
                sfreq_minus_p_[i_type][1] = node["crosslink"][i_type]["frequency_minus_parallel"][1].as<double>();
                break;
            }

            // Minus end directed transition frequency in the antiparallel case
            switch (node["crosslink"][i_type]["frequency_minus_antiparallel"].Type()) {
            case YAML::NodeType::Scalar:
                sfreq_minus_ap_[i_type][0] = sfreq_minus_ap_[i_type][1] =
                    node["crosslink"][i_type]["frequency_minus_antiparallel"].as<double>();
                break;
            case YAML::NodeType::Sequence:
                sfreq_minus_ap_[i_type][0] = node["crosslink"][i_type]["frequency_minus_antiparallel"][0].as<double>();
                sfreq_minus_ap_[i_type][1] = node["crosslink"][i_type]["frequency_minus_antiparallel"][1].as<double>();
                break;
            }

        } // bidirectional or not
    }
}

void XlinkManagement::PushOneStageNeighb(int n_dim, int bond_1,
                                         nl_entry p,
                                         double **u_bond,
                                         double *length) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        if (p.r_min_mag2 < SQR(r_cutoff_1_2_[i_type])) {
            double r_dum[3] = {0.0, 0.0, 0.0};
            double s_dum[3] = {0.0, 0.0, 0.0};
            int bond_2 = p.label;
            p.value = PolarAffinityOn(n_dim, i_type, u_bond[bond_1], u_bond[p.label]);
            if (p.value > 0) {
                p.value *=
                    XlinkEntry::Stage_0_2_Probability(n_dim, 0, NULL,
                                                      (1.0 - barrier_weight_[i_type]) *
                                                      k_stretch_[i_type],
                                                      r_equil_[i_type],
                                                      p.r_min_mag2,
                                                      r_dum, s_dum, u_bond[bond_1], length[bond_1],
                                                      p.dr, s_dum, u_bond[bond_2], length[bond_2]);
                one_stage_probability_[i_type][bond_1].push_back(p);
                n_exp_bond_[i_type][bond_1] += p.value;
            }
        }
    }
}

void XlinkManagement::OneStageKMC(system_parameters *parameters,
                                  system_properties *properties) {

    /* Get relevant data shortcuts */
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        if (barrier_weight_[i_type] == 0.0) {
            /* Make sure that the probability to remove one crosslink in the unit
               time is not greater than one. This would be bad. */
            double p_off_single = on_rate_[i_type] * parameters->delta;
            if (p_off_single >= 1.0) {
                printf("removal probability greater than 1, use smaller timestep or k_naught\n");
                exit(1);
            }
            /* Each crosslink has p_off_single chance to be removed. Instead of
               generating n_links random numbers, note that this is just a chain
               of Bernoulli trials each with the same probability. This leads to
               a binomial distribution of the number of removal events so we can
               just pull a random number from a binomial distribution for the number
               of removal events. Then select n_trials_off random crosslinks for removal
            */
            int n_trials_off = gsl_ran_binomial(properties->rng.r,
                                                p_off_single,
                                                n_bound_2_[i_type]);

            /* Remove n_trials_off links */
            for(int i_trial = 0; i_trial < n_trials_off; ++i_trial) {
                int curr_link = 0;
                int n_links = n_bound_2_[i_type];
                /* select random link */
                int i_link = (int) (gsl_rng_uniform(properties->rng.r) * n_links);

                /* find link and remove it */
                for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                    curr_link += n_on_bond_2_[i_type][i_bond];
                    /* If your current link is greater than the desired link, you've
                       found the first bond of the pair. */
                    if (curr_link > i_link) {
                        /* Find number of link to remove in linked list */
                        int j_link = i_link - (curr_link - n_on_bond_2_[i_type][i_bond]);

                        int k_link = 0;
                        for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                             xlink < stage_2_xlinks_[i_type][i_bond].end();
                             xlink++) {
                            if(xlink->IsActive()) {
                                if (k_link == j_link) {
                                    DeleteStage2(&*xlink);

                                    break;
                                }
                                k_link++;
                            }
                        }

                        /* No longer need to search. Move to next removal attempt */
                        break;
                    }
                }
            }
        }
        else {
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                     xlink < stage_2_xlinks_[i_type][i_bond].end();
                     xlink++) {
                    if(xlink->IsActive()) {
                        double p_off = on_rate_[i_type] *
                            parameters->delta * exp(xlink->u_ * barrier_weight_[i_type]);

                        if (gsl_rng_uniform(properties->rng.r) < p_off)
                            DeleteStage2(&*xlink);
                    }
                }
            }
        }

        /*  From wikipedia:
            In probability theory and statistics, the Poisson distribution
            is a discrete probability distribution that expresses the probability
            of a given number of events occurring in a fixed interval of time and/or space
            if these events occur with a known average rate and independently of the time
            since the last event. If the average number of events is the
            product of the time, rate, and total expected particles but each insertion site
            has an infinitesimal probability of insertion, this process matches perfectly.
            Therefore we draw from the Poisson distribution for the number of insertion attempts
        */
        double n_insert = on_rate_[i_type] * parameters->delta *
            n_exp_[i_type] * eps_eff_[i_type];
        int n_trials_on = gsl_ran_poisson(properties->rng.r, n_insert);

        /* Insert n_trials_on new crosslinks */
        for (int i_trial = 0; i_trial < n_trials_on; ++i_trial) {
            /* select bond pair depending on contribution to partition function */
            int i_bond, j_bond;
            this->SelectBonds(one_stage_probability_[i_type],
                              i_type, &i_bond, &j_bond,
                              properties->rng.r);
            /* attempt insertion. If after a certain number of attempts
               the crosslink isnt inserted, try another bond to avoid a (near)
               infinite loop */
            int direction_state[2] = {0};
            XlinkEntry new_link(this,
                                parameters->n_dim, parameters->n_periodic,
                                properties->unit_cell.h,
                                properties->bonds.r_bond,
                                properties->bonds.s_bond,
                                properties->bonds.u_bond,
                                properties->bonds.length,
                                i_type, i_bond, j_bond,
                                properties->rng.r,
                                direction_state); // FIXME hardcode in minus direction state for stochastic direction siwtching
            PushStage2(&new_link);
        }
    }
}

void XlinkManagement::SelectBonds(nl_list *neighbs,
                                  int i_type,
                                  int *bond_1, int *bond_2,
                                  gsl_rng *r_rng) {
    /* Find random location along Z */
    double loc = gsl_rng_uniform(r_rng) * n_exp_[i_type];
    /* Loop over all pairs of bonds in neighbor lists. */
    double curr_loc = 0.0;
    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
        /* Add contribution to location searcher */
        curr_loc += n_exp_bond_[i_type][i_bond];

        /* If curr_loc > loc, then you've found the bond the loc
           resides on */
        if (curr_loc > loc) {
            /* Move back to beginning of current segment */
            curr_loc -= n_exp_bond_[i_type][i_bond];

            for (nl_list::iterator p = neighbs[i_bond].begin();
                 p < neighbs[i_bond].end();
                 ++p) {

                /* find other bond s.t. curr_loc lies between loc and loc+z_pair */
                curr_loc += p->value;
                if (curr_loc > loc) {
                    /* Set bond_1 to be lower numbered bond and bond_2 the higher */
                    int j_bond = p->label;
                    if (j_bond > i_bond) {
                        *bond_1 = i_bond;
                        *bond_2 = j_bond;
                    }
                    else {
                        *bond_1 = j_bond;
                        *bond_2 = i_bond;
                    }

                    return;
                }

            }
        }
    }

    error_exit("Unable to select bonds\n");
}

void XlinkManagement::ClearNeighbors() {
    if (attachment_model_)
        memset(&n_exp_bond_[0][0], 0, n_types_ * n_bonds_ * sizeof(double));
    else
        return;

    if (attachment_model_ == 1) {
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            n_exp_[i_type] = 0.0;
#pragma omp parallel for schedule(runtime)
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                one_stage_probability_[i_type][i_bond].clear();
            }
        }
    }
    else {
        // Not necesssary to clear two stage neighbors. They're copied every
        // time.

        //         for (int i_type = 0; i_type < n_types_; ++i_type) {
        //             n_exp_[i_type] = 0.0;

        // #pragma omp parallel for schedule(runtime)
        //             for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
        //                 for (xlink_list::iterator xliter = stage_1_xlinks_[i_type][i_bond].begin();
        //                      xliter < stage_1_xlinks_[i_type][i_bond].end();
        //                              ++xliter) {
        //                     xliter->ClearNeighbors();
        //                 }
        //             }
        //         }
    }
}

void XlinkManagement::TwoStageKMC(system_parameters *parameters,
                                  system_properties *properties) {
    /* Generate random order for crosslink moves */
    int g[4] = {0, 1, 2, 3};
    for (int i = 0; i < 4; ++i) {
        int j = gsl_rng_uniform_int(properties->rng.r, 4);
        int swap = g[i];
        g[i] = g[j];
        g[j] = swap;
    }

    UpdateArrest(parameters, properties);

    for (int i = 0; i < 4; ++i) {
        switch (g[i]) {
        case 0:
            KMC_0_1(parameters, properties);
            break;
        case 1:
            KMC_1_0(parameters, properties);
            break;
        case 2:
            KMC_1_2(parameters, properties);
            break;
        case 3:
            KMC_2_1(parameters, properties);
            break;
        }
    }
}

void XlinkManagement::KMC_0_1_implicit(system_parameters *parameters,
                                       system_properties *properties) {
    double length_tot = 0.0;
    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond)
        length_tot += properties->bonds.length[i_bond];

    for (int i_type = 0; i_type < n_types_; ++i_type) {
        double concentration = (n_tot_[i_type] - n_bound_1_[i_type][0] -
                                n_bound_1_[i_type][1] - n_bound_2_[i_type])/
            (1.0/6.0*M_PI*CUBE(properties->unit_cell.h[0][0]));
        unsigned int n_on[2] =
            {gsl_ran_poisson(properties->rng.r,
                             on_rate_1_[i_type][0] * eps_eff_1_[i_type][0] *
                             concentration * parameters->delta * length_tot),
             gsl_ran_poisson(properties->rng.r,
                             on_rate_1_[i_type][1] * eps_eff_1_[i_type][1] *
                             concentration * parameters->delta * length_tot)};

        for (unsigned int i = 0; i < (n_on[0] + n_on[1]); ++i) {
            int head_type = (i < n_on[1]);
            double length_pos = gsl_rng_uniform(properties->rng.r) * length_tot;
            double curr_pos = 0.0;
            int i_bond = -1;
            do {
                i_bond++;
                curr_pos += properties->bonds.length[i_bond];
            } while (curr_pos < length_pos);

            double pos = gsl_rng_uniform(properties->rng.r) *
                properties->bonds.length[i_bond];
            XlinkEntry new_link(this, parameters->n_dim,
                                parameters->n_periodic,
                                properties->unit_cell.h,
                                properties->bonds.r_bond,
                                properties->bonds.s_bond,
                                properties->bonds.u_bond,
                                properties->bonds.length,
                                i_type, head_type,
                                i_bond,
                                pos,
                                properties->rng.r,
                                properties->neighbors.neighbs,
                                0); // FIXME: Minus end directed original state, probaby shoudjnt' be hardcoded
            PushStage1(&new_link);
        }
    }
}

void XlinkManagement::KMC_0_1_explicit(system_parameters *parameters,
                                       system_properties *properties) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
             xl_iter < stage_0_xlinks_[i_type].end();
             ++xl_iter) {
            if (xl_iter->IsActive() && xl_iter->n_exp_ > 0.0) {
                double roll = gsl_rng_uniform(properties->rng.r);
                if (roll < xl_iter->n_exp_) {
                    // FIXME: This might not be the appropriate way to do this.
                    // Select head according to binding affinity
                    int head_type = gsl_rng_uniform(properties->rng.r) <
                    (eps_eff_1_[i_type][1] /
                     (eps_eff_1_[i_type][0]+eps_eff_1_[i_type][1]));

                    // Select random neighbor and fall on that neighbor
                    XlinkEntry new_link = *xl_iter;
                    new_link.Convert_0_1(parameters, properties,
                                         head_type, roll);

                    //Add to list for stage 1
                    PushStage1(&new_link);

                    //Remove from list for stage0
                    DeleteStage0(&*xl_iter);
                }
            }
        }
    }
}

void XlinkManagement::KMC_0_1(system_parameters *parameters,
                              system_properties *properties) {
    if (implicit_0_)  // Implicit background of xlinks
        KMC_0_1_implicit(parameters, properties);
    else //Use explicit stage 0 crosslinks
        KMC_0_1_explicit(parameters, properties);
}

void XlinkManagement::Insert_1_2(system_parameters *parameters,
                                 system_properties *properties,
                                 int type) {
    double ran_loc = gsl_rng_uniform(properties->rng.r) * n_exp_[type];
    double loc = 0.0;
    /* Loop over all crosslink lists */
    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
        /* Loop over all crosslinks for i_bond */
        for (xlink_list::iterator xl_iter = stage_1_xlinks_[type][i_bond].begin();
             xl_iter < stage_1_xlinks_[type][i_bond].end();
             xl_iter++) {
            if (xl_iter->IsActive()) {
                loc += xl_iter->n_exp_;
                if (loc > ran_loc) {
                    loc -= xl_iter->n_exp_;

                    for (nl_list::iterator nl_iter = xl_iter->neighbs_.begin();
                         nl_iter < xl_iter->neighbs_.end();
                         nl_iter++) {

                        loc += nl_iter->value;
                        if (loc > ran_loc) {
                            XlinkEntry test_link = *xl_iter;

                            test_link.Convert_1_2(parameters,
                                                  properties,
                                                  nl_iter->label);
                            if (test_link.IsActive()) {
                                DeleteStage1(&*xl_iter);
                                PushStage2(&test_link);
                            }

                            return;
                        }
                    }
                }
            }
        }
    }
}

void XlinkManagement::Push(XlinkEntry *link, xlink_list *list) {
    /* Search for inactive link */
    if (list == &stage_0_xlinks_[link->type_] && list->size() == n_free_[link->type_] - 1) {
        list->push_back(*link);
    }
    else if (list == &stage_1_xlinks_[link->type_][link->head_parent_[0]] &&
             list->size() == (n_bound_1_[link->type_][0] + n_bound_1_[link->type_][1] - 1)) {
        list->push_back(*link);
    }
    else {
        for (xlink_list::iterator xl_iter = list->begin();
             xl_iter < list->end();
             xl_iter++) {
            if (!xl_iter->IsActive()) {
                /* found an inactive link, copy new link into hole and return */
                *xl_iter = *link;
                return;
            }
        }
        /* No link found. Just put the new link at the end of the list */
        list->push_back(*link);
    }
}

void XlinkManagement::KMC_1_2(system_parameters *parameters,
                              system_properties *properties) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        int N_tot = gsl_ran_poisson(properties->rng.r,
                                    n_exp_[i_type] * parameters->delta);
        for (int i_trial = 0; i_trial < N_tot; ++i_trial)
            Insert_1_2(parameters, properties, i_type);
    }
}

void XlinkManagement::KMC_1_0(system_parameters *parameters,
                              system_properties *properties) {
    for (int i_type = 0; i_type < n_types_; ++i_type) {

        double p_off_single_a = parameters->delta * on_rate_1_[i_type][0] * 1.299;
        double p_off_single_b = parameters->delta * on_rate_1_[i_type][1] * 1.299;

        unsigned int n_off[2] =
            {gsl_ran_binomial(properties->rng.r, p_off_single_a, n_bound_1_[i_type][0]),
             gsl_ran_binomial(properties->rng.r, p_off_single_b, n_bound_1_[i_type][1])};

        double r_cutoff = r_cutoff_0_1_[i_type];
        double r_cutoff2 = SQR(SKIN + r_cutoff);

        for (unsigned int i = 0; i < (n_off[0] + n_off[1]); ++i) {
            int head_type = i < n_off[1];
            int i_off = (int) (gsl_rng_uniform(properties->rng.r) * n_bound_1_[i_type][head_type]);
            int loc = -1;

            bool found = false;
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                for (xlink_list::iterator xlink = stage_1_xlinks_[i_type][i_bond].begin();
                     xlink < stage_1_xlinks_[i_type][i_bond].end();
                     ++xlink) {
                    if (xlink->IsActive() && xlink->head_type_[0] == head_type) {
                        loc++;
                        if (loc == i_off) {
                            if (!implicit_0_) {
                                int head_bond = xlink->head_parent_[0];

                                double randr[3];
                                double mag2 = 0.0;
                                do {
                                    mag2 = 0.0;
                                    for (int i = 0; i < parameters->n_dim; ++i) {
                                        randr[i] = 2*r_cutoff *
                                            (gsl_rng_uniform(properties->rng.r) - 0.5);
                                        mag2 += SQR(randr[i]);
                                    }
                                } while (mag2 > r_cutoff2);

                                for (int i = 0; i < parameters->n_dim; ++i) {
                                    xlink->r_cross_[i] =
                                        properties->bonds.r_bond[head_bond][i] +
                                        properties->bonds.u_bond[head_bond][i] *
                                        (xlink->cross_position_[0] -
                                         0.5 * properties->bonds.length[head_bond]) + randr[i];
                                }
                                // Check for pathological crosslink behavior outside of boundary!
                                xlink->CheckBoundary(parameters->n_dim, properties->unit_cell.h);

                                xlink->dr_[0] = xlink->dr_[1] = xlink->dr_[2] = 0.0;

                                xlink->UpdateNeighbs(parameters->n_dim,
                                                     parameters->n_periodic,
                                                     properties->bonds.n_bonds,
                                                     properties->unit_cell.h,
                                                     properties->bonds.r_bond,
                                                     properties->bonds.s_bond,
                                                     properties->bonds.u_bond,
                                                     properties->bonds.length,
                                                     r_cutoff2);

                                PushStage0(&*xlink);
                            }

                            DeleteStage1(&*xlink);

                            found = true;
                            break;
                        }
                    }
                }
                if (found)
                    break;
            }
        }
    }
}

// Choose if we're doing energy or force dependent unbinding
void XlinkManagement::KMC_2_1(system_parameters *parameters,
                              system_properties *properties) {
    if (force_dependent_) {
        KMC_2_1Fdep(parameters, properties);
    } else {
        KMC_2_1Edep(parameters, properties);
    }
}

void XlinkManagement::KMC_2_1Edep(system_parameters *parameters,
                              system_properties *properties) {

    for (int i_type = 0; i_type < n_types_; ++i_type) {
        if (barrier_weight_[i_type] == 0.0 && pa_off_[i_type] == 1.0) { // All detachments equally probable
            int n_heads[2] = {n_bound_2_[i_type], n_bound_2_[i_type]};
            gsl_rng *r = properties->rng.r;

            // FIXME check to make sure that on_rate_2_[i_type][0] * parameters->delta < 1
            unsigned int n_off[2] =
                {gsl_ran_binomial(r, on_rate_2_[i_type][0] * parameters->delta, n_heads[0]),
                 gsl_ran_binomial(r, on_rate_2_[i_type][1] * parameters->delta, n_heads[1])};
            for (unsigned int i = 0; i < (n_off[0] + n_off[1]); ++i) {
                uint8_t head_type = (i < n_off[1]);
                int curr_link = 0;
                int n_links = n_bound_2_[i_type];
                /* select random link */
                int i_link = (int) (gsl_rng_uniform(properties->rng.r) * n_links);
                int found = 0;
                /* find link and remove it */
                for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                    curr_link += n_on_bond_2_[i_type][i_bond];
                    /* If your current link is greater than the desired link, you've
                       found the first bond of the pair. */
                    if (curr_link > i_link) {
                        /* Find number of link to remove in linked list */
                        int j_link = i_link - (curr_link - n_on_bond_2_[i_type][i_bond]);
                        int k_link = 0;
                        for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                             xlink < stage_2_xlinks_[i_type][i_bond].end();
                             xlink++) {

                            if(xlink->IsActive()) {
                                if (k_link == j_link) {
                                    XlinkEntry new_link = *xlink;
                                    /* Remove link from stage 2 list */
                                    DeleteStage2(&*xlink);

                                    /* Our lucky link should pop off either '0' or '1' head */
                                    new_link.Convert_2_1(head_type);

                                    /* Copy converted link to stage 1 list */
                                    PushStage1(&new_link);

                                    found = 1;
                                    break;
                                }
                                k_link++;
                            }
                        }
                    }
                    if (found)
                        break;
                }
            }
        }
        else { // Force dependendent detachment. Must try to remove each xlink individually
            double p_off_base[2] = //probability to remove single xlink with no force
                {on_rate_2_[i_type][0] * parameters->delta,
                 on_rate_2_[i_type][1] * parameters->delta};
            //double r_cutoff2 = SQR(SKIN + r_cutoff_0_1_[i_type]);;
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                     xlink < stage_2_xlinks_[i_type][i_bond].end();
                     xlink++) {
                    if(xlink->IsActive()) {
                        double boltz = exp(barrier_weight_[i_type] * xlink->u_);
                        double pa_off = 1.0;
                        if (pa_off_[i_type] != 1.0){
                            pa_off = PolarAffinityOff( n_dim_, i_type,  
                                                       properties->bonds.u_bond[xlink->head_parent_[0]],
                                                       properties->bonds.u_bond[xlink->head_parent_[1]]);
                        }
                        uint8_t off[2] =
                            {gsl_rng_uniform(properties->rng.r) < p_off_base[0] * boltz * pa_off,
                             gsl_rng_uniform(properties->rng.r) < p_off_base[1] * boltz * pa_off};

                        if (off[0] && off[1]) {
                            XlinkEntry new_link = *xlink;
                            /* Remove link from stage 2 list */
                            DeleteStage2(&*xlink);
                            if (!implicit_0_) { //If using explicit background,
                                // Choose randomly which head to unbind
                                new_link.Convert_2_1(gsl_rng_uniform(properties->rng.r) > .5);
                                PushStage1(&new_link);
                            }
                        }
                        else if (off[0] || off[1]) {
                            XlinkEntry new_link = *xlink;
                            /* Remove link from stage 2 list */
                            DeleteStage2(&*xlink);
                            /* If off[1] is true, remove head '1', else remove head '0' */
                            new_link.Convert_2_1(off[1]);
                            /* Copy converted link to stage 1 list */
                            PushStage1(&new_link);
                        }
                    }
                }
            }
        }
    }
}

void XlinkManagement::KMC_2_1Fdep(system_parameters *parameters,
                              system_properties *properties) {

    for (int i_type = 0; i_type < n_types_; ++i_type) {
        // Force dependendent detachment. Must try to remove each xlink individually
        double p_off_base[2] = //probability to remove single xlink with no force
            {on_rate_2_[i_type][0] * parameters->delta,
             on_rate_2_[i_type][1] * parameters->delta};
        double r_cutoff2 = SQR(SKIN + r_cutoff_0_1_[i_type]);;
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
            for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                 xlink < stage_2_xlinks_[i_type][i_bond].end();
                 xlink++) {
                if(xlink->IsActive()) {
                    double pa_off = 1.0;
                    if (pa_off_[i_type] != 1.0){
                        pa_off = PolarAffinityOff( n_dim_, i_type,  
                                                   properties->bonds.u_bond[xlink->head_parent_[0]],
                                                   properties->bonds.u_bond[xlink->head_parent_[1]]);
                    }
                    //double boltz = exp(barrier_weight_[i_type] * xlink->u_);
                    //double fmag = 0.0;
                    //for (int idim = 0; idim < n_dim_; ++idim) {
                    //    fmag += SQR(xlink->flink_[idim]);
                    //}
                    //fmag = sqrt(fmag);
                    double fmag = xlink->feff_;
                    double boltz = exp(xc_[i_type] * fmag);
                    uint8_t off[2] =
                        {gsl_rng_uniform(properties->rng.r) < p_off_base[0] * boltz * pa_off,
                         gsl_rng_uniform(properties->rng.r) < p_off_base[1] * boltz * pa_off};

                    if (off[0] && off[1]) {
                        XlinkEntry new_link = *xlink;

                        if (!implicit_0_) { //If using explicit background,
                            // fall off halfway between two heads
                            int head_bond = xlink->head_parent_[0];
                            for (int i = 0; i < parameters->n_dim; ++i)
                                xlink->r_cross_[i] = 0.5 * xlink->r_cross_[i] +
                                    properties->bonds.r_bond[head_bond][i] +
                                    properties->bonds.u_bond[head_bond][i] *
                                    (xlink->cross_position_[0] -
                                     0.5 * properties->bonds.length[head_bond]);
                            xlink->CheckBoundary(parameters->n_dim,
                                                 properties->unit_cell.h);

                            xlink->UpdateNeighbs(parameters->n_dim,
                                                 parameters->n_periodic,
                                                 properties->bonds.n_bonds,
                                                 properties->unit_cell.h,
                                                 properties->bonds.r_bond,
                                                 properties->bonds.s_bond,
                                                 properties->bonds.u_bond,
                                                 properties->bonds.length,
                                                 r_cutoff2);

                            PushStage0(&*xlink);
                        }
                        /* Remove link from stage 2 list */
                        DeleteStage2(&*xlink);
                    }
                    else if (off[0] || off[1]) {
                        XlinkEntry new_link = *xlink;
                        /* Remove link from stage 2 list */
                        DeleteStage2(&*xlink);

                        /* If off[1] is true, remove head '1', else remove head '0' */
                        new_link.Convert_2_1(off[1]);

                        /* Copy converted link to stage 1 list */
                        PushStage1(&new_link);
                    }
                }
            }
        }
    }
}

void XlinkManagement::PushStage0(XlinkEntry *xl) {
    xl->stage_ = 0;
    xl->n_exp_ = 0;
    n_free_[xl->type_]++;
    xl->SetActive();

    Push(xl, &stage_0_xlinks_[xl->type_]);
}

void XlinkManagement::PushStage1(XlinkEntry *xl) {
    xl->stage_ = 1;
    n_bound_1_[xl->type_][xl->head_type_[0]] += 1;
    //std::cout << "nbound1 set: " << n_bound_1_[xl->type_][xl->head_type_[0]] << std::endl;
    n_exp_[xl->type_] += xl->n_exp_;
    n_exp_bond_[xl->type_][xl->head_parent_[0]] += xl->n_exp_;

    xl->SetActive();
    Push(xl, &stage_1_xlinks_[xl->type_][xl->head_parent_[0]]);
}

void XlinkManagement::PushStage2(XlinkEntry *xl) {
    xl->stage_ = 2;
    n_on_bond_2_[xl->type_][xl->head_parent_[0]]++;
    n_bound_2_[xl->type_]++;
    xl->SetActive();

    Push(xl, &stage_2_xlinks_[xl->type_][xl->head_parent_[0]]);
}

void XlinkManagement::DeleteStage0(XlinkEntry *xl) {
    xl->stage_ = 0;
    n_free_[xl->type_]--;
    n_exp_0_1_[xl->type_] -= xl->n_exp_;

    xl->SetInactive();
}

void XlinkManagement::DeleteStage1(XlinkEntry *xl) {
    xl->stage_ = 0;
    n_bound_1_[xl->type_][xl->head_type_[0]] -= 1;
    n_exp_[xl->type_] -= xl->n_exp_;
    n_exp_bond_[xl->type_][xl->head_parent_[0]] -= xl->n_exp_;
    xl->SetInactive();
}

void XlinkManagement::DeleteStage2(XlinkEntry *xl) {
    n_bound_2_[xl->type_]--;
    n_on_bond_2_[xl->type_][xl->head_parent_[0]]--;
    xl->SetInactive();
}

void XlinkManagement::PrepKMC(system_parameters *parameters,
                              system_properties *properties) {
    if (attachment_model_ == 2) {
        Update_0_1_Probability(parameters, properties);
        Update_1_2_Probability(parameters, properties);
    }
    else {
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            n_exp_[i_type] = 0.0;
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                n_exp_[i_type] += n_exp_bond_[i_type][i_bond];
            }
        }
    }
}

void XlinkManagement::StepKMC(system_parameters *parameters,
                              system_properties *properties) {
    PrepKMC(parameters, properties);

    if (attachment_model_ == 1)
        this->OneStageKMC(parameters, properties);
    else if (attachment_model_ == 2)
        this->TwoStageKMC(parameters, properties);
}

void XlinkManagement::Clear() {
    ClearNeighbors();
}

void XlinkManagement::CheckStage0Neighbs(system_parameters *parameters,
                                         system_properties *properties) {
    if (implicit_0_) {
        double half_skin2 = 0.25 * SQR(SKIN);

        for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
            int i_site = properties->bonds.bond_site_1[i_bond];
            int j_site = properties->bonds.bond_site_1[i_bond];

            double dri[3];
            double drj[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                dri[i] = properties->bonds.r_bond[i_bond][i] -
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i] -
                    r_init_[i_site][i];
                drj[i] = properties->bonds.r_bond[i_bond][i] +
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i] -
                    r_init_[j_site][i];
            }
            if ( (dot_product(parameters->n_dim, dri, dri) > half_skin2) ||
                 (dot_product(parameters->n_dim, drj, drj) > half_skin2) ) {
                UpdateStage0Neighbs(parameters, properties);
                return;
            }
        }
    }
}

void XlinkManagement::UpdateStage0Neighbs(system_parameters *parameters,
                                          system_properties *properties) {
    if (!implicit_0_) {
        for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
            int i_site = properties->bonds.bond_site_1[i_bond];
            int j_site = properties->bonds.bond_site_1[i_bond];

            for (int i = 0; i < parameters->n_dim; ++i) {
                r_init_[i_site][i] = properties->bonds.r_bond[i_bond][i] -
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

                r_init_[j_site][i] = properties->bonds.r_bond[i_bond][i] +
                    0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            }
        }

        for (int i_type = 0; i_type < n_types_; ++i_type) {
            n_exp_0_1_[i_type] = 0.0;
            double r_cutoff = r_cutoff_0_1_[i_type];
            double r_cutoff2 = SQR(SKIN + r_cutoff);

            for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
                 xl_iter < stage_0_xlinks_[i_type].end();
                 xl_iter++) {
                xl_iter->UpdateNeighbs(parameters->n_dim,
                                       parameters->n_periodic,
                                       properties->bonds.n_bonds,
                                       properties->unit_cell.h,
                                       properties->bonds.r_bond,
                                       properties->bonds.s_bond,
                                       properties->bonds.u_bond,
                                       properties->bonds.length,
                                       r_cutoff2);
            }
        }
    }
}

void XlinkManagement::Update_0_1_Probability(system_parameters *parameters,
                                             system_properties *properties) {
    if (!implicit_0_) {
        CheckStage0Neighbs(parameters, properties);
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            n_exp_0_1_[i_type] = 0.0;

            for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
                 xl_iter < stage_0_xlinks_[i_type].end();
                 xl_iter++) {
                if (xl_iter->IsActive()) {
                    xl_iter->StepUnbound(parameters->n_dim, parameters->delta,
                                         properties->unit_cell.h, properties->rng.r);
                    xl_iter->Update_0_1(parameters->n_dim,
                                        parameters->n_periodic,
                                        properties->bonds.n_bonds,
                                        properties->unit_cell.h,
                                        properties->bonds.r_bond,
                                        properties->bonds.s_bond,
                                        properties->bonds.u_bond,
                                        properties->bonds.length,
                                        parameters->delta);
                    n_exp_0_1_[i_type] += xl_iter->n_exp_;
                }
            }
        }
    }
}

void XlinkManagement::Update_1_2_Probability(system_parameters *parameters,
                                             system_properties *properties) {
    // Clear expected number to bind
    //for (int i_type = 0; i_type < n_types_; ++i_type) {
    //    n_exp_[i_type] = 0.0;
    //    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond)
    //        n_exp_bond_[i_type][i_bond] = 0.0;
    //}
    
    // Loop of xlink types 
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        n_exp_[i_type] = 0.0; //FIXME why do we clear it twice?
        // Loop over bonds to collect all the stage1 crosslinks on each bond
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
            n_exp_bond_[i_type][i_bond] = 0.0;
            for (xlink_list::iterator xl_iter = stage_1_xlinks_[i_type][i_bond].begin();
                 xl_iter < stage_1_xlinks_[i_type][i_bond].end();
                 xl_iter++) {
                //If xlink is active, step the motor and see if it has fallen off
                if (xl_iter->IsActive()) {
                    xl_iter->StepSingly(parameters->n_dim,
                                        parameters->delta,
                                        properties->bonds.length,
                                        properties->rng.r);
                    if (xl_iter->IsActive()) {
                        // Update probability of binding to neighbor rod
                        xl_iter->Update_1_2(parameters->n_dim,
                                            properties->bonds.u_bond,
                                            properties->bonds.length,
                                            properties->neighbors.neighbs);
                        // Add to total number of xlinks expected to bind rod
                        n_exp_bond_[i_type][i_bond] += xl_iter->n_exp_;
                        // Add to total number of xlinks expected to bind
                        n_exp_[i_type] += xl_iter->n_exp_; 
                    }
                    else { // Fallen off the end, remove.
                        if (!implicit_0_) {
                            int head_bond = xl_iter->head_parent_[0];
                            for (int i = 0; i < parameters->n_dim; ++i){
                                xl_iter->r_cross_[i] =
                                    properties->bonds.r_bond[head_bond][i] +
                                    properties->bonds.u_bond[head_bond][i] *
                                    (xl_iter->cross_position_[0] -
                                     0.5 * properties->bonds.length[head_bond]);
                            }
                            // Check to see if xlink unbound outside of system
                            xl_iter->CheckBoundary(parameters->n_dim,
                                                   properties->unit_cell.h);
                            //double r_cutoff = sqrt(10.0 * parameters->delta * diffusion_free_[i_type]);
                            double r_cutoff = 1.0; // Cutoff for neighborlists
                            double r_cutoff2 = SQR(SKIN + r_cutoff); 
                            xl_iter->UpdateNeighbs(parameters->n_dim,
                                                   parameters->n_periodic,
                                                   properties->bonds.n_bonds,
                                                   properties->unit_cell.h,
                                                   properties->bonds.r_bond,
                                                   properties->bonds.s_bond,
                                                   properties->bonds.u_bond,
                                                   properties->bonds.length,
                                                   r_cutoff2);

                            PushStage0(&*xl_iter);
                        }
                        DeleteStage1(&*xl_iter);
                    }
                }
            }
        }
    }
}

void XlinkManagement::WriteState(system_parameters *parameters,
                                 system_properties *properties,
                                 FILE **f_crosslink) {
    if (!write_posit_) {
        return;
    }

    if (f_crosslink == nullptr) {
        WriteStage0(parameters->n_dim);
        WriteStage1();
        WriteStage2();
    } else {
        WriteStageOriginal(parameters,
                           properties,
                           f_crosslink);
    }
}

void XlinkManagement::ReadState(system_parameters *parameters,
                                system_properties *properties,
                                FILE **f_crosslink) {

    if (!write_posit_)
        return;

    else if (f_crosslink == nullptr) {
        ReadStage0(parameters->n_dim);
        ReadStage1();
        ReadStage2();
    } else {
        ReadStageOriginal(parameters,
                          properties,
                          f_crosslink);
    }
}

void XlinkManagement::Restart(system_parameters *parameters,
                              system_properties *properties,
                              const std::string& restart_name) {
    // IF we have anything open, close it
    CloseAll();
    std::string oldname = xlink_filename_;
    xlink_filename_ = restart_name;

    // Load the state
    ReadState(parameters, properties);
    PrintSimulationStep(false);

    // Close it all again and restore the old name
    CloseAll();
    xlink_filename_ = oldname;
}

// Restart based on the frame number, will copy the old information into RESTART files and then replace
void XlinkManagement::Restart(system_parameters *parameters,
                              system_properties *properties,
                              int nframes) {
    // Close everything, this is going to get interesting
    CloseAll();

    // Read through and load the states, note, this will set the file pointers
    // to their properlocations....
    for (int iframe = 0; iframe < nframes; ++iframe) {
        ReadState(parameters, properties);
    }
    // This SHOULD position the file pointers correctly in the files...., so just continue now
}

void XlinkManagement::PrintSimulationStep(bool print_sub) {
    std::cout << "****************\n";
    std::cout << "Printing simulation step: " << properties_->i_current_step << std::endl;
    for (int itype = 0; itype < n_types_; ++itype) {
        std::cout << "Xlinks Type[" << itype << "]\n";
        // Always print the numbers in each category
        // Stage 0
        int nfreecalc = 0;
        for (auto xit =  stage_0_xlinks_[itype].begin();
                  xit != stage_0_xlinks_[itype].end();
                  xit++) {
            if (xit->IsActive()) {
                nfreecalc++;
            }
        }
        std::cout << "  nfree:   " << n_free_[itype] << " calc(" << nfreecalc << "), len(" << stage_0_xlinks_[itype].size() << ")\n";
        if (n_free_[itype] != nfreecalc) {
            std::cout << "Mismatch nfree calc, exiting!\n";
            exit(1);
        }

        // Stage 1
        int nbound1calc = 0;
        int totalstage1len = 0;
        int direction_plus_singly = 0;
        int direction_minus_singly = 0;
        for (int ibond = 0; ibond < n_bonds_; ++ibond) {
            totalstage1len += stage_1_xlinks_[itype][ibond].size();
            for (auto xit =  stage_1_xlinks_[itype][ibond].begin();
                      xit != stage_1_xlinks_[itype][ibond].end();
                      xit++) {
                if (xit->IsActive()) {
                    nbound1calc++;
                    if (!xit->CheckEndPosition(properties_->bonds.length, 0)){
                        std::cout<<" ---- Singly bound tip tracking problems\n";
                    }

                    if (bi_directional_[itype] == 1) {
                        if (xit->direction_state_[xit->head_type_[0]] == 0) {
                            direction_minus_singly++;
                        } else {
                            direction_plus_singly++;
                        }
                    }
                }
            }
        }
        std::cout << "  nbound1: " << (n_bound_1_[itype][0] + n_bound_1_[itype][1]) << ", calc(" << nbound1calc << "), len(" << totalstage1len << ")\n";
        if (bi_directional_[itype] == 1) {
            std::cout << "    plus_directed: " << direction_plus_singly << ", minus_directed: " << direction_minus_singly << std::endl;
        }
        if ((n_bound_1_[itype][0] + n_bound_1_[itype][1]) != nbound1calc) {
            std::cout << "Mismatch nbound1 calc, exiting!\n";
            exit(1);
        }

        // Stage 2
        int nbound2calc = 0;
        int totalstage2len = 0;
        int direction_minus = 0;
        int direction_plus = 0;
        for (int ibond = 0; ibond < n_bonds_; ++ibond) {
            totalstage2len += stage_2_xlinks_[itype][ibond].size();
            for (auto xit =  stage_2_xlinks_[itype][ibond].begin();
                      xit != stage_2_xlinks_[itype][ibond].end();
                      xit++) {
                if (xit->IsActive()) {
                    nbound2calc++;
                    if ( !xit->CheckEndPosition(properties_->bonds.length, 0) ||
                         !xit->CheckEndPosition(properties_->bonds.length, 1) ){
                        std::cout<<" ---- Doubly bound tip tracking problems\n";
                    }

                    // Check bidirectionality
                    if (bi_directional_[itype] == 1) {
                        if (xit->direction_state_[0] == 0) {
                            direction_minus++;
                        } else {
                            direction_plus++;
                        }
                        if (xit->direction_state_[1] == 0) {
                            direction_minus++;
                        } else {
                            direction_plus++;
                        }
                    }
                }
            }
        }
        std::cout << "  nbound2: " << n_bound_2_[itype] << ", calc(" << nbound2calc << "), len(" << totalstage2len << ")\n";
        if (bi_directional_[itype] == 1) {
            std::cout << "    plus_directed: " << direction_plus << ", minus_directed: " << direction_minus << std::endl;
        }
        if (n_bound_2_[itype]!= nbound2calc) {
            std::cout << "Mismatch nbound2 calc, exiting!\n";
            exit(1);
        }

        std::cout << "  FINAL: " << (n_free_[itype] + n_bound_1_[itype][0] + n_bound_1_[itype][1] + n_bound_2_[itype]) << ", calc(" << (nfreecalc + nbound1calc + nbound2calc) << ")\n";
        if((n_free_[itype] + n_bound_1_[itype][0] + n_bound_1_[itype][1] + n_bound_2_[itype]) != (nfreecalc + nbound1calc + nbound2calc)) {
            std::cout << "derp!\n";
            exit(11);
        }

        if (print_sub) {
            PrintStage0Xlinks(itype);
            PrintStage1Xlinks(itype);
            PrintStage2Xlinks(itype);
        }
    }
}

void XlinkManagement::PrintStage0Xlinks(int itype) {
    if (attachment_model_ == 2) {
        std::cout << "Stage 0:\n";
    }
    std::cout << "derp\n";
}

void XlinkManagement::PrintStage1Xlinks(int itype) {

    std::cout << "derp\n";
}

void XlinkManagement::PrintStage2Xlinks(int itype) {
    // Always have this one
    int n_xlink_tot = 0;
    int n_max = 0;
    for (int ibond = 0; ibond < n_bonds_; ++ibond) {
        for (auto xit  = stage_2_xlinks_[itype][ibond].begin();
                  xit != stage_2_xlinks_[itype][ibond].end();
                  ++xit) {
            if (xit->IsActive()) {
                int bond_1 = xit->head_parent_[0];
                int bond_2 = xit->head_parent_[1];
                double crosspos[2] = {0.0};
                crosspos[0] = xit->cross_position_[0];
                crosspos[1] = xit->cross_position_[1];
                std::cout << "bond1: " << bond_1 << ", bond2: " << bond_2 << ", crosspos("
                    << crosspos[0] << ", " << crosspos[1] << ")\n";
                n_xlink_tot++;
            }
                n_max = MAX(n_max, (int)stage_2_xlinks_[itype][ibond].size());
        }
    }
    std::cout << "Total xlink number = "<< n_xlink_tot << std::endl;
    std::cout << "  Max xlink number = "<< n_max << std::endl;
}

void XlinkManagement::ConsistencyCheck() {
    // Check if all the xlinks are consistent with what we expect
    for (int itype = 0; itype < n_types_; ++itype) {
        // Both stages 1 and 2 require looping over bonds
        for (int ibond = 0; ibond < n_bonds_; ++ibond) {
            // Stage 1 xlinks
            for (auto xit =  stage_1_xlinks_[itype][ibond].begin();
                      xit != stage_1_xlinks_[itype][ibond].end();
                      xit++) {
                if (xit->IsActive()) {
                    if (xit->cross_position_[0] > properties_->bonds.length[ibond]) {
                        std::cout << std::setprecision(16);
                        std::cout << "Crosslink[" << itype << "] stage1 off end of bond step: " << properties_->i_current_step << std::endl;
                        std::cout << "  xlink: " << std::distance(xit, stage_1_xlinks_[itype][ibond].begin()) << std::endl;
                        std::cout << "  bond: " << ibond << std::endl;
                        std::cout << "  length: " << properties_->bonds.length[ibond] << ", crosspos: " << xit->cross_position_[0] << std::endl;
                        //exit(1);
                    }
                }
            } // stage 1

            // Stage 2
            for (auto xit  = stage_2_xlinks_[itype][ibond].begin();
                      xit != stage_2_xlinks_[itype][ibond].end();
                      ++xit) {
                if (xit->IsActive()) {
                    int bond_1 = xit->head_parent_[0];
                    int bond_2 = xit->head_parent_[1];
                    if (xit->cross_position_[0] > properties_->bonds.length[bond_1]) {
                        std::cout << std::setprecision(16);
                        std::cout << "Crosslink stage2 head0 off end of bond step: " << properties_->i_current_step << std::endl;
                        std::cout << "  ibond: " << ibond << ", bond1: " << bond_1 << std::endl;
                        std::cout << "  xlink: " << std::distance(xit, stage_2_xlinks_[itype][ibond].begin()) << std::endl;
                        std::cout << "  bond: " << bond_1 << std::endl;
                        std::cout << "  length: " << properties_->bonds.length[bond_1] << ", crosspos: " << xit->cross_position_[0] << std::endl;
                        exit(1);
                    }
                    if (xit->cross_position_[1] > properties_->bonds.length[bond_2]) {
                        std::cout << std::setprecision(16);
                        std::cout << "Crosslink stage2 head1 off end of bond step: " << properties_->i_current_step << std::endl;
                        std::cout << "  ibond: " << ibond << ", bond2: " << bond_2 << std::endl;
                        std::cout << "  xlink: " << std::distance(xit, stage_2_xlinks_[itype][ibond].begin()) << std::endl;
                        std::cout << "  bond: " << bond_2 << std::endl;
                        std::cout << "  length: " << properties_->bonds.length[bond_2] << ", crosspos: " << xit->cross_position_[0] << std::endl;
                        exit(1);
                    }
                }
            } // stage2
        }
    }
}

void XlinkManagement::WriteStageOriginal(system_parameters *parameters,
                                         system_properties *properties,
                                         FILE **f_crosslink) {
    for (int i_type = 0; i_type < n_types_; ++i_type)
        fwrite(&(n_bound_2_[i_type]),
                sizeof(int), 1, f_crosslink[i_type]);

    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
            for (xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                    xlink < stage_2_xlinks_[i_type][i_bond].end();
                    xlink++) {

                if (xlink->IsActive()) {
                    int bond_1 = xlink->head_parent_[0];
                    int bond_2 = xlink->head_parent_[1];

                    /* Note that positions are stored relative to center of bonds
                       in legacy code, but relative to the first bond site in new code,
                       so to keep compatibility with old files the old format is maintained */
                    float cross_position[2];
                    cross_position[0] = (float) (xlink->cross_position_[0] -
                                                 0.5 * properties->bonds.length[bond_1]);
                    cross_position[1] = (float) (xlink->cross_position_[1] -
                                                 0.5 * properties->bonds.length[bond_2]);

                    if (i_bond == bond_1) {
                        fwrite(&bond_1, sizeof(int), 1, f_crosslink[i_type]);
                        fwrite(&bond_2, sizeof(int), 1, f_crosslink[i_type]);
                        fwrite(cross_position, sizeof(float), 2, f_crosslink[i_type]);
                        //std::cout << "bond1: " << bond_1 << ", bond2: " << bond_2 << ", crosspos[0]: " << cross_position[0]
                        //    << ", crosspos[1]: " << cross_position[1] << ", original :(" << xlink->cross_position_[0] << ", "
                        //    << xlink->cross_position_[1] << std::endl;
                    }
                }
            }
        }
    }
}

void XlinkManagement::ReadStageOriginal(system_parameters *parameters,
                                        system_properties *properties,
                                        FILE **f_crosslink) {
    // Read the original crosslinks in
    //std::cout << "DEBUG, Read Stage Original!\n";
    //std::cout << "Found " << n_types_ << " types\n";

    std::vector<int> nbound2(n_types_, 0);

    for (int i_type = 0; i_type < n_types_; ++i_type) {
        n_bound_2_[i_type] = 0;
        int nbound2_loc = 0;
        fread(&nbound2_loc, sizeof(int), 1, f_crosslink[i_type]);
        //std::cout << "nbound2: " << nbound2_loc << std::endl;
        nbound2[i_type] = nbound2_loc;
        //n_bound_2_[i_type] = nbound2;
    }

    // Reset all the bonds
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
            stage_2_xlinks_[i_type][i_bond].clear();
        }
    }

    // Loop again to read in information
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (int xit = 0; xit < nbound2[i_type]; ++xit) {
            int bond1 = 0;
            int bond2 = 0;
            float crosspos[2] = {0.0};

            // Read in the information, then figure out the bond
            fread(&bond1, sizeof(int), 1, f_crosslink[i_type]);
            fread(&bond2, sizeof(int), 1, f_crosslink[i_type]);
            fread(crosspos, sizeof(float), 2, f_crosslink[i_type]);
            //std::cout << "b1: " << bond1 << ", b2: " << bond2 << ", crosspos[0]: " << crosspos[0]
            //    << ", crosspos[1]: " << crosspos[1] << std::endl;
            double lambda = (double) crosspos[0];
            double mu = (double) crosspos[1];
            XlinkEntry new_link(this,
                                 parameters->n_dim,
                                 parameters->n_periodic,
                                 properties->unit_cell.h,
                                 properties->bonds.r_bond, properties->bonds.s_bond,
                                 properties->bonds.u_bond, properties->bonds.length,
                                 i_type, bond1, bond2, lambda, mu,
                                 0); // FIXME: Minus end directed original state
            PushStage2(&new_link);
        }
    }

    // Some quick checks
    //for (int i_type = 0; i_type < n_types_; ++i_type) {
    //    std::cout << "nbound2[" << i_type << "] = " << n_bound_2_[i_type] << std::endl;
    //    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
    //        for (auto xlink = stage_2_xlinks_[i_type][i_bond].begin();
    //                 xlink != stage_2_xlinks_[i_type][i_bond].end();
    //                 ++xlink) {
    //            int bond_1 = xlink->head_parent_[0];
    //            int bond_2 = xlink->head_parent_[1];
    //            float cross_position[2];
    //            cross_position[0] = (float) (xlink->cross_position_[0] -
    //                                         0.5 * properties->bonds.length[bond_1]);
    //            cross_position[1] = (float) (xlink->cross_position_[1] -
    //                                         0.5 * properties->bonds.length[bond_2]);
    //            std::cout << "bond1: " << bond_1 << ", bond2: " << bond_2 << ", crosspos[0]: " << cross_position[0]
    //                      << ", crosspos[1]: " << cross_position[1] << ", original :(" << xlink->cross_position_[0] << ", "
    //                      << xlink->cross_position_[1] << std::endl;
    //        }
    //    }
    //}
}

void XlinkManagement::OpenStage(int stage, std::string opent) {
    std::vector<FILE*> *file_vector_p = NULL;
    if (stage == 0)
        file_vector_p = &f_crosslink_0_;
    else if (stage == 1)
        file_vector_p = &f_crosslink_1_;
    else if (stage == 2)
        file_vector_p = &f_crosslink_2_;
    else
        error_exit("Invalid stage for crosslink output: \"%d\"", stage);

    (*file_vector_p).resize(n_types_);
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        char outfile[256];
        sprintf(outfile,"%s.posit.stage%d.type%d", xlink_filename_.c_str(), stage, i_type);
        //sprintf(outfile,"crosslinks.posit.stage%d.type%d", stage, i_type);
        //(*file_vector_p)[i_type] = gfopen(outfile,"w");
        (*file_vector_p)[i_type] = gfopen(outfile, opent.c_str());
    }
}

// Close all of the open xlink files
void XlinkManagement::CloseAll() {
    for (auto fit = f_crosslink_0_.begin(); fit != f_crosslink_0_.end(); ++fit) {
        fflush(*fit);
        fclose(*fit);
    }
    f_crosslink_0_.clear();
    for (auto fit = f_crosslink_1_.begin(); fit != f_crosslink_1_.end(); ++fit) {
        fflush(*fit);
        fclose(*fit);
    }
    f_crosslink_1_.clear();
    for (auto fit = f_crosslink_2_.begin(); fit != f_crosslink_2_.end(); ++fit) {
        fflush(*fit);
        fclose(*fit);
    }
    f_crosslink_2_.clear();
}

void XlinkManagement::WriteStage0(int n_dim) {
    if (!implicit_0_ && n_types_ > 0) {
        if (f_crosslink_0_.size() == 0) {
            OpenStage(0, "w");
        }
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            //if (std::isnan(n_free_[i_type])) {
            //    std::cout << "Encountered NaN in writing nfree Xlink[" << i_type << "] Stage0\n";
            //    exit(1);
            //}
            fwrite(&(n_free_[i_type]), sizeof(int), 1, f_crosslink_0_[i_type]);
            //std::cout << "nfree[" << i_type << "] " << n_free_[i_type] << std::endl;
            //int ntotal = 0;
            for (xlink_list::iterator xlink = stage_0_xlinks_[i_type].begin();
                 xlink < stage_0_xlinks_[i_type].end();
                 ++xlink) {
                if (xlink->IsActive()) {
                    // Do a quick check to make sure we aren't having problems with NaN cropping up
                    for (int i = 0; i < n_dim; ++i) {
                        if (std::isnan(xlink->r_cross_[i])) {
                            std::cout << "Encountered NaN in Xlink[" << i_type << "] Stage0, idx[" << std::distance(stage_0_xlinks_[i_type].begin(), xlink) << "]\n";
                            exit(1);
                        }
                    }
                    float pos[3];
                    std::copy(xlink->r_cross_, xlink->r_cross_+3, pos);
                    fwrite(pos, sizeof(float), n_dim, f_crosslink_0_[i_type]);
                    //ntotal++;
                }
            }
            //std::cout << "nwritten: " << ntotal << std::endl;
        }
    }
}

void XlinkManagement::ReadStage0(int ndim) {
    if (!implicit_0_ && n_types_ > 0) {
        if (f_crosslink_0_.size() == 0) {
            OpenStage(0, "r+");
        }

        for (int i_type = 0; i_type < n_types_; ++i_type) {
            n_free_[i_type] = 0;
            int nfree = 0;
            fread(&nfree, sizeof(int), 1, f_crosslink_0_[i_type]);

            //if (nfree != nfree) {
            //    std::cout << "Encountered NaN in reading nfree Xlink[" << i_type << "] Stage0\n";
            //    exit(1);
            //}

            stage_0_xlinks_[i_type].clear();
            for (int xit = 0; xit < nfree; ++xit) {
                XlinkEntry new_link(this);
                new_link.SetActive();
                new_link.type_ = i_type;
                float pos[3] = {0.0};
                fread(pos, sizeof(float), ndim, f_crosslink_0_[i_type]);
                for (int i = 0; i < ndim; ++i) {
                    //if (std::isnan(pos[i])) {
                    //    std::cout << "Encountered NaN in reading pos Xlink[" << i_type << "] Stage0, idx[" << xit << "]\n";
                    //    exit(1);
                    //}
                    new_link.r_cross_[i] = pos[i];
                }
                PushStage0(&new_link);
            }
        }
    }
}

void XlinkManagement::WriteStage1() {
    if (n_types_ > 0) {
        if (f_crosslink_1_.size() == 0) {
            OpenStage(1, "w");
        }
        for (int i_type = 0; i_type < n_types_; ++i_type) {
            fwrite(&(n_bound_1_[i_type][0]), sizeof(int), 1, f_crosslink_1_[i_type]);
            fwrite(&(n_bound_1_[i_type][1]), sizeof(int), 1, f_crosslink_1_[i_type]);
            //int totalcount = 0;
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                for (xlink_list::iterator xlink = stage_1_xlinks_[i_type][i_bond].begin();
                     xlink < stage_1_xlinks_[i_type][i_bond].end();
                     ++xlink) {
                    if (xlink->IsActive()) {
                        int headtype = xlink->head_type_[0];
                        int parent = xlink->head_parent_[0];
                        float pos = xlink->cross_position_[0];
                        // Small test for NaN
                        if (std::isnan(pos)) {
                            std::cout << "Encountered NaN in Xlink[" << i_type << "] Stage1, bond: " << i_bond << ", idx: " << std::distance(stage_1_xlinks_[i_type][i_bond].begin(), xlink) << std::endl;
                            exit(1);
                        }
                        fwrite(&headtype, sizeof(int), 1, f_crosslink_1_[i_type]);
                        fwrite(&parent, sizeof(int), 1, f_crosslink_1_[i_type]);
                        fwrite(&pos,  sizeof(float), 1, f_crosslink_1_[i_type]);
                        //totalcount++;
                        //std::cout << "ibond: " << i_bond << ", parent: " << parent << std::endl;
                    }
                }
            }
            //std::cout << "totalcount: " << totalcount << std::endl;
            //std::cout << "nbound1[0]: " << n_bound_1_[i_type][0] << std::endl;
            //std::cout << "nbound1[1]: " << n_bound_1_[i_type][1] << std::endl;
        }
    }
}

void XlinkManagement::ReadStage1() {
    if (n_types_ > 0) {
        if (f_crosslink_1_.size() == 0) {
            OpenStage(1, "r+");
        }

        for (int i_type = 0; i_type < n_types_; ++i_type) {
            // Clear the bonds of their stage 1 xlinks
            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                n_bound_1_[i_type][0] = 0;
                n_bound_1_[i_type][1] = 0;
                stage_1_xlinks_[i_type][i_bond].clear();
            }

            //std::cout << "nbound1 setup: " << n_bound_1_[i_type][0] << ", " << n_bound_1_[i_type][1] << std::endl;

            int nbound10 = 0;
            int nbound11 = 0;
            fread(&nbound10, sizeof(int), 1, f_crosslink_1_[i_type]);
            fread(&nbound11, sizeof(int), 1, f_crosslink_1_[i_type]);

            // Figure out which bond we're on, for each one
            int totalcount = nbound10 + nbound11;
            for (int xit = 0; xit < totalcount; ++xit) {
                int headtype = 0;
                int parent = 0;
                float pos = 0.0;
                fread(&headtype, sizeof(int), 1, f_crosslink_1_[i_type]);
                fread(&parent, sizeof(int), 1, f_crosslink_1_[i_type]);
                fread(&pos,  sizeof(float), 1, f_crosslink_1_[i_type]);
                XlinkEntry new_link(this);
                new_link.SetActive();
                new_link.type_ = i_type;
                new_link.head_type_[0] = headtype;
                new_link.head_parent_[0] = parent;
                new_link.cross_position_[0] = pos;

                PushStage1(&new_link);
            }
        }
    }
}

void XlinkManagement::WriteStage2() {
    if (n_types_ > 0) {
        if (f_crosslink_2_.size() == 0) {
            OpenStage(2, "w");
        }

        for (int i_type = 0; i_type < n_types_; ++i_type) {
            fwrite(&(n_bound_2_[i_type]),
                   sizeof(int), 1, f_crosslink_2_[i_type]);

            for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
                for(xlink_list::iterator xlink = stage_2_xlinks_[i_type][i_bond].begin();
                    xlink < stage_2_xlinks_[i_type][i_bond].end();
                    xlink++) {

                    if (xlink->IsActive()) {
                        int head_1 = xlink->head_type_[0];
                        int head_2 = xlink->head_type_[1];

                        int bond_1 = xlink->head_parent_[0];
                        int bond_2 = xlink->head_parent_[1];

                        float cross_position[2];
                        cross_position[0] = (float) (xlink->cross_position_[0]);
                        cross_position[1] = (float) (xlink->cross_position_[1]);
                        if (std::isnan(cross_position[0])) {
                            std::cout << "Encountered NaN in Xlink{0}[" << i_type << "] Stage1, bond: " << i_bond << ", idx: " << std::distance(stage_2_xlinks_[i_type][i_bond].begin(), xlink) << std::endl;
                            exit(1);
                        }
                        if (std::isnan(cross_position[1])) {
                            std::cout << "Encountered NaN in Xlink{1}[" << i_type << "] Stage1, bond: " << i_bond << ", idx: " << std::distance(stage_2_xlinks_[i_type][i_bond].begin(), xlink) << std::endl;
                            exit(1);
                        }

                        if (i_bond == bond_1) {
                            //std::cout << "Writing nbound2: " << head_1 << ", " << head_2 << std::endl;
                            fwrite(&head_1,        sizeof(int),   1, f_crosslink_2_[i_type]);
                            fwrite(&head_2,        sizeof(int),   1, f_crosslink_2_[i_type]);
                            fwrite(&bond_1,        sizeof(int),   1, f_crosslink_2_[i_type]);
                            fwrite(&bond_2,        sizeof(int),   1, f_crosslink_2_[i_type]);
                            fwrite(cross_position, sizeof(float), 2, f_crosslink_2_[i_type]);
                        }
                    }
                }
            }
        }
    }
}

void XlinkManagement::ReadStage2() {
    if (n_types_ > 0) {
        if (f_crosslink_2_.size() == 0) {
            OpenStage(2, "r+");
        }

        for (int i_type = 0; i_type < n_types_; ++i_type) {
            // Clear the Stage2 xlinks
            for (int ibond = 0; ibond < n_bonds_; ++ibond) {
                stage_2_xlinks_[i_type][ibond].clear();
                n_bound_2_[i_type] = 0;
            }

            int nbound2 = 0;
            fread(&nbound2, sizeof(int), 1, f_crosslink_2_[i_type]);

            for (int xit = 0; xit < nbound2; ++xit) {
                int head_1 = 0;
                int head_2 = 0;
                int bond_1 = 0;
                int bond_2 = 0;

                float cross_position[2] = {0.0};
                fread(&head_1, sizeof(int), 1, f_crosslink_2_[i_type]);
                fread(&head_2, sizeof(int), 1, f_crosslink_2_[i_type]);
                fread(&bond_1, sizeof(int), 1, f_crosslink_2_[i_type]);
                fread(&bond_2, sizeof(int), 1, f_crosslink_2_[i_type]);

                fread(cross_position, sizeof(float), 2, f_crosslink_2_[i_type]);

                XlinkEntry new_link(this);
                new_link.SetActive();
                new_link.type_ = i_type;
                new_link.head_type_[0] = head_1;
                new_link.head_type_[1] = head_2;
                new_link.head_parent_[0] = bond_1;
                new_link.head_parent_[1] = bond_2;
                new_link.cross_position_[0] = cross_position[0];
                new_link.cross_position_[1] = cross_position[1];

                //Calculate force and more importantly energy for new xlink
                double f[3];
                new_link.CalcForce(parameters_->n_dim,
                                   parameters_->n_periodic,
                                   properties_->unit_cell.h,
                                   properties_->bonds.r_bond,
                                   properties_->bonds.s_bond,
                                   properties_->bonds.u_bond,
                                   properties_->bonds.length,
                                   f);

                //std::cout << "Reading bound2: " << head_1 << ", " << head_2 << std::endl;

                PushStage2(&new_link);
            }
        }
    }
}

// Get the forces on all of the crosslinker species
// This is done as a (ipole x ntypes x dimension)
std::vector<std::vector<std::vector<double>>> XlinkManagement::GenerateForces() {
    // Initialize the system
    std::vector<std::vector<std::vector<double>>> forces;
    for (int ipole = 0; ipole < properties_->anchors.n_anchors; ++ipole) {
        std::vector<std::vector<double>> polefake;
        for (int itype = 0; itype < n_types_; ++itype) {
            std::vector<double> fake_force_insert(3, 0.0);
            polefake.push_back(fake_force_insert);
        }
        forces.push_back(polefake);
    }

    // Loop over the types
    int nmts_per_spb = n_bonds_ / properties_->anchors.n_anchors;
    for(int itype = 0; itype < n_types_; ++itype) {
        // Loop over the bonds
        for (int ibond = 0; ibond < n_bonds_; ++ibond) {
            // Loop over the stage2 xlinks
            for (auto xit  = stage_2_xlinks_[itype][ibond].begin();
                      xit != stage_2_xlinks_[itype][ibond].end();
                      xit++) {
                if(xit->IsActive()) {
                    // We know this is active, but we need to make sure of the pole order...
                    int bond_1 = xit->head_parent_[0];
                    int bond_2 = xit->head_parent_[1];
                    
                    // Set the force to be the correct directionality like what we see in the crosslink
                    // calculation
                    double fbond1[3] = {0.0};
                    double fbond2[3] = {0.0};
                    for (int i = 0; i < n_dim_; ++i) {
                        fbond1[i] +=  xit->flink_[i]; 
                        fbond2[i] += -xit->flink_[i];
                    }

                    // Uniquely determine the pole;
                    int pole_1 = bond_1 / nmts_per_spb;
                    int pole_2 = bond_2 / nmts_per_spb;

                    // Load the information onto the forces
                    for (int i = 0; i < n_dim_; ++i) {
                        forces[pole_1][itype][i] += fbond1[i];
                        forces[pole_2][itype][i] += fbond2[i];
                    }
                }
            }
        }
    }
    return forces;
}

double XlinkManagement::GetOneStageProbability(int i_type) {
    double prob = 0.0;
    for (int i_bond = 0; i_bond < n_bonds_; ++i_bond) {
        for (nl_list::iterator neighb = one_stage_probability_[i_type][i_bond].begin();
             neighb < one_stage_probability_[i_type][i_bond].end();
             ++neighb) {
            prob += neighb->value;
        }
    }
    return prob;
}

void XlinkManagement::CenterXlinks(){
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        double r_cutoff = r_cutoff_0_1_[i_type];
        double r_cutoff2 = SQR(SKIN + r_cutoff);
        for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
             xl_iter < stage_0_xlinks_[i_type].end();
             ++xl_iter) {
            for (int i=0; i<n_dim_; ++i){
                xl_iter->r_cross_[i] = 0.0;
                xl_iter->UpdateNeighbs(parameters_->n_dim,
                                       parameters_->n_periodic,
                                       properties_->bonds.n_bonds,
                                       properties_->unit_cell.h,
                                       properties_->bonds.r_bond,
                                       properties_->bonds.s_bond,
                                       properties_->bonds.u_bond,
                                       properties_->bonds.length,
                                       r_cutoff2);
            }
        }
    }
    return;
}

void XlinkManagement::PrintXlinks(){
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
             xl_iter < stage_0_xlinks_[i_type].end();
             ++xl_iter) {
            std::cout << "  xlink: [ ";
            double rmag2=0.0;
            for (int i=0; i<n_dim_; ++i){
                std::cout<<xl_iter->r_cross_[i]<<", ";
                rmag2 += SQR(xl_iter->r_cross_[i]);
            }

            std::cout << sqrt(rmag2) <<" ]"<<std::endl;
        }
    }
}

void XlinkManagement::FillFreeXlinkSqrdDisplacementVec(std::vector<double> &dis2arr){
    int i_xl = 0;
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        for (xlink_list::iterator xl_iter = stage_0_xlinks_[i_type].begin();
             xl_iter < stage_0_xlinks_[i_type].end();
             ++xl_iter) {
            for (int i=0; i<n_dim_; ++i){
                dis2arr[i_xl] += SQR(xl_iter->r_cross_[i]);
            }
            i_xl++;
        }
    }
}

struct diffusion_params {
    double diff;
    double delta_max;
    double lambda;
    double rmin;
};

double integrand_0_1(double delta_t, void *params) {
    struct diffusion_params* myparams = (struct diffusion_params *) params;
    double rmin = myparams->rmin;
    double difft = myparams->diff * delta_t;
    double lambda = myparams->lambda;

    return exp(-0.25*SQR(rmin)/difft) * erf(0.5 * lambda / sqrt(difft)) / (8*M_PI*difft);
}

double prob_0_1(std::vector<double> &x, void *params) {
    gsl_integration_workspace * w
        = gsl_integration_workspace_alloc (10000);

    struct diffusion_params *inparams = (struct diffusion_params*) params;
    struct diffusion_params gslparams;
    gslparams.diff = inparams->diff;
    gslparams.delta_max = inparams->delta_max;
    gslparams.lambda = x[0];
    gslparams.rmin = x[1];
    if (x[1] == 0.0)
        x[1] = 1e-20;

    gsl_function F;
    F.function = &integrand_0_1;
    F.params = &gslparams;

    double result, error;
    if (x[1] > 0 && x[0] > 0)
        gsl_integration_qags (&F, 1E-200, gslparams.delta_max, 0, 1e-9, 10000,
                              w, &result, &error);
    else
        result = 0;

    gsl_integration_workspace_free (w);

    return result;
}

void XlinkManagement::BuildTables() {
    for (int i_type = 0; i_type < n_types_; ++i_type) {
        if (r_equil_[i_type] != 0.0 && attachment_model_ == 2) {

            if (force_dependent_) {
                // Force dependent version
                std::vector<double> x[2];
                double bin_size = 0.05;
                double alpha = k_stretch_[i_type] / 2.;
                //const double eps = 1e-5;
                const double eps = 1e-5;
                double a_cutoff = exp(alpha*xc_[i_type]*xc_[i_type])/sqrt(alpha) * my_erfinv(1 - 4.0*sqrt(alpha/M_PI)*eps) +
                    r_equil_[i_type] + xc_[i_type];
                double y_cutoff = r_cutoff_1_2_[i_type];

                //std::cout << "alpha: " << alpha << std::endl;
                //std::cout << "a_cutoff: " << a_cutoff << std::endl;
                //std::cout << "y_cutoff: " << y_cutoff << std::endl;

                xlink_params params;
                params.alpha = alpha;
                params.xc = xc_[i_type];
                params.r0 = r_equil_[i_type];

                for (double a = 0.0; a <= a_cutoff; a += bin_size)
                    x[0].push_back(a);

                for (double y0 = 0.0; y0 <= y_cutoff; y0 += bin_size)
                    x[1].push_back(y0);

                n_exp_lookup_[i_type].Init(2, x, &prob_1_2_fdep, &params);
            } else {
                // Energy dependent version
                std::vector<double> x[2];
                double bin_size = 0.05;
                double alpha = k_stretch_[i_type] * (1 - barrier_weight_[i_type]) / 2;
                const double eps = 1e-5;
                // Maximum distance you integrate along neighbor rod
                double a_cutoff = 1/sqrt(alpha) * my_erfinv(1 - 4.0*sqrt(alpha/M_PI)*eps) +
                    r_equil_[i_type];
                double y_cutoff = r_cutoff_1_2_[i_type];

                //std::cout << "alpha: " << alpha << std::endl;
                //std::cout << "a_cutoff: " << a_cutoff << std::endl;
                //std::cout << "y_cutoff: " << y_cutoff << std::endl;
                //std::cout << "r_equil: " << r_equil_[i_type]<< std::endl;

                xlink_params params;
                params.alpha = alpha;
                params.r0 = r_equil_[i_type];

                for (double a = 0.0; a <= a_cutoff; a += bin_size)
                    x[0].push_back(a);

                for (double y0 = 0.0; y0 <= y_cutoff; y0 += bin_size)
                    x[1].push_back(y0);

                n_exp_lookup_[i_type].Init(2, x, &prob_1_2, &params);
                //double temp[2] = {a_cutoff, 3};
                //std::cout << "lookup_value: " << 2*n_exp_lookup_[i_type].Lookup(temp)<< std::endl;
                //std::cin.get();
            }
        }
    }
}

// Update the arrest states
void XlinkManagement::UpdateArrest(system_parameters *parameters,
                                   system_properties *properties) {
    for (int itype = 0; itype < n_types_; ++itype) {
        // Check to see if we are arresting or not
        if (arrest_[itype]) {
            if (properties->time > arrest_time_[itype]) {
                // If we have gone past the arrest time, restore the original values
                std::cout << "Crosslink type: " << itype << " turning on!\n";
                std::cout << "  previous velocity = [" << velocity_[itype][0] << ", " << velocity_[itype][1] << "]\n";
                velocity_[itype][0] = arrest_vel_original_[itype][0];
                velocity_[itype][1] = arrest_vel_original_[itype][1];
                std::cout << "  current velocity = [" << velocity_[itype][0] << ", " << velocity_[itype][1] << "]\n";
                arrest_[itype] = false;
            } else {
                velocity_[itype][0] = arrest_vel_[itype][0];
                velocity_[itype][1] = arrest_vel_[itype][1];
            }
        }
    }
}

#undef SKIN
