#ifndef _XLINK_MANAGEMENT_H
#define _XLINK_MANAGEMENT_H

#include <vector>
#include <yaml-cpp/yaml.h>

class XlinkEntry;

#ifndef _PARAMETERS_H
typedef struct system_parameters system_parameters;
#endif
#ifndef _SYSTEM_PROPERTIES_H
typedef struct system_properties system_properties;
#endif

#include "neighbors.h"
#include <gsl/gsl_rng.h>

typedef std::vector<XlinkEntry> xlink_list;
#ifndef _LOOKUP_TABLE_H
class LookupTable;
#endif

class XlinkManagement {
 private:
    int n_bonds_;
    int n_dim_;

    /* The <crosslink_properties> structure contains attributes of crosslinks. */
 public:
    system_parameters *parameters_;
    system_properties *properties_;
    #ifdef ENABLE_OPENMP
    omp_lock_t *steplock_;
    #endif
    int attachment_model_;       /* Attachment model, 0: None, 1: one-stage, 2: two-stage */
    bool implicit_0_;       /* Background concentration flag. If set, use math
                              for uniform concentration */
    bool force_dependent_ = false;
    bool write_posit_ = false;
    int n_types_,                /* total number of species of crosslinks */
        **n_on_bond_2_;                /* number of links of a type attached to
                                          a given bond */
    int *n_free_;                      /* Total free links */
    int **n_bound_1_;                  /* Total number of heads attached to any
                                          filament in stage 1 for each
                                          head_type */
    int *n_bound_2_;                  /* Total number of heads attached to any
                                         filament in stage 1 for each head_type */
    int *n_tot_;                /* total number of xlinks available to the
                                   system for each type */

    int *stall_type_;               /* how crosslinks stall when moving, 0: no
                                       stall. 1: parallel component of force
                                       stalls, 2: absolute magnitude of force
                                       stalls */
    int **end_pause_;           /* Behavior of xlink head when reaching end of MT
                                0: Fall off
                                1: Pause at end
                               */
    int *bi_directional_;       // Bi directionality of this motor, 0 for not anything (traditional)
                                // 1 for bidirectional stochastic
    double *r_cutoff_0_1_;
    double *r_cutoff_1_2_;

    LookupTable *n_exp_lookup_;
    
    double *n_exp_,                   /* Expected number by type */
        **n_exp_bond_,
        *n_exp_0_1_;

    double **r_init_;

    // Arresting various quantities (termperature sensitive perhaps) of quantities
    bool *arrest_;                   // If we've gone past the time that we were arrested 
    double **arrest_vel_;            // Arrest the velocity to this value
    double **arrest_vel_original_;   // Original before arrested velocity of the crosslink
    double *arrest_time_;           // How long to wait in the arresting of various parameters
    
    double **velocity_,           /* velocity of crosslink heads */
        **velocity_p_scale_,           /* scale factor for velocity of polar aligned pairs */
        **velocity_ap_scale_,          /* scale factor for velocity of anti-polar pairs */
        **velocity_switch_costheta_,    /* cos of angle between filaments where
                                           heads switch between polar and antipolar velocities */
        **svelocity_plus_,               // Plus end directed stochastic velocity
        **svelocity_minus_,              // Minus end directed stochastic velocity
        **sfreq_plus_,                   // Transition rate from minus to plus directed motion
        **sfreq_minus_,                  // Transition rate from plus to minus directed motion
        **sfreq_plus_p_,                   // Transition rate from minus to plus directed motion parallel
        **sfreq_minus_p_,                  // Transition rate from plus to minus directed motion parallel
        **sfreq_plus_ap_,                   // Transition rate from minus to plus directed motion antiparallel
        **sfreq_minus_ap_,                  // Transition rate from plus to minus directed motion antiparallel
        **diffusion_bound_,           /* diffusion constant of xlink heads */
        **diffusion_bound_2_,           /* diffusion constant of xlink heads */
        *diffusion_free_,           /* diffusion constant of xlink complex */
        *polar_affinity_,            /* Polar affinity for antiparallel binding (1.0 is symmetric) */
        *pa_off_,                    /* Polar affinity contribution to the off rate */
        *barrier_weight_,             /* Scaling for force dependent detachment. 
                                        0.0: No force dependence
                                        1.0: Pure force dependence */
        *xc_,                        // Characteristic length scale for problem, if force dependent
        *on_rate_,                   /*  */
        **on_rate_1_,                   /*  */
        **on_rate_2_,                   /*  */
        *eps_eff_,                   /* Effective Concentration */
        **eps_eff_1_,                   /* Effective Concentration */
        **eps_eff_2_,                   /* Effective Concentration */
        **f_stall_,                   /* Stall force. v = v0 * (1 - f/f_stall) */
        *r_equil_,                   /* equilibrium length of crosslinks. */
        *k_stretch_,                 /* spring constant of crosslinks. */
        *f_stabilize_vg_,                 /* Scaling of dynamic instability
                                             grow velocity when crosslinks near
                                             ends. */
        *f_stabilize_vs_,                 /* Scaling of dynamic instability shrinking velocity */
        *f_stabilize_fc_,                 /* Scaling of dynamic instability catastrophe frequency */
        *f_stabilize_fr_,                 /* Scaling of dynamic instability rescue frequency */
        *l_stabilize_;                 /* Proximity to + end to stabilize
                                          dynamic instability (see
                                          f_stabilize_). */
    float **color_;                    /* 4d color vector for each xlink species (RGBa) */

    std::vector<XlinkEntry> *stage_0_xlinks_; /* N_types array of
                                                  dynamic arrays of stage 0
                                                  motors */
    std::vector<XlinkEntry> **stage_1_xlinks_; /* N_types x N_bonds array of
                                                 dynamic arrays of stage 1
                                                 motors */
    std::vector<XlinkEntry> **stage_2_xlinks_; /* N_types x N_bonds array of
                                                  dynamic arrays of stage 2
                                                  motors */
    std::vector<nl_entry> **one_stage_probability_; /* N_types x N_bonds array
                                                       of dynamic arrays of
                                                       probabilities to insert a
                                                       one-stage crosslink */

    std::vector<FILE*> f_crosslink_0_;
    std::vector<FILE*> f_crosslink_1_;
    std::vector<FILE*> f_crosslink_2_;


    std::string xlink_filename_;
    YAML::Node node_;
    
    XlinkManagement();
    XlinkManagement(system_parameters *parameters,
                    system_properties *properties,
                    const char *crosslink_file,
                    const char *crosslink_config);
    ~XlinkManagement();
    
    void Init(system_parameters *parameters,
              system_properties *properties,
              const char *crosslink_file,
              const char *crosslink_config);

    void Init(system_parameters *parameters,
              system_properties *properties,
              YAML::Node *pnode);

    void Init(system_parameters *parameters,
              system_properties *properties,
              int n_types, int attachment_model,
              FILE **f_crosslink);

    //void ParseParams(const char *crosslink_file);
    void ParseParams();

    void ReadConfig(system_parameters *parameters,
                    system_properties *properties,
                    FILE **crosslink_config);

    void RemoveLink(int stage, int bond, int index);

    void RemoveLink(int stage, XlinkEntry *xlink);

    void InsertLink(int stage, XlinkEntry *xlink);

    void PrepKMC(system_parameters * parameters, system_properties *properties);

    void Update_0_2_Probability(int n_dim, int n_periodic, double **h,
                                double r_min_mag2,
                                int bond_1,
                                double *r_1, double *s_1, double *u_1, double length_1,
                                double *r_2, double *s_2, double *u_2, double length_2,
                                double *probability);

    void Update_0_1_Probability(system_parameters *parameters,
                                system_properties *properties);

    void Update_1_2_Probability(system_parameters *parameters,
                                system_properties *properties);

    void OneStageKMC(system_parameters *parameters,
                     system_properties *properties);

    void SelectBonds(nl_list *neighbs,
                     int i_type,
                     int *bond_1, int *bond_2,
                     gsl_rng *r);

    void Insert_1_2(system_parameters *parameters,
                    system_properties *properties,
                    int type);

    void Clear();
    
    void ClearNeighbors();

    void TwoStageKMC(system_parameters *parameters,
                     system_properties *properties);

    void UpdateArrest(system_parameters *parameters,
                      system_properties *properties);

    void StepKMC(system_parameters *parameters,
                 system_properties *properties);

    void KMC_0_1(system_parameters *parameters,
                 system_properties *properties);

    void KMC_0_1_implicit(system_parameters *parameters,
                          system_properties *properties);

    void KMC_0_1_explicit(system_parameters *parameters,
                          system_properties *properties);

    void KMC_1_2(system_parameters *parameters,
                 system_properties *properties);

    void KMC_1_0(system_parameters *parameters,
                 system_properties *properties);

    void KMC_2_1(system_parameters *parameters,
                 system_properties *properties);
    void KMC_2_1Edep(system_parameters *parameters,
                     system_properties *properties);
    void KMC_2_1Fdep(system_parameters *parameters,
                     system_properties *properties);

    void PushStage0(XlinkEntry *xl);

    void DeleteStage0(XlinkEntry *xl);

    void PushStage1(XlinkEntry *xl);

    void DeleteStage1(XlinkEntry *xl);

    void PushStage2(XlinkEntry *xlink);

    void DeleteStage2(XlinkEntry *xl);

    void PushOneStageNeighb(int n_dim, int bond_1,
                            nl_entry p, 
                            double **u_bond,
                            double *length);

    // Calculate the contribution of polar affinity to kon
    inline double PolarAffinityOn(int n_dim, int i_type, double *u_1, double *u_2) {
        if (polar_affinity_[i_type] != 1.0 && polar_affinity_[i_type]*pa_off_[i_type] != 1.0) {
            double ui_dot_uj = u_1[0] * u_2[0] + u_1[1] * u_2[1] +
                ((n_dim == 3) ? u_1[2] * u_2[2] : 0.0);
            return (ui_dot_uj < 0) ? 1.0 : polar_affinity_[i_type]*pa_off_[i_type];
        }
        return 1.0;
    }

    // Calculate the contribution of polar affinity to koff
    inline double PolarAffinityOff(int n_dim, int i_type, double *u_1, double *u_2) {
        if (polar_affinity_[i_type] != 1.0 && pa_off_[i_type] != 1.0) {
            double ui_dot_uj = u_1[0] * u_2[0] + u_1[1] * u_2[1] +
                ((n_dim == 3) ? u_1[2] * u_2[2] : 0.0);
            return (ui_dot_uj < 0) ? 1.0 : pa_off_[i_type];
        }
        return 1.0;
    }

    double GetOneStageProbability(int i_type);

    void WriteState(system_parameters *parameters,
                    system_properties *properties,
                    FILE **f_crosslink = nullptr);
    void ReadState(system_parameters *parameters,
                   system_properties *properties,
                   FILE **f_crosslink = nullptr);
    void Restart(system_parameters *parameters,
                 system_properties *properties,
                 const std::string& restart_name);
    void Restart(system_parameters *parameters,
                 system_properties *properties,
                 int nframes);
    void CloseAll();
    std::vector<std::vector<std::vector<double>>> GenerateForces();

    void PrintSimulationStep(bool print_sub=false);
    void PrintStage0Xlinks(int itype);
    void PrintStage1Xlinks(int itype);
    void PrintStage2Xlinks(int itype);
    void ConsistencyCheck();

    //Testing functions
    void CenterXlinks();
    void PrintXlinks();
    void FillFreeXlinkSqrdDisplacementVec(std::vector<double> &dis2arr);

 private:
    void Allocate();

    void ParseShared(YAML::Node &node);

    void ParseOneStage(YAML::Node &node);

    void ParseTwoStage(YAML::Node &node);

    void Push(XlinkEntry *link, xlink_list *list);

    void CalcCutoff(double *length, double length_max);

    void BuildTables();

    void InitPositions(system_parameters *parameters,
                       system_properties *properties);

    void CheckStage0Neighbs(system_parameters *parameters,
                              system_properties *properties);

    void UpdateStage0Neighbs(system_parameters *parameters,
                             system_properties *properties);

    void OpenStage(int stage, std::string opent);

    void WriteStage0(int n_dim);
    void WriteStage1();
    void WriteStage2();

    void ReadStage0(int ndim);
    void ReadStage1();
    void ReadStage2();

    void WriteStageOriginal(system_parameters *parameters,
                            system_properties *properties,
                            FILE **f_crosslink);
    void ReadStageOriginal(system_parameters *parameters,
                           system_properties *properties,
                           FILE **f_crosslink);
};

#endif
