% Source code for Jaiman & Thattai, Figure 2B
% Effect of compartment residence time on the entropy of glycan profiles

% List of constants and variables
%
% N: number of distinct oligomer types
%
% Amat: NxN transition rate matrix of the continuous-time Markov process
% Amat(i,j): transition rate constant to oligomer i from oligomer j for j≠i
% Amat(i,i) = -sum(Amat(i,j)) over j≠i, so columns of Amat sum to zero
%
% tvec: 1xM vector of M residence times at which the state is evaluated
% xini: Nx1 vector of the initial probability distribution over N oligomers
% xmat: NxM matrix of Nx1 probability distributions for M residence times
%
% svec: 1xM vector of Shannon entropies for M residence times

% Initialize
clear;

% Define case
% R=1: cystic fibrosis mucin
% R=2: human chorionic gonadotropin
% R=3: horse chorionic gonadotropin
R=1;

% Define Amat and xini for each case, corresponding to 
% reactions in final compartments of Figure 1B

switch(R)
    
    case(1)
        
        % R=1: cystic fibrosis mucin
        N=9;
        
        Amat=zeros(N);
        Amat(2,1)=1;
        Amat(3,1)=1;
        Amat(4,2:3)=1;
        Amat(6,5)=1;
        Amat(7,5)=1;
        Amat(9,8)=1;
        
        xini=zeros(N,1); xini(1)=1/2; xini(5)=1/4; xini(8)=1/4;
        
    case(2)
        
        % R=2: human chorionic gonadotropin
        N=10;
        
        Amat=zeros(N);
        Amat(2,1)=1;
        Amat(3,1)=1;
        Amat(4,2:3)=1;
        Amat(6,5)=1;
        Amat(8,7)=1;
        Amat(9,7)=1;
        Amat(10,8:9)=1;
        
        xini=zeros(N,1); xini(1)=1/2; xini(5)=1/4; xini(7)=1/4;
        
    case(3)
        
        % R=3: horse chorionic gonadotropin
        N=61; % cut off at some arbitrary oligomer size
        
        Amat=zeros(N);
        Jmax=floor((N-4)/3+1); % number of tandem repeats
        zmat=[1 0 0 0; 0 1 0 0; 0 1 0 0];    
        for j=1:15      
            Amat(3*(j-1)+(2:4),3*(j-1)+(1:4))=zmat; 
        end
        
        xini=zeros(N,1); xini(1)=1;
end

% Enforce condition that columns of Amat sum to zero
Amat=Amat-diag(sum(Amat));

% Evaluate NxM matrix of Nx1 probability distributions at M residence times

tvec=0:0.01:5;
xmat=zeros(N,length(tvec));
for i=1:length(tvec)
    
    % Find state at time t by matrix exponentiation
    t=tvec(i);
    xmat(:,i)=expm(t*Amat)*xini;
    
end

% Calculate Shannon entropy for each possible residence time
svec=sum(-xmat.*log2(xmat+eps));

% Plot results

plot(tvec,svec,'r');
xlabel('Compartment residence time (T)');
ylabel('Glycan entropy (bits)');