function [yield,time_needed,CompStat] = SelfAssembly_MeanField(alpha,mu,N,L,S,critical_size,delta,MakeMovie,MaxYield,MaxSimulTime)      
% function integrates the mean field dynamics (deterministc simulation) of
% the self-assembly process of the ring model with the parameters described below. For the
% simulation the program exploits the equivalence between homogeneous and heterogeneous systems that holds in the deterministic limit (see paper) and therefore simulates always the equivalent homogeneous system which is computationally easier  
% makes a movie of the evoution of the complex size distribution if
% desired. To create the movie make sure that template_Wave.fig is in the
% same folder, otherwise use the simple movie makeup below by commenting it in.
% as a test run to get started you may call   
% SelfAssembly_MeanField(1e-1,1,10000,80,20,1,0.,1,1.0,60)
% in the matlab console

% OUTPUT: 
% yield: the final determinsitc yield
% time_needed: time to reach the final state
% CompStat: vector that contains the final distribution of the sizes of complexes

% INPUT:
% alpha: activation rate, 
% mu: dimerization rate; 
% N: particle copy number,
% L: size of structure, 
% S: number of species (S is relevant only for the assembly time but not for yield in this deterministic setup)
% critical_size: critical size such that complexes of size <= critical size can decay into monomers, delta: decay rate for subcritical complexes,
% MakeMovie = 0,1 indicates if a movie of the time evolution of the complex size distribution is made, 
% MaxYield (optional parameter, default 1.0): stop the simulation if yield exceeds MaxYield. 
% MaxSimulTime (optional parameter, default 1e20): maximal simulation time  

% further parameters: 
rho = 1.0;    % reaction rate 
DECAY_MODE = 1;  % 1: decay of subcritical structures into monomers; 2: detachment at the ends of subcritical structures; 3: fragmentation of subcritical structures at any bond
NO_ACTIVATION = false;      % if true effctively set the activation rate to infinity for simulation of the Dimerization_scenario; 

if nargin==8        % if MaxSimulTime has not been provided use large default value
    MaxSimulTime = 1e20;
    MaxYield = 1.0;
elseif nargin==9
    MaxSimulTime = 1e20;
end

cutoff = 0.55/S;    % cutoff required to determine the assembly time (time_needed) in accordance with the average assembly time in a stochastic simulation (the factor is approximately the Euler Mascheroni cconstant; it can be shown analytically that when the concentration of active monomers has dropped below this value a stochastic simulation would have reached its final state on average)    

% to make a movie: only relevant if MakeMovie==true
timestep = -1;    % timestep between subsequent frames in the movie; timestep==-1 indicates that the timestep shall be determined as timestep=stoptime/400 such that there will be a total of 200 frames in the movie.
ylim_movie = -1;    % ylim of the frames in the movie;   % ylim_movie==-1 indicates that the ylim for the frames in the movie will be chosen as the maximum value of all complex concentrations that appear in the simulation
MaxMovieTime = MaxSimulTime;    % time up to which the movie is created. If you only want a movie it doesn't make sense to simulate longer than the movie time, however, if you want a movie but at the same time determine the final yield very accurately you can choose MaxMovieTime << MaxSimulTime 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% set up initial conditions
x0 = zeros(L+1,1);    % x(1): concentration of active monomers; x(n) with 1<n<L+1: concentration of complexes of size n; x(L+1): concentration of inactive monomers
if(NO_ACTIVATION==true)
    x0(1) = N;
else
    x0(end) = N;
end

    function [value,isterminal,direction] = myEventsFcn(~,x)     % stop the simulation if the total monomer content in all subcritical complexes is smaller than cutoff (this reproduces approximately the assembly time that would be obtained in an average stochastic simulation)
        value=zeros(1,2);
        isterminal=[1,1];    
        direction=[0,0];
        if sum([1:critical_size,1]'.*x([1:critical_size,L+1]))<cutoff
            value(1) = 1;
        end
        if MaxYield<1.0 && x(L)*L>=MaxYield*N   % to stop simulation as soon as MaxYield yield is achieved
            value(1) = 1;
        end
    end

% perform the integration 
options_ode = odeset('RelTol',1e-12,'AbsTol',1e-12,'NonNegative',(1:L+1),'Events',@myEventsFcn);   % alternatively choose: 'RelTol',1e-6,'AbsTol',1e-8   % ,'Stats','on' ,'OutputFcn',@odeplot,'OutputSel',[size+1,1,2,3,20,size] % 'Jacobian',@Jacobian%
if NO_ACTIVATION==true
    [T,X,time_needed,xe,ie] = ode15s(@(t,x) dynamics_Dimerization(t,x),[0 MaxSimulTime],x0, options_ode);
else
    [T,X,time_needed,xe,ie] = ode15s(@(t,x) dynamics_Activation_Dimerization(t,x),[0 MaxSimulTime],x0, options_ode);
end
CompStat = X(end,:);       % save the complex statistics, i.e. the distribution of complexes of different sizes at the end of the simulation

Y = X*diag([1:L,1]);     % multiply each complex by the number of particles it contains 
mass_conservation_violation = sum(Y(end,:))-N       % check if mass is conserved
if abs(mass_conservation_violation) > N*1e-3
    disp('Mass Conservation is violated');
   % return;
end
yield = Y(end,L)/N       % calculate and print the yield
if isempty(time_needed)   % determine and print the assembly time
    time_needed = inf;
end
time_needed

% if MakeMovie is true show a movie of the time evolution of the complex size distribution
% make sure that template_wave.fig is in the same folder; if this does not
% work or if template_wave is not available you can also comment this out
% and use the simple movie makeup below that is currently commented out
if MakeMovie == 1
    M = VideoWriter('Movie_SelfAssembly');
    open(M);
    fig=openfig('template_Wave.fig');
    axes = get(gcf,'Children');
% main plot (middle)
    set(fig,'CurrentAxes',axes(2));
    set(get(axes(2),'Children'),'xdata',1:L-1,'ydata',X(1,1:L-1));
    
    xlim([0 L]);
    xlabel('polymer size')
    %pos_xlabel = get(get(axes(2),'xlabel'),'position');
    if(ylim_movie==-1)
        ylim_movie = max(max(X(:,2:end-2)));  % determine ylim as the maximum of all complex concentrations
    end
    set(axes(2),'ylim',[0 ylim_movie]);
    xlabel('polymer size','position',[30 -2, -1]);
% left: inactive particles
    set(fig,'CurrentAxes',axes(1));
    set(get(axes(1),'Children'),'xdata',0,'ydata',X(1,end)/N);
% right: complete structures
    set(fig,'CurrentAxes',axes(3));
    set(get(axes(3),'Children'),'xdata',0,'ydata',X(1,L)/(N/L));
    
    stoptime = min([time_needed,MaxMovieTime,MaxSimulTime]);
    if(timestep==-1)
        timestep=stoptime/200;
    end
    for k=1:ceil(stoptime/timestep)        
        writeVideo(M,getframe);
        index = find(T>=k*timestep,1);
        set(get(axes(2),'Children'),'xdata', 1:L-1, 'ydata', X(index,1:L-1) );
        set(get(axes(1),'Children'),'xdata', 0, 'ydata', X(index,end)/N );
        set(get(axes(3),'Children'),'xdata', 0, 'ydata', X(index,L)/(N/L) );
        pause(0.01);
    end
    close(M);
%    save('Wave.mat','T','X','timestep','alpha','N','L','critical_size');      % save data if desired
end

% simple movie makeup. use this instead if the above movie does not work (comment this in and the above movie out).
%{
if MakeMovie == 1
    M = VideoWriter('Movie_SelfAssembly');
    open(M);
    figure(3);
    fig = plot(0:L,X(1,[end,1:L]),'o','erase','xor','LineWidth',2,'Markersize',7);
    xlim([0 L]);
    if(ylim_movie==-1)
        ylim_movie = max(max(X(:,2:end-2)));  % determine ylim as the maximum of all complex concentrations
    end
    ylim([0 ylim_movie]);
    xlabel('polymer size');
    ylabel('concentration');
    set(gca,'Fontsize',17,'Linewidth',2)
    stoptime = min([time_needed,MaxMovieTime,MaxSimulTime]);
    if(timestep==-1)
        timestep=stoptime/200;
    end
    for k=1:ceil(stoptime/timestep)        % alternatively: t = linspace(0,stoptime,400) and then: index = find(T>=t,1);
        writeVideo(M,getframe);
        index = find(T>=k*timestep,1);
        set(fig,'xdata',0:L, 'ydata', X(index,[end,1:L]) );
        pause(0.02);
    end
    close(M);
%    save('Wave.mat','T','X','timestep','alpha','N','L','critical_size');    % save the data if desired
end
%}



%%%%%%%%%%%%%%%%%% dynamics function for ode solvers %%%%%%%%%%%%%%%%%% %%%%%%%%%%%%%%%%%%%%%%%



    function dxdt = dynamics_Dimerization(t,x)      % dynamics for dimerization scenario (used if NO_ACTIVATION==true) (both dynamical functions are quite similar)
        dxdt = zeros(L+1,1);
        dxdt(end) = 0;   % inactive particles (at last position in vector)
        dxdt(1) = - 2*mu*x(1)*x(1) - 2*rho*x(1)*(sum(x(2:L-1)));      % dynamics of active monomers; when one monomer reacts with another monomer to a 2-complex the concentration of monomers diminishes about 2 (stoichiometric factor)! 
        dxdt(2) = mu*x(1)*x(1) - 2*rho*x(1)*x(2);       % dynamics of dimers
        dxdt(3:L) = 2*rho*x(1)*(x(2:L-1)-[x(3:L-1);0]);  % reaction dynamics of all larger complexes; note that the complete complex has no loss term as it does not further react.    
        if DECAY_MODE==1        % decay into monomers
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);   % add decay terms to subcritcal complexes
            dxdt(1) = dxdt(1) + delta*((2:critical_size)*x(2:critical_size));
        elseif DECAY_MODE==2     % back rates; detachment of monomers at the ends
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);   % add decay terms to subcritcal complexes
            dxdt(2:critical_size-1) = dxdt(2:critical_size-1) + delta*x(3:critical_size);
            dxdt(1) = dxdt(1) + 2*delta*x(2) + delta*sum(x(3:critical_size));
        elseif DECAY_MODE==3        % fragmentation at any bond or subcritical structures
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);
            dxdt(1:critical_size-1) = dxdt(1:critical_size-1) + 2*delta*cumsum(x(2:critical_size)./(1:critical_size-1)','reverse');
        end
    end

    function dxdt = dynamics_Activation_Dimerization(t,x)      % dynamics for activation or 'mixed' scenario (used if NO_ACTIVATION==false) (both dynamical functions are quite similar)
        dxdt = zeros(L+1,1);
        dxdt(L+1) = -alpha*x(L+1);          % inactive particles (at last position in vector)
        dxdt(1) = alpha*x(L+1) - 2*mu*x(1)^2 - 2*rho*x(1)*(sum(x(2:L-1)));     % dynamics of active monomers; when one monomer reacts with another monomer to a 2-complex the concentration of monomers diminishes about 2! 
        dxdt(2) = mu*x(1)^2 - 2*rho*x(1)*x(2);
        dxdt(3:L) = 2*rho*x(1)*(x(2:L-1)-[x(3:L-1);0]);  % reaction dynamics of all larger complexes; note that the complete complex has no loss term as it does not further react.    
        if DECAY_MODE==1        % decay into monomers
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);   % add decay terms to subcritcal complexes
            dxdt(1) = dxdt(1) + delta*((2:critical_size)*x(2:critical_size));
        elseif DECAY_MODE==2     % back rates; detachment of monomers at the ends
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);   % add decay terms to subcritcal complexes
            dxdt(2:critical_size-1) = dxdt(2:critical_size-1) + delta*x(3:critical_size);
            dxdt(1) = dxdt(1) + 2*delta*x(2) + delta*sum(x(3:critical_size));
        elseif DECAY_MODE==3        % fragmentation at any bond or subcritical structures
            dxdt(2:critical_size) = dxdt(2:critical_size) - delta*x(2:critical_size);
            dxdt(1:critical_size-1) = dxdt(1:critical_size-1) + 2*delta*cumsum(x(2:critical_size)./(1:critical_size-1)','reverse');
        end
    end

end










