function [centered_stack, mean_profile] = Center_pkf(filename, writeIO)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                                                 %%%
%%%               This code was written to be used in our publication               %%%
%%%       "Rapid regulation of vesicle priming explains synaptic facilitation       %%%
%%%               despite heterogeneous vesicle:Ca2+ channel distances"             %%%
%%%                          Kobbersmed et al., eLife 2020                          %%%
%%%                             DOI: XXXXXXXXXXXXXXXXXX                             %%%
%%%                                                                                 %%%
%%%             When reusing this code in a publication, in original or             %%%
%%%                       modified form, please cite our work.                      %%%
%%%                                                                                 %%%
%%%                                     Authors:                                    %%%
%%%                  Andreas T. Grasskamp (grasskamp@fmp-berlin.de)                 %%%
%%%                   Alexander M. Walter (awalter@fmp-berlin.de)                   %%%
%%%         Leibniz-Forschungsinstitut fr Molekulare Pharmakologie, Berlin         %%%
%%%                                  November 2019                                  %%%
%%%                                                                                 %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Optimization function returning centered images

% This function takes as an input the stack of images to be evaluated,
% ideally containing deconvolved STED images of single active zones cut 
% out with a square ROI. It returns the centered image stack 
% after shifting all images to the center.
% Input 'filename' should be the directory/filename of the original .tif-Stack,
% for example 'strcat(pwd,'\Input stacks\Stack_1.tif')'.
% Input 'writeIO' should be 1 if the centered stack is supposed to be
% written to the current working directory ('centered.tif'), and averaged ('avg_centered.tif').

%%
imsize = size(imread(filename,1),1); % get image size in px
n_slices = numel(imfinfo(filename)); % get number of slices in stack

peakfinder_edge = 1; % in px; This excludes an edge of the evaluated image
peakfinder_dist = 5; % in px; This ensures a minimum distance between clusters
peakfinder_thresh1 = 25; % in grey values; Threshold of background subtraction
peakfinder_thresh2 = 0; % Not currently in use, was used as relative threshold
       
coord_array = [];
for aa = 1:n_slices
    [category_vector(aa), coord_array{aa}] = peakfinder(imread(filename,aa), peakfinder_edge, peakfinder_dist, peakfinder_thresh1, peakfinder_thresh2, 0, aa);
    if category_vector(aa)~=0
        xshift(aa) = mean(coord_array{aa}(:,1));
        yshift(aa) = mean(coord_array{aa}(:,2));
        for ff = 1:size(coord_array{aa},1)
            coord_array{aa}(ff,1) = (imsize/2 - xshift(aa)) + coord_array{aa}(ff,1);
            coord_array{aa}(ff,2) = (imsize/2 - yshift(aa)) + coord_array{aa}(ff,2);
        end
        original_stack(:,:,aa) = imread(filename,aa);
        centered_stack(:,:,aa) = imtranslate(imread(filename,aa),[length(imread(filename,aa))/2-yshift(aa), length(imread(filename,aa))/2-xshift(aa)]); % x and y are switched here because of the way MATLAB treats image origins
    else
        xshift(aa) = 0;
        yshift(aa) = 0;
        coord_array{aa} = [];
        original_stack(:,:,aa) = imread(filename,aa);
        centered_stack(:,:,aa) = original_stack(:,:,aa);
    end
end

if writeIO == 1
    for ll = 1:n_slices
        if ~isempty(coord_array{ll})
            imwrite(centered_stack(:,:,ll),'centered.tif','WriteMode','append');
        end
    end
    avgImgs('centered.tif');
end

[profile, mean_profile, sem_profile] = get_profile('centered.tif');

end

function [] = avgImgs(CurrentName)

if fopen(CurrentName) ~= -1 % check if file exists
    avg_Temp = zeros(size(imread(CurrentName)));
    slices = numel(imfinfo(num2str(CurrentName)));
    close all;
    for n=1:slices
        InputImg=imread(CurrentName, n);
        InputImg=double(InputImg);
        avg_Temp=avg_Temp+(InputImg);
    end
    avg_Temp=avg_Temp./max(max(avg_Temp));
    imwrite(avg_Temp,strcat('avg_',CurrentName));
else
    disp('ERROR - file does not exist');
    return
end

fclose all;

end

function [profiles, mean_profile, sem_profile] = get_profile(stack_name)

n_AZs = numel(imfinfo(stack_name));

for ll = 1:n_AZs
    temp_img = imread(stack_name,ll);
    for kk = 1:360
        profiles_temp(kk,:) = temp_img(26,26:end);
    end
    profiles(ll,:) = mean(profiles_temp);
end

mean_profile = mean(profiles);
sem_profile = std(profiles)./sqrt(size(profiles,1));

% % % figure; hold on; plot(0:10:250, mean(profiles));

end