function [peak_count, imax, zmax, distances] = peakfinder(image, edge, mindist, threshold_1, threshold_2, visualize, stack_position)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                                                 %%%
%%%               This code was written to be used in our publication               %%%
%%%       "Rapid regulation of vesicle priming explains synaptic facilitation       %%%
%%%               despite heterogeneous vesicle:Ca2+ channel distances"             %%%
%%%                          Kobbersmed et al., eLife 2020                          %%%
%%%                             DOI: XXXXXXXXXXXXXXXXXX                             %%%
%%%                                                                                 %%%
%%%             When reusing this code in a publication, in original or             %%%
%%%                       modified form, please cite our work.                      %%%
%%%                                                                                 %%%
%%%                                     Authors:                                    %%%
%%%                  Andreas T. Grasskamp (grasskamp@fmp-berlin.de)                 %%%
%%%                   Alexander M. Walter (awalter@fmp-berlin.de)                   %%%
%%%         Leibniz-Forschungsinstitut fr Molekulare Pharmakologie, Berlin         %%%
%%%                                  November 2019                                  %%%
%%%                                                                                 %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% image = matrix in workspace or matrix input from calling function
% edge = exclusion around image in pixels, default 1
% mindist = minimal distance between two peaks, default 5
% threshold_1 & threshold_2 = background subtraction & peak exclusion values, default 0.33
% visualize = 0 or 1 depending on whether result should be displayed

for ii = 1:edge
    image(:,ii) = 0;
    image(ii,:) = 0;
    image(:,length(image)-ii+1) = 0;
    image(length(image)-ii+1,:) = 0;
end

try
    [zmax,imax,~,~] = extrema2(image-(threshold_1),1);
    zmax = zmax+threshold_1;
    if isempty(zmax)
        zmax = 0;
        imax = 0;
        peak_count = 0;
        distances = 0;
        disp(strcat('NO PEAKS DETECTED IN IMAGE', {' '}, num2str(stack_position)));
        return
    end
catch
    zmax = 0;
    imax = 0;
    peak_count = 0;
    distances = 0;
    disp(strcat('NO PEAKS DETECTED IN IMAGE', {' '}, num2str(stack_position)));
    return
end

[imax(:,1),imax(:,2)] = ind2sub([size(image), size(image)], imax);

if length(zmax) > 1
    
    distances = zeros(length(imax(:,1)), length(imax(:,2)));
    dX = zeros(length(distances),length(distances));
    dY = zeros(length(distances),length(distances));
    
    for i=1:length(imax);
        for j=1:length(imax)
            dX(i,j)=imax(i,1)-imax(j,1);
            dY(i,j)=imax(i,2)-imax(j,2);
            distances(i,j)=1*sqrt(dX(i,j)^2+dY(i,j)^2);
        end
    end
    
    %_{
    % new version
    
    % list all ROIs within mindist range
    for ww = 1:length(distances)
        [row{ww},~] = find((distances(:,ww)>0 & distances(:,ww)<mindist));
    end
    
    % make sorted list of ROIs with zmax values (amplitudes)
    % and delete all but the biggest value. If some values are
    % equal, this loop deletes all but one. If one value is close to two others,
    % the distance between these two others needs to be evaluated. If
    % it is higher than mindist, this loop keeps both.
    for ss = 1:length(distances)
        vals{1,ss} = [zmax(ss) zmax(row{ss})'];
        vals{2,ss} = [ss row{1,ss}'];
        if length(vals{1,ss}) < 2
            vals{1,ss} = [];
        end
        [~,idx]=sort(vals{1,ss});
        vals{1,ss} = sort(vals{1,ss});
        vals{2,ss} = vals{2,ss}(idx); % sort ROI indices by max amps
        for pp = 1:length(vals{2,ss})-1
            % Only delete ROI if it is less than mindist away from max value
            % -> this prevents both ROIs even if only one fulfills mindist.
            if distances(vals{2,ss}(pp),vals{2,ss}(length(vals{2,ss}))) < mindist
                imax(vals{2,ss}(pp),:) = 0;
                zmax(vals{2,ss}(pp)) = 0;
            end
        end
    end
    
    imax2(:,1) = nonzeros(imax(:,1));
    imax2(:,2) = nonzeros(imax(:,2));
    imax = imax2;
    zmax = nonzeros(zmax);
    %}
    %%%%%%%%%%%
    
    %%% Exclude peaks with an amplitude below threshold_2 times the
    %%% average of the 3 brightest peaks:
    
    if length(zmax) > 3
        brightest_three = sort(zmax,'descend');
        brightest_three = brightest_three(1:3);
        cutoff = mean(brightest_three)*threshold_2;
        [idx,~] = find(zmax<cutoff);
        zmax(idx) = 0;
        imax(idx,:) = 0;
    end
    
    zmax = nonzeros(zmax);
    imax3(:,1) = nonzeros(imax(:,1));
    imax3(:,2) = nonzeros(imax(:,2));
    imax = imax3;
else
    distances = 0;
end

peak_count = length(zmax);

if visualize == 1
    % plot result
    figure
    
    hold on
    plot3((imax(:,2)),(imax(:,1)),zmax,'ro');
    for i = 1:length(zmax)
        text(double((imax(i,2))),double((imax(i,1))),double(zmax(i)),['  ' num2str(zmax(i))]);
    end
    
    hold off
    
    hold on
    surf(image), axis equal, shading interp
end

end

function [xymax,smax,xymin,smin] = extrema2(xy,varargin)

%   Modified from a script written by
%   Lic. on Physics Carlos Adrin Vargas Aguilera
%   Physical Oceanography MS candidate
%   UNIVERSIDAD DE GUADALAJARA
%   Mexico, 2005
%
%   nubeobscura@hotmail.com

% From       : http://www.mathworks.com/matlabcentral/fileexchange
% File ID    : 12275
% Submitted at: 2006-09-14

% % % % Copyright (c) 2016, Carlos Adrian Vargas Aguilera
% % % % All rights reserved.
% % % %
% % % % Redistribution and use in source and binary forms, with or without
% % % % modification, are permitted provided that the following conditions are met:
% % % %
% % % % * Redistributions of source code must retain the above copyright notice, this
% % % %   list of conditions and the following disclaimer.
% % % %
% % % % * Redistributions in binary form must reproduce the above copyright notice,
% % % %   this list of conditions and the following disclaimer in the documentation
% % % %   and/or other materials provided with the distribution
% % % % THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% % % % AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% % % % IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% % % % DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
% % % % FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
% % % % DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
% % % % SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
% % % % CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
% % % % OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
% % % % OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

M = size(xy);
if length(M) ~= 2
    error('Entry must be a matrix.')
end
N = M(2);
M = M(1);

% Search peaks through columns:
[smaxcol,smincol] = extremos(xy);

% Search peaks through rows, on columns with extrema points:
im = unique([smaxcol(:,1);smincol(:,1)]); % Rows with column extrema
[smaxfil,sminfil] = extremos(xy(im,:).');

% Conversion from 2 to 1 index:
smaxcol = sub2ind([M,N],smaxcol(:,1),smaxcol(:,2));
smincol = sub2ind([M,N],smincol(:,1),smincol(:,2));
smaxfil = sub2ind([M,N],im(smaxfil(:,2)),smaxfil(:,1));
sminfil = sub2ind([M,N],im(sminfil(:,2)),sminfil(:,1));

% Peaks in rows and in columns:
smax = intersect(smaxcol,smaxfil);
smin = intersect(smincol,sminfil);

% Extrema points:
xymax = xy(smax);
xymin = xy(smin);

% Descending order:
[~,inmax] = sort(-xymax); clear temp
xymax = xymax(inmax);
smax = smax(inmax);
[xymin,inmin] = sort(xymin);
smin = smin(inmin);
end

function [smax,smin] = extremos(matrix)
% Peaks through columns or rows.

smax = [];
smin = [];

for n = 1:length(matrix(1,:))
    [~,imaxfil,~,iminfil] = extrema(matrix(:,n)); clear temp
    if ~isempty(imaxfil)     % Maxima indexes
        imaxcol = repmat(n,length(imaxfil),1);
        smax = [smax; imaxfil imaxcol];
    end
    if ~isempty(iminfil)     % Minima indexes
        imincol = repmat(n,length(iminfil),1);
        smin = [smin; iminfil imincol];
    end
end
end

function [xmax,imax,xmin,imin] = extrema(x)

%   Modified from a script written by
%   Lic. on Physics Carlos Adrin Vargas Aguilera
%   Physical Oceanography MS candidate
%   UNIVERSIDAD DE GUADALAJARA
%   Mexico, 2004
%
%   nubeobscura@hotmail.com

% From       : http://www.mathworks.com/matlabcentral/fileexchange
% File ID    : 12275
% Submitted at: 2006-09-14

% % % % Copyright (c) 2016, Carlos Adrian Vargas Aguilera
% % % % All rights reserved.
% % % %
% % % % Redistribution and use in source and binary forms, with or without
% % % % modification, are permitted provided that the following conditions are met:
% % % %
% % % % * Redistributions of source code must retain the above copyright notice, this
% % % %   list of conditions and the following disclaimer.
% % % %
% % % % * Redistributions in binary form must reproduce the above copyright notice,
% % % %   this list of conditions and the following disclaimer in the documentation
% % % %   and/or other materials provided with the distribution
% % % % THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% % % % AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% % % % IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% % % % DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
% % % % FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
% % % % DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
% % % % SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
% % % % CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
% % % % OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
% % % % OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

xmax = [];
imax = [];
xmin = [];
imin = [];

% Vector input?
Nt = numel(x);
if Nt ~= length(x)
    error('Entry must be a vector.')
end

% NaNs:
inan = find(isnan(x));
indx = 1:Nt;
if ~isempty(inan)
    indx(inan) = [];
    x(inan) = [];
    Nt = length(x);
end

% Difference between subsequent elements:
dx = diff(x);
dxs = zeros(1,length(x));
for hh = 1:length(dx)
    dxs(hh+1) = dx(hh);
end

dx = dxs;

% Is the input vector a horizontal line?
if ~any(dx)
    return
end

% Flat peaks? Put the middle element:
a = find(dx~=0);              % Indices where x changes
lm = find(diff(a)~=1) + 1;    % Indices where a does not change
d = a(lm) - a(lm-1);          % Number of elements in the flat peak
a(lm) = a(lm) - floor(d/2);   % Save middle elements
a(end+1) = Nt;

% Peaks?
xa  = x(a);             % Series without flat peaks
b = (diff(xa) > 0);     % 1  =>  positive slopes (minima begin)
xb  = diff(b);          % -1 =>  maxima indexes (but one)
imax = find(xb == -1) + 1; % maxima indices
imin = find(xb == +1) + 1; % minima indices
imax = a(imax);
imin = a(imin);

nmaxi = length(imax);
nmini = length(imin);

% Maximum or minumim on a flat peak at the ends?
if (nmaxi==0) && (nmini==0)
    if x(1) > x(Nt)
        xmax = x(1);
        imax = indx(1);
        xmin = x(Nt);
        imin = indx(Nt);
    elseif x(1) < x(Nt)
        xmax = x(Nt);
        imax = indx(Nt);
        xmin = x(1);
        imin = indx(1);
    end
    return
end

% Maximum or minimum at the ends?
if (nmaxi==0)
    imax(1:2) = [1 Nt];
elseif (nmini==0)
    imin(1:2) = [1 Nt];
else
    if imax(1) < imin(1)
        imin(2:nmini+1) = imin;
        imin(1) = 1;
    else
        imax(2:nmaxi+1) = imax;
        imax(1) = 1;
    end
    if imax(end) > imin(end)
        imin(end+1) = Nt;
    else
        imax(end+1) = Nt;
    end
end
xmax = x(imax);
xmin = x(imin);

% NaN's:
if ~isempty(inan)
    imax = indx(imax);
    imin = indx(imin);
end

% Same size as x:
imax = reshape(imax,size(xmax));
imin = reshape(imin,size(xmin));

% Descending order:
[~,inmax] = sort(-xmax);
clear temp
xmax = xmax(inmax);
imax = imax(inmax);
[xmin,inmin] = sort(xmin);
imin = imin(inmin);

end
