% GluSnFR simulations (Cartesian coordinates, multisynapse)
% December 16, 2019

%---------- General Model Parameters -------------------
tsf=1;                                      %time step (s)
numtrials=10;                               %Run simulations multiple times and average
multi=4;                                    %# different synaptic configurations
savesingles=1;                              %save individual trials separately
particlesper=5000;                          %# molecules released at each synapse
simdur=100000;                              %Simulation duration (us)
worldsize = 30;                             %width of cubic diffusion world (um)
boxsize=0.1;                                %width of cubic box (um)
boxblock=20;                                %number of boxes to group for segregation experiments
segregated=3;                               %flag for whether to segregate EAATs and iGluSnFRs
GluSnFRfract=0.5;                           %0.1 means put GluSnFR in 10% of the boxes
NND=.465;                                   %separation between synapses (um)
numsteps=ceil(simdur/tsf);                  %time steps
numboxes=worldsize/boxsize;                 %boxes in each direction
numblocks=numboxes/boxblock;
edgearray=(0:boxsize:worldsize);            %world is in positive quadrant
radialvector=[0.5 1 2 5 10 25 50 100 10000];%ROI sizes to track
tort=1.5;                                   %tortuosity index
vf=0.21;                                    %volume fraction
boxvolume=1e-15*boxsize^3*vf;               %Liters
Q10=1;                                      %Room temp = 1, Phys temp = 3
unitconc=(1/6.022e23)/boxvolume;            %effective concentration of each glutamate molecule in a box
gVm=-90;                                    %glial resting potential
nVm=-70;                                    %neuronal resting potential
D=0.000253/(tort^2);                        %Room temp
rdiff = sqrt(2*D*tsf);                      %1-D rms distance in one time step

%Ion/glu concentrations
gluin= 0.00001;                             %glial intracellular glu = 10 uM
Naout=0.1462;                               %extracellular Na = 146.2 mM 
Nain = 0.005;                               %glial intracellular Na = 5 mM
nNain = 0.005;                              %neuronal intracellular Na = 5 mM
Kout = 0.0025;                              %extracellular K = 2.5 mM
Kin = 0.120;                                %glial intracellular K = 120 mM
nKin = 0.120;                               %neuronal intracellular K = 120 mM
Hout = 3.98107e-08;                         %extracellular pH 7.4
Hin = 3.98107e-08;                          %glial pH 7.4
nHin = 3.98107e-08;                         %neuronal intracellular pH 7.4
ngluin= 0.00001;                            %neuronal intracellular glu = 10 uM

%------------ Arrays for averages (multiple trials)--------
ave=zeros(numsteps,size(radialvector,2),multi);                   %time steps, ROI size, multi
if(savesingles==1)
    singles=zeros(numsteps,numtrials,size(radialvector,2));    %time steps, trial, channel
    singleuptake=zeros(numsteps,numtrials);
end

%Synapse array coordinates
%lattice15 is a 15x3 array containing the 3D coordinates for hexagonally
%arranged release sites. NND = 1.0 um.
%Note: NND in hippocampus CA1 is 0.465 um (Rusakov and Kullmann, 1998).
lattice15=[-1.101,0,0;0,0,0;1.101,0,0;
-0.551,0.954,0;0.551,0.954,0;-0.551,-0.954,0;
0.551,-0.954,0;-0.551,-0.477,0.898;0.551,-0.477,0.898;
-0.551,0.477,0.898;0.551,0.477,0.898;-0.551,-0.477,-0.898;
0.551,-0.477,-0.898;-0.551,0.477,-0.898;0.551,0.477,-0.898];

%----------------------------------------------------------

for k=1:numtrials               %---------- MULTIPLE TRIALS LOOP ------------
counter1=zeros(numsteps,1);                         %counters for glutamate, etc.
if(segregated==0)                               %EAATs and iGluSnFRs are co-localized
    GluSnFRBoolean = ones(numboxes,numboxes,numboxes);
    EAATBoolean = ones(numboxes,numboxes,numboxes);
elseif(segregated==4)                               %Place EAATs and iGluSnFR in every other box
    GluSnFRBoolean = zeros(numboxes,numboxes,numboxes);
    EAATBoolean = zeros(numboxes,numboxes,numboxes);
    for x=1:numboxes
        GluSnFRBoolean((mod(x,2)==1),(mod(x,2)==1),(mod(x,2)==1))=2;
        GluSnFRBoolean((mod(x,2)==0),(mod(x,2)==0),(mod(x,2)==1))=2;
        EAATBoolean((mod(x,2)==1),(mod(x,2)==0),(mod(x,2)==1))=2;
        EAATBoolean((mod(x,2)==0),(mod(x,2)==1),(mod(x,2)==1))=2;
        EAATBoolean((mod(x,2)==1),(mod(x,2)==1),(mod(x,2)==0))=2;
        EAATBoolean((mod(x,2)==0),(mod(x,2)==0),(mod(x,2)==0))=2;
        GluSnFRBoolean((mod(x,2)==1),(mod(x,2)==0),(mod(x,2)==0))=2;
        GluSnFRBoolean((mod(x,2)==0),(mod(x,2)==1),(mod(x,2)==0))=2;
    end
elseif(segregated==1)                               %Place EAATs and iGluSnFR in every other box;
    GluSnFRBoolean = zeros(numboxes,numboxes,numboxes);
    EAATBoolean = zeros(numboxes,numboxes,numboxes);
    for z=0:numblocks-1
        for y=0:numblocks-1
            for x=0:numblocks-1
                for xp=1:boxblock
                    for yp=1:boxblock
                        for zp=1:boxblock
                            if(mod(z,2)==1)
                                if(mod(y,2)==1)
                                    if(mod(x,2)==1)
                                        GluSnFRBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    else
                                        EAATBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    end
                                else
                                    if(mod(x,2)==0)
                                        GluSnFRBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2; 
                                    else
                                        EAATBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    end
                                end
                            else
                                if(mod(y,2)==0)
                                    if(mod(x,2)==1)
                                        GluSnFRBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    else
                                        EAATBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    end
                                else
                                    if(mod(x,2)==0)
                                        GluSnFRBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2; 
                                    else
                                        EAATBoolean((x*boxblock)+xp,(y*boxblock)+yp,(z*boxblock)+zp)=2;
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end
    end
    EAATBoolean = ones(numboxes,numboxes,numboxes);
elseif(segregated==2)    % "interneuron" arrangement iGluSnFR added randomly to some % of boxes, EAATs everywhere
    GluSnFRBoolean = zeros(numboxes,numboxes,numboxes);
    EAATBoolean = ones(numboxes,numboxes,numboxes);
    for y=1:boxblock:numboxes
        for x=1:boxblock:numboxes
            for z=1:boxblock:numboxes
                rn=rand;
                for xp=1:boxblock
                    for yp=1:boxblock
                        for zp=1:boxblock
                            if(rn<GluSnFRfract)
                                GluSnFRBoolean(x+xp,y+yp,z+zp)=(1/GluSnFRfract);
                            end
                        end
                    end
                end
            end
        end
    end
elseif((segregated==3)&&(GluSnFRfract~=0))    % "random but perfect segregation
    GluSnFRBoolean = zeros(numboxes,numboxes,numboxes);
    EAATBoolean = zeros(numboxes,numboxes,numboxes);
    for y=1:boxblock:numboxes
        for x=1:boxblock:numboxes
            for z=1:boxblock:numboxes
                if(rand<GluSnFRfract)
                    for xp=1:boxblock
                        for yp=1:boxblock
                            for zp=1:boxblock
                                GluSnFRBoolean(x+xp,y+yp,z+zp)=(1/GluSnFRfract);
                            end
                        end
                    end
                else
                    for xp=1:boxblock
                        for yp=1:boxblock
                            for zp=1:boxblock
                                EAATBoolean(x+xp,y+yp,z+zp)=(1/(1-GluSnFRfract));
                            end
                        end
                    end
                end
            end
        end
    end
EAATBoolean = ones(numboxes,numboxes,numboxes);
end

switch mod(k,multi)
        case 1
            latticepos=2;
        case 2
            latticepos=[2,3,10];
        case 3
            latticepos=[1,2,3,10,11,14,15];
        case 0
            switch multi
                case 1
                    latticepos=2;
                case 2
                    latticepos=[2,3,10];
                case 3    
                    latticepos=[1,2,3,10,11,14,15];
                case 4
                    latticepos=[1,2,3,4,5,6,7,8,9,10,11,12,13,14,15];
            end
end

lattarray=(lattice15.*NND)+(worldsize/2);
A=size(latticepos);
numsynapses=A(2);                           %# synapses
numparticles=numsynapses*particlesper;      %total number of transmitter molecules

releasepoint=zeros(numsynapses,3);          %3-D position of each active synapse     
for i=1:numsynapses
    releasepoint(i,1)=lattarray(latticepos(i),1);
    releasepoint(i,2)=lattarray(latticepos(i),2);
    releasepoint(i,3)=lattarray(latticepos(i),3);
end

delete(findall(gcf,'type','annotation'))
dim=[.1 .9 .3 .1];
str="Trial number: "+num2str(k);
h=annotation('textbox',dim,'LineStyle','none','String',str);
drawnow

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Transporter density/concentration
EAATconc = EAATBoolean.*1e-4;              %EAAT concentration (M)
GluSnFRconc = GluSnFRBoolean.*3e-4;
GluSnFRtype = 2;                           %1=iGluSnFR@34C, 2=iGluSnFR@20C, 3=Gluf@34C, 4=Gluf@20C, 5=iGluu@34C, 6=iGluu@20C
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

radial = zeros(numparticles,1);             %distance of each transmitter particle from release site (origin)
radial(:)= 1;
whichorder = zeros(numparticles,1);
whichcounter = 0;

glunum = zeros(numboxes,numboxes,numboxes);       %number of glutamate moldecules in each box
transtrac = zeros(numsteps,1);
GluSnFRtrac = zeros(numsteps,1);
boxtrans = 6.022e23*boxvolume.*EAATconc;         %calculate the number of transporters in each box (note: .* is vector multiplication)
boxGluSnFR = 6.022e23*boxvolume.*GluSnFRconc;    %calculate the number of GluSnFR molecules in each box (note: .* is vector multiplication)
EAAT0=boxtrans;                                  %original number of transporters
GluSnFR0=boxGluSnFR;                             %original number of GluSnFR molecules

%----- Position each glutamate molecule to begin the simulation -----
position=ones(numparticles,2,3);                  %3-D position of each transmitter particle
for i=1:numparticles                              %allot transmitter to each synapse
    position(i,:,1)=releasepoint(ceil(i/(particlesper)),1);
    position(i,:,2)=releasepoint(ceil(i/(particlesper)),2);
    position(i,:,3)=releasepoint(ceil(i/(particlesper)),3);
end
position0=position;                               %original 3-D position of each transmitter particle
positionecho=position;
box=ones(numparticles,2,3);                       %which box contains each transmitter particle

 %----------- Counter arrays, etc. -----------------------
dFF=zeros(numsteps,10);                             %GluSnFR signal: time steps, volumes, multi
STC=zeros(numsteps,1);                              %surrogate for the glial transporter current
cumSTC=zeros(numsteps,1);
GLTtouches = zeros(numsteps,1);                     %number of times EAAC1 binds glutamate
cumGLTtouches = zeros(numsteps,1);                  %number of times EAAC1 binds glutamate
GLTEfficiency = zeros(numsteps,1);                  %cumNTC/cumEAAC1touches
GLTEfficiency(:)=NaN;
trialcounter=zeros(numtrials,1);                    %counters for glutamate, etc.
counter=zeros(numsteps,8,4);
endpoint=zeros(numparticles,1);                     %site of uptake, radial distance from release
endpoint(:)=NaN;
endtime=zeros(numparticles,1);                      %time of uptake
endtime(:)=NaN;
bound=zeros(numparticles,1);                        %unbound=0; glial EAAT=1; GluSnFR=2
timesbound=zeros(numparticles,1);                   %number of transporter hits before getting transported
bp = zeros(numparticles,1);                         %EAAT binding prob for each particle
gbp = zeros(numparticles,1);                        %iGluSnFR binding prob for each particle
transported=zeros(numparticles,1);                  %flag that a particle has been taken up
free=ones(numparticles,1);
EAATbound=zeros(numparticles,1);
EAATtouches=zeros(numparticles,1);
iGlubound=zeros(numparticles,1);
snifftouch=zeros(numparticles,1);
cycled=zeros(numparticles,1);

%---------- Transporter Kinetic Schemes -------------------

%GLT-1, according to Bergles, et al. (2002) J. Neurosci. 
%each rate is multiplied by Q10, the time step fraction, and a s-->s conversion

vfactor1= exp((96486*(gVm+70)/1000*0.46)/(2*8.3143*298));
vfactor7 = exp((96486*(gVm+70)/1000*0.55)/(2*8.3143*298));
%vfactor9=  exp((0.1*96486*(gVm+70)/1000*0.4)/(2*8.3143*298));
vfactor9f=  exp((0.1*96486*(gVm+70)/1000*0.4)/(2*8.3143*298));
vfactor9b=  exp(((1-0.1)*96486*(gVm+70)/1000*0.4)/(2*8.3143*298));
vfactor15f= exp((0.9*96486*(gVm+70)/1000*0.59)/(2*8.3143*298));
vfactor15b= exp(((1-0.9)*96486*(gVm+70)/1000*0.59)/(2*8.3143*298));

k1f = Q10*tsf*1e-6*Naout*1e4/vfactor1;          %extracellular Na dependent (120 mM)
k1b = Q10*tsf*1e-6*1e2*vfactor1;
k2f = Q10*tsf*1e-6*Naout*1e4;                   %extracellularNa dependent
k2b = Q10*tsf*1e-6*5e2;
k3f = Q10*tsf*1e-6*6e6;                         %extracellular glutamate dependent
k3b = Q10*tsf*1e-6*5e2;
k4f = Q10*tsf*1e-6*Hout*6e11;                   %proton dependent (pH 7.4)
k4b = Q10*tsf*1e-6*7e2;
%k5f = Q10*tsf*1e-6*Hout*6e11;                   %proton dependent (pH 7.4)
%k5b = Q10*tsf*1e-6*7e2;
k6f = Q10*tsf*1e-6*6e6;                         %extracellular glutamate dependent
k6b = Q10*tsf*1e-6*1e2;                         %glu unbinding, default 5e2
k7f = Q10*tsf*1e-6*Naout*1e4/vfactor7;          %extracellular Na
k7b = Q10*tsf*1e-6*1e3*vfactor7;
k8f = Q10*tsf*1e-6*2e3;                         %translocation
k8b = Q10*tsf*1e-6*1.9e3;
k9f = Q10*tsf*1e-6*1e3/vfactor9f;                %Na unbinding
k9b = Q10*tsf*1e-6*Nain*4e4*vfactor9b;           %intracellular Na dependent
k10f = Q10*tsf*1e-6*3e3;
k10b = Q10*tsf*1e-6*Hin*9e10;
k11f = Q10*tsf*1e-6*3e3;                        %intracellular glutamate unbinding
k11b = Q10*tsf*1e-6*gluin*1e5;
k12f = Q10*tsf*1e-6*1e5;
k12b = Q10*tsf*1e-6*Nain*2e7;
k13f = Q10*tsf*1e-6*1e5;
k13b = Q10*tsf*1e-6*Nain*1e8;
k14f = Q10*tsf*1e-6*Kin*1e6;
k14b = Q10*tsf*1e-6*1e3;
k15f = Q10*tsf*1e-6*40/vfactor15f;
k15b = Q10*tsf*1e-6*10*vfactor15b;
k16f = Q10*tsf*1e-6*2e4;
k16b = Q10*tsf*1e-6*Kout*1e6;
%k17f = Q10*tsf*1e-6*1.4;
%k17b = Q10*tsf*1e-6*Kout*0.01;
EAATState = ones(numparticles,1);              %transporter state for bound glutamate molecules
GLTfrate=[k1f;k2f;k4f;k6f;k7f;k8f;k9f;k10f;k11f;k12f;k13f;k14f;k15f;k16f];
GLTbrate=[k16b;k1b;k2b;k4b;k6b;k7b;k8b;k9b;k10b;k11b;k12b;k13b;k14b;k15b];

%GluSnFR kinetic scheme (Halassa et al (2018) PNAS)
switch GluSnFRtype
    case 1                          %iGluSnFR @ 34C
        gk1f = tsf*1e-6*2.8e7;
        gk1b = tsf*1e-6*8161;
        gk2f = tsf*1e-6*756;
        gk2b = tsf*1e-6*220;
    case 2                          %iGluSnFR @ 20C
        gk1f = tsf*1e-6*2.7e7;
        gk1b = tsf*1e-6*5965;
        gk2f = tsf*1e-6*569;
        gk2b = tsf*1e-6*110;
    case 3                          %Gluf @ 34C
        gk1f = tsf*1e-6*3.63e6;
        gk1b = tsf*1e-6*3018;
        gk2f = tsf*1e-6*1254;
        gk2b = tsf*1e-6*566;
    case 4                          %Gluf @ 20C
        gk1f = tsf*1e-6*3.5e6;
        gk1b = tsf*1e-6*2206;
        gk2f = tsf*1e-6*944;
        gk2b = tsf*1e-6*283;
    case 5                          %Gluu @ 34C
        gk1f = tsf*1e-6*2.28e6;
        gk1b = tsf*1e-6*2331;
        gk2f = tsf*1e-6*180.7;
        gk2b = tsf*1e-6*936;
    case 6                          %Gluu @ 20C
        gk1f = tsf*1e-6*2.2e6;
        gk1b = tsf*1e-6*1704;
        gk2f = tsf*1e-6*136;
        gk2b = tsf*1e-6*468;
    case 7                          %FrankenSnFR
        gk1f = tsf*1e-6*3.5e7;
        gk1b = tsf*1e-6*2500;
        gk2f = tsf*1e-6*2000;
        gk2b = tsf*1e-6*600;
end
GluSnFRstate = ones(numparticles,1);
GluSnFRfrate=[gk1f;gk2f;0];
GluSnFRbrate=[0;gk1b;gk2b];
tvect=tsf.*(1:numsteps);

%-------------------------------- TRIAL LOOP --------------------------------
trialcounter(k)=1;
for i=2:numsteps                            %loop for every point in time
 
    %Calculate new 3-D position of particles
    position(:,1,:)=position(:,2,:);        %1 is old position, 2 is new
    box(:,1,:)=box(:,2,:);                  %1 is old box, 2 is new
    diff=free.*rdiff.*randn(numparticles,1,3); %only unbound particles move
    position(:,2,:)=position(:,1,:)+diff;
    X=find(transported==0);                 %subset of untransported particles
    positionecho(X,:,:)=position(X,:,:);    %for diffusion graph
    box(:,2,:)=ceil(position(:,2,:)./boxsize);
    box(box<1)=1;
    box(box>numboxes)=numboxes;
    disp_x=(position(:,2,1)-position0(:,1,1));  %dispersion in (x,y,z)
    disp_y=(position(:,2,2)-position0(:,1,2));
    disp_z=(position(:,2,3)-position0(:,1,3));
    radial=sqrt((disp_x.^2)+(disp_y.^2)+(disp_z.^2)); %total dist from start
    
    %Binding probabilities 
    bp = (k6f*unitconc).*boxtrans(sub2ind(size(boxtrans),box(:,2,1),box(:,2,2),box(:,2,3)));                %EAAT2
    gbp = (gk1f*unitconc).*boxGluSnFR(sub2ind(size(boxGluSnFR),box(:,2,1),box(:,2,2),box(:,2,3)));          %iGluSnFR
    mprob=rand(numparticles,1);
    inplay=free.*(1-cycled);                %Free particles
   
    %Determine fate of each particle during the time step
    %Bind to EAATs
    X=find(free.*(EAATbound==0).*(mprob<(bp.*inplay)));     %Particles that bind EAAT during this time step
    %Particles that bound EAAT during the last time step
    Xf=find((EAATbound==1).*(mprob<GLTfrate(EAATState))); %subset going forward (bound previously)
    Xb=find((EAATbound==1).*(mprob>=GLTfrate(EAATState))&(mprob<(GLTfrate(EAATState)+GLTbrate(EAATState)))); %subset going backward (bound previously)
    nXf=(EAATbound==1).*(mprob<GLTfrate(EAATState)); %binary: forward=1 (bound previously)
    nXb=(EAATbound==1).*(mprob>=GLTfrate(EAATState)).*(mprob<(GLTfrate(EAATState)+GLTbrate(EAATState))); %binary: backward=1 (bound previously)
    trans_ind=sub2ind(size(boxtrans),box(X,2,1),box(X,2,2),box(X,2,3));
    EAATState(X)=5;     %EAAT bound state
    EAATbound(X)=1;    
    free(X)=0;
    EAATtouches(X)=EAATtouches(X)+1;
    boxtrans(trans_ind)=boxtrans(trans_ind)-1;  %reduce available transporters
    Xf14=find(nXf.*(EAATState==14));             %EAAT state 14-->1
    nXf14=nXf.*(EAATState==14);                  %binary
    STC(i)=STC(i)-sum(nXf14);                   %K countertransport
    Xf14_ind=sub2ind(size(boxtrans),box(Xf14,2,1),box(Xf14,2,2),box(Xf14,2,3));
    boxtrans(Xf14_ind)=boxtrans(Xf14_ind)+1;   %increment EAAT
    cycled(Xf14)=1;
    Xb1=find(nXb.*(EAATState==1));                          %1-->14
    EAATState(Xf)=EAATState(Xf)+1;
    EAATState(Xb)=EAATState(Xb)-1;
    EAATState(Xf14)=1;
    EAATState(Xb1)=14;
    X54=find(nXb&(EAATState==4));                        %5-->4
    X54_ind=sub2ind(size(boxtrans),box(X54,2,1),box(X54,2,2),box(X54,2,3));
    boxtrans(X54_ind)=boxtrans(X54_ind)+1;   %increment EAAT
    EAATbound(X54)=0;
    free(X54)=1;
    X910=find(nXf.*(EAATState==10));                     %transported
    nX910=nXf.*(EAATState==10);                        
    transported(X910)=1;
    STC(i)=STC(i)+sum(nX910);
    EAATbound(X910)=0;
    endpoint(X910)=sqrt((position(X910,2,1)-position0(X910,1,1)).^2+(position(X910,2,2)-position0(X910,1,2)).^2+(position(X910,2,3)-position0(X910,1,3)).^2);
    whichcounter=whichcounter+sum(nX910);
    whichorder(X910)=whichcounter;
    endtime(X910)=i*tsf;
    positionecho(X910,1,:)=NaN;

    %Particles bound to iGluSnFR
    Y12=find(free.*(iGlubound==0).*(mprob>=(bp.*inplay)).*(mprob<((bp+gbp).*inplay))); %Particles that bind iGlu
        %Particles that bound iGluSnFR previously
    Y23=find((GluSnFRstate==2)&(mprob<GluSnFRfrate(2)));    %iGlu 2-->3
    Y21=find((GluSnFRstate==2)&(mprob>=GluSnFRfrate(2))&(mprob<(GluSnFRfrate(2)+GluSnFRbrate(2))));   %iGlu 2-->1
    Y32=find((GluSnFRstate==3)&(mprob<GluSnFRbrate(3)));
    iglu_12_ind=sub2ind(size(boxGluSnFR),box(Y12,2,1),box(Y12,2,2),box(Y12,2,3));
    iglu_23_ind=sub2ind(size(boxGluSnFR),box(Y23,2,1),box(Y23,2,2),box(Y23,2,3));
    iglu_21_ind=sub2ind(size(boxGluSnFR),box(Y21,2,1),box(Y21,2,2),box(Y21,2,3));

    GluSnFRstate(Y12)=2;                  %iGluSnFR bound state (for those bound this time step)
    iGlubound(Y12)=1;
    free(Y12)=0;
    snifftouch(Y12)=snifftouch(Y12)+1;
    boxGluSnFR(iglu_12_ind)=boxGluSnFR(iglu_12_ind)-1;   %reduce available GluSnFR
    boxGluSnFR(iglu_21_ind)=boxGluSnFR(iglu_21_ind)+1;   %increment GluSnFR      
    GluSnFRstate(Y23)=3;
    GluSnFRstate(Y21)=1;
    GluSnFRstate(Y32)=2;
    iGlubound(Y21)=0;
    free(Y21)=1;

    %Update iGluSnFR counters
    for v=1:size(radialvector,2)
        dFF(i,v)=sum((GluSnFRstate==3).*(radial<radialvector(v)));
    end
    cumSTC(i)=size(find(transported>0),1);
    counter1(i)=sum(free.*(radial>.4)&(radial<=.5));

    %---------------- Update plot ----------------------------
    if(mod(i,500)==0 || i==2)
        delete(findall(gcf,'type','annotation'))
        
        subplot(3,2,1)
        plot(tvect, STC)
        axis([0 (numsteps*tsf) 0 5])
        axis 'auto y'
        title('Transporter Currents')
        
        subplot(3,2,3)
        scatter(whichorder, endpoint,1)
        axis([0 (numparticles) 0 5])
        title('Diffusion distance')
               
        subplot(3,2,[2,4])
        scatter3(positionecho(:,1,1),positionecho(:,1,2),positionecho(:,1,3),4,'.')
        axis([0 worldsize 0 worldsize 0 worldsize])
        title('3-D diffusion map')

        subplot(3,2,5)
        plot(tvect, cumSTC)
        axis([0 (numsteps*tsf) 0 numparticles])
        axis 'auto y'99
        title('Glu molecules taken up')

        subplot(3,2,6)
        plot(tvect,dFF(:,1),tvect,dFF(:,2),tvect,dFF(:,3),tvect,dFF(:,4),tvect,dFF(:,5),tvect,dFF(:,6),tvect,dFF(:,7),tvect,dFF(:,8))
        axis([0 (numsteps*tsf) 0 5])
        axis 'auto y'
        title('GluSnFR signal')
        
        dim=[.1 .9 .3 .1];
        str="Trial number: "+num2str(k);
        h=annotation('textbox',dim,'LineStyle','none','String',str);
        drawnow

    end
end         % -------------- END OF THE TIME STEP LOOP -----------------
    
    if(savesingles==1)
        for c=1:size(radialvector,2)
            singles(:,k,c)=dFF(:,c)/tsf;
        end
        singleuptake(:,k)=cumSTC;
    end
    if(multi==1)
        for h=1:size(radialvector,2)
            ave(:,h,1)= squeeze(ave(:,h,1) + dFF(:,h));
        end
    end
    if(multi>1)
        n=mod(k,multi);
        if(n==0)
            n=4;
        end
        for m=1:size(radialvector,2)
            ave(:,m,n)=ave(:,m,n)+dFF(:,m);
        end    
    end
end             %------------ END OF MULTIPLE TRIALS LOOP
    ave=ave/(numtrials/multi);
    if(savesingles==1)
        writematrix(singles,'singles.dat');
        writematrix(singleuptake,'singleSTCs.dat');
    end
    writematrix(ave,"aves.dat")
    csvwrite('CumUptake.dat',cumSTC);
    beep;
    csvwrite('counter1.dat',counter1);
