% This function generates graphical output showing cell periphery, cortical force-generators, and centrosomes
%
% INPUT:
% Cell_BD: cell periphery
% CFG: positions and normal vectors of cortical force-generators
% CFG_Num: number of cortical force-generators
% CFG_R: force-generator size (radius)
% Cent_Pos: position of centrosomes
%
% LAST UPDATE:
% 09/07/2020
% Reza Farhadifar (rfarhadifar@flatironinstitute.org) 

function Graphics(Cell_BD, CFG, CFG_Num, CFG_R, Cent_Pos)

clf;
hold on;

% cell surface
h = surf(Cell_BD{1}, Cell_BD{2}, Cell_BD{3}, 'EdgeColor', 'none');
h.FaceColor = [231/255, 177/255, 33/255];
h.FaceLighting = 'gouraud';
h.FaceAlpha = 0.5;
h.SpecularStrength = 0.5;

% centrosomes
centR = 2;
[x, y, z] = sphere(100);
surf(x*centR + Cent_Pos(1, 1), y*centR + Cent_Pos(1, 2), z*centR + Cent_Pos(1, 3), 'FaceColor', 'r', 'EdgeColor', 'none', 'FaceAlpha', 1);
surf(x*centR + Cent_Pos(2, 1), y*centR + Cent_Pos(2, 2), z*centR + Cent_Pos(2, 3), 'FaceColor', 'r', 'EdgeColor', 'none', 'FaceAlpha', 1);

% cortical force-generators
theta = 0:2*pi/100:2*pi;
for i=1:CFG_Num
    center = CFG(i, 1:3);
    normal = CFG(i, 4:6);
    v = null(normal);
    points = repmat(center', 1, size(theta,2)) + CFG_R*(v(:,1)*cos(theta) + v(:,2)*sin(theta));
    g = fill3(points(1,:), points(2,:), points(3,:), [0.6 0.6 1]);
    g.FaceAlpha = 1;
    g.EdgeColor = 'none';    
end

% light and camera
A = 0; B = 90;
light('Position',[ 0 0 30], 'Style', 'local');
material shiny;
view([A, B]);
X = 30; Y = 18; Z = 18;
axis([-X X -Y Y -Z Z]);
axis off;
pbaspect([X Y Z])
set(gcf,'color','w');

hold off;

end