% This function uploads the precomputed positions of cortical
% force-generators and calculates their normal vectors.
%
% INPUT:
% filename: file name for precomputed cortical force-generators
% Cell_Radii: radius of the cell in xyz directions
% Cell_ShapeFactor: cell shape factor in xyz directions
% 
% OUTPUT:
% CFG: positions and normal vectors of cortical force-generators
% CFG_Num: number of cortical force-generators
%
% LAST UPDATE:
% 09/07/2020
% Reza Farhadifar (rfarhadifar@flatironinstitute.org) 

function [CFG, CFG_Num] = CFG_Config(filename, Cell_Radii, Cell_ShapeFactor)

f = load(filename); 
p = f.V; % position of force-generators
ps = (abs(p)./Cell_Radii).^Cell_ShapeFactor; % project position of force-generators to the surface of the cell
for i=1:length(ps)
    xs = fzero(@(x) betaEqu(x, Cell_ShapeFactor, ps(i,:)), 0.5 );
    p(i, :) = xs*p(i, :);
end

% calculate the normal vectors of force-generators
nx = Cell_ShapeFactor(1)*sign(p(:, 1)).*abs(p(:, 1)).^(Cell_ShapeFactor(1)-1)/(Cell_Radii(1)^Cell_ShapeFactor(1));
ny = Cell_ShapeFactor(2)*sign(p(:, 2)).*abs(p(:, 2)).^(Cell_ShapeFactor(2)-1)/(Cell_Radii(2)^Cell_ShapeFactor(2));
nz = Cell_ShapeFactor(3)*sign(p(:, 3)).*abs(p(:, 3)).^(Cell_ShapeFactor(3)-1)/(Cell_Radii(3)^Cell_ShapeFactor(3));
n = [nx, ny, nz];
n = n./vecnorm(n, 2, 2);

CFG_Num = size(p, 1);
CFG(1:CFG_Num, :) = [p, n];

end

% projection function
function F = betaEqu(x, P, Rs)
F = x.^P(1)*Rs(1) + x.^P(2)*Rs(2) + x.^P(3)*Rs(3) - 1;
end
