% This function calculates the probability of attachment of a microtubule to
% cortical force-generators. 
%
% INPUT:
% Cent_Pos: position of centrosomes
% CFG: cortical force-generator configuration
% CFG_R: cortical force-generator size (radius)
% CFG_P: probability of attachment of a microtubule to force-generators
% CFG_K: detachment rate of cortical force-generators
% MT_NucRate: nucleation rate of microtubules
% MT_AveLength: average length of microtubules
%
% OUTPUT: 
% CFG_P: probability of attachment of a microtubule to force-generators
%
% LAST UPDATE:
% 09/07/2020
% Reza Farhadifar (rfarhadifar@flatironinstitute.org) 

function CFG_P = update_P(Cent_Pos, CFG, CFG_R, CFG_P, CFG_K, MT_NucRate, MT_AveLength)

CFG_Pos = CFG(:, 1:3);
CFG_Normal = CFG(:, 4:6);

% impingement rate for centrosome 1
kesi1 = CFG_Pos - Cent_Pos(1, :);
d1 = vecnorm(kesi1, 2, 2);
kesi1 = kesi1./d1;
Xi1 = 0.5*dot(kesi1, CFG_Normal, 2) .* ( 1-1./sqrt(1+(CFG_R./d1).^2) );
Xi1 = max(Xi1, 0);
Omega1 = MT_NucRate * Xi1 .* exp(-d1/MT_AveLength);

% impingement rate for centrosome 2
kesi2 = CFG_Pos - Cent_Pos(2, :);
d2 = vecnorm(kesi2, 2, 2);
kesi2 = kesi2./d2;
Xi2 = 0.5*dot(kesi2, CFG_Normal, 2) .* ( 1-1./sqrt(1+(CFG_R./d2).^2) );
Xi2 = max(Xi2, 0);
Omega2 = MT_NucRate * Xi2 .* exp(-d2/MT_AveLength);

% probability of attachment
CFG_P(:, 1) = Omega1./(Omega1 + Omega2 + CFG_K);
CFG_P(:, 2) = Omega2./(Omega1 + Omega2 + CFG_K);

