% Simulation of the Stoichiometric Model of spindle.
%
% Cortical force-generators are generated by distmesh (http://persson.berkeley.edu/distmesh/).
%
% A precomputed distribution of cortical force-generators for Cell_Radii=[25 15 15] and Cell_ShapeFactor=[3 2 2] 
% with 50% asymmetry between the posterior and anterior halves of the cell is located in 'Input' folder.
% 
% LAST UPDATE:
% 09/07/2020
% Reza Farhadifar (rfarhadifar@flatironinstitute.org)


clc;
close all;
clear;

addpath('InternalSubroutine', 'Input');

% simulation parameters
Cell_Radii = [25 15 15]; % radii of the cell in xyz directions
Cell_ShapeFactor = [3 2 2]; % shape factor in xyz directions
MT_Vg = 0.5;  % microtubule growth rate [micron/s]
MT_AveLength = 20;  % microtubule average length [micron]
MT_Ninf = 10000;  % average number of microtubules per centrosome
MT_CatRate = MT_Vg/MT_AveLength;  % microtubule catastrophe rate [1/s]
MT_NucRate = MT_Ninf*MT_CatRate; % microtubule nucleation rate [1/s]
CFG_F0 = 10;  % maximum pulling force exerted by occupied force-generator [pN]
CFG_R = 1.5;  % force-generator radius [micron]
CFG_K = 0.1;  % force-generator detachment rate [1/s]
Cent_Drag = [150 100]; % drag on centrosome; spindle viscous friction coefficient [pNs/micron]
T_Off = 40; % time period from the start of the simulation for which CFG_F0=0 [s]
TMAX = 500; % maximum simualtion time [s]
deltat = 0.05; % time step [s]
GRAPHICS = true; % save graphical output
GRAPHICS_Time = 5; % time interval for saving graphical output

%  initialization of cortical force-generator configuration
[CFG, CFG_Num] = CFG_Config('FG_100_Asym.mat', Cell_Radii, Cell_ShapeFactor);

%  initialization of cell surface for graphical output
[Cell_BD{1}, Cell_BD{2}, Cell_BD{3}] = CellShape(Cell_Radii, Cell_ShapeFactor);

% initialization of centrosome position and velocity
Cent_Pos = [5 0 0; -5 0 0];
Cent_Vel = [0 0 0; 0 0 0];

if (GRAPHICS)
    figure('visible', 'off', 'position', [0 0 250 250]);
end

n = 1;
tp = 0;
CENT_POS = [];
CFG_P = zeros(CFG_Num, 2);
for t=0:deltat:TMAX
    fprintf('time elapsed = %f\n', t)
    
    if(t<T_Off)
        f0 = 0;
    end
    if(t>=T_Off && t<TMAX)
        f0 = CFG_F0*(t-T_Off)/(TMAX-T_Off);
    end
    
    CFG_P = update_P(Cent_Pos, CFG, CFG_R, CFG_P, CFG_K, MT_NucRate, MT_AveLength);                  
    Cent_Force = update_Force(Cent_Pos, CFG, CFG_P, f0);
    [Cent_Pos, Cent_Vel] = update_CentPosVel (Cent_Force, Cent_Pos, Cent_Drag, deltat);
    CENT_POS = [CENT_POS; t, Cent_Pos(1, :), Cent_Pos(2, :)];
    
    if (GRAPHICS && t-tp>GRAPHICS_Time)        
        clf;
        Graphics(Cell_BD, CFG, CFG_Num, CFG_R, Cent_Pos);
        H = getframe(gcf);
        imwrite(H.cdata, sprintf('./Output/Frame%d.jpg', n), 'jpg');
        tp = t;
        n = n + 1;
    end
end

figure; 
Time = CENT_POS(:, 1);
SpindleLength = vecnorm(CENT_POS(:, 2:4) - CENT_POS(:, 5:7), 2, 2);
SpindleCenter = 0.5*(CENT_POS(:, 2) + CENT_POS(:, 5));
subplot(1, 2, 1); plot(Time, SpindleLength); 
axis([0 TMAX 8 27]); pbaspect([1 1 1]); xlabel('time [s]'); ylabel('spindle length [\mu m]');

subplot(1, 2, 2); plot(SpindleCenter, Time); axis([-1 4 0 TMAX]); 
pbaspect([1 1 1]); set(gca,'YDir','reverse');
xlabel('spindle center [\mu m]'); ylabel('time [s]'); 





