import os
import numpy as np
from tifffile import tifffile as tif 
from skimage.filters import gaussian
import json

DATA = '../data'
# choose most homogeneous subregion
CROP = [53, 453, 26, 426]

join = os.path.join

rel_path = lambda s: join('..', s)
# prepare dark images and field corrections
DARK = [
        rel_path(join('20200316-CaMKII repeats', 'Dark-images', 'dark-images*')),
        rel_path(join('20200316-CaMKII repeats', 'Dark-images', 'dark-images*'))
       ]
FIELDS = [
              [
               rel_path(join('20200304-CaMKII_data', 'Field_illumination_control-3', 'field control*')),
               rel_path(join('20200304-CaMKII_data', 'Field_illumination_control-4', 'field control*'))
              ],
              [
               rel_path(join('20200316-CaMKII repeats', 'Field-control-1', 'filed-control*')),
               rel_path(join('20200316-CaMKII repeats', 'Field-control-2', 'filed-control*'))
              ]
         ]


if not os.path.exists(DATA):
    os.mkdir(DATA)

def load(paths):
    array = []
    if not isinstance(paths, (list, tuple)):
        paths = [paths]
    for p in paths:
        array.append(tif.TiffSequence(p).asarray())
    return np.concatenate(array, axis=0).squeeze()

dark = [load(d) for d in DARK]
field = [load(f) for f in FIELDS]


mdark, mfield, ffield = [], [], []
for i, (d, f) in enumerate(zip(dark, field)):
    md = np.median(d, axis=0).astype('uint16')
    mdark.append(md)
    f = f - md
    mf = np.median(f, axis=0).astype('uint16')
    mfi = gaussian(mf, sigma=2, preserve_range=True).astype('uint16')
    tif.imsave(join(DATA, f'mdark{i}.tif'), md, imagej=True)
    tif.imsave(join(DATA, f'mfield{i}.tif'), mfi, imagej=True)
    mff = gaussian(mf, sigma=2) # float field
    mff /= mff.max()
    # reduce "range" of field max/min
    mff -= 1
    # this factor was determined by looking at the corrected images in imagej (esp. monomers)
    # 1. perform a 1 pixel radius gaussian filter (to not incorporate too much background)
    # 2. then start with a high threshold and scan downward
    # the thresholded spots should appear randomly across the field of view,
    # not dominated from a particular region. I'm guessing this factor is necessary due to the 
    # nature of EMCCD gain
    mff *= 0.7
    mff += 1
    ffield.append(mff)


# load experimental data
D = {
        'Activated WT 1': {'file': rel_path(join('20200304-CaMKII_data', 'activated-CaMKII', 'act*')), 'field': 0, 'dark': 0},
        'Activated WT 2': {'file': rel_path(join('20200316-CaMKII repeats', 'Activated-WT-1', 'Act*')), 'field': 1, 'dark': 1},
        'Activated WT 3': {'file': rel_path(join('20200316-CaMKII repeats', 'Activated-WT-2', 'Act*')), 'field': 1, 'dark': 1},
        'Activated FA 1': {'file': rel_path(join('20200316-CaMKII repeats', 'Activated-FA', 'Act*')), 'field': 1, 'dark': 1},
        'Activated FA 2': {'file': rel_path(join('FA_repeats', 'Activated FA_repeat1', 'fa-act*')), 'field': 1, 'dark': -1},
        'Activated FA 3': {'file': rel_path(join('FA_repeats', 'Activated FA_repeat2', 'Act*')), 'field': 1, 'dark': -1},
        'Activated KMDN': {'file': rel_path(join('20200316-CaMKII repeats', 'Activated-WT-KMDN', 'Act*')), 'field': 1, 'dark': 1},
        'Activated FA-KMDN': {'file': rel_path(join('20200316-CaMKII repeats', 'Activated-FA-KMDN', 'Act*')), 'field': 1, 'dark': 1},
        'Unactivated WT 1': {'file': rel_path(join('20200304-CaMKII_data', 'unactivated-CaMKII', 'unact*')), 'field': 0, 'dark': 0},
        'Unactivated WT 2': {'file': rel_path(join('20200316-CaMKII repeats', 'Unactivated-WT-1', 'unact*')), 'field': 1, 'dark': 1},
        'Unactivated WT 3': {'file': rel_path(join('20200316-CaMKII repeats', 'Unactivated-WT-2', 'unact*')), 'field': 1, 'dark': 1},
        'Unactivated FA 1': {'file': rel_path(join('20200316-CaMKII repeats', 'Unactivated-FA', 'Unact*')), 'field': 1, 'dark': 1},
        'Unactivated FA 2': {'file': rel_path(join('FA_repeats', 'Unactivated FA_repeat1', 'fa-unact*')), 'field': 1, 'dark': -1},
        'Unactivated FA 3': {'file': rel_path(join('FA_repeats', 'Unactivated FA_repeat2', 'Unact*')), 'field': 1, 'dark': -1},
        'Unactivated KMDN': {'file': rel_path(join('20200316-CaMKII repeats', 'Unactivated-KMDN', 'Unact*')), 'field': 1, 'dark': 1},
        'Unactivated FA-KMDN': {'file': rel_path(join('20200316-CaMKII repeats', 'Unactivated-FA-KMDN', 'Unact*')), 'field': 1, 'dark': 1},
        'Monomers': {'file': rel_path(join('20200304-CaMKII_data', 'Monomer-control', 'Hck*')), 'field': 0, 'dark': 0}
    }


def correct(array, mdark, ffield):
    print(f'\tArray max: {array.max()},    min: {array.min()}')
    array_range = array.max() - array.min()
    array = array - mdark
    array = array / array.max()
    array = array / ffield
    print(f'\tArray max: {array.max():.06f},    min: {array.min():.06f}')
    array = np.clip(array, 0, array.max())
    array = (array * int(array_range / array.max())).astype('uint16')
    print(f'\tArray max: {array.max()},    min: {array.min()}')
    return array

bg = {}
def process(name, attrib):
    print('\nprocessing', name, attrib['file'])
    arr = load(attrib['file']).astype('int64')
    if attrib['dark'] == -1:
        # no dark images were gathered, use best guess
        md = np.ones_like(arr) * np.percentile(arr, 1) * 0.8
    else:
        md = mdark[attrib['dark']]
    arr = correct(arr, md, ffield[attrib['field']])
    arr = arr[:, CROP[0]:CROP[1], CROP[2]:CROP[3]]
    T, Y, X = arr.shape
    arr = arr.reshape(T, 1, 1, Y, X, 1)
    bg[name] = np.median(arr)
    print('\tbackground', bg[name])
    tif.imsave(join(DATA, name+'.tif'), arr, imagej=True)


for name in D:
    process(name, D[name])

with open(join(DATA, 'bg.json'), 'w') as jf:
    json.dump(bg, jf)

with open(join(DATA, 'names.json'), 'w') as jf:
    json.dump(D, jf)

