import numpy as np
import scipy as sp
from scipy.integrate import odeint
import matplotlib.pyplot as plt
from math import floor, log10, isnan


###----------------- TO REPRODUCE FIGURE 7a (based on Equation 1a)

##------------------ PARAMETERS 


Nsp = 100           # Number of species initially
NbRep = 500         # Number of replicate per combination of parameters

d = 0.5             # basal mortality rate

gamma = 1           # gamma value (non linear mortality function analyzed in supplementary analyses)
linearFunction = 0  # if 0: nonlinear mortality function; if 1: linear mortality function (see Figure S1)

FractionSpeciesDensityDependence = 1.0  # Fraction of species undergoing conspecific positive density dependence (see Figure S21)


# range of the species characteristics (following a uniform distribution)

alphaMin = 0.0      # minimum value of alpha  (competition for resources) 
alphaMax = 0.1      # maximum value of alpha 


# sensitivity to changes in the density dependence factor (s in the manuscript; here l)
# /!\ log scale; here, from s = 10^-2 to s= 10^4

lVecMin = -2                # minimum value of s
lVecMax = 4                 # maximum value of s
nbIntervalNumerical = 7     # number of values tested

# initial conditions 

IniY = 1         # "0": random initial densities; "1": initial densities = 1; "2": recurrent invasion (Figure S22)

# characteristics of each simulation

Tmax = 5000      # length of the run
epsCoex = 1e-3   # threshold below which species are declared as extinct






##------------------ DO NOT CHANGE THE CODE BELOW


#------------------ differential equation describing Equation (1) in the manuscript

def dydt(y, t, d, l, alphaMatrix):
    
    # term n^Gamma in nonlinear model (Figure S1)
    Ngamma = [0 for n in y]
    for i in range(len(y)):
        if y[i]<0.0:
            y[i]=0.0
        Ngamma[i] = y[i]**gamma
        if isnan(Ngamma[i])==True:
            Ngamma[i]=0.0

    # D(n)    
    if linearFunction==1:
        deathTerms = [-d*(1-l*y[i]) for i in range(len(y))]
        for i in range(len(deathTerms)):
            if deathTerms[i]>0.0:
                deathTerms[i] = 0.0
    else:
        deathTerms = [-d/(1+l*Ngamma[i]) for i in range(len(y))]

    # Fraction of species undergoing conspecific positive density dependence (see Figure S21) 
    if FractionSpeciesDensityDependence<1.0:
        for i in range(0,int(round(Nsp*(1-FractionSpeciesDensityDependence)))):
            deathTerms[i] = -(d)    
    
    dydt = [1+deathTerms[i] for i in range(len(y))]
    
    # Competition for resources
    for i in range(len(y)):
        dydt = dydt - alphaMatrix[i,] * y[i]
    dydt = [dydt[i]*y[i] for i in range(len(y))]
    
    return dydt




#------------------ other functions

# function used to draw the plot (to sort values of s in the x-axis; does not work for all Python version)
def unique(list1): 
    unique_list = [] 
    for x in list1: 
        if x not in unique_list: 
            unique_list.append(x) 
    return unique_list 

# function used to draw the plot (10^-2, ... 10^4 in the x-axis)
def sci_notation(num, decimal_digits=1, precision=None, exponent=None):
    if not exponent:
        exponent = int(floor(log10(abs(num))))
    coeff = round(num / float(10**exponent), decimal_digits)
    if not precision:
        precision = decimal_digits
    return r"$10^{{{1:d}}}$".format(coeff, exponent, precision)



#------------------ simulations


lVec = np.logspace(lVecMin, lVecMax, nbIntervalNumerical) # values of density dependence factor (s) tested
t = np.linspace(0, Tmax, 2)                               # duration of the run
tInv = np.linspace(0, 200, 2)                             # duration of the run during the invasion phase if IniY=2

lVecPlot = [0] * NbRep * nbIntervalNumerical              # variable where all s values will be stored
NbSpeciesPlot = [0] * NbRep * nbIntervalNumerical         # variable where all remaining species richness will be stored
r_ = 0
for rep in range(NbRep):
    alphaMatrix = np.random.uniform(low=alphaMin,high=alphaMax,size=(Nsp,Nsp))  # competition for resources between each pair of species
    np.fill_diagonal(alphaMatrix,1)                                             # intraspecific competition = 1
    
    # initialization of species densities
    if IniY==0:
        y0=[np.random.uniform() for i in range(Nsp)] 
    elif IniY==1:
        y0=[1] * Nsp
    elif IniY==2:
        y0=[0] * Nsp
    
    # sensitivity analysis on the positive density dependence factor (s; in the code called l)
    for l in lVec:
        if l==0:
            l=0.00001   # because a log scale will be used
            
        y = y0          # initialization of densities
        
        if IniY==2:     # if recurrent invasion
            for sp in range(Nsp):
                y[sp] = 0.001   # initial density
                sol = odeint(dydt, y, tInv, args=(d, l, alphaMatrix))    # simulation until t = end(tInv)
                y = sol[1]
        
        sol = odeint(dydt, y, t, args=(d, l, alphaMatrix))                # simulation until t = Tmax
        
        # count of the number of remaining species
        nbSpSimu = 0
        for sp in range(Nsp):
            if sol[1][sp]>epsCoex:
                nbSpSimu = nbSpSimu+1
                
        # storage of the density dependence factor and the number of remaining species
        lVecPlot[r_] = l
        NbSpeciesPlot[r_] = nbSpSimu
        r_ = r_ + 1



#------------------ plot

# to display the x-axis
dict1 = {}
lUnique = unique(lVecPlot)
newData = 1
for l in lUnique:
    listNbSpOneL = [] 
    for i in range(len(lVecPlot)):
        if lVecPlot[i] == l:
            listNbSpOneL.append(NbSpeciesPlot[i])
    dict1[sci_notation(l,1)] = listNbSpOneL

    
labels, data = dict1.keys(), dict1.values()

# plot
f1 = plt.figure(figsize=(4,3.5))
plt.boxplot(data, patch_artist=True, medianprops = dict(color='black'), flierprops = dict(marker='o', markerfacecolor='black', markersize=1,linestyle='none'), boxprops=dict(facecolor="gainsboro", color="black"),widths = 0.85)
plt.xticks(range(1, len(labels) + 1), labels)
plt.ylim(-3, Nsp+4)
plt.rc('xtick', labelsize=30) 
plt.rc('ytick', labelsize=30) 
plt.xticks(fontsize=11, rotation=0)
plt.yticks([1,50,100], fontsize=10, rotation=0)

# storage of the plot in the folder "Graphes"
f1.savefig('Graphes/simulationMultiSpeciesResb1_Nsp'+str(Nsp)+'_Ini'+str(IniY)+'_Cmax'+str(alphaMax)+'_D'+str(d)+'_gamma'+str(gamma)+'linearFunction'+str(linearFunction)+'Fraction'+str(FractionSpeciesDensityDependence)+'.png', bbox_inches='tight',dpi=300)


