import numpy as np
import scipy as sp
from scipy.integrate import odeint
import matplotlib.pyplot as plt
import math



###----------------- TO REPRODUCE FIGURE 6a-c (based on Equation 2)
###                  take > 48 hours to run; to get faster run, lower the resolution (see parameters below)

##------------------ PARAMETERS 

d = 0.9                         # basal mortality rate
alpha = 0.5                     # symmetric competition for resources

gamma = 1                       # gamma value (non linear mortality function analyzed in supplementary analyses)
linearFunction = 1              # if 0: nonlinear mortality function; if 1: linear mortality function (see Figure S1)

cVecMin = 0                     # minimum value of delta  (difference in mortality rate)
cVecMax = 1                     # maximum value of delta


# sensitivity to changes in the density dependence factor (s in the manuscript; here l)
# /!\ log scale; here, from s = 10^-2 to s= 10^4

lVecMin = -2                    # minimum value of s
lVecMax = 4                     # maximum value of s

# Numerical integration

nbIntervalNumerical = 201       # precision of the sensitivity analysis (number of pixels) 
sizeMarkerNumerical = 2         # size of each pixel

nbIntervalY0 = 20               # range of initial densities value tested (see full numerical method in Appendix S3)
tMax = 5e6                      # length of the run
epsCoex = 1e-5                  # threshold below which species are declared as extinct

freqInv = 1e-12             # initial frequency of the less competitive species to test globality criterion
TBeforeInv = 1000           # burn-in period before included a rare competitor
TInv = 1                    # length of the run before assessing the invasion success

## ## ## ## ## ## ## ## DO NOT CHANGE THE CODE BELOW






#------------------ differential equation describing Equation (2) in the manuscript

def dydt(y, t, d, l, c, alpha):
    N1, N2 = y
    
    if N1<0.0:
        N1=0.0
    if N2<0.0:
        N2=0.0
    
    # term n^Gamma in nonlinear model (Figure S1)
    N1gamma = N1**gamma
    N2gamma = N2**gamma
    if math.isnan(N1gamma)==True:
        N1gamma=0.0
    if math.isnan(N2gamma)==True:
        N2gamma=0.0
    
    # D(n)
    if linearFunction==1:
        deathTerms = [d*(1-l*N1), (d+c*(1-d))*(1-l*N2)]
        if deathTerms[0]<0.0:
            deathTerms[0] = 0.0
        if deathTerms[1]<0.0:
            deathTerms[1] = 0.0
    else:
        deathTerms = [d/(1+l*N1gamma), (d+c*(1-d))/(1+l*N2gamma)]    

    # differential equation
    dydt = [ N1*(1-N1-alpha*N2-deathTerms[0]), N2*(1-alpha*N1-N2-deathTerms[1]) ]
    return dydt




#------------------ Numerical integration

# all the initial conditions tested to investigate global stability criterion
y0Test = np.linspace(1.0,0.0001, nbIntervalY0)
y0Test_1=[0]*nbIntervalY0*nbIntervalY0
y0Test_2=[0]*nbIntervalY0*nbIntervalY0
r_=0
for y1 in y0Test:
    for y2 in y0Test:
        y0Test_1[r_] = y1
        y0Test_2[r_] = y2
        r_ = r_ + 1


t = np.linspace(0, tMax, 2)                                 # duration of the run

cVec = np.linspace(cVecMin, cVecMax, nbIntervalNumerical)   # values of delta tested
lVec = np.logspace(lVecMin, lVecMax, nbIntervalNumerical)   # values of density dependence factor (s) tested

cVecPlot = [0] * nbIntervalNumerical * nbIntervalNumerical  # variable where all delta values will be stored
lVecPlot = [0] * nbIntervalNumerical * nbIntervalNumerical  # variable where all s values will be stored
eqPlot = [0] * nbIntervalNumerical * nbIntervalNumerical    # variable equilibrium state will be stored

tBefore = np.linspace(0, TBeforeInv, 2)                     # duration of the burn-in perion before invasion test
tInvasion = np.linspace(0, TInv, 2)                         # duration of the run during the invasion test

r_ = 0
for c in cVec:
    print(c)
    for l in lVec:
        if l==0:
            l=0.00001
        if c==0:
            c=0.00001

        eqState = 0 # extinction (if so, eqState will remain = 0)
    
        found = 0
        rt_ = 0 
        while found==0 and rt_<len(y0Test_1):
            y0 = [y0Test_1[rt_], y0Test_2[rt_]]                 # initialization of densities
            yb = y0
            
            sol = odeint(dydt, yb, t, args=(d, l, c, alpha))    # simulation until t = Tmax
            yb = sol[1]
                
            if yb[0]>epsCoex and yb[1]>epsCoex:                 # condition for coexistence
                eqState = 5
                found = 1
                
            rt_ = rt_ + 1
            
        # test for global attractor
        if eqState == 5: # is coexistence a global attractor (if so, eqState will remain = 5)?
            y0 = [1,0]
            sol = odeint(dydt, y0, tBefore, args=(d, l, c, alpha))
            y0=sol[1]
            y0[1] = freqInv
            sol = odeint(dydt, y0, tInvasion, args=(d, l, c, alpha))
            if sol[1][1]<y0[1]:
                eqState = 4 # local attractor (eqState will remain = 4)
        
        # storage of the density dependence factor and the number of remaining species
        cVecPlot[r_] = c
        lVecPlot[r_] = l
        eqPlot[r_] = eqState
        r_ = r_ + 1





#------------------ plot

cVecPlot.append(-1000)
lVecPlot.append(-1000)
eqPlot.append(5)
cVecPlot.append(-1000)
lVecPlot.append(-1000)
eqPlot.append(4)
cVecPlot.append(-1000)
lVecPlot.append(-1000)
eqPlot.append(0)

# plot
f1 = plt.figure(figsize=(5,4.5))
plt.scatter(cVecPlot, lVecPlot, c=eqPlot, s=sizeMarkerNumerical, lw = 0 ,marker="s", cmap='gray')
plt.xlim(0,1)
plt.ylim(min(lVec), max(lVec))
plt.yscale('log')
plt.rc('xtick', labelsize=30) 
plt.rc('ytick', labelsize=30) 
plt.xticks(fontsize=13, rotation=0)
plt.yticks(fontsize=14, rotation=0)

# storage of the plot in the folder "Graphes"
f1.savefig('Graphes/sensitivityNumericalPred_D'+str(d)+'_A'+str(alpha)+'_gamma'+str(gamma)+'linearFunction'+str(linearFunction)+'.png', bbox_inches='tight',dpi=300)

