function [PReject, FPR, RR] = OutlierRejection(...  
          alpha1, alpha2, power2, baserate, k, d, n, twosample, pOutlier, sdOutlier, cThresholds, nIterations)

% Assumptions & limitations:
%   the twosample parameter is not used; only twosample is implemented
%   the user is responsible for control of the rng seed

if baserate < 1
   AlphaInflation = PRejectHo(alpha1,0,k,n,pOutlier, sdOutlier, cThresholds, nIterations);
else
   AlphaInflation = 0;
end
if baserate > 0
   PowerInflation = PRejectHo(alpha1,d,k,n,pOutlier, sdOutlier, cThresholds, nIterations);
else
   PowerInflation = 0;
end

FPR = FalsePositiveRate(baserate,AlphaInflation,PowerInflation);
RR = ReplicationRate(baserate,AlphaInflation,alpha2,PowerInflation,power2);
PReject = PowerInflation;

end


function [PReject] = PRejectHo(alpha,d,k,n,p,s,c, nIterations)

parfor i=1:nIterations  %parallel loop computation
    
    %Sample values for experimental and control group
    E = SimSample(n,d,p,s);
    C = SimSample(n,0,p,s);
      
    Reject(i)=0;
    
    %for k<4 try t-tests with Z-thresholds
    for kk = 1:min(k,3)
        pvalue = ZThreshold(E,C,c(kk));
        if pvalue < alpha
            Reject(i) = 1;
            break
        end
    end
    
    %for k=4 try t-test with Tukey's fences
    if k>3 && Reject(i) == 0
        pvalue = Tukey(E,C);
        if pvalue < alpha
            Reject(i)=1;
        end
    end
    
    %for k=5 try U-test
    if k>4 && Reject(i) == 0
         pvalue = ranksum(E,C,'tail','right');
        if pvalue < alpha
            Reject(i)=1;
        end
    end
    
end

%Probability of rejecting Ho
PReject=mean(Reject);

end

function [X] = SimSample(n,d,p,s)
NO = binornd(n,p);           %number of outliers
O = normrnd(0,s,[1,NO]);     %outlier contamination
T = normrnd(d,1,[1,n-NO]);   %true values
X = [O,T];                   %sample values
end

function [p] = ZThreshold(E,C,c)
E1 = Eliminate(E,c);  %eliminate outliers from E
C1 = Eliminate(C,c);  %eliminate outliers from C
[~,p] = ttest2(E1,C1,'Tail','right'); %test difference
end

function [Y] = Eliminate(X,c)
Z = zscore(X);     %convert observations to z-values
Y = X(abs(Z)<c);   %eliminate extreme observations
end

function [p] = Tukey(E,C)
E1=TukeyFence(E);
C1=TukeyFence(C);
[~,p] = ttest2(E1,C1,'Tail','right'); %test difference
end

function [Z]=TukeyFence(X)
k=1.5; %indicates an outlier according to Tukey
Q1=quantile(X,.25);
Q3=quantile(X,.75);
l=Q1-k*(Q3-Q1);
u=Q3+k*(Q3-Q1);
Y=X(X > l); %Take observations above l  and
Z=Y(Y < u); %take observations below u
end

function [FPR] = FalsePositiveRate(baserate,alpha1,power1)
FPR=(1-baserate).*alpha1./((1-baserate).*alpha1+baserate.*power1);
end

function [RR] = ReplicationRate(baserate,alpha1,alpha2,power1,power2)
RR = (baserate.*power1*power2 + (1-baserate).*alpha1*alpha2)./(baserate.*power1 + (1-baserate).*alpha1);
end
