% figure1.m reproduces Figure 1D,F using:
% - parasol_scores: a matrix of reconstruction performance scores (correlation)
% for each of the 15 recordings. Each cell contains a matrix with
% dimensions (condition x image x trial), where conditions refers to which
% parasol cells' responses were included in the reconstruction (ON+OFF, ON, or OFF).
% - RF_size: a vector of average parasol RF size (radius, in microns) for each recording

clear
load('figure1.mat');
figure;

n_exps = size(parasol_scores,1);
n_test_ims = 150;

% calculate trial averaged scores
trial_averaged_scores = zeros(n_exps, n_test_ims);
variations_across_trials = zeros(n_exps, n_test_ims);
n = zeros(n_exps, 20);
for i= 1:n_exps
    trial_averaged_scores(i,:) = mean(parasol_scores{i}(1,:,:), 3); % average scores across trials
    [n(i,:),edges] = histcounts(trial_averaged_scores(i,:), 0:0.05:1); % score distribution for each experiment
    variations_across_trials(i,:) = std(parasol_scores{i}(1,:,1:5), [], 3); % variation across trials

end
[total_n,~] = histcounts(trial_averaged_scores(:),edges); % score distribution across all experiments

% FIGURE 1D: Reconstruction performance
subplot(1,3,1)
[n_tot,edges] = histcounts(trial_averaged_scores(:), 0:0.05:1);
centers = (edges(2:end) + edges(1:(end-1)))/2;
fill([centers centers(end) centers(1)], [n_tot/(150*15) 0 0], 0.8*[1 1 1], 'LineWidth', 1.1, 'EdgeColor', 'k', 'FaceAlpha', 0);

% FIGURE 1F: Reconstruction performance for each experiment, 
% ordered by RF size
subplot(1,3,2)
[~,score_order] = sort(RF_size, 'descend');
hold on
for i = 15:-1:1
    fill([centers centers(end) centers(1)], [n(score_order(i),:)/(300) 0 0]+i*(0.05), 0.8*[1 1 1], 'LineWidth', 1.1, 'EdgeColor', 'k', 'FaceAlpha', 0.7);
end
axis off

% FIGURE 1F: RF size vs. performance
subplot(1,3,3)
plot(RF_size, mean(trial_averaged_scores,2), '.')
ylim([0.65 0.85])

% Calculations
mean_scores = mean(trial_averaged_scores(:));
std_scores = std(trial_averaged_scores(:));
fprintf('Reconstruction performance across the set of test images: %0.3g +/- %0.3g\n', mean_scores, std_scores);
fprintf('Trial-to-trial variation: %0.3g\n', mean(variations_across_trials(:)));
fprintf('RF size vs performance correlation: %0.1g\n',corr(RF_size, mean(trial_averaged_scores,2)));


