# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# Code to generate Figure 1 of Price, Shearer, et al. (2020)
# 
# Authors: David J Price & Freya M Shearer
# Date: 20 April, 2020
# 
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

## Figure 1 ----

library(ggplot2)
library(ggrepel)
library(lubridate)

theme_set(cowplot::theme_cowplot())

cases.dat <- read.csv("cases.csv")

cases.dat <- cases.dat %>% mutate(date = as_date(date))

dd = ymd(c("2020-02-01", "2020-03-01", "2020-03-05", "2020-03-11", "2020-03-15", "2020-03-16", "2020-03-19", "2020-03-24", "2020-03-27", "2020-03-29"))
dd.height <- sapply(X = dd, function(x){sum(cases.dat$cases[which(cases.dat$date==x)])})
shift.end <- c(35, 30, 50, 55, 70, 70, 75, 25, 40, 35)
shift.point <- rep(10, length(dd))
event.dates <- data.frame("date"=dd, "y" = dd.height + shift.end, "yend" = dd.height + shift.point,
                          "event.lab"=c("1/2: Travel restrictions on arrivals\nfrom mainland China",
                                        "1/3: Travel restrictions extended to Iran",
                                        "5/3: Travel restrictions extended to South Korea",
                                        "11/3: Travel restrictions extended to Italy",
                                        "15/3: Self-quarantine advice\nextended to all international arrivals" ,
                                        "16/3: Public gatherings\nlimited to 500 people",
                                        "19/3: Borders closed to\nnon-citizens and non-residents",
                                        "24/3: Stay home except for\nessential activities",
                                        "27/3: Mandatory quarantine\nof all international arrivals",
                                        "29/3: Public gatherings\nlimited to 2 people"))

manual.cols <- c("Overseas acquired"="#440154FF", "Locally acquired"= "#31688EFF", "Unknown origin"="#1B9E77")


ggplot(cases.dat, aes(x=date, y=cases, fill=import_status)) + geom_col(position=position_stack()) +
  scale_y_continuous("Daily New Confirmed Cases", limits=c(0, 500)) + 
  scale_x_date("Date", date_labels = "%d-%b", breaks = seq.Date(min(cases.dat$date)-1,max(cases.dat$date), by="1 week")) + 
  geom_segment(data=event.dates, aes(x =date, xend = date, y = y, yend = yend), inherit.aes = F, 
               arrow=arrow(type = "open", length = unit(x = 0.1, units = "cm")), lineend = "butt", linejoin = "round") +
  geom_text_repel(data=event.dates, aes(x=date, y=y, label = event.lab),
                           nudge_y = 15, nudge_x = c(rep(-9,6), -10, -5, 5, 6), inherit.aes = F , direction = "both") +
  scale_fill_manual("",values=manual.cols) +
  theme(text=element_text(size=16), legend.position = "top")





#### Supplementary Figure 1 ----
# States sorted by population
cases.dat <- cases.dat %>% mutate(region.ord = factor(region, levels=c("NSW","VIC","QLD","WA","SA","TAS","ACT","NT")))

ggplot(cases.dat, aes(x=date, y=cases, fill=import_status)) + geom_col(position=position_stack()) +
  scale_y_continuous("Daily New Reported Cases", breaks = scales::pretty_breaks(4)) + 
  scale_x_date("Date", date_labels = "%d-%b", breaks = seq.Date(min(cases.dat$date)-1,max(cases.dat$date),by="2 weeks")) + 
  scale_fill_manual("",values=manual.cols) +
  facet_wrap(region.ord~., ncol=1, scales = "free_y")+
  theme(text=element_text(size=14), legend.position = "top")

