# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# Code to generate Figure 2 of Price, Shearer, et al. (2020)
# 
# Authors: David J Price & Freya M Shearer
# Date: 20 April, 2020
# 
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


library(ggplot2)
# devtools::install_github("epiforecasts/EpiNow")
library(EpiNow)

cases.dat <- read.csv("cases.csv")
cases.dat <- cases.dat %>% mutate(date=ymd(date), import_status = recode(import_status, 
                                                                        "Unknown origin"="local",  # conservative approach assuming all unknown cases are locally acquired
                                                                        "Locally acquired"="local",
                                                                        "Overseas acquired"="imported")) %>% 
              group_by(date,region,import_status) %>% tally(wt=cases) %>% rename(cases=n) %>% ungroup()

# Load gamma distributed reporting delays
delay_samples <- read.csv("sampled_report_delay.csv")

# Create a "linelist" from 
linelist <- cases.dat %>% group_by(date,import_status,region) %>% 
              expand(count = seq(1:cases)) %>% select(-count) %>% 
              ungroup() %>% 
              mutate(report_delay = sample(x = delay_samples$x, size = nrow(.), replace = T)) %>% 
              mutate(date_onset = ymd(date) - report_delay) %>% 
              rename(date_confirm = date)
  
  
EpiNow::regional_rt_pipeline(cases=cases.dat, linelist=linelist, case_limit = 20)



## Figure 2 ----


date <- "2020-04-14"

results.folder <- "~/results"

states <- list.files(results.folder)


bigr_estimates <- NULL
for (ss in states){
  bigr_estimates <- rbind(bigr_estimates, cbind(readRDS(paste0(results.folder,ss,"/",date,"/bigr_estimates.rds")), "State"=ss))
}

state.labels=c("NSW"="New South Wales (NSW)", "QLD"="Queensland (QLD)", "SA"="South Australia (SA)", 
               "VIC"="Victoria (VIC)", "WA"="Western Australia (WA)")

plot.start.date <- max((aggregate(bigr_estimates$date, by=list(bigr_estimates$State),min))$x)
plot.end.date <- max((aggregate(bigr_estimates$date, by=list(bigr_estimates$State),max))$x)

plot.start.date <- ymd("2020-03-01")

ddd <- seq.Date(plot.start.date, plot.end.date, by="2 week")
dd.ticks <- seq.Date(plot.start.date, plot.end.date, by="1 week")

plot_bigr <- vector("list", length=length(states))
k<-1
for (ss in states){
  
  plot_bigr[[k]] <- bigr_estimates %>% 
    dplyr::filter(State == ss) %>%
    dplyr::filter(type %in% "nowcast") %>%
    EpiNow::plot_confidence(plot_median = FALSE) +
    ggplot2::theme(legend.position = "none") +
    ggplot2::labs(y = "", x = "",title = state.labels[ss]) +
    ggplot2::scale_y_continuous(name = expression(R[eff]), position = "right") +
    ggplot2::scale_x_date(labels=NULL, breaks=dd.ticks, limits=c(plot.start.date, plot.end.date)) +
    ggplot2::geom_hline(yintercept = 1, linetype = 2) +
    ggplot2::geom_vline(xintercept = ymd("2020-03-15"), linetype = 1, colour="grey80") +
    ggplot2::geom_vline(xintercept = ymd("2020-03-27"), linetype = 1, colour="grey80") +
    ggplot2::coord_cartesian(xlim=c(plot.start.date, plot.end.date), ylim=c(0,2))+
    ggplot2::expand_limits(y = 0) +
    ggplot2::theme(axis.text.x = element_text(angle=0),
                   axis.title.y.right = element_text(angle=0, vjust=0.5),
                   text=element_text(size=14),
                   plot.margin = margin(t=2, r=2, b=0 , l=4))
  
  if (k == length(states)){
    plot_bigr[[k]] <- plot_bigr[[k]] + ggplot2::scale_x_date(breaks=dd.ticks, 
                                                             labels=c(rbind(paste0(day(ddd),"-",month(ddd, abbr = T, label = T)),c("","",""))),
                                                             limits=c(plot.start.date, plot.end.date))
  }
  
  k<-k+1
}

(all.bigr <- do.call("plot_grid", c(plot_bigr, ncol=2)))




## Supplementary Figures 3-5 ----


linelist.import <- linelist %>% filter(import_status=="imported")
linelist.local <- linelist %>% filter(import_status=="local")

# Sensitivity probabilities: these values represent probability of quarantine
pp <- matrix(c(0.1, 0.5, 0.99,
               0.2, 0.5, 0.99,
               0.5, 0.8, 0.99), ncol=3, byrow=TRUE)

# Scenario i corresponds to row i of pp matrix above as to what proportion of 
## individuals were quarantined
scenario <- 1
set.seed(scenario)

cases.import <- cases.dat %>% filter(import_status=="imported") %>% ungroup() %>% group_by(date, region) 

# pp represents probability of not being quarantined (i.e., contributing to transmission)
cases.import$pp <- 1 - pp[scenario,1]
cases.import$pp[cases.import$date>ymd("2020-03-15")] <- 1 - pp[scenario,2]
cases.import$pp[cases.import$date>ymd("2020-03-27")] <- 1 - pp[scenario,3]

cases.import <- cases.import %>% mutate(sub.cases = rbinom(n(), cases,  pp))

new.import.linelist <- NULL
for (rr in unique(import.linelist$region)){
  for (dd in unique(import.linelist$date_confirm)){
    
    tmp.import <- linelist.import %>% filter(date_confirm == dd, region == rr)
    tmp.sample <- cases.import %>% filter(date == dd, region == rr)
    
    tmp.import.ll <- sample_n(tbl = tmp.import, size = tmp.sample$sub.cases, replace = F)
    
    new.import.linelist <- rbind(new.import.linelist, tmp.import.ll)
  }
}


# Re-create the linelist with the subset data
sub.linelist <- rbind(local.linelist, new.import.linelist)  

# Re-generate case dataframe
sub.cases <- linelist %>% select(date=date_confirm, import_status, region) %>%
  group_by(date, import_status, region) %>%
  count() %>% rename(cases = n)





EpiNow::regional_rt_pipeline(cases=sub.cases, linelist=sub.linelist, results=paste0("results_sensitivity",scenario), case_limit = 20)

save("sub.linelist","sub.cases", file=paste0("/results_sensitivity",scenario,"/subsampled_data.Rdata"))






