# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# Code to generate Figure 3 of Price, Shearer, et al. (2020)
# 
# Authors: David J Price & Freya M Shearer
# Date: 20 April, 2020
# 
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

library(ggplot2)
library(readr)
library(lubridate)

source('icu_ward_occupancy_model.R')

# Load data
cases.dat <- read.csv("cases.csv")
cases.dat <- cases.dat %>% mutate(date=ymd(date))

age.specific.cases <- read.csv("cases_by_age.csv")
age.specific.probs <- read.csv("ward_icu_by_age.csv")

projected.dat <- read.csv("forecast_14Apr.csv")
projected.dat <- projected.dat %>% mutate(date=ymd(date)) %>% filter(date>=ymd("2020-04-14") & date<ymd("2020-04-29"))

obs.dat <- read.csv("icu_ward_data.csv")
obs.dat <- obs.dat %>% mutate(date=ymd(date))


# Set occupancy model parameters
nsims <- 20  # per replicate
days.forward <- 50

start.date <- min(cases.dat$date)


# Summarise cases

# For each region
cases.dat <- cases.dat %>% ungroup() %>% mutate(region=NULL) %>% group_by(date) %>% tally(wt=cases) %>% rename(cases=n)
# merge with projections
cases.dat <- full_join(cases.dat, projected.dat, by="date") %>%  
              rename("projected1"=Median, "projected2"=CI50_L, "projected3"=CI95_L, "projected4"=CI50_U, "projected5"=CI95_U) %>% 
              replace_na(list(cases=0,projected1=0,projected2=0,projected3=0,projected4=0,projected5=0))


occupancy.dat <- ward.icu.occupancy(cases.dat, age.specific.cases, age.specific.probs, nsims, days.forward)


plot.start.date <- ymd("2020-03-20")
plot.end.date <- ymd("2020-04-28")
proj.start.date <- min(projected.dat$date)

ddd <- seq.Date(plot.start.date, plot.end.date, by="1 week")


ward.fill <- "darkorchid"
icu.fill <- "green4"

ward.plot <- occupancy.dat %>%
  filter(src=="Ward Occupancy") %>% 
  ggplot() +
  geom_vline(aes(xintercept=proj.start.date), lty=2) +
  geom_ribbon(aes(x=date, ymin=lower, ymax=upper), alpha=0.4, fill=ward.fill)+
  geom_ribbon(aes(x=date, ymin=bottom, ymax=top), alpha=0.2, fill=ward.fill)+
  geom_point(data=obs.dat, inherit.aes = F, aes(x=date, y=ward)) +
  scale_y_continuous("Number Occupied Beds") +
  scale_x_date("Date", date_labels = "%d-%b", labels = ddd,
               breaks = ddd, limits = c(plot.start.date, plot.end.date)) +
  ggtitle("Ward Occupancy") +
  theme(legend.position = "none",
        axis.text.x = element_text(angle=0),
        text=element_text(size=18),
        axis.text=element_text(size=14),
        strip.background =element_rect(fill="white"))

icu.plot <- occupancy.dat %>%
  filter(src=="ICU Occupancy") %>%
  ggplot() + 
  geom_vline(aes(xintercept=proj.start.date), lty=2) +
  geom_ribbon(aes(x=date, ymin=lower, ymax=upper), alpha=0.4, fill=icu.fill)+
  geom_ribbon(aes(x=date, ymin=bottom, ymax=top), alpha=0.2, fill=icu.fill)+
  geom_point(data=obs.dat, inherit.aes = F, aes(x=date, y=ICU)) +
  scale_y_continuous("Number Occupied Beds") +
  scale_x_date("Date", date_labels = "%d-%b", labels = ddd,
               breaks = ddd, limits = c(plot.start.date, plot.end.date)) +
  ggtitle("ICU Occupancy") +
  theme(legend.position = "none",
        axis.text.x = element_text(angle=0),
        text=element_text(size=18),
        axis.text=element_text(size=14),
        strip.background =element_rect(fill="white"))

(occ.plot <- plot_grid(ward.plot, icu.plot, nrow = 1))



