ward.icu.occupancy <- function(cases.dat, age.specific.cases, age.specific.probs, nsims, days.forward){
  
  start.date <- min(cases.dat$date)
  end.date <- as_date(ymd(max(cases.dat$date)) + as.difftime(days.forward, unit="days"))
  
  hosp.occupied.beds.df <- data.frame("Date"=as_date(start.date:end.date))
  icu.occupied.beds.df <- hosp.occupied.beds.df
  
  hosp.occupied.beds.conf.df <- hosp.occupied.beds.df
  icu.occupied.beds.conf.df <- hosp.occupied.beds.df
  
  # Grab number of replicates from cases.dat
  replicates <- max(parse_number(colnames(cases.dat)[3:ncol(cases.dat)]))
  
  ward.pars <- c(10.331915, 1.064662) # Gamma with 2.5-97.5 quantiles ~(5.35, 18.65)
  icu.pars <- c(7.189935, 1.947166) # Gamma with 2.5-97.5 quantiles ~(5.71, 25.94)
  
  # Mean number of days in ward bed prior to admission to ICU (Poisson distributed)
  icu.ward.stopover <- 2.5
  
  set.seed(1)
  for(j in 1:replicates){
    for (i in 1:nsims){
      
      hosp.load.list <- vector("list", length=nrow(hosp.occupied.beds.df))
      icu.load.list <- vector("list", length=nrow(icu.occupied.beds.df))
      names(hosp.load.list) <- as.character(hosp.occupied.beds.df$Date)
      names(icu.load.list) <- names(hosp.load.list)
      
      proj.hosp.load.list <- hosp.load.list
      proj.icu.load.list <- icu.load.list
      
      for (dd in as.character(cases.dat$date)){
        # Cases by age of observed
        cases.by.age <- as.numeric(rmultinom(1, as.numeric(cases.dat[which(cases.dat$date==dd), "cases"]), prob = age.specific.cases$p))
        
        # Simulated proportion of hosp/icu/deaths from observed cases
        hosp.by.age <- sapply(X=1:length(cases.by.age), 
                              FUN=function(x){ rbinom(size = cases.by.age[x], n = 1, prob = runif(n = 1, min = age.specific.probs$ph.l[x], max = age.specific.probs$ph.u[x]))})
        icu.by.age <- sapply(X=1:length(cases.by.age), 
                             FUN=function(x){ rbinom(size = cases.by.age[x], n = 1, prob = runif(n = 1, min = age.specific.probs$pi.l[x], max = age.specific.probs$pi.u[x]))})
        
        # Cases by age of projected
        proj.cases.by.age <- as.numeric(rmultinom(1, as.numeric(cases.dat[which(cases.dat$date==dd), paste0("projected",j)]), prob = age.specific.cases$p))
        
        # Simulated proportion of hosp/icu/deaths from projected cases
        proj.hosp.by.age <- sapply(X=1:length(proj.cases.by.age), FUN=function(x){ rbinom(size = proj.cases.by.age[x], n = 1, prob = runif(n = 1, min = age.specific.probs$ph.l[x], max = age.specific.probs$ph.u[x]))})
        proj.icu.by.age <- sapply(X=1:length(proj.cases.by.age), FUN=function(x){ rbinom(size = proj.cases.by.age[x], n = 1, prob = runif(n = 1, min = age.specific.probs$pi.l[x], max = age.specific.probs$pi.u[x]))})
        
        
        # Ward bed dates occupied for observed cases
        hosp <- round(rgamma(n = sum(hosp.by.age), shape=ward.pars[1],  scale=ward.pars[2])) # cases that are just hospitalised
        hosp.to.icu <- rpois(n = sum(icu.by.age), icu.ward.stopover) # cases that are subsequently admitted to ICU after some time in ward bed
        hosp.length.stay <- c(hosp, hosp.to.icu)
        hosp.end.date <- as_date(ymd(dd)+hosp.length.stay, origin=lubridate::origin)
        
        icu.length.stay <- round(rgamma(n = sum(icu.by.age), shape=icu.pars[1]  , scale=icu.pars[2]))
        icu.end.date <- as_date(ymd(dd)+hosp.to.icu+icu.length.stay, origin=lubridate::origin)
        
        ward.icu.end.date <- icu.end.date + rpois(sum(icu.by.age), icu.ward.stopover)
        
        # Store ward occupied days
        if (length(hosp.end.date)==0){
          hosp.occupied.dates <- "NA"
        } else{ 
          if (length(hosp.end.date)!=0 & length(icu.end.date)==0){
            hosp.occupied.dates <-  as_date(unlist(sapply(X = hosp.end.date, 
                                                          FUN = function(x){ ymd(dd):x })), origin=lubridate::origin)
          } 
          else{
            hosp.occupied.dates <-  c(as_date(unlist(sapply(X = hosp.end.date, 
                                                            FUN = function(x){ ymd(dd):x })), origin=lubridate::origin),
                                      as_date(unlist(mapply(x = icu.end.date, y = ward.icu.end.date, 
                                                            FUN = function(x,y){ x:y })), origin=lubridate::origin))
          }
        }
        hosp.load.list[[as.character(dd)]] <- hosp.occupied.dates
        
        # Store icu occupied days
        if (length(icu.end.date)==0){
          icu.occupied.dates <- "NA"
        } else{
          icu.occupied.dates <-  as_date(unlist(mapply(x = icu.end.date, y=hosp.to.icu, 
                                                       FUN = function(x,y){ (ymd(dd)+y):x })), origin=lubridate::origin)
        }
        icu.load.list[[as.character(dd)]] <- icu.occupied.dates
        
        # As above, but for projected
        proj.hosp <- round(rgamma(n = sum(proj.hosp.by.age), shape=ward.pars[1],  scale=ward.pars[2])) # projected cases that are just hospitalised
        proj.hosp.to.icu <- rpois(n = sum(proj.icu.by.age), icu.ward.stopover) #  projected cases that are subsequently admitted to ICU after some time in ward bed
        proj.hosp.length.stay <- c(proj.hosp, proj.hosp.to.icu)
        proj.hosp.end.date <- as_date(ymd(dd)+proj.hosp.length.stay, origin=lubridate::origin)
        
        proj.icu.length.stay <- round(rgamma(n = sum(proj.icu.by.age), shape=icu.pars[1], scale=icu.pars[2]))
        proj.icu.end.date <- as_date(ymd(dd)+proj.hosp.to.icu+proj.icu.length.stay, origin=lubridate::origin)
        
        proj.ward.icu.end.date <- proj.icu.end.date + rpois(sum(proj.icu.by.age), icu.ward.stopover)
        
        # Store projected ward occupied days
        if (length(proj.hosp.end.date)==0){
          proj.hosp.occupied.dates <- "NA"
        } else{ if(length(proj.hosp.end.date)!=0 & length(proj.icu.end.date)==0){
          proj.hosp.occupied.dates <-  as_date(unlist(sapply(X = proj.hosp.end.date, 
                                                             FUN = function(x){ ymd(dd):x })), origin=lubridate::origin)
        } else{
          proj.hosp.occupied.dates <-  c(as_date(unlist(sapply(X = proj.hosp.end.date, 
                                                               FUN = function(x){ ymd(dd):x })), origin=lubridate::origin),
                                         as_date(unlist(mapply(x = proj.icu.end.date, y = proj.ward.icu.end.date, 
                                                               FUN = function(x,y){ x:y })), origin=lubridate::origin))
          }
        }
        proj.hosp.load.list[[as.character(dd)]] <- proj.hosp.occupied.dates
        
        # Store projected icu occupied days
        if (length(proj.icu.end.date)==0){
          proj.icu.occupied.dates <- "NA"
        } else{
          proj.icu.occupied.dates <-  as_date(unlist(mapply(x = proj.icu.end.date, y=proj.hosp.to.icu, 
                                                            FUN = function(x,y){ (ymd(dd)+y):x })), origin=lubridate::origin)
        }
        proj.icu.load.list[[as.character(dd)]] <- proj.icu.occupied.dates
        
      }
      
      # Hospitalisations df
      hosp.counts <- table(unlist(hosp.load.list))
      hosp.counts <- hosp.counts[names(hosp.counts)!="NA"]
      hosp.occupied.beds.sim <- data.frame("Date"=as_date(as.numeric(names(hosp.counts))), "count"=as.numeric(hosp.counts))
      
      proj.hosp.counts <- table(unlist(proj.hosp.load.list))
      proj.hosp.counts <- proj.hosp.counts[names(proj.hosp.counts)!="NA"]
      proj.hosp.occupied.beds.sim <- data.frame("Date"=as_date(as.numeric(names(proj.hosp.counts))), "count"=as.numeric(proj.hosp.counts))
      
      hosp.occ.beds.tmp <- merge(hosp.occupied.beds.sim, proj.hosp.occupied.beds.sim, by="Date", all=T) %>%
        replace_na(list(count.x=0,count.y=0)) %>%
        mutate(total = count.x+count.y)
      
      hosp.occupied.beds.df <- suppressWarnings(merge(hosp.occupied.beds.df, hosp.occ.beds.tmp[,c("Date","total")], by="Date", all=T))
      
      # ICU df
      icu.counts <- table(unlist(icu.load.list))
      icu.counts <- icu.counts[which(names(icu.counts)!="NA")]
      icu.occupied.beds.sim <- data.frame("Date"=as_date(as.numeric(names(icu.counts))), "count"=as.numeric(icu.counts))
      
      proj.icu.counts <- table(unlist(proj.icu.load.list))
      proj.icu.counts <- proj.icu.counts[names(proj.icu.counts)!="NA"]
      proj.icu.occupied.beds.sim <- data.frame("Date"=as_date(as.numeric(names(proj.icu.counts))), "count"=as.numeric(proj.icu.counts))
      
      icu.occ.beds.tmp <- merge(icu.occupied.beds.sim, proj.icu.occupied.beds.sim, by="Date", all=T) %>% 
        replace_na(list(count.x=0,count.y=0)) %>% 
        mutate(total = count.x+count.y)
      
      icu.occupied.beds.df <- suppressWarnings(merge(icu.occupied.beds.df, icu.occ.beds.tmp[,c("Date","total")], by="Date", all=T))
      
    }
  }
  
  names(hosp.occupied.beds.df) <- c("Date", 1:(nsims*replicates))
  hosp.occ.bed.df <- reshape2:::melt.data.frame(hosp.occupied.beds.df, id.vars="Date")
  hosp.occ.bed.df <- hosp.occ.bed.df[complete.cases(hosp.occ.bed.df),]
  
  names(icu.occupied.beds.df) <- c("Date", 1:(nsims*replicates))
  icu.occ.bed.df <- reshape2:::melt.data.frame(icu.occupied.beds.df, id.vars="Date")
  icu.occ.bed.df <- icu.occ.bed.df[complete.cases(icu.occ.bed.df),]
  
  # Generate summary statistics on ward occupancy
  hosp.dat <- hosp.occ.bed.df %>% 
    group_by(Date) %>% 
    summarise(median = median(value), 
              lower = quantile(value, 0.25), 
              upper = quantile(value, 0.75),
              bottom = quantile(value, 0.025), 
              top = quantile(value, 0.975))
  
  # Generate summary statistics on ICU occupancy
  icu.dat <- icu.occ.bed.df %>% 
    group_by(Date) %>% 
    summarise(median = median(value), 
              lower = quantile(value, 0.25), 
              upper = quantile(value, 0.75),
              bottom = quantile(value, 0.025), 
              top = quantile(value, 0.975))
  
  hosp.dat$src <- "Ward Occupancy"
  icu.dat$src <- "ICU Occupancy"
  
  all.dat <- rbind(hosp.dat, icu.dat)
  all.dat$src <- factor(all.dat$src, levels = unique(all.dat$src), ordered = T)
  
  all.dat <- all.dat %>% rename(date=Date)
  
  return(all.dat)
  
}