%
% SCRIPT TO POSTPROCESS & VISUALIZE THE RESULTS OF THE ORGAN, CELLULAR AND 
% MOLECULAR LEVEL ANALYSIS OF A 3D IMAGE STACK FOLLOWING THE PROVIDED TUTORIAL.
%
% The script is divided into 4 parts and prompts the user to introduce
% a number in the Command Window upon execution:
%
% 1. PLOT ORGAN-LEVEL RESULTS ALONG PROXIMODISTAL (PD) AXIS:
%    + cross-sectional area (CSA) of slices
%    + circularity of CSA in slices (R1/R2 and Imin/Imax)
%
% 2. PLOT CELLULAR-LEVEL RESULTS ALONG PD AXIS: 
%    + #cells per slice 
%    + #cells / slice volume
%    + volume of cells / slice volume
%    + slice volume (for verification)
%    + distribution of all object volumes in file (for verification)
%    + distribution of object volumes selected for plotting (for verification)
%
%
% 3. PLOT MOLECULAR-LEVEL RESULTS ALONG PD AXIS: 
%    + mean intensity
%    + histogram
%
% 4. PLOT 3D views of organ and cell surfaces, including slicing effect
%    + 3D plot of organ surface 
%    + 3D ploint cloud of organ surface
%    + 3D plot of cell surfaces 
%    + 3D ploint cloud of cell surfaces
%
% GENERAL REQUIREMENTS:
%
% - Install Matlab plugin MinGW-w64 Compiler to run the "mex" command
% - The following toolboxes are needed:
%    + geom3d
%    + Computer Vision Toolbox
% - The following files must be in the same folder as this script:
%   For PART 1:
%    + .csv file with list of BoneJ Slice Geometry results for organ slices
%   For PART 2:
%    + Excel file with cell's center of mass from 3D Objects Counter
%   For PART 3:
%    + .csv file with list of ImageJ script results for pixel analysis
%   For PART 4:
% 	 + stlread.m
%    + intersectPlaneSurf.m
%    + IntersectPlaneTriangle.cpp
%    + .stl file with segmented organ surface
%    + .stl file with segmented cell surfaces
%
% - Please update the file names in "USER INPUT DATA" sections.
% - If a folder named "results" exists, its contents will be rewritten.
%
%  Some things to be careful about:
% - If the .csv headers are different to the original files, these must be
%   changed in the code when the readtable function is called.
% - 3D Objects Counter provided the z-dimension data in pixels, not
%   microns for my specific processing pipeline. I double-checked the
%   image stack used as input for 3D objects counter, and the properties
%   are correct. The (temporary) solution I found is to multiply the z-dim
%   of the CoM by the pixel resolution in the Zdim.
%
% Script written by E.Comellas, with input from J. Garnier and M. Joetzjer
% Northeastern University, 2020.
% Contact: e.comellas@northeastern.edu

%% Initiate
% Close all windows
close all; 

%% USER INPUT DATA

% Note: all values should be in microns 
% (assuming that the input data from Fiji is in microns)

% Axis formatting for 3D plots
% (assuming that PD axis is aligned along y-axis, with base at origin
%  and tip in positive quadrant)
axLIMx = 650;   % max value of x-axis
axLIMtx = 600;  % max value of ticks in x-axis
axLIMy = 1400;  % max value of y-axis
axLIMty = 1400; % max value of ticks in y-axis
axLIMz = 650;   % max value of z-axis 
axLIMtz = 600;  % max value of ticks in z-axis    

% Axis formatting for 2D plots
PDdistance = 1250;   % max value from distal tip of humerus, along PD axis
sliceThickness = 50; % slicing perpendicular to PD axis
init = 50;           % start plotting at "init" microns from distal tip


AxisIncrem = 200;    % grid spacing in the 3D and 2D plots 

% Pixel resolution from .czi file.
pixelsize_czi_XYdim = 0.9154;  % Assuming x and y resolutions are the same
pixelsize_czi_Zdim = 4.9454;
  
% Name for output files, no extension
outputname = '2018-12-11_Fig4_DMSO_L3_elbow';  

% In case we want to filter the cell count results, removing small and/or
% large volumes
MinVolCell = 0; %4*pi()*(5^3)/3;
MaxVolCell = 1e10; %4*pi()*(25^3)/3; %0.5e3;


%% Request user which analysis part must be run
run = input('Enter the part you want to run (1,2, 3 or 4): ');
switch run
%--- PART 1 ----------------------------------------------------------------------
    case 1
        % Path to Slice Geometry results .csv file, with extension
        inputBONEJ ='MATLAB-INPUT/SliceGeom_mask.csv'; 

%--- PART 2 ----------------------------------------------------------------------
    case 2
        % Path to Slice Geometry results .csv file, with extension
        % (needed to compute volume of slices)
        inputBONEJ ='MATLAB-INPUT/SliceGeom_mask.csv'; 
        
        % Path to results file from 3D Objects Counter, with extension
        inputPARTICLES ='MATLAB-INPUT/3DObjCount.csv'; 

%--- PART 3 ----------------------------------------------------------------------
    case 3
        % Path to ImageJ script (intensity analysis) results in .csv file, with extension
        inputIMAGEJ ='MATLAB-INPUT/ImageJ-results-intensity.csv'; %
        max_intensity = 2^16;
        
%--- PART 4 ----------------------------------------------------------------------
    case 4
        % Path to segmented organ .stl file, with extension          
        inputSURF = 'MATLAB-INPUT/Humerus_mask-smooth.stl'; % 

        % Path to segmented cells .stl file, with extension
        inputBLOB = 'MATLAB-INPUT/cell-surfaces-smooth.stl';      
        
    otherwise 
        warning('Unexpected PART number.')
end 

%---------------------------------------------------------------------------------
%% RUN
switch run

    %% PART 1. PLOT ORGAN-LEVEL RESULTS
    case 1
        %% Some initial actions
        %If results folder doesn't exist, create a new one with subfolders
        if (exist('results', 'dir')~=7) 
            mkdir results;
            mkdir ./results eps_images;
            mkdir ./results png_images;
            mkdir ./results fig_matlab;
        else
            if (exist('results/eps_images', 'dir')~=7) 
                mkdir ./results eps_images;
            end
            if (exist('results/png_images', 'dir')~=7) 
                mkdir ./results png_images;
            end
            if (exist('results/fig_matlab', 'dir')~=7) 
                mkdir ./results fig_matlab;
            end
        end
        
        %% Store relevant data
        %Read .csv file from Slice Geometry
        dataGEOM = readtable(inputBONEJ);
        CSA(:) = dataGEOM.CSA_microns___; 
        R1(:) = dataGEOM.R1_microns_; 
        R2(:) = dataGEOM.R2_microns_; 
        Imin(:) = dataGEOM.Imin_microns_4_; 
        Imax(:) = dataGEOM.Imax_microns_4_; 

        %% Produce plots 
        marker = 'o';
        color = [0, 103, 205]/255; %blue

        %Invert direction of PD axis
        SliceVolBoneJ = fliplr(CSA);
        
        % Define X axis
        PDaxisPos = 0:pixelsize_czi_XYdim:pixelsize_czi_XYdim*(size(CSA,2)-1);
        initPos = round(init/pixelsize_czi_XYdim); 
        if (initPos==0) 
            initPos=1;
        end
        
        %Plot CSA and save to file
        fig1 = figure(1);
        fig1.Color = 'w';
        plot(PDaxisPos(initPos:end),SliceVolBoneJ(initPos:end),...
             'Color',color,'LineStyle','none','LineWidth',1.5,...
             'Marker',marker,'MarkerSize',5,'MarkerFaceColor',color);
        xlabel('dist. from distal tip (m)');
        ylabel('cross-sect. area (m^{2})');
        title('');
        
        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YAxis.Exponent=5;
        ax.YLim  = [0.5e5 2.0e5];
        ax.YTick =  0.5e5:0.75e5:2e5; 
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%1.1f';
        ax.Position = [0.2002 0.2276 0.7048 0.6600];

        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;
        %The manipulation of the labels means the exponent is "lost".
        %The best solution I found is to manually print it.
        Yexponent = floor(log10(ax.YLim(2)));
        s1='\times';    
        s2 = sprintf('%d^{%d}', 10, Yexponent);
        exponent = strcat(s1,s2);
        annotation('textbox',[.19 0.67 .1 .2],...
                   'String', exponent,...
                   'EdgeColor','none',...
                   'FontSize',20)
        filename_eps=[pwd '/results/eps_images/' outputname '-CSA.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-CSA.png'];
        saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-CSA.fig'];
         savefig(filename_matlab);
        movegui(fig1,'northwest');
        
        %Compute R1/R2 from BoneJ data
        ratioR = R1./R2;
        ratioRflip = fliplr(ratioR);
          
        %Plot R1/R2 and save to file
        fig2 = figure(2);
        fig2.Color = 'w';
        plot(PDaxisPos(initPos:end),ratioRflip(initPos:end),...
            'Color',color,'LineStyle','none','LineWidth',1.5,...
            'Marker',marker,'MarkerSize',5,'MarkerFaceColor',color);
        xlabel('dist. from distal tip (m)');
        ylabel('R_{1} / R_{2}');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%.1f';
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YLim = [0.0 1.0];
        ax.YTick =  0:0.5:1.0;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;     
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-ratioR.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-ratioR.png'];
        saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-ratioR.fig'];
         savefig(filename_matlab);
        movegui(fig2,'west');
        
        %Compute Imin/Imax from BoneJ 
        ratioI = Imin./Imax;
        ratioIflip = fliplr(ratioI);
        
        %Plot Imin/Imax and save to file
        fig3 = figure(3);
        fig3.Color = 'w';
        plot(PDaxisPos(initPos:end),ratioIflip(initPos:end),...
            'Color',color,'LineStyle','none','LineWidth',1.5,...
            'Marker',marker,'MarkerSize',5,'MarkerFaceColor',color);
        xlabel('dist. from distal tip (m)');
        ylabel('I_{min} / I_{max}');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%.1f';
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YLim = [0.0 1.0];
        ax.YTick =  0:0.5:1.0;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;    
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-ratioI.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-ratioI.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-ratioI.fig'];
        savefig(filename_matlab);
        movegui(fig3,'southwest');
        
        disp('PART 1 completed.')
        
        %Clear all variables from workspace
        clear variables;
        
        
    %% PART 2. PLOT CELLULAR-LEVEL RESULTS
    case 2        
        
        %% Some initial actions
        %If results folder doesn't exist, create a new one with subfolders
        if exist('results', 'dir')~=7
            mkdir results;
            mkdir ./results eps_images;
            mkdir ./results png_images;
            mkdir ./results fig_matlab;
        end
        
        %% Import center of mass positions
        %Read csv file from 3D Object counter
        dataPARTICLES = readtable(inputPARTICLES);
        Vol(:,1) = dataPARTICLES.Volume_microns_3_; 
        CoM(:,1) = dataPARTICLES.X; 
        CoM(:,2) = dataPARTICLES.Y; 
        % It seems that 3D Objects Counter provides the
        % z-dimension values in pixels, instead of microns.
        % It could be a bug/error in the version I used.
        CoM(:,3) = dataPARTICLES.Z*pixelsize_czi_Zdim; 
         
        % Plot a histogram to see the distribution of #cells for each
        % volume. It should roughly give a Gaussian distribution.
        fig100 = figure(100);
        fig100.Color = 'w';
        histfit(Vol, size(Vol,1),'normal');
        xlabel('object volume (m^{3})');
        ylabel('number of objects');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        movegui(fig100,'northwest');

        % Save to file
        filename_png=[pwd '/results/png_images/' outputname '-cell-volumes-original.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-volumes-original.fig'];
        savefig(filename_matlab);

        %Only keep particles below MaxVolCell and above MinVolCell 
        DeleteVol = (MaxVolCell > Vol) & (Vol > MinVolCell);
        i = 1;
        
        endval = size(DeleteVol,1);
        while (i <= endval)
             if (DeleteVol(i,1) == 0)
                CoM(i,:) = [];
                DeleteVol(i,:) = [];
                Vol(i,:) = [];
                endval = endval - 1;
             else
                 i = i+1;
             end
        end

        % Plot the new distribution once the small and big volumes are
        % removed
        fig101 = figure(101);
        fig101.Color = 'w';
        histfit(Vol, size(Vol,1),'normal');
        xlabel('object volume (m^{3})');
        ylabel('number of objects');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        movegui(fig101,'west');

        filename_png=[pwd '/results/png_images/' outputname '-cell-volumes-processed.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-volumes-processed.fig'];
        savefig(filename_matlab);
        
        % Print to command window the min and max volumes, which will
        % correspond to min and max of the legend in the scatter plot.
        X = ['Maximum cell volume is ',num2str(max(Vol)),...
            ' and minimum cell volume is ',num2str(min(Vol)),'.'];
        disp(X);

        %Create scatter plot with CoM positions of cells
        % Volume is proportional 
        ptCloud_CoM = pointCloud(CoM,'Intensity',Vol);

        size_vector = Vol(:,1)*150/max(Vol(:,1));
        color_vector = rescale(Vol(:,1));
        fig4 = figure(4);
        length = 50;
        color1 = [244, 125, 0]/255; % #f47d00
        color2 = [9, 5, 0]/255;  % #090500
        colors_p = [linspace(color1(1),color2(1),length)',...
                    linspace(color1(2),color2(2),length)',...
                    linspace(color1(3),color2(3),length)'];
        colormap(colors_p)
        scatter3(CoM(:,1),CoM(:,2),CoM(:,3),...
                  size_vector, color_vector, 'filled');

        % Axis limits and formatting
        ax = gca;
        ax.Color = 'w';
        ax.XAxis.Color = 'k';
        ax.YAxis.Color = 'k';
        ax.ZAxis.Color = 'k';
        fig4.Color = 'w';
        grid on;
        box on;
        axis image;
        view([-135 35]);
        xlabel('x (m)','VerticalAlignment','baseline',...
                        'HorizontalAlignment','center');
        ylabel('y (m)','VerticalAlignment','baseline',...
                        'HorizontalAlignment','left');
        zlabel('z (m)');

        ax.FontSize = 24;

        % To plot only labels at axis limits
        ax.XLim  = [0 axLIMx];
        ax.XTick =  0:AxisIncrem:axLIMtx;
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
         
        ax.YLim  = [0 axLIMy];
        ax.YTick =  0:AxisIncrem:axLIMty;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;

        ax.ZLim  = [0 axLIMz];
        ax.ZTick =  0:AxisIncrem:axLIMtz;
        labels = string(ax.ZAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.ZAxis.TickLabels = labels;
        movegui(fig4,'northeast');

        filename_eps=[pwd '/results/eps_images/' outputname '-cell-3Dpos.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-cell-3Dpos.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-3Dpos.fig'];
        savefig(filename_matlab);

        %Define number of iterations
        max_count = floor(PDdistance/sliceThickness);

        %Initialize some vectors
        cellcount = zeros(1,max_count);
        volumecount = zeros(1,max_count);
        PDaxisPos = zeros(1,max_count);
     
        count = 0;
        for SliceXPos = PDdistance:(-sliceThickness):0
            
            count = count+1;
            if count == max_count
                break
            end
            %Count cell center of mass points in this slice.
            %Slice is counted as volume between this slice and the next one.
            roi = [-inf,inf;SliceXPos-sliceThickness,SliceXPos;-inf,inf];
            indices = findPointsInROI(ptCloud_CoM,roi);
            ptCloud_selected = select(ptCloud_CoM,indices);
            cellcount(count) = ptCloud_selected.Count;
            volumecount(count) = sum(ptCloud_selected.Intensity);

            %Store position along PD-axis
            if (count == 1)
                PDaxisPos(count) = sliceThickness/2;
            else
                PDaxisPos(count) = PDaxisPos(count-1) + sliceThickness;
            end

            %Figure out "count" value at which we reach PDdistance   
            if (PDaxisPos(count)<=PDdistance)
                PDaxisPosMaxCount = count;
            else
                break
            end
        end
        
        %Compute volumes of each slice based on CSA sections
        %Read .csv file from Slice Geometry
        dataGEOM = readtable(inputBONEJ);
        CSA(:) = dataGEOM.CSA_microns___; 
        
        %Invert direction of PD axis
        SliceArea = fliplr(CSA);
        SliceVol = SliceArea.*pixelsize_czi_XYdim;
        
        % Define X axis in microns corresponding to CSA
        PDaxisPosCSA = 0:pixelsize_czi_XYdim:pixelsize_czi_XYdim*(size(CSA,2)-1);
        

        % To compute equal volume sections we must interpolate data because
        % (sliceThickness/pixelsize_czi_XYdim) is not an integer!
        % If we used SliceVol, some slices would use more CSA than others,
        % resulting in uneven volumes
        PDaxisPosInterp = 0:1:(PDaxisPosCSA(1,end));
        SliceVolInterp = interp1(PDaxisPosCSA,SliceVol,PDaxisPosInterp);
        
        sectionvolume = zeros(1,max_count);
        j = 1;
        for i=1:max_count
            while (PDaxisPosInterp(1,j)<=(sliceThickness*i))
                sectionvolume(1,i) = sectionvolume(1,i)+SliceVolInterp(1,j);
                j=j+1;
            end
        end
        
        color = [205, 102, 0]/255; %orange
        
        %Plot center of mass count from Fiji and save to file
        initCell = floor(init/sliceThickness);
        fig5 = figure(5);
        fig5.Color = 'w';
        bar(PDaxisPos(initCell:PDaxisPosMaxCount),cellcount(initCell:PDaxisPosMaxCount),...
            'FaceColor',color,'EdgeColor','w','LineWidth',0.01);
        xlabel('dist. from distal tip (m)');
        ylabel('cell num.');
        title('');
        
        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.XRuler.TickLabelFormat = '%d';         
        ax.YRuler.TickLabelFormat = '%d';
        ax.YLim = [0.0 60.0];
        ax.YTick =  0:20:60;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-cell-num.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-cell-num.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-num.fig'];
        savefig(filename_matlab);
        movegui(fig5,'north');

        %Compute cell "density" (cell num./ slice vol.)
        celldensity = cellcount./sectionvolume;

        %Plot cell density and save to file
        fig6 = figure(6);
        fig6.Color = 'w';
        bar(PDaxisPos(initCell:PDaxisPosMaxCount),celldensity(initCell:PDaxisPosMaxCount),...
            'FaceColor',color,'EdgeColor','w','LineWidth',0.01);
        xlabel('dist. from distal tip (m)');
        ylabel('cell num. / vol. (m^{-3})'); 
        title('');
        
        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%.1f';
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YLim = [0.0 0.00001];
        ax.YTick =  0:0.000005:0.00001;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;   
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(ax.YAxis.TickLabels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels; 
        %The manipulation of the labels means the exponent is "lost".
        %The best solution I found is to manually print it.
        Yexponent = floor(log10(ax.YLim(2)));
        s1='\times';    
        s2 = sprintf('%d^{%d}', 10, Yexponent);
        exponent = strcat(s1,s2);
        annotation('textbox',[.19 0.67 .1 .2],...
                   'String', exponent,...
                   'EdgeColor','none',...
                   'FontSize',20)
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-cell-num-dens.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-cell-num-dens.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-num-dens.fig'];
        savefig(filename_matlab);
        movegui(fig6,'center');
        
        %Compute EdU cell volume ratio
        cellvolratio = volumecount./sectionvolume;

        %Plot cell volume and save to file
        fig7 = figure(7);
        fig7.Color = 'w';
        bar(PDaxisPos(initCell:PDaxisPosMaxCount),cellvolratio(initCell:PDaxisPosMaxCount),...
            'FaceColor',color,'EdgeColor','w','LineWidth',0.01);
        xlabel('dist. from distal tip (m)');
        ylabel('cell vol. / total vol.'); 
        title('');
        
        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%.1f';
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YLim = [0.0 0.1];
        ax.YTick =  ax.YLim(1):((ax.YLim(2)-ax.YLim(1))/2):ax.YLim(2);
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;   
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(ax.YAxis.TickLabels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels; 
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-cell-vol-dens.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-cell-vol-dens.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-cell-vol-dens.fig'];
        savefig(filename_matlab);
        movegui(fig7,'south');
        
        
        %Plot cell volume and save to file
        fig102 = figure(102);
        fig102.Color = 'w';
        bar(PDaxisPos(initCell:PDaxisPosMaxCount),sectionvolume(initCell:PDaxisPosMaxCount));
        xlabel('dist. from distal tip (m)');
        ylabel('slice vol.'); 
        title('');
        
        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XRuler.TickLabelFormat = '%d';
        ax.YRuler.TickLabelFormat = '%.1f';
        ax.XLim  = [init PDdistance];
        ax.XTick =  init:AxisIncrem:PDdistance;
        ax.YLim = [0.0 inf];
        ax.YTick =  ax.YLim(1):((ax.YLim(2)-ax.YLim(1))/2):ax.YLim(2);
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;   
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(ax.YAxis.TickLabels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels; 
        %
        movegui(fig102,'southwest');
        
        disp('PART 2 completed.')
        
        %Clear all variables from workspace
        clear variables;
        
        
    %% PART 3. PLOT MOLECULAR-LEVEL RESULTS
    case 3        
        
        %% Some initial actions
        %If results folder doesn't exist, create a new one with subfolders
        if exist('results', 'dir')~=7
            mkdir results;
            mkdir ./results eps_images;
            mkdir ./results png_images;
            mkdir ./results fig_matlab;
        end
        
        % Each slice of a 3D image has a specific histogram.
        % Each column of the HistogramTable files correspond to the histogram of
        % a proximal to distal slice of the humerus. 
        % This information can be used to measure a distal to proximal intensity 
        % distribution, or to recreate the histogram of the entire humerus.
        
        %% Store relevant data       
        % Reading the histogram tables
        % This section read the input files in a table and a matrix and flip this matrix 
        % distal to proximal

        HistogramTable = readtable(inputIMAGEJ); 
        array = table2array(HistogramTable(2:end-1,2:end-1));
        
        % flip the matrix distal to proximal
        array = flip(array,2);
        
        % Defining the total number of slices and creating an axis with the slices numbers
        total_slices = size(array,2);
        xaxis = linspace(1,total_slices,total_slices)'; 
        
        % Convert axis to microns
        AxisInMicrons = xaxis*pixelsize_czi_XYdim;
        StartPointMicrons = init;
        FinalPointMicrons = PDdistance;
        
        % Start and end point in terms of slices
        StartPointSlices = round(StartPointMicrons/pixelsize_czi_XYdim)+1;
        FinalPointSlices = round(FinalPointMicrons/pixelsize_czi_XYdim)+1;

        %Remove positions in Axis below/above defined limits
        AxisInMicrons = AxisInMicrons(StartPointSlices:FinalPointSlices);
        
        %% Measure mean intensity of each plane
        Sum = 0;
        Tot = 0;
        n = 1;
        lengthMI = FinalPointSlices-StartPointSlices;
        MeanIntensity = zeros(1,lengthMI);
        
        for i = StartPointSlices:FinalPointSlices
            for j = 1:size(array,1)
             Sum = Sum + (j-1)*array(j,i);
             Tot = Tot + array(j,i) ;
            end
            MeanIntensity(n) = Sum/Tot;
            Sum = 0;
            Tot = 0;
            n = n+1;
        end

        MeanIntensity = MeanIntensity';
        
        %% Measure histogram
        Sum = 0;
        lengthHT = size(array,1);
        HistogramTotal = zeros(1,lengthHT);
        
        for j = 1:lengthHT
            for i = 1:FinalPointSlices
                Sum = Sum + array(j,i);
            end
            HistogramTotal(j) = Sum;
            Sum = 0;
        end

        HistogramTotal = HistogramTotal';
        
        %% Produce plots 
        marker = 'o';
        color = [103, 205, 0]/255; %green
        
        %Plot mean intensity and save to file
        fig8 = figure(8);
        fig8.Color = 'w';
        plot(AxisInMicrons,MeanIntensity,...
         'Color',color,'LineStyle','none','LineWidth',1.5,'Marker',marker,...
         'MarkerSize',5,'MarkerFaceColor',color); 
        xlabel('dist. from distal tip (m)');
        ylabel('mean intensity');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XLim  = [StartPointMicrons FinalPointMicrons];
        ax.XTick =  StartPointMicrons:AxisIncrem:FinalPointMicrons; 
        ax.YLim  = [200 500];
        ax.YTick =  200:150:500; 
        ax.XRuler.TickLabelFormat = '%d'; 
        ax.YRuler.TickLabelFormat = '%d';
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-mean-intensity.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-mean-intensity.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-mean-intensity.fig'];
        savefig(filename_matlab);
        movegui(fig8,'northeast');
        
        % Computing stuff for histogram
        X_axis_histogram = linspace(0,max_intensity-2,max_intensity-1)';
        Area = trapz(X_axis_histogram,HistogramTotal);
        Y_axis_histogram_norm = HistogramTotal./Area; 
        Y_axis_histogram = HistogramTotal;
        
        % Plot histogram and save to file
        fig9 = figure(9);
        fig9.Color = 'w';
        area(X_axis_histogram,Y_axis_histogram,...
             'FaceColor',color,'FaceAlpha',0.5,...
             'EdgeColor',color,'LineWidth',1.5); 
        xlabel('pixel intensity');
        ylabel('num. of pixels');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XLim  = [0 1000];
        ax.XTick =  0:250:1000; 
        ax.YLim = [0.0 4e5];
        ax.YTick =  0:2e5:4e5; 
        ax.XRuler.TickLabelFormat = '%d'; 
        ax.YRuler.TickLabelFormat = '%1.1f';
        ax.YAxis.Exponent = 5;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;
        %The manipulation of the labels means the exponent is "lost".
        %The best solution I found is to manually print it.
        Yexponent = floor(log10(ax.YLim(2)));
        s1='\times';    
        s2 = sprintf('%d^{%d}', 10, Yexponent);
        exponent = strcat(s1,s2);
        annotation('textbox',[.19 0.67 .1 .2],...
                   'String', exponent,...
                   'EdgeColor','none',...
                   'FontSize',20)
        %
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-histogram.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-histogram.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-histogram.fig'];
        savefig(filename_matlab);
        movegui(fig9,'east');
        
        fig10 = figure(10);
        fig10.Color = 'w';
        area(X_axis_histogram,Y_axis_histogram_norm,...
             'FaceColor',color,'FaceAlpha',0.5,...
             'EdgeColor',color,'LineWidth',1.5); 
        xlabel('pixel intensity');
        ylabel('norm. num. of pixels');
        title('');

        % Axis limits and formatting
        ax = gca;
        ax.PlotBoxAspectRatio = [1 0.5 1];
        ax.FontSize = 24;
        ax.TickLength = [0.02 0.02];
        ax.XLim  = [0 1000];
        ax.XTick =  0:250:1000; 
        ax.YLim = [0.0 1.2e-2];
        ax.YTick =  0:0.6e-2:1.2e-2; 
        ax.XRuler.TickLabelFormat = '%d'; 
        ax.YRuler.TickLabelFormat = '%1.1f';
        ax.YAxis.Exponent = -2;
        ax.Position = [0.2002 0.2276 0.7048 0.6600];
        
        % To plot only labels at axis limits
        labels = string(ax.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.XAxis.TickLabels = labels;
        labels = string(ax.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax.YAxis.TickLabels = labels;
        %The manipulation of the labels means the exponent is "lost".
        %The best solution I found is to manually print it.
        Yexponent = floor(log10(ax.YLim(2)));
        s1='\times';    
        s2 = sprintf('%d^{%d}', 10, Yexponent);
        exponent = strcat(s1,s2);
        annotation('textbox',[.19 0.67 .1 .2],...
                   'String', exponent,...
                   'EdgeColor','none',...
                   'FontSize',20)
        %
        filename_eps=[pwd '/results/eps_images/' outputname '-histogram-norm.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-histogram-norm.png'];
        saveas(gcf,filename_png);
        filename_matlab=[pwd '/results/fig_matlab/' outputname '-histogram-norm.fig'];
        savefig(filename_matlab);
        movegui(fig10,'southeast');
        
        %%  Save tables to files, in case we want to plot them again
        ExportHistogram(:,1) = X_axis_histogram;
        ExportHistogram(:,2) = HistogramTotal;

        ExportMeanIntensity(:,1) = AxisInMicrons;
        ExportMeanIntensity(:,2) = MeanIntensity;

        writematrix(ExportHistogram,'results/Histogram.xlsx');
        writematrix(ExportMeanIntensity,'results/MeanIntensity.xlsx');
        
        disp('PART 3 completed.')
        
        %Clear all variables from workspace
        clear variables;
        
    %% PART 4. PLOT 3D views of organ and cell surfaces, including slicing effect
    case 4
        %% Some initial actions
        %If results folder doesn't exist, create a new one with subfolders
        if exist('results', 'dir')~=7
            mkdir results;
            mkdir ./results eps_images;
            mkdir ./results png_images;
            mkdir ./results fig_matlab;
        end
        
        %Compile this c++ file, needed for the "intersectPlaneSurf" function
        %More info at https://www.mathworks.com/matlabcentral/fileexchange/32256-intersectplanesurf-ii
        mex('IntersectPlaneTriangle.cpp','-v');
        
        %% Load & plot STL meshes
        [stlSURFimport] = stlread(inputSURF);
        [stlBLOBimport] = stlread(inputBLOB);

         %Plot imported organ surface
         fig10 = figure(10);
         patch(stlSURFimport,'FaceColor',       [0.92 0.92 0.92], ...
                             'FaceAlpha',       1,                ...
                             'EdgeColor',       'none',           ...
                             'FaceLighting',    'gouraud',        ...
                             'AmbientStrength', 0.15);
         camlight('headlight');
         material('dull');
         fig10.Color = 'w'; 
         grid on;
         box on;
         axis equal;
         view([-135 35]);
        
         % Axis labels
         xlabel('x (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','center');
         ylabel('y (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','left');
         zlabel('z (m)');
            
        % Axis limits and formatting
         ax = gca;
         ax.FontSize = 24;

         ax.XLim  = [0 axLIMx]; 
         ax.XTick =  0:AxisIncrem:axLIMtx;
         labels = string(ax.XAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.XAxis.TickLabels = labels;

         ax.YLim  = [0 axLIMy]; 
         ax.YTick =  0:AxisIncrem:axLIMty; 
         labels = string(ax.YAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.YAxis.TickLabels = labels;

         ax.ZLim  = [0 axLIMz];  
         ax.ZTick =  0:AxisIncrem:axLIMtz; 
         labels = string(ax.ZAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.ZAxis.TickLabels = labels;
         movegui(fig10,'northwest');
    
         % Save plot to files
         filename_eps=[pwd '/results/eps_images/' outputname '-3D-organ.eps'];
         saveas(gcf,filename_eps,'epsc');
         filename_png=[pwd '/results/png_images/' outputname '-3D-organ.png'];
         saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-3D-organ.fig'];
         savefig(filename_matlab);

         %Plot imported cell surfaces
         fig11 = figure(11);
         patch(stlBLOBimport,'FaceColor',       [0.92 0.92 0.92], ...
                             'FaceAlpha',       1,                ...
                             'EdgeColor',       'none',           ...
                             'FaceLighting',    'gouraud',        ...
                             'AmbientStrength', 0.15);
         camlight('headlight');
         material('dull');
         fig11.Color = 'w';
         grid on;
         box on;
         axis image;
         view([-135 35]);
         
         % Axis labels
         xlabel('x (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','center');
         ylabel('y (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','left');
         zlabel('z (m)');

        % Axis limits and formatting
         ax = gca;
         ax.FontSize = 24;

         ax.XLim  = [0 axLIMx];
         ax.XTick =  0:AxisIncrem:axLIMtx;
         labels = string(ax.XAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.XAxis.TickLabels = labels;

         ax.YLim  = [0 axLIMy];
         ax.YTick =  0:AxisIncrem:axLIMty;
         labels = string(ax.YAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.YAxis.TickLabels = labels;

         ax.ZLim  = [0 axLIMz];
         ax.ZTick =  0:AxisIncrem:axLIMtz;
         labels = string(ax.ZAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.ZAxis.TickLabels = labels;
         movegui(fig11,'west');

         filename_eps=[pwd '/results/eps_images/' outputname '-3D-cells.eps'];
         saveas(gcf,filename_eps,'epsc');
         filename_png=[pwd '/results/png_images/' outputname '-3D-cells.png'];
         saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-3D-cells.fig'];
         savefig(filename_matlab);

         %% Create point clouds with imported stl surfaces & plot
         ptCloud_SURF = pointCloud(stlSURFimport.vertices);
         ptCloud_BLOB = pointCloud(stlBLOBimport.vertices);

         fig12 = figure(12);
         pcshowpair(ptCloud_BLOB,ptCloud_SURF)
         
         % Axis limits and formatting
         ax = gca;
         ax.Color = 'w';
         ax.XAxis.Color = 'k';
         ax.YAxis.Color = 'k';
         ax.ZAxis.Color = 'k';
         fig12.Color = 'w';
         grid on;
         box on;
         axis image;
         view([-135 35]);
         
         % Axis labels
         xlabel('x (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','center');
         ylabel('y (m)','VerticalAlignment','baseline',...
                         'HorizontalAlignment','left');
         zlabel('z (m)');

         % Axis formatting
         ax = gca;
         ax.FontSize = 24;

         ax.XLim  = [0 axLIMx];
         ax.XTick =  0:AxisIncrem:axLIMtx;
         labels = string(ax.XAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.XAxis.TickLabels = labels;

         ax.YLim  = [0 axLIMy];
         ax.YTick =  0:AxisIncrem:axLIMty;
         labels = string(ax.YAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.YAxis.TickLabels = labels;

         ax.ZLim  = [0 axLIMz];
         ax.ZTick =  0:AxisIncrem:axLIMtz;
         labels = string(ax.ZAxis.TickLabels);
         max_labels = size(labels);
         labels(2:max_labels-1) = '';
         ax.ZAxis.TickLabels = labels;
         movegui(fig12,'southwest');
    
         % Save to files
         filename_eps=[pwd '/results/eps_images/' outputname '-3D-SurfPointClouds.eps'];
         saveas(gcf,filename_eps,'epsc');
         filename_png=[pwd '/results/png_images/' outputname '-3D-SurfPointClouds.png'];
         saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-3D-SurfPointClouds.fig'];
         savefig(filename_matlab);


        %% Plot slice intersections
        fv.faces = stlSURFimport.faces; %Use face definitions from original stl data
        fv.vertices = ptCloud_SURF.Location; %Use vertices position from aligned data

        % Plot surface to slice
        fig13 = figure(13);
        set(0, 'CurrentFigure', fig13); 
        fig13.Color = 'w';
        fig14 = patch(fv);
        set(fig14,'FaceColor',[.8 .8 .8],'EdgeColor','none');
        camlight,lighting gouraud
        grid on;
        box on;
        axis image;
        view([-135 35]);
        xlabel('x (m)','VerticalAlignment','baseline',...
                     'HorizontalAlignment','center');
        ylabel('y (m)','VerticalAlignment','baseline',...
                     'HorizontalAlignment','left');
        zlabel('z (m)');

        ax13 = gca;
        ax13.FontSize = 24;

        ax13.XLim  = [0 axLIMx];
        ax13.XTick =  0:AxisIncrem:axLIMtx;
        labels = string(ax13.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax13.XAxis.TickLabels = labels;

        ax13.YLim  = [0 axLIMy];
        ax13.YTick =  0:AxisIncrem:axLIMty;
        labels = string(ax13.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax13.YAxis.TickLabels = labels;

        ax13.ZLim  = [0 axLIMz];
        ax13.ZTick =  0:AxisIncrem:axLIMtz;
        labels = string(ax13.ZAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax13.ZAxis.TickLabels = labels;
        movegui(fig13,'center');
        hold on;


        fvCell.faces = stlBLOBimport.faces; %Use face definitions from original stl data
        fvCell.vertices = ptCloud_BLOB.Location; %Use vertices position from aligned data

        % Plot cell surfaces to slice
        fig15 = figure(15);
        set(0, 'CurrentFigure', fig15); 
        fig15.Color = 'w';
        fig16 = patch(fvCell);
        set(fig16,'FaceColor',[.8 .8 .8],'EdgeColor','none');
        camlight,lighting gouraud
        grid on;
        box on;
        axis image;
        view([-135 35]);
        xlabel('x (m)','VerticalAlignment','baseline',...
                     'HorizontalAlignment','center');
        ylabel('y (m)','VerticalAlignment','baseline',...
                     'HorizontalAlignment','left');
        zlabel('z (m)');

        ax8 = gca;
        ax8.FontSize = 24;

        ax8.XLim  = [0 axLIMx];
        ax8.XTick =  0:AxisIncrem:axLIMtx;
        labels = string(ax8.XAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax8.XAxis.TickLabels = labels;

        ax8.YLim  = [0 axLIMy];
        ax8.YTick =  0:AxisIncrem:axLIMty;
        labels = string(ax8.YAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax8.YAxis.TickLabels = labels;

        ax8.ZLim  = [0 axLIMz];
        ax8.ZTick =  0:AxisIncrem:axLIMtz;
        labels = string(ax8.ZAxis.TickLabels);
        max_labels = size(labels);
        labels(2:max_labels-1) = '';
        ax8.ZAxis.TickLabels = labels;
        movegui(fig15,'south');
        hold on;

        %Get coordinate limits to compute loop min and max
        XLim = ptCloud_SURF.YLimits;
        YLim = ptCloud_SURF.ZLimits;
        ZLim = ptCloud_SURF.XLimits;

        % Min and max determine the limits of the slicing.
        % The code will not work if these values are outside the range of
        % the surface being sliced, i.e. an error will occur when trying to
        % slice "empty space".
        %Round to nearest integer more than or equal to the min value of the PD axis.
        min_slice = ceil(XLim(1));
        %Round to the nearest integer less than or equal to the max value of the PD axis.
        max_slice = floor(XLim(2)); 

        % Compute axis limits to define constant area of 2D plots
        % Minimum values need to be smaller than the axis min (floor) and 
        % maximum values need to be larger than the axis max (ceil).
        xmin = floor(YLim(1));
        xmax = ceil(YLim(2));
        ymin = floor(ZLim(1));
        ymax = ceil(ZLim(2));
        
        %Add a little of margin so 2D sections have some space around them
        xaux = (xmax-xmin)*0.1;
        yaux = (ymax-ymin)*0.1;
        xmin = xmin - xaux;
        xmax = xmax + xaux;
        ymin = ymin - yaux;
        ymax = ymax + yaux;
        
        n = [0 -1 0]; %vector containing plane normal (slice along y-plane)

        %Define number of iterations
        max_count = floor((max_slice-min_slice)/sliceThickness);

        %Initialize some vectors
        PDaxisPos = zeros(1,max_count);
        
        count = 0;
        for SliceXPos = max_slice:(-sliceThickness):min_slice
            
            count = count+1;

            p = [0 SliceXPos 0];

            %Plot plane intersecting 3D surface 
            plane = createPlane(p,n(:).');
            lin = intersectPlaneSurf(fv,p,n);
            
            set(0, 'CurrentFigure', fig13)
            hold on, hpl=drawPlane3d(plane,'FaceAlpha', 0.1,    ...
                                           'FaceColor', 'blue', ...
                                           'EdgeColor','none'); 
            set(hpl,'FaceAlpha',0.1)
            daspect([1 1 1]); axis('vis3d')
            set(gcf,'Renderer','OpenGL')
            for m=1:size(lin,2)
                hold on,
                plot3(lin{m}(1,:),lin{m}(2,:),lin{m}(3,:),'b','Linewidth',2)
            end
            hold off;

            set(0, 'CurrentFigure', fig15)
            hold on, hpl=drawPlane3d(plane,'FaceAlpha', 0.1,    ...
                                           'FaceColor', 'blue', ...
                                           'EdgeColor','none'); 
            set(hpl,'FaceAlpha',0.1)
            daspect([1 1 1]); axis('vis3d')
            set(gcf,'Renderer','OpenGL')
            for m=1:size(lin,2)
                hold on,
                plot3(lin{m}(1,:),lin{m}(2,:),lin{m}(3,:),'b','Linewidth',2)
            end
            hold off;
        end

        set(0, 'CurrentFigure', fig13)
        filename_eps=[pwd '/results/eps_images/' outputname '-3D-organ-sliced.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-3D-organ-sliced.png'];
        saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-3D-organ-sliced.fig'];
         savefig(filename_matlab);


        set(0, 'CurrentFigure', fig15)
        filename_eps=[pwd '/results/eps_images/' outputname '-3D-cells-sliced.eps'];
        saveas(gcf,filename_eps,'epsc');
        filename_png=[pwd '/results/png_images/' outputname '-3D-cells-sliced.png'];
        saveas(gcf,filename_png);
         filename_matlab=[pwd '/results/fig_matlab/' outputname '-3D-cells-sliced.fig'];
         savefig(filename_matlab);

        disp('PART 4 completed.')
        
        %Clear all variables from workspace
        clear variables;
        
    otherwise 
        warning('Unexpected PART number.')
end        
clear run;
