% SCRIPT TO PLOT RESULTS FROM Pixel_intensity_blastemas_CUBE_16bits.ijm
% for Figure 6b results. 
%
% REQUIREMENTS:
% - The following files must be in the same folder as this script:
%
% 	 + .csv with results from the ImageJ macro
%
%    +  breakyaxis.m available from Matlab File Exchange
%    https://www.mathworks.com/matlabcentral/fileexchange/45760-break-y-axis
%
%    +  distributionPlot_OnlyPositiveMod.m 
%    A modified version of the script available from Matlab File Exchange
%    https://www.mathworks.com/matlabcentral/fileexchange/69682-distributionplot_onlypositive/
%
%    + colorCode2rgb.m
%    + myHistogram.m   
%    + countEntries.m
%    all available from Matlab File Exchange
%    https://fr.mathworks.com/matlabcentral/fileexchange/23661-violin-plots-for-plotting-multiple-distributions-distributionplot-m
%
% Ester Comellas, Northeastern University, 2020
% Contact: e.comellas@northeastern.edu
%
%% USER INPUT DATA
% Indicate the root name for the output files:
outputname = 'Fig6';

plothist = 0; % 1 for 'yes' or 0 'no'
plotviolin = 1; % 1 for 'yes' or 0 'no'

% Indicate the name of the input files, including the .csv extension
% Ensure that the order is consistent, i.e., always denervated first and
% inmediately after the corresponding innervated limb. Or viceversa, but
% same dpa and same repl consecutive in odd+even positions.
inputfile = ["ImageJ-macro-results/9dpa_rep1_EdU_L-denervated_side0_tile1_RS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/9dpa_rep1_EdU_R-control_side0_tile1_RS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/9dpa_rep2_EdU_L-denervated_side180_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/9dpa_rep2_EdU_R-control_side0_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/12dpa_rep2_EdU_L-denervated_side0_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/12dpa_rep2_EdU_R-control_side0_tile2_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/12dpa_rep3_EdU_L-denervated_side0_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/12dpa_rep3_EdU_R-control_side0_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/15dpa_rep1_EdU_L-denervated_side180_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/15dpa_rep1_EdU_R-control_side180_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/15dpa_rep2_EdU_L-denervated_side0_tile1_RCS_cube_resliced_intensity-values.csv",...
             "ImageJ-macro-results/15dpa_rep2_EdU_R-control_side0_tile1_RCS_cube_resliced_intensity-values.csv" ];

%Indicate the names for the legend in the plots:         
legend_names_histograms = [ "9dpa rep1 denervated",...
                            "9dpa rep1 innervated",...
                            "9dpa rep2 denervated",... 
                            "9dpa rep2 innervated",...
                            "12dpa rep2 denervated",...
                            "12dpa rep2 innervated",...
                            "12dpa rep3 denervated",... 
                            "12dpa rep3 innervated",...
                            "15dpa rep1 denervated",...
                            "15dpa rep1 innervated",...
                            "15dpa rep2 denervated",... 
                            "15dpa rep2 innervated"];  
legend_names_violin = [ "9 dpa rep.1",...
                        "9 dpa rep.2",... 
                        "12 dpa rep.1",...
                        "12 dpa rep.2",... 
                        "15 dpa rep.1",...
                        "15 dpa rep.2"];   
                    
dpa_ratios_axis = [9, 9, 12, 12, 15, 15];

color_9dpa_rep1_den  = [189, 63, 67]/255;   %Red
color_9dpa_rep2_den  = [189, 63, 67]/255;   %Red
color_12dpa_rep1_den = [240, 137, 52]/255;  %Orange
color_12dpa_rep2_den = [240, 137, 52]/255;  %Orange
color_15dpa_rep1_den = [244, 216, 91]/255;  %Yellow
color_15dpa_rep2_den = [244, 216, 91]/255;  %Yellow

color_9dpa_rep1_in   = [44, 132, 129]/255;  %Turquoise
color_9dpa_rep2_in   = [44, 132, 129]/255;  %Turquoise
color_12dpa_rep1_in  = [63, 98, 189]/255;   %Blue
color_12dpa_rep2_in  = [63, 98, 189]/255;   %Blue
color_15dpa_rep1_in  = [95, 50, 171]/255;   %Purple
color_15dpa_rep2_in  = [95, 50, 171]/255;   %Purple

color_9dpa_rep1_r    = [0, 0, 0];  %black
color_9dpa_rep2_r    = [1, 1, 1];  %white
color_12dpa_rep1_r   = [0, 0, 0];  %black
color_12dpa_rep2_r   = [1, 1, 1];  %white
color_15dpa_rep1_r   = [0, 0, 0];  %black
color_15dpa_rep2_r   = [1, 1, 1];  %white

% Indicate the name for the second line of the y-axis label in the ratio plot:
% Must be odd legend_names / even legend_names          
axis_for_ratio = "(innerv./denerv.)"; 


%% Initial actions 
%Close all windows
close all; 

%Create a new results folder if it doesn't exist
if exist('results', 'dir')~=7
    mkdir results;
end


%% About the input tables
% Each slice of a 3D image has a specific histogram.
% Each column of the HistogramTable files corresponds to the histogram of
% slice along the a proximal-distal axis of the blastema. 
% This information can be used to measure a distal to proximal pixel intensity
% distribution, or to recreate the histogram of the entire blastema.
% 
%% Read the pixel intensity data from the results file
if exist('StoredVars.mat', 'file')==2
    load('StoredVars.mat');
    clear inputfile;
else
    
    %Create cell of empty matrices.
    %All the data from each input files will be stored in each cell
    % position as a matrix
    NumSamples = size(inputfile,2);
    SliceIntensities = cell(NumSamples,4);
    total_slices = zeros(NumSamples,1);
    
    for n=1:NumSamples
        
        % Load the data into a table       
        HistogramTable = readtable(inputfile(n)); 
        % Copy data from table into array, but skip:
        % - first column (intensity values, because they're not stored as numbers)
        % - first row (slice numbering)
        IntensityCount = table2array(HistogramTable(2:end,2:end));
        
        %Read intensity values
        IntensityValues = str2double(table2array(HistogramTable(2:end,1)));
        HistogramArray = cat(2,IntensityValues,IntensityCount);
        clear IntensityCount IntensityValues;
        
        % Store each array in a cell position
        % Flip the matrix in HistogramArray to have the histogram of 
        % each slice in a row instead of in a column, i,e,
        % columns are intensity values; rows indicate slices of the stack
        SliceIntensities{n,1} = num2cell(HistogramArray',[size(HistogramTable,1) size(HistogramTable,2)]);
        
        % Save total number of slices for posterior check
        total_slices(n,1) = size(SliceIntensities{n,1},1);
        
        clear HistogramTable HistogramArray;
    end
    
    %Check that all files have same number of slices, if not print error
    if (range(total_slices) == 1)
        disp('Not all samples have the same number of slices. Here is a list of slice numbers for each file:')
        total_slices 
        error('Script ended with an error!');
    end
    
    clear inputfile n total_slices;

    %% Measure mean, median and mode of histograms
    TotalIntensities = cell(NumSamples,5);

    %Compute number of slices, which should be the same for all samples
    NumSlices = size(SliceIntensities{2,1},1)-1;

    for k=1:NumSamples

        % Save current sample to an array for easier manipulation
        SliceHistogram = cell2mat(SliceIntensities{k,1});

        % Obtain histogram of whole cube by adding all pixels with same
        % intensity across slices (add columns for rows 2 to end)
        % First row is intensity values of the histogram
        TotalHistogram = zeros(2,size(SliceHistogram,2));
        TotalHistogram(1,:) = SliceHistogram(1,:);
        TotalHistogram(2,:) = sum(SliceHistogram(2:end,:));

        % Save total histogram in cell for plotting later
        TotalIntensities{k,1} = num2cell(TotalHistogram);

        %Compute mean of histogram for whole cube and for each slice
        TotalBins = sum(TotalHistogram(2,:));
        TotalMean = sum(TotalHistogram(1,:).*TotalHistogram(2,:))/TotalBins;
        TotalIntensities{k,2} = TotalMean;

        SliceMean = zeros(NumSlices,1);
        SliceBins = zeros(NumSlices,1);
        for i = 1:NumSlices
         SliceBins(i,1) = sum(SliceHistogram(i+1,:));
         SliceMean(i,1) = sum(SliceHistogram(1,:).*SliceHistogram(i+1,:))./SliceBins(i,1);
        end

        % We could do a check here and ensure that all SliceBins are the same!
        SliceBins(2:end)=[];
        if ( (TotalMean - (sum(SliceMean)/NumSlices))/TotalMean > 1e-6 )
            disp('Ooops! Something might have gone wrong computing the means. Check the code!');
            TotalMean
            disp('Computed mean');
            (sum(SliceMean)/NumSlices)
        end
        SliceIntensities{k,2} = SliceMean;

        %Compute median of histogram for whole cube and for each slice
        TotalArrayBins = zeros(TotalBins,1);

        % Create an array with all pixels, where each position of array contains the value of each pixel.
        % Delete pixel values that have no pixel count to reduce for-loop
        auxTotalHist = TotalHistogram;
        indices = auxTotalHist(2,:)==0;
        auxTotalHist(:,indices) = [];
        clear indices;

        i=1;
        for j = 1:size(auxTotalHist,2)
           for m = 1:auxTotalHist(2,j)
                    TotalArrayBins(i,1)= auxTotalHist(1,j);
                    i = i+1;
           end
        end
        clear i j m auxTotalHist;

        TotalMedian = median(TotalArrayBins);
        TotalIntensities{k,3} = TotalMedian;
        TotalMode = mode(TotalArrayBins);
        TotalIntensities{k,4} = TotalMode;
        TotalIntensities{k,5} = max(TotalHistogram(1,:));
        % We could do a check here and ensure that
        % mean(TotalArrayBins) == TotalMean

        clear TotalArrayBins TotalMean TotalMedian TotalMode TotalHistogram  TotalBins;

        % Create an array with all pixels, where each position of array contains the value of each pixel.
        % Delete pixel values that have no pixel count to reduce for-loop
        SliceMedian = zeros(NumSlices,1);
        SliceMode = zeros(NumSlices,1);

        for n=1:NumSlices

            SliceArrayBins = zeros(SliceBins,1); 
            auxSliceHist = cat(1,SliceHistogram(1,:),SliceHistogram(n+1,:));
            indices = auxSliceHist(2,:)==0;
            auxSliceHist(:,indices) = [];
            clear indices;

            i=1;
            for j = 1:size(auxSliceHist,2)
               for m = 1:auxSliceHist(2,j)
                        SliceArrayBins(i,1)= auxSliceHist(1,j);
                        i = i+1;
               end
            end
            clear i j m auxSliceHist;

            SliceMedian(n,1) = median(SliceArrayBins);
            SliceMode(n,1) = mode(SliceArrayBins);
            % We could do a check here and ensure that
            % mean(SliceArrayBins) == SliceMean(n,1)
        end

        SliceIntensities{k,3} = SliceMedian;
        SliceIntensities{k,4} = SliceMode;
        clear SliceArrayBins SliceMean SliceMedian SliceMode SliceHistogram  SliceBins;
    end
    clear k n ;


    %% Measure ratios (mean,median, mode)

    TotalRatios = zeros((NumSamples/2),4);
    TotalRatios(:,1) = dpa_ratios_axis';
    i = 1;
    for n = 1:(NumSamples/2)
        TotalRatios(n,2) =  TotalIntensities{i+1,2}./TotalIntensities{i,2};
        TotalRatios(n,3) =  TotalIntensities{i+1,3}./TotalIntensities{i,3};
        TotalRatios(n,4) =  TotalIntensities{i+1,4}./TotalIntensities{i,4};
        i = i+2;
    end
    clear i n;

    SliceRatios = zeros((NumSamples/2),4,NumSlices);
    SliceRatios(:,1) = dpa_ratios_axis';
    i = 1;
    for n = 1:(NumSamples/2)
        SliceRatios(n,2,:) =  SliceIntensities{i+1,2}./SliceIntensities{i,2};
        SliceRatios(n,3,:) =  SliceIntensities{i+1,3}./SliceIntensities{i,3};
        SliceRatios(n,4,:) =  SliceIntensities{i+1,4}./SliceIntensities{i,4};
        i = i+2;
    end
    clear i n NumSlices SliceRatios SliceIntensities;

    %Save cell array created for faster loading in the future.
    save('StoredVars.mat',...
         'TotalIntensities', 'TotalRatios', 'NumSamples');
   
end

%% PLOT HISTOGRAMS
if plothist == 1
    % Plot total ratios
    colors = vertcat( color_9dpa_rep1_r,...
                      color_9dpa_rep2_r,...
                      color_12dpa_rep1_r,...
                      color_12dpa_rep2_r,...
                      color_15dpa_rep1_r,...
                      color_15dpa_rep2_r);

    fig1 = figure(1);
    %fig1.Color = 'w';
    scatter(TotalRatios(:,1),TotalRatios(:,2),100,...
            colors,'filled','MarkerEdgeColor','k','LineWidth',1.5);
    xlabel('dpa');
    ylabel({'mean intensity','ratio '+axis_for_ratio});
    title('');
    box on;
    set(gcf, 'Position',  [100, 100, 250, 300])
    hAx = gca;
    hAx.FontSize = 24;
    hAx.TickLength = [0.02 0.02];
    hAx.XLim  = [7 17];
    hAx.XTick = [9 12 15]; %This should be automatized...
    hAx.YLim  = [1 4];
    hAx.YTick = 1:1:4;      
    hAx.XRuler.TickLabelFormat = '%d';
    hAx.YRuler.TickLabelFormat = '%.1f';  

    labels = string(hAx.YAxis.TickLabels);
    max_labels = size(labels);
    labels(2:max_labels-1) = '';
    hAx.YAxis.TickLabels = labels;   

    filename_png=[pwd '/results/' outputname '-TOTAL-MEAN.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-TOTAL-MEAN.fig'];
    savefig(filename_matlab);
    movegui(fig1,'northwest');
    clear hAx filename_png filename_matlab fig1 max_labels labels;


    fig2 = figure(2);
    %fig2.Color = 'w';
    scatter(TotalRatios(:,1),TotalRatios(:,3),100,...
            colors,'filled','MarkerEdgeColor','k','LineWidth',1.5);
    xlabel('dpa');
    ylabel({'median intensity','ratio '+axis_for_ratio});
    title('');
    box on;
    set(gcf, 'Position',  [100, 100, 250, 300])
    hAx = gca;
    hAx.FontSize = 24;
    hAx.TickLength = [0.02 0.02];
    hAx.XLim  = [8 16];
    hAx.XTick = [9 12 15]; %This should be automatized...
    hAx.YLim  = [1 4];
    hAx.YTick = 1:1:4;
    hAx.XRuler.TickLabelFormat = '%d';
    hAx.YRuler.TickLabelFormat = '%.1f';

    labels = string(hAx.YAxis.TickLabels);
    max_labels = size(labels);
    labels(2:max_labels-1) = '';
    hAx.YAxis.TickLabels = labels;    

    filename_png=[pwd '/results/' outputname '-TOTAL-MEDIAN.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-TOTAL-MEDIAN.fig'];
    savefig(filename_matlab);
    movegui(fig2,'west');
    clear hAx filename_png filename_matlab fig2 max_labels labels;

    fig3 = figure(3);
    %fig3.Color = 'w';
    scatter(TotalRatios(:,1),TotalRatios(:,4),100,...
            colors,'filled','MarkerEdgeColor','k','LineWidth',1.5);
    xlabel('dpa');
    ylabel({'mode intensity','ratio '+axis_for_ratio});
    title('');
    box on;
    set(gcf, 'Position',  [100, 100, 250, 300])
    hAx = gca;
    hAx.FontSize = 24;
    hAx.TickLength = [0.02 0.02];
    hAx.XLim  = [7 17];
    hAx.XTick = [9 12 15]; %This should be automatized...
    hAx.YLim  = [1 4];
    hAx.YTick = 1:1:4;
    hAx.XRuler.TickLabelFormat = '%d';
    hAx.YRuler.TickLabelFormat = '%.1f';

    labels = string(hAx.YAxis.TickLabels);
    max_labels = size(labels);
    labels(2:max_labels-1) = '';
    hAx.YAxis.TickLabels = labels;    

    filename_png=[pwd '/results/' outputname '-TOTAL-MODE.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-TOTAL-MODE.fig'];
    savefig(filename_matlab);
    movegui(fig3,'southwest');
    clear hAx filename_png filename matlab fig3  max_labels labels;
    clear colors;

    % Plot the histograms comparison
    fig4 = figure(4);
    %fig4.Color = 'w';
    i = 1;
    for n = 1:(NumSamples/2)

        switch n
            case 1
                colorFirst  = color_9dpa_rep1_den;
                colorSecond = color_9dpa_rep1_in;
                linetype = '-';
            case 2
                colorFirst  = color_9dpa_rep2_den; 
                colorSecond = color_9dpa_rep2_in;
                linetype = '--';
            case 3
                colorFirst  = color_12dpa_rep1_den;
                colorSecond = color_12dpa_rep1_in;
                linetype = '-';
            case 4
                colorFirst  = color_12dpa_rep2_den;
                colorSecond = color_12dpa_rep2_in;
                linetype = '--';
            case 5
                colorFirst  = color_15dpa_rep1_den;
                colorSecond = color_15dpa_rep1_in;
                linetype = '-';
            case 6
                colorFirst  = color_15dpa_rep2_den;
                colorSecond = color_15dpa_rep2_in;
                linetype = '--';
            otherwise
                disp('error')
        end

        HistFirst = (cell2mat(TotalIntensities{i,1}))';
        HistSecond = (cell2mat(TotalIntensities{i+1,1}))';

        %First compute areas under the curve and print to ensure they're the same
        AreaFirst = trapz(HistFirst(:,1),HistFirst(:,2));
        AreaSecond = trapz(HistSecond(:,1),HistSecond(:,2));

        if AreaFirst ~= AreaSecond
            error ('Areas under the histogram curve do not coincide.')
        end
        clear AreaFirst AreaSecond;

        area(HistFirst(:,1),HistFirst(:,2),...
         'FaceColor',colorFirst,'FaceAlpha',0.2,...
         'EdgeColor',colorFirst,'LineWidth',1.0,'LineStyle',linetype); 
        hold on; 

        area(HistSecond(:,1),HistSecond(:,2),...
             'FaceColor',colorSecond,'FaceAlpha',0.2,...
             'EdgeColor',colorSecond,'LineWidth',1.0,'LineStyle',linetype); 
        hold on; 
        clear colorFirst colorSecond;

        i = i+2;
    end
    clear i n;

    hold off;

    xlabel('pixel intensity');
    ylabel('num. of pixels');
    title('');

    set(gcf, 'Position',  [100, 100, 600, 800])
    hAx = gca;
    hAx.FontSize = 24;
    hAx.TickLength = [0.02 0.02];
    hAx.XLim  = [0 4500];
    hAx.XTick =  0:1500:4500;
    hAx.YLim = [0 5.8e4];
    hAx.YTick =  0:1e4:5.8e4;

    filename_png=[pwd '/results/' outputname '-HISTOGRAMS.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS.fig'];
    savefig(filename_matlab);
    movegui(fig4,'north');
    clear filename_png filename_matlab fig4 hAx;

    set(gcf, 'Position',  [100, 100, 600, 350])
    %This function doesn't work properly with legend and axis formatting
    breakaxisinfo = breakyaxis([2.4e4 5.4e4]); 
    clear breakaxisinfo
    filename_png=[pwd '/results/' outputname '-HISTOGRAMS-break.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-break.fig'];
    savefig(filename_matlab);
    clear filename_png filename_matlab fig4 hAx;

    i = 1;
    name = '9dpa';
    fig5 = figure(5);
    %fig5.Color = 'w';
    set(gcf, 'Position',  [100, 100, 600, 350])

    for n = 1:(NumSamples/2)

        switch n
            case 1
                colorFirst  = color_9dpa_rep1_den;
                colorSecond = color_9dpa_rep1_in;
                linetype = '-';
            case 2
                colorFirst  = color_9dpa_rep2_den; 
                colorSecond = color_9dpa_rep2_in;
                linetype = '--';
            case 3
                colorFirst  = color_12dpa_rep1_den;
                colorSecond = color_12dpa_rep1_in;
                linetype = '-';
            case 4
                colorFirst  = color_12dpa_rep2_den;
                colorSecond = color_12dpa_rep2_in;
                linetype = '--';
            case 5
                colorFirst  = color_15dpa_rep1_den;
                colorSecond = color_15dpa_rep1_in;
                linetype = '-';
            case 6
                colorFirst  = color_15dpa_rep2_den;
                colorSecond = color_15dpa_rep2_in;
                linetype = '--';
            otherwise
                disp('error')
        end

        HistFirst = (cell2mat(TotalIntensities{i,1}))';
        HistSecond = (cell2mat(TotalIntensities{i+1,1}))';

        figure(fig5);
        area(HistFirst(:,1),HistFirst(:,2),...
         'FaceColor',colorFirst,'FaceAlpha',0.2,...
         'EdgeColor',colorFirst,'LineWidth',1.0, 'LineStyle',linetype); 
        xlabel('pixel intensity');
        ylabel('num. of pixels');
        title('');

        hAx = gca;
        hAx.FontSize = 24;
        hAx.TickLength = [0.02 0.02];
        hAx.XLim  = [0 4500];
        hAx.XTick =  0:1500:4500;
        hAx.YLim = [0.0 inf];
        hold on; 

        area(HistSecond(:,1),HistSecond(:,2),...
             'FaceColor',colorSecond,'FaceAlpha',0.2,...
             'EdgeColor',colorSecond,'LineWidth',1.0, 'LineStyle',linetype); 
        hold on; 
        clear colorFirst colorSecond;

         if (i == 3)
            legend(legend_names_histograms(1,1:4),'Location','northeast','FontSize',20);
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;

            hAx.YLim = [0.0 2.5e4];
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;
            movegui(fig5,'northeast');
            hold off;

            fig5 = figure(6);
            %fig5.Color = 'w';
            set(gcf, 'Position',  [100, 100, 600, 350])
            name = '12dpa';


         elseif (i == 7)
            legend(legend_names_histograms(1,5:8),'Location','northeast','FontSize',20);
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;

            hAx.YLim = [0.0 2.5e4];
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;
            movegui(fig5,'east');
            hold off;

            fig5 = figure(7);
            %fig5.Color = 'w';
            set(gcf, 'Position',  [100, 100, 600, 350])
            name = '15dpa';

         elseif (i == 11)
            legend(legend_names_histograms(1,9:12),'Location','northeast','FontSize',20);
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;

            hAx.YLim = [0.0 2.5e4];
            filename_png=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.png'];
            saveas(gcf,filename_png);
            filename_matlab=[pwd '/results/' outputname '-HISTOGRAMS-BY-DPA-' name '-bis.fig'];
            savefig(filename_matlab);
            clear filename_png filename_matlab;
            movegui(fig5,'southeast');
            hold off;
         end

        i = i+2;
    end
    
    
clear HistFirst HistSecond i n linetype;
clear fig5 hAx name fig10;
end

%% PLOT VIOLIN PLOTS
if plotviolin == 1
    
    % Convert histogram data from pixel num. vs intensty to "bin storage"
    % The violin plotting function requires histogram stored in this manner
    
    % Compute first case to be able to preallocate HistBins
    % All images should have same num of total pixels
    HistPixels = (cell2mat(TotalIntensities{1,1}))';
    HistBinAux = repelem(HistPixels(:,1),HistPixels(:,2));
    HistBins = zeros (size(HistBinAux,1),NumSamples);
    HistBins(:,1) = HistBinAux;
    clear HistBinAux;
        
    for i=2:NumSamples
        HistPixels = (cell2mat(TotalIntensities{i,1}))';
        HistBins(:,i) = repelem(HistPixels(:,1),HistPixels(:,2));
    end

    
    % See distributionPlot.m documentation for info on these options:
    divfact=0.5;
    histopt=1.1;
    globalNorm = 0;
    showMM = 7; %modified original function
    i = 1;
    xdist = 0.5;
    xpos = 2*xdist;
    
    fig8 = figure(8);
    
    for n = 1:(NumSamples/2)

        switch n
            case 1
                colorFirst  = color_9dpa_rep1_in;
                colorSecond = color_9dpa_rep1_den;
            case 2
                colorFirst  = color_9dpa_rep2_in; 
                colorSecond = color_9dpa_rep2_den;
            case 3
                colorFirst  = color_12dpa_rep1_in;
                colorSecond = color_12dpa_rep1_den;
            case 4
                colorFirst  = color_12dpa_rep2_in;
                colorSecond = color_12dpa_rep2_den;
            case 5
                colorFirst  = color_15dpa_rep1_in;
                colorSecond = color_15dpa_rep1_den;
            case 6
                colorFirst  = color_15dpa_rep2_in;
                colorSecond = color_15dpa_rep2_den;
            otherwise
                disp('error')
        end
        
        
        distributionPlot_OnlyPositiveMod(HistBins(:,i+1),...
                                      'histOpt', histopt,...
                                      'globalNorm', globalNorm,...
                                      'divFactor', divfact,...
                                      'showMM', showMM,...
                                      'xValues', xpos,...
                                      'histOri', 'left',...
                                      'widthDiv', [2 1],...
                                      'color', colorFirst);
        hold on;
        
        %'FaceColor',colorFirst,'FaceAlpha',0.2,...
        %'EdgeColor',colorFirst,'LineWidth',1.0, 'LineStyle',linetype
     
     
        distributionPlot_OnlyPositiveMod(HistBins(:,i),...
                                      'histOpt', histopt,...
                                      'globalNorm', globalNorm,...
                                      'divFactor', divfact,...
                                      'xValues', xpos,...
                                      'showMM', showMM,...
                                      'histOri', 'right',...
                                      'widthDiv', [2 2],...
                                      'color', colorSecond);
       hold on;
       i=i+2;
       xpos=xpos+2*xdist;
    end
    hold off;
    hAx = gca;
    
    ylabel('pixel intensity');
    %ylabel('num. of pixels');
    title('');
    box on;

    set(gcf, 'Position',  [100, 100, 800, 700])
    hAx = gca;
    hAx.FontSize = 24;
    hAx.TickLength = [0.02 0.02];
    hAx.YLim  = [0 5000];
    hAx.YTick =  0:1000:5000;
    hAx.XLim  = [0 7];
    hAx.XTick =  1:1:7;
    set(gca,'xtick',[1:7],'xticklabel',legend_names_violin);
    xtickangle(45);
    
    filename_png=[pwd '/results/' outputname '-VIOLIN-PLOTS.png'];
    saveas(gcf,filename_png);
    filename_matlab=[pwd '/results/' outputname '-VIOLIN-PLOTS.fig'];
    savefig(filename_matlab);
    clear filename_png filename_matlab;
            
    clear hAx fig8 colorFirst colorSecond i n xdist xpos;
    clear HistBins HistPixels divfact globalNorm histopt showMM;
    
end
%% Clear variables
clear plothist plotviolin;

clear color_12dpa_rep1_den color_12dpa_rep1_in color_12dpa_rep1_r;
clear color_12dpa_rep2_den color_12dpa_rep2_in color_12dpa_rep2_r;
clear color_15dpa_rep1_den color_15dpa_rep1_in color_15dpa_rep1_r; 
clear color_15dpa_rep2_den color_15dpa_rep2_in color_15dpa_rep2_r;
clear color_9dpa_rep1_den color_9dpa_rep1_in color_9dpa_rep1_r;
clear color_9dpa_rep2_den color_9dpa_rep2_in color_9dpa_rep2_r;
clear legend_names_histograms legend_names_violin;
clear dpa_ratios_axis  axis_for_ratio outputname;
clear NumSamples TotalIntensities TotalRatios;

disp("Analysis complete.")