######################
#
# Plot efficiencies of LiCre
# at various duty cycle values
#
########################

library(ggplot2)

for (subdir in c("figs/")){ 
  if (! (subdir %in% dir()) ) system(paste("mkdir -p ", subdir, sep = " "));
};

# standard error of the mean
sem = function(x) return(sd(x)/sqrt(length(x)))

#read data
res = read.table("outputs/dutycycle_twocolors-counts.txt", header =TRUE, sep = "\t")

# We discard 4 samples (see justification in data/2020-11-06/comments.txt)
disc = which(res$date == "2020-11-06" & res$file %in% c("PL1.0003", "PL1.0006", "PL1.0027", "PL1.0030"))
res = res[-disc,]

# set period
res$period = res$time.on.s + res$time.off.s
res$period = as.character(res$period)

# fix ambiguity about dark annotation:
# control samples of duty.cycle value v are 
# annotated with duty.cycle.value=v whereas
# they actually have a duty.cycle.value of zero
res$duty.cycle.value[ res$condition == "dark" ] <- 0

# assign continuously-illuminated samples
# (equivalent to dutycycle = 100%)
# to the period=10 and period=120 series
# (half the samples to each series)
i = which(res$period == 3600)
ni = length(i)
halfni = floor(ni/2)
res$period[ i[1:halfni] ] <- 10
res$period[ i[(halfni+1):ni] ] <- 120

ymax = 0.4
# Figure for LiCre at intensity 255
constr = "LiCre"
intensity = 255
sx = subset(res, construct == constr & illumination.intensity == intensity)

sx$meanEff   = ave(sx$f.RED, sx$duty.cycle.value, sx$period, FUN = mean)
sx$sem       = ave(sx$f.RED, sx$duty.cycle.value, sx$period, FUN = sem)

file = paste("figs/dutycycle_", constr, "_", intensity, ".pdf", sep = "")
pdf(file)      
p = ggplot( sx, aes(x=duty.cycle.value, y = f.RED, colour = period, fill = period)) +
       theme(panel.grid.major.x = element_blank(),panel.grid.minor.x = element_blank()) +
       scale_colour_manual(values = c("darkblue", "blue")) +
       stat_summary(fun.y=mean, geom = "line", position= "identity") + 
       geom_errorbar(aes(ymin=meanEff-sem,ymax=meanEff+sem), width = 2.5, position = "identity") +
       scale_x_continuous(breaks = c(0,1,5,10,25,50,75,100)) +
       theme(panel.grid.major.x = element_line(colour = "white")) +
       geom_jitter(shape = 1, size = 1, position ="identity") +
       ylim(0,ymax) +
       ggtitle(paste(constr, "intensity", intensity))

print(p)
dev.off()

# Figure for Mag_Cre and LiCre at intensity 15
constr = c("LiCre", "Mag_Cre")
intensity = 15
ymax = 0.28
sx = subset(res, construct %in% constr & illumination.intensity == intensity)

sx$meanEff   = ave(sx$f.RED, sx$construct, sx$duty.cycle.value, sx$period, FUN = mean)
sx$sem       = ave(sx$f.RED, sx$construct, sx$duty.cycle.value, sx$period, FUN = sem)

file = paste("figs/dutycycle_", intensity, ".pdf", sep = "")
pdf(file)      
p = ggplot( sx, aes(x=duty.cycle.value, y = f.RED, colour = construct)) +
       theme(panel.grid.major.x = element_blank(),panel.grid.minor.x = element_blank()) +
       scale_colour_manual(values = c("blue", "magenta")) +
       stat_summary(fun.y=mean, geom = "line", position= "identity") + 
       geom_errorbar(aes(ymin=meanEff-sem,ymax=meanEff+sem), width = 2.5, position = "identity") +
       scale_x_continuous(breaks = c(0,5,10,25,50,75,100)) +
       theme(panel.grid.major.x = element_line(colour = "white")) +
       geom_jitter(shape = 1, size = 1, position = "identity") +
       ylim(0,ymax) +
       ggtitle(paste("intensity", intensity))

print(p)
dev.off()

 
