//Physical model of the PSM and tailbud
//written by Koichiro Uriu

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include "mt64.h" //using Mersenne twister

#define PI 3.14159265358979

//maximum cell number 
#define NUMPTS 4300

int N; //cell number
int Nar; //number of arrested cells

double Lx, Ly, Lz; //domain size

double Lx0, Lx1; //parameters in the equation of PSM length
double Lxa; //anterior border of the PSM
double vxa; // rate of PSM shortening
double Ls; //segment size
double density, etalx, tlx;

double v0; //self-propelling velocity
double vd0; // drift
double Dv0; //diffusion coefficient of velocity angle
double mu, sl;
double mub; //boundary force coefficient
double rb0; //lengthscale of boundary forces

double Xv, hv;

double w0; //intrinsic frequency at tailbud
double k0; //coupling strength
double k0r; //recovered coupling strength
double Dp; //phase diffusion coefficient

double cr; // coupling range

double sigf; //shape parameter for frequency profile
double kfrq; //length scale of frequency profile

double rt; // smaller radius for a torus
double Rt; // larger radius for a torus
double ptor; //[-pi/2, pi/2]
double qtor; //[0, 2*pi]

double Xc, Yc, Zc; //position of torus center

double rois, roil; // for msdd calculation

double xq, vd1; //non-uniform growth
double waa; //weight for cell addition in tailbud
double xal; //anterior limit of cell addition

int Twash; //timing of DAPT washout
int Tgs; //timing of growth mode switch

int Ntor, Ncr, Ncl; //cell number at the tailbud, right and left PSM

struct state {
	
	double x; //AP
	double y; //medial-lateral
	double z; //DV
	
	double phi; //velocity angle for xy plane
	double varphi; //velocity angle from z-axis
	double theta; //phase of oscillation
	
	double wn; //natural frequency
	
	double nx; //cell polarity
	double ny;
	double nz;
	
	double vs; //self-propulsion speed
	double vx; //velocity
	double vy;
	double vz;
	
	double vd; //drift velocity
	
};

double DT; //time step for Euler method

//function for Gaussian noise
double Box_Muller_Method(double m, double sd)
{
	static int count_BM = 0;
	double r1, r2;
	double nrand;
	
	do {
		
		r1 = genrand64_real3(); 
		r2 = genrand64_real3();
		
		if (count_BM == 0) {
			count_BM = 1;
			nrand = sd*sqrt(-2*log(r1))*cos(2*PI*r2) + m;
		}
		else {
			count_BM = 0;
			nrand = sd*sqrt(-2*log(r1))*sin(2*PI*r2) + m;
		}
		
	} while (isinf(sqrt(-2*log(r1))) != 0);
	
	return(nrand);
	
}

//function to calculate polarity angle for cell movement
void polarity_dynamics(struct state a[NUMPTS+1], double sddv0, int i)
{
	
	//for acos to prevent nan when a[i].ny is close to zero
	double tolerance = 0.000001;
	double ntemp;
	
	ntemp = a[i].nz;
	//polarity dynamics
	if (-0.7071 < ntemp && ntemp <= 0.7071) { //cos(0.25*PI) == 0.7071
		
		a[i].phi = acos(a[i].nz);
		if (fabs(a[i].ny) > tolerance) {
			a[i].varphi = acos(a[i].nx/sin(a[i].phi));
			if (a[i].ny < 0.0) {
				a[i].varphi = 2.0*PI - a[i].varphi;
			}
		}
		else {
			if (a[i].nx > 0.0) a[i].varphi = 0.0;
			else a[i].varphi = PI;
		}
		
		a[i].varphi = a[i].varphi + (sddv0/sin(a[i].phi))*Box_Muller_Method(0.0, 1.0);
		a[i].phi = a[i].phi + DT*(Dv0/tan(a[i].phi)) + sddv0*Box_Muller_Method(0.0, 1.0);
		
		//pi mode phi
		if (a[i].phi < 0.0) a[i].phi = - a[i].phi;
		if (a[i].phi > PI) a[i].phi = 2.0*PI - a[i].phi;
		
		//2pi mod for varphi
		if (a[i].varphi < 0.0) a[i].varphi = 2.0*PI + a[i].varphi;
		if (a[i].varphi > 2.0*PI) a[i].varphi = a[i].varphi - 2.0*PI;
		
		a[i].nx = sin(a[i].phi)*cos(a[i].varphi);
		a[i].ny = sin(a[i].phi)*sin(a[i].varphi);
		a[i].nz = cos(a[i].phi);
		
	}
	else { //when the polarity vector is pointing north or south pole
		
		//transform angles
		a[i].phi = acos(a[i].nx);
		a[i].varphi = acos(a[i].ny/sin(a[i].phi));
		if (a[i].nz < 0.0) {
			a[i].varphi = 2.0*PI - a[i].varphi;
		}
		
		a[i].varphi = a[i].varphi + (sddv0/sin(a[i].phi))*Box_Muller_Method(0.0, 1.0);
		a[i].phi = a[i].phi + DT*(Dv0/tan(a[i].phi)) + sddv0*Box_Muller_Method(0.0, 1.0);
		
		//pi mode phi
		if (a[i].phi < 0.0) a[i].phi = - a[i].phi;
		if (a[i].phi > PI) a[i].phi = 2.0*PI - a[i].phi;
		
		//2pi mod for varphi
		if (a[i].varphi < 0.0) a[i].varphi = 2.0*PI + a[i].varphi;
		if (a[i].varphi > 2.0*PI) a[i].varphi = a[i].varphi - 2.0*PI;
		
		a[i].nx = cos(a[i].phi);
		a[i].ny = sin(a[i].phi)*cos(a[i].varphi);
		a[i].nz = sin(a[i].phi)*sin(a[i].varphi);
		
	}
	
	
}

//boundary force calculation
void compute_boundary_force(struct state a[NUMPTS+1], int i, double *Fbx, double *Fby, double *Fbz)
{
	
	double dx, dy, dz;
	double rti; // smaller radius for a torus
	double ptori; //[-pi/2, pi/2]
	double qtori, qtori_temp; //[0, 2*pi]
	double tlr = 0.0001; //tolerance for angle calculation
	double temp, rx, ry, rz;
	
	//calculation of angles and radius for cell i
	if (a[i].x >= Xc) { // within the torus
		
		dx = a[i].x - Xc;
		dy = a[i].y - Yc;
		dz = a[i].z - Zc;
		
		if (dx > tlr) {ptori = atan(dy/dx);}
		else {
			if (dy > 0.0) ptori = PI/2.0;
			else ptori = -PI/2.0;
		}
		
		if (fabs(dx-Rt*cos(ptori)) >= fabs(dy-Rt*sin(ptori))) {
			temp = dz*cos(ptori)/(dx-Rt*cos(ptori));
		}
		else {
			temp = dz*sin(ptori)/(dy-Rt*sin(ptori));
		}
		
		qtori_temp = atan(temp);
		if (dz > 0.0 && qtori_temp > 0.0) qtori = qtori_temp;
		if (dz > 0.0 && qtori_temp < 0.0) qtori = PI + qtori_temp;
		if (dz < 0.0 && qtori_temp > 0.0) qtori = PI + qtori_temp;
		if (dz < 0.0 && qtori_temp < 0.0) qtori = 2.0*PI + qtori_temp;
		
		rti = dz/sin(qtori);
		
		rx = fabs(Rt*cos(ptori)+rt*cos(ptori)*cos(qtori) - dx);
		*Fbx = -mub*exp(-rx/rb0)*cos(ptori)*cos(qtori);
		
		ry = fabs(Rt*sin(ptori)+rt*sin(ptori)*cos(qtori) - dy);
		*Fby = -mub*exp(-ry/rb0)*cos(qtori)*sin(ptori);
		
		rz = fabs(rt*sin(qtori) - dz);
		*Fbz = -mub*exp(-rz/rb0)*sin(qtori);
		
		if (fabs(Rt*cos(ptori)-dx)+fabs(Rt*sin(ptori)-dy)+fabs(dz) < tlr) {
			*Fbx = 0.0; *Fby = 0.0; *Fbz = 0.0;
		}
		
	}
	else { //within columns
		
		if (a[i].y < Yc) {
			dy = a[i].y - rt;
		}
		else {
			dy = a[i].y - (rt+2*Rt);
		}
		
		dz = a[i].z - Zc;
		
		if (fabs(dy) + fabs(dz) > tlr) {
			
			if (fabs(dy) > tlr) {
				qtori_temp = atan(dz/dy);
				if (dz > 0.0 && qtori_temp > 0.0) qtori = qtori_temp;
				if (dz > 0.0 && qtori_temp < 0.0) qtori = PI + qtori_temp;
				if (dz < 0.0 && qtori_temp > 0.0) qtori = PI + qtori_temp;
				if (dz < 0.0 && qtori_temp < 0.0) qtori = 2.0*PI + qtori_temp;
			}
			else {
				if (dz > 0.0) qtori = PI/2.0;
				else qtori = 3*PI/2.0;
			}
			
			ry = fabs(rt*cos(qtori) - dy);
			*Fby = -mub*exp(-ry/rb0)*cos(qtori);
			
			rz = fabs(rt*sin(qtori) - dz);
			*Fbz = -mub*exp(-rz/rb0)*sin(qtori);
			
		}
		else {
			*Fby = 0.0;
			*Fbz = 0.0;
		}
		
	}
	
	
}

// function to create initial cell positions
void initial_relaxation(struct state a[NUMPTS+1])
{
	
	int i, j;
	double Fijx, Fijy, Fijz;
	double Fbx, Fby, Fbz;
	double rij, rijx, rijy, rijz, uijx, uijy, uijz;
	double temp;
	double vm;
	double sddv0 = sqrt(2.0*Dv0)*sqrt(DT);
	
	double *new_x, *new_y, *new_z;
	
	new_x = (double *)malloc(sizeof(double) * (N+1)); if (new_x == NULL) { printf("memory allocation error: new_x\n"); exit(0); }
	new_y = (double *)malloc(sizeof(double) * (N+1)); if (new_y == NULL) { printf("memory allocation error: new_y\n"); exit(0); }
	new_z = (double *)malloc(sizeof(double) * (N+1)); if (new_z == NULL) { printf("memory allocation error: new_z\n"); exit(0); }
	
	for (i = 1; i <= N; i ++) {
		
		Fijx = 0.0; 
		Fijy = 0.0;
		Fijz = 0.0;
		
		for (j = 1; j <= N; j ++) {
			
			if (j != i) {
				
				rijx = a[j].x - a[i].x;
				rijy = a[j].y - a[i].y;
				rijz = a[j].z - a[i].z;
				
				rij = sqrt(rijx*rijx + rijy*rijy + rijz*rijz);
				uijx = rijx/rij;
				uijy = rijy/rij;
				uijz = rijz/rij;
				
				if (rij <= sl) {
					temp = mu*(rij/sl - 1.0);
					Fijx += temp*uijx;
					Fijy += temp*uijy;
					Fijz += temp*uijz;
				}
				else {
					Fijx += 0.0;
					Fijy += 0.0;
					Fijz += 0.0;
				}
				
			}
			
		}
		
		//force from boundary
		Fbx = 0.0; Fby = 0.0; Fbz = 0.0;
		compute_boundary_force(a, i, &Fbx, &Fby, &Fbz);
		
		if (a[i].x <= 2.0*rb0) { Fbx = mub*exp(-a[i].x/rb0); } //to confin cells in the domain
		
		new_x[i] = a[i].x + DT*Fijx + DT*Fbx + DT*v0*a[i].nx;
		new_y[i] = a[i].y + DT*Fijy + DT*Fby + DT*v0*a[i].ny;
		new_z[i] = a[i].z + DT*Fijz + DT*Fbz + DT*v0*a[i].nz;
		
		a[i].vx = (new_x[i] - a[i].x)/DT;
		a[i].vy = (new_y[i] - a[i].y)/DT;
		a[i].vz = (new_z[i] - a[i].z)/DT;
		
		vm = sqrt(a[i].vx*a[i].vx + a[i].vy*a[i].vy + a[i].vz*a[i].vz);
		
		polarity_dynamics(a, sddv0, i);
		
	}
	
	for (i = 1; i <= N; i ++) {
		a[i].x = new_x[i];
		a[i].y = new_y[i];
		a[i].z = new_z[i];
		
	}
	
	free(new_x); free(new_y); free(new_z);
	
}

//function to determine the position of added cells
double cell_addition(struct state a[NUMPTS+1], double dift, double difr, double difl, int i)
{
	double rti, ptori, qtori;
	double new_x;
	
	//default: tail bud
	rti = (rt - 1.0)*genrand64_real3();
	ptori = -PI/2.0 + PI*genrand64_real3();
	qtori = 2.0*PI*genrand64_real3();
	
	new_x = Xc + Rt*cos(ptori) + rti*cos(ptori)*cos(qtori);
	a[i].y = Yc + Rt*sin(ptori) + rti*sin(ptori)*cos(qtori);
	a[i].z = Zc + rti*sin(qtori);
	
	//PSM
	if (difr <= difl && difr <= dift && Lxa+xal <= Xc){
		new_x = Lxa + xal + (Xc-(Lxa+xal))*genrand64_real3();
		
		rti = rt*genrand64_real3();
		qtori = 2.0*PI*genrand64_real3();
		
		a[i].y = (rt+2*Rt) + rti*cos(qtori);
		a[i].z = rt + rti*sin(qtori);
	}
	
	if (difl <= difr && difl <= dift && Lxa+xal <= Xc){
		new_x = Lxa + xal + (Xc-(Lxa+xal))*genrand64_real3();
		
		rti = rt*genrand64_real3();
		qtori = 2.0*PI*genrand64_real3();
		
		a[i].y = rt + rti*cos(qtori);
		a[i].z = rt + rti*sin(qtori);
	}
	
	return(new_x);
	
}

//function for cell advection anterior to the PSM
void arrested_cell_movement(struct state *ar)
{
	int i;
	
	for (i = 1; i <= Nar; i ++) {
		ar[i].x = ar[i].x - vd0*DT;
	}
}

//function to set the states of added cells
void relocation(struct state a[NUMPTS+1], struct state *ar, double t)
{
	int i;
	int ncl, ncr, nct;
	double new_x, xi;
	
	int Ntemp, Nts;
	int difr, difl, dift;
	
	//count the number of cells in each region
	ncl = 0; ncr = 0; nct = 0;
	for (i = 1; i <= N; i ++) {
		
		if (Xc < a[i].x) {
			nct += 1;
		}
		
		if (Lxa < a[i].x && a[i].x <= Xc && 0.0 < a[i].y && a[i].y <= 2*rt) {
			ncl += 1;
		}
		
		if (Lxa < a[i].x && a[i].x <= Xc && 2*Rt < a[i].y) {
			ncr += 1;
		}
		
	}
	
	//target cell number with a given cell density
	Ntemp = density*(Xc-Lxa)*PI*rt*rt;
	Nts = density*PI*PI*rt*rt*Rt;
	
	i = 1;
	
	while (i <= N) {
		
		new_x = a[i].x;
		
		if (a[i].x < Lxa) {
			
			Nar += 1;
			ar[Nar] = a[i];
			
			difr = (ncr - Ntemp);
			difl = (ncl - Ntemp);
			dift = (nct - Nts);
			
			if (difr < 0 || difl < 0 || dift < 0) {
				
				nct += 1;
				if (difr <= difl && difr <= dift && Lxa+xal <= Xc) { ncr += 1; nct -= 1;}
				if (difl <= difr && difl <= dift && Lxa+xal <= Xc) { ncl += 1; nct -= 1;}
				
				new_x = cell_addition(a, dift, difr, difl, i);
				
				xi = (Lx - new_x)/(Lx - Lxa);
				//autonomous frequency
				if (xi >= 0.0) a[i].wn = w0*(sigf+(1.0-sigf)*(1.0-exp(kfrq*(xi-1.0)))/(1.0-exp(-kfrq)));
				else a[i].wn = 0.0;
				
				//drift velocity
				
				if (xi < xq) {  a[i].vd = vd1*xi;  }
				else { a[i].vd = ((vd0-vd1*xq)/(1.0-xq))*(xi-xq) + vd1*xq; }
				
				//self-propulsion
				a[i].vs = v0/(1.0 + pow(xi/Xv,hv));
				
				a[i].phi = PI*genrand64_real3();
				a[i].varphi = 2.0*PI*genrand64_real3();
				
				a[i].nx = cos(a[i].varphi)*sin(a[i].phi);
				a[i].ny = sin(a[i].varphi)*sin(a[i].phi);
				a[i].nz = cos(a[i].phi);
				
				a[i].vx = a[i].vs*a[i].nx; 
				a[i].vy = a[i].vs*a[i].ny;
				a[i].vz = a[i].vs*a[i].nz;
				
				a[i].theta = 2.0*PI*genrand64_real3();
				
				a[i].x = new_x;
				
				i += 1;
				
			}
			else {
				
				a[i] = a[N];
				N = N - 1;
				
			}
			
		}
		else {
			i += 1;
		}
		
	}
	
}

//function to remove cells when the PSM becomes short enough
void cell_remove(struct state a[NUMPTS+1], struct state *ar)
{
	int i;
	
	i = 1;
	
	while (i <= N) {
		
		if (a[i].x < Lxa) {
			
			Nar += 1;
			ar[Nar] = a[i];
			
			a[i] = a[N];
			N = N - 1;
		}
		else {
			i += 1;
		}
		
	}
	
}

//function to perform Euler method for cell postions, phase and polarity
void dynamics(struct state a[NUMPTS+1])
{
	
	int i, j;
	int theta_int;
	int numnei;
	double Fijx, Fijy, Fijz;
	double Fbx, Fby, Fbz;
	double rij, rijx, rijy, rijz, uijx, uijy, uijz;
	
	
	double rbx, xi;
	
	double temp;
	double vm;
	
	double phase_coupling;
	double sddv0 = sqrt(2.0*Dv0)*sqrt(DT);
	double sddp = sqrt(2.0*Dp)*sqrt(DT);
	
	double *new_x, *new_y, *new_z;
	double *new_theta;
	
	new_x = (double *)malloc(sizeof(double) * (N+1)); if (new_x == NULL) { printf("memory allocation error: new_x\n"); exit(0); }
	new_y = (double *)malloc(sizeof(double) * (N+1)); if (new_y == NULL) { printf("memory allocation error: new_y\n"); exit(0); }
	new_z = (double *)malloc(sizeof(double) * (N+1)); if (new_z == NULL) { printf("memory allocation error: new_z\n"); exit(0); }
	
	new_theta = (double *)malloc(sizeof(double) * (N+1)); if (new_theta == NULL) { printf("memory allocation error: new_theta\n"); exit(0); }
	
	for (i = 1; i <= N; i ++) {
		
		Fijx = 0.0; Fijy = 0.0; Fijz = 0.0;
		phase_coupling = 0.0;
		numnei = 0;
		
		for (j = 1; j <= N; j ++) {
			
			if (j != i) {
				
				rijx = a[j].x - a[i].x;
				rijy = a[j].y - a[i].y;
				rijz = a[j].z - a[i].z;
				
				rij = sqrt(rijx*rijx + rijy*rijy + rijz*rijz);
				uijx = rijx/rij;
				uijy = rijy/rij;
				uijz = rijz/rij;
				
				if (rij <= sl) {
					temp = mu*(rij/sl - 1.0);
					Fijx += temp*uijx;
					Fijy += temp*uijy;
					Fijz += temp*uijz;
				}
				else {
					Fijx += 0.0;
					Fijy += 0.0;
					Fijz += 0.0;
				}
				
				if (rij <= cr) {
					phase_coupling += k0*sin(a[j].theta - a[i].theta);
					numnei += 1;
				}
				
				
			}
		}
		
		//force from boundary
		Fbx = 0.0; Fby = 0.0; Fbz = 0.0;
		
		compute_boundary_force(a, i, &Fbx, &Fby, &Fbz);
		
		new_x[i] = a[i].x + DT*Fijx + DT*Fbx + DT*a[i].vs*a[i].nx - DT*a[i].vd;
		new_y[i] = a[i].y + DT*Fijy + DT*Fby + DT*a[i].vs*a[i].ny;
		new_z[i] = a[i].z + DT*Fijz + DT*Fbz + DT*a[i].vs*a[i].nz;
		
		a[i].vx = (new_x[i] - a[i].x)/DT;
		a[i].vy = (new_y[i] - a[i].y)/DT;
		a[i].vz = (new_z[i] - a[i].z)/DT;
		
		vm = sqrt(a[i].vx*a[i].vx + a[i].vy*a[i].vy + a[i].vz*a[i].vz);
		
		polarity_dynamics(a, sddv0, i);
		
		//phase dynamics of oscillators
		if (numnei > 0) {
			new_theta[i] = a[i].theta + DT*a[i].wn + DT*phase_coupling/numnei + sddp*Box_Muller_Method(0.0, 1.0);
		}
		else {
			new_theta[i] = a[i].theta + DT*a[i].wn + sddp*Box_Muller_Method(0.0, 1.0);
		}
		
		theta_int = new_theta[i]/(2.0*PI);
		new_theta[i] = (new_theta[i]/(2.0*PI) - theta_int)*2.0*PI;
		if (new_theta[i] < 0) new_theta[i] += 2.0*PI;
		
		//update velocity modulus
		rbx = (Lx - a[i].x)/(Lx - Lxa);
		a[i].vs = v0/(1.0 + pow((rbx/Xv),hv));
		
		//update of autonomus frequency
		xi = (Lx - a[i].x)/(Lx - Lxa);
		if (xi >= 0.0) {
			a[i].wn = w0*(sigf+(1.0-sigf)*(1.0-exp(kfrq*(xi-1.0)))/(1.0-exp(-kfrq)));
		}
		else {
			a[i].wn = 0.0;
		}
		
		if (xi < xq) {  a[i].vd = vd1*xi;  }
		else { a[i].vd = ((vd0-vd1*xq)/(1.0-xq))*(xi-xq) + vd1*xq; }
		
	}
	
	for (i = 1; i <= N; i ++) {
		a[i].x = new_x[i]; a[i].y = new_y[i]; a[i].z = new_z[i];
		a[i].theta = new_theta[i];
	}
	
	free(new_x); free(new_y); free(new_z);
	free(new_theta);
	
}

//calculate mean squared distances between cells
double compute_msdd(struct state a[NUMPTS+1], struct state *ini_posi, double r)
{
	int i, j;
	int count;
	
	double dxi, dxj, dyi, dyj, dzi, dzj;
	double dxij, dyij, dzij;
	double inidxij, inidyij, inidzij, inidij;
	double ave_dist;
	
	count = 0;
	ave_dist = 0.0;
	
	for (i = 1; i <= N-1; i ++) {
		
		if (rois <= ini_posi[i].x && ini_posi[i].x <= roil && 0.0 <= ini_posi[i].y && ini_posi[i].y <= Yc) {
			if (rois <= a[i].x && a[i].x <= roil && 0.0 <= a[i].y && a[i].y <= Yc) {
				
				dxi = a[i].x - ini_posi[i].x;
				dyi = a[i].y - ini_posi[i].y;
				dzi = a[i].z - ini_posi[i].z;
				
				for (j = i+1; j <= N; j ++) {
					
					if (rois <= ini_posi[j].x && ini_posi[j].x <= roil && 0.0 <= ini_posi[j].y && ini_posi[j].y <= Yc) {
						if (rois <= a[j].x && a[j].x <= roil && 0.0 <= a[j].y && a[j].y <= Yc) {
							
							inidxij = ini_posi[i].x - ini_posi[j].x;
							inidyij = ini_posi[i].y - ini_posi[j].y;
							inidzij = ini_posi[i].z - ini_posi[j].z;
							
							inidij = sqrt(inidxij*inidxij + inidyij*inidyij + inidzij*inidzij);
							
							if (inidij <= r) {
								dxj = a[j].x - ini_posi[j].x;
								dyj = a[j].y - ini_posi[j].y;
								dzj = a[j].z - ini_posi[j].z;
								
								dxij = dxj - dxi;
								dyij = dyj - dyi;
								dzij = dzj - dzi;
								
								ave_dist += (dxij*dxij + dyij*dyij + dzij*dzij);
								count += 1;
							}
							
						}
					}
					
					
				} //for (j = i+1; j <= N; j ++) {
				
				
			} //if (rois <= a[i].x && a[i].x <= roil) {
		} //if (rois <= ini_posi[i].x && ini_posi[i].x <= roil) {
		
	} //for (i = 1; i <= N-1; i ++) {
	
	if (count > 0) {
		ave_dist /= count;
	}
	else {
		ave_dist = 0.0;
	}
	
	return(ave_dist);
	
}


//local phase order for left and right PSM
void local_phase_order(struct state a[NUMPTS+1], int jmin, int jmax, double *opr, double *opl, double *psir, double *psil)
{
	
	int i, cr, cl, j;
	double ave_cosr, ave_sinr;
	double ave_cosl, ave_sinl;
	double xl, xr, delx;
	
	delx = sl;
	
	*opr = 0.0;
	*opl = 0.0;
	
	for (j = jmin; j < jmax; j ++) {
		
		xl = j*delx + Lxa; xr = (j+1)*delx + Lxa;
		
		ave_cosr = 0.0; ave_sinr = 0.0;
		ave_cosl = 0.0; ave_sinl = 0.0;
		cr = 0; cl = 0;
		
		for (i = 1; i <= N; i ++) {
			
			if (xl <= a[i].x && a[i].x < xr) {
				
				if (a[i].y >= Yc) {
					ave_cosr += cos(a[i].theta);
					ave_sinr += sin(a[i].theta);
					cr += 1;
				}
				else {
					ave_cosl += cos(a[i].theta);
					ave_sinl += sin(a[i].theta);
					cl += 1;
				}
				
			}
			
		}
		
		if (cr > 0) { ave_cosr /= cr; ave_sinr /= cr; }
		
		if (cl > 0) { ave_cosl /= cl; ave_sinl /= cl; }
		
		*opr = *opr + sqrt(ave_cosr*ave_cosr + ave_sinr*ave_sinr);
		*opl = *opl + sqrt(ave_cosl*ave_cosl + ave_sinl*ave_sinl);
		
		if (j == jmin) {
			
			//mean angle for right PSM
			if (fabs(ave_cosr) < 0.0001) {
				if (ave_sinr > 0.0) *psir = PI/2.0;
				else *psir = 3*PI/2.0;
			}
			else {
				if (ave_cosr > 0.0 && ave_sinr > 0.0) *psir = atan(ave_sinr/ave_cosr);
				if (ave_cosr < 0.0 && ave_sinr > 0.0) *psir = PI + atan(ave_sinr/ave_cosr);
				if (ave_cosr < 0.0 && ave_sinr < 0.0) *psir = PI + atan(ave_sinr/ave_cosr);
				if (ave_cosr > 0.0 && ave_sinr < 0.0) *psir = 2*PI + atan(ave_sinr/ave_cosr);
			}
			
			//mean angle for left PSM
			if (fabs(ave_cosl) < 0.0001) {
				if (ave_sinl > 0.0) *psil = PI/2.0;
				else *psil = 3*PI/2.0;
			}
			else {
				if (ave_cosl > 0.0 && ave_sinl > 0.0) *psil = atan(ave_sinl/ave_cosl);
				if (ave_cosl < 0.0 && ave_sinl > 0.0) *psil = PI + atan(ave_sinl/ave_cosl);
				if (ave_cosl < 0.0 && ave_sinl < 0.0) *psil = PI + atan(ave_sinl/ave_cosl);
				if (ave_cosl > 0.0 && ave_sinl < 0.0) *psil = 2*PI + atan(ave_sinl/ave_cosl);
			}
			
			
		}
		
		
	}
	
	*opr = *opr/(jmax-jmin);
	*opl = *opl/(jmax-jmin);
	
}

void simulation(FILE *gp1, FILE *gp2, FILE *gp3, FILE *gp4, FILE *gp5, FILE *gp6, FILE *gp7, FILE *gp8)
{
	struct state a[NUMPTS+1];
	struct state *ar; //record phase of cells that exit from the PSM
	int i;
	
	int timestep, snap;
	int tend;
	
	double xi;
	
	double t, te;
	double Tp;
	
	double rti;
	double ptori, qtori;
	
	double rop, lop;
	double psir, psil; //mean phase
	
	//initial cell number
	Ntor = 555;
	Ncr = 883;
	Ncl = 883;
	
	N = Ntor + Ncr + Ncl;
	
	density = 0.0015;
	
	Tp = 2.0*PI/(w0); //period at anterior end
	
	//advection speed
	vd0 = Ls/Tp - vxa;
	//vd1 = vd0/xq;
	
	//initial positions in a torus
	for (i = 1; i <= N; i ++) {
		
		if (1 <= i && i <= Ntor) {
			
			rti = rt*genrand64_real3();
			
			ptori = -PI/2.0 + PI*genrand64_real3();
			qtori = 2.0*PI*genrand64_real3();
			
			a[i].x = Xc + Rt*cos(ptori) + rti*cos(ptori)*cos(qtori);
			a[i].y = Yc + Rt*sin(ptori) + rti*sin(ptori)*cos(qtori);
			a[i].z = Zc + rti*sin(qtori);
		}
		else if (Ntor < i && i <= Ntor+Ncr) {
			a[i].x = Xc*genrand64_real3();
			
			rti = rt*genrand64_real3();
			qtori = 2.0*PI*genrand64_real3();
			
			a[i].y = (rt+2*Rt) + rti*cos(qtori);
			a[i].z = rt + rti*sin(qtori);
			
		}
		else {
			a[i].x = Xc*genrand64_real3();
			
			rti = rt*genrand64_real3();
			qtori = 2.0*PI*genrand64_real3();
			
			a[i].y = rt + rti*cos(qtori);
			a[i].z = rt + rti*sin(qtori);
			
		}
		
		a[i].phi = PI*genrand64_real3();
		a[i].varphi = 2.0*PI*genrand64_real3();
		
		a[i].nx = cos(a[i].varphi)*sin(a[i].phi);
		a[i].ny = sin(a[i].varphi)*sin(a[i].phi);
		a[i].nz = cos(a[i].phi);
		
		a[i].vx = v0*a[i].nx;
		a[i].vy = v0*a[i].ny;
		a[i].vz = v0*a[i].nz;
		
		//random initial
		a[i].theta = 2.0*PI*genrand64_real3();
		
		//sync ini
		//a[i].theta = 3.0*PI/2.0;
		
	}
	
	//create initial cell positions by relaxation
	DT = 0.01;
	tend = 10/DT;
	
	for (timestep = 1; timestep <= tend; timestep ++) {
		initial_relaxation(a);
	}
	
	//set frequency profile and drift velocity
	for (i = 1; i <= N; i ++) {
		xi = (Lx - a[i].x)/Lx;
		if (xi >= 0.0) {
			a[i].wn = w0*(sigf+(1.0-sigf)*(1.0-exp(kfrq*(xi-1.0)))/(1.0-exp(-kfrq)));
		}
		else {
			a[i].wn = 0.0;
		}
		
		if (xi < xq) {  a[i].vd = vd1*xi;  }
		else { a[i].vd = ((vd0-vd1*xq)/(1.0-xq))*(xi-xq) + vd1*xq; }
		
		a[i].vs = v0/(1.0 + pow(xi/Xv,hv));
		
	}
	
	//output initial condition
	//compute order parameters
	local_phase_order(a, 0, 5, &rop, &lop, &psir, &psil);
	fprintf(gp1, "%f %f %f %f %f %f\n", 0.0, 0.0, rop, lop, psir, psil);
	
	//cell position
	for (i = 1; i <= N; i ++) { fprintf(gp3, "%.2f %.2f %.2f\n", a[i].x, a[i].y, a[i].z); }
	fprintf(gp3, "\n\n");
	
	//intensity profile
	for (i = 1; i <= N; i ++) { fprintf(gp4, "%.2f\n", (1.0+sin(a[i].theta))/2.0); }
	fprintf(gp4, "\n\n");
	
	//cell number
	fprintf(gp5, "%d\n", N);
	
	//PSM length
	fprintf(gp6, "%.2f %.2f %.2f\n", 0.0, Lxa, (Lx-Lxa));
	
	//advection speed at anterior
	fprintf(gp7, "%.2f %.2f %.2f %.2f\n", 0.0, vd0, vd1, vxa);
	
	tend = 780;
	
	// to record phase of cells that exited from the PSM
	ar = (struct state*)malloc(sizeof(struct state) * 15000);
	if (ar == NULL) { printf("memory allocation error: ar\n"); exit(0); }
	
	Nar = 0; //initialize Nar
	
	DT = 0.01;
	tend = tend/DT;
	te = tend*DT; t = 0.0;
	snap = 1/DT;
	
	k0 = 0.0; //turn off coupling strength
	
	//calculation of phase dynamics under collective movement
	for (timestep = 1; timestep <= tend; timestep ++) {
		
		//printf("t = %.2f\n", timestep*DT);
		
		if (timestep*DT >= Twash*Tp) {
			k0 = k0r;
			//k0 = 0.06*(timestep-1)*DT/780.0 + 0.035;
		}
		
		vd0 = Ls/Tp - vxa;
		
		//growth mode switch
		//if (timestep*DT >= Tgs*Tp) { xq = 0.3; vd1 = 3.0; }
		
		dynamics(a);
		arrested_cell_movement(ar);
		
		//PSM shortening
		t += DT;
		Lxa = vxa*t;
		
		if (Lxa >= Xc) { break; }
		
		if ((Xc-Lxa) >= xal) { relocation(a, ar, t); }
		else { cell_remove(a, ar); }
		
		//OUTPUT
		if (timestep%snap == 0) {
			
			//printf("t = %.1f: N = %d Lxa = %.3f PSM length = %.3f\n", timestep*DT, N, Lxa, Lx-Lxa);
			
			//phase order at the anterior end
			local_phase_order(a, 0, 5, &rop, &lop, &psir, &psil);
			fprintf(gp1, "%f %f %f %f %f %f\n", timestep*DT, timestep*DT*w0/(2.0*PI), rop, lop, psir, psil);
			
			//formed segment
			for (i = 1; i <= Nar; i ++) { fprintf(gp2, "%.3f %.3f %.3f %.3f\n", ar[i].x, ar[i].y, ar[i].z, (1.0+sin(ar[i].theta))/2.0);}
			fprintf(gp2, "\n\n");
			
			//output cell positions
			for (i = 1; i <= N; i ++) { fprintf(gp3, "%.3f %.3f %.3f\n", a[i].x, a[i].y, a[i].z); }
			fprintf(gp3, "\n\n");
			
			//intensity
			for (i = 1; i <= N; i ++) { fprintf(gp4, "%.3f\n", (1.0+sin(a[i].theta))/2.0); }
			fprintf(gp4, "\n\n");
			
			//cell number
			fprintf(gp5, "%d\n", N);
			
			//PSM length
			fprintf(gp6, "%.2f %.2f %.2f\n", timestep*DT, Lxa, (Lx-Lxa));
			
			//advection speed at anterior
			fprintf(gp7, "%.2f %.2f %.2f %.2f\n", timestep*DT, vd0, vd1, vxa);
			
			//arrested cell number
			fprintf(gp8, "%d\n", Nar);
			
		}
		
		
	}
	
	fprintf(gp1, "\n\n");
	fprintf(gp5, "\n\n");
	fprintf(gp6, "\n\n");
	fprintf(gp7, "\n\n");
	fprintf(gp8, "\n\n");
	
	free(ar);
	
}


int main(void)
{
	
	int seed;
	int i, imax;
	
	char str1[1000], str2[1000], str3[1000], str4[1000];
	char str5[1000], str6[1000], str7[1000], str8[1000];
	
	FILE *fp1, *fp2, *fp3, *fp4, *fp5, *fp6, *fp7, *fp8;
	
	seed = time(NULL);
	init_genrand64(seed);
	
	v0 = 1.0; //intrinsic cell velocity min-1
	Dv0 = 0.026; //polarity noise intensity min-1
	mu = 8.71;  //coefficient of forces
	
	sl = 11.0; //cell diameter um
	cr = 11.0; //coupling range um
	
	w0 = 0.2094; //autonomous frequency min-1
	k0 = 0.07; //coupling strength min-1
	k0r = 0.07;
	Dp = 0.0013; //phase noise intensity min-1
	
	mub = 20.0; //boundary force coefficient
	rb0 = 1.0; //lengthscale of boundary force
	
	Xv = 0.4; //lengthscale of cell mixing gradient
	hv = 3.0; //Hill coefficient for spontaneous velocity modulus
	
	//frequency profile
	sigf = 0.66;
	kfrq = 3.07;
	
	Rt = 60.0; //torus radius um
	rt = 25.0; //psm radius um
	
	//torus center
	Xc = 300; //um
	Yc = Rt + rt;
	Zc = rt;
	
	Lx = Xc + Rt + rt; //um
	Ly = 50.0;
	Lz = 50.0;
	
	Ls = 50.0; //segment size (um)
	
	Lxa = 0.0;
	
	//anterior limit of cell addition
	xal = 100.0;
	
	vxa = 0.0; //speed of PSM shortening (um)
	
	Twash = 0;
	Tgs = 0;
	
	imax = 1;
	
	sprintf(str1, "anterior_phase_order_DAPT.dat");
	sprintf(str2, "somite_shapes_DAPT.dat");
	sprintf(str3, "cell_position.dat");
	sprintf(str4, "intensity.dat");
	sprintf(str5, "cell_number.dat");
	sprintf(str6, "PSM_length.dat");
	sprintf(str7, "t_vs_vd0.dat");
	sprintf(str8, "arrested_cell_number.dat");
	
	fp1 = fopen(str1, "w");
	fp2 = fopen(str2, "w");
	fp3 = fopen(str3, "w");
	fp4 = fopen(str4, "w");
	fp5 = fopen(str5, "w");
	fp6 = fopen(str6, "w");
	fp7 = fopen(str7, "w");
	fp8 = fopen(str8, "w");
	
	for (i = 1; i <= imax; i ++) {
		
		k0 = k0r;
		Lxa = 0.0;
		xq = 0.3;
		vd1 = 3.00;
		
		simulation(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8);
		
	}
	
	fclose(fp1); fclose(fp2); fclose(fp3); fclose(fp4);
	fclose(fp5); fclose(fp6); fclose(fp7); fclose(fp8); 
	
	return(0);
	
}

