function doplots(means, savesetid1, savesetid2, savedir, xr, nfigs);
% Usage: doplots(means, savesetid1, savesetid2, savedir, xr, nfigs);
% Use this function after MAIN has completed to view the results.
% If savesetid2 is empty, plots comparisons of truth, prior and posterior from savesetid1.
% If savesetid2 is not empty, plots comparisons of posteriors in the two sets.
% If both savesetids are cell arrays containing two savesetids, then plots
%  and comparisons are done on differences in the resulting curves
%  between corresponding samples in the two savesets.
% xr is the maximum time plotted.
% nfigs is as for plot_survival_curve if savesetid2 is empty.
%  If savesetid2 is empty, nfigs is of length 10, 
%   with the first five for comparisons of prior and truth, 
%   and the next five for comparisons of posterior and truth.
%   In each five, the first is survival probability against time,
%   the second lifetime distribution,
%   the third hazard rate,
%   and the last two for the comparison probabilities of 
%   lifetime and hazard rate respectively against truth.
%  If savesetid2 is not empty, nfigs is of length 5, with
%   the first three for survival curve, lifetime distributions, 
%   and hazard rate, and the last two for comparison probabilities
%   for lifetime and hazard rate respectively.
%  Sensible defaults are provided for nfigs empty or of shorter length.

% Change Log:
%
%     1.1          12:oct:19    rfs34    First version.
%     1.7          12:oct:19    rfs34    Now done some actual work.
%     1.11         14:oct:19    rfs34    Modified to use extended version of plot_survival_curve.m,
%                                        and to handle various comparisons.
%     1.16         14:oct:19    rfs34    Added means option.
%     1.17         15:oct:19    rfs34    Now uses synthetic to plot truth or not; titles fixed.
%     1.23         17:oct:19    rfs34    Added deaths curve to both plots for comparisons also.
%     1.25         17:oct:19    rfs34    Made right limit of plots a parameter also.
%     1.26         17:oct:19    rfs34    Added comparison of hazard rates also.
%     1.37         21:oct:19    rfs34    Fixed plotting death data on wrong plot and bad titles.
%     1.43         23:oct:19    rfs34    Comments only changed.
%     1.50         13:nov:19    rfs34    Difference plots added.
%     1.51         13:nov:19    rfs34    Trivial bug fixed.
%     1.54         15:nov:19    rfs34    Comments in usage instructions fixed.

mytitl = ' /home/rfs/ramakrishnan/software/survival/jc2062/SCCS/s.doplots.m 1.57 20/06/20 12:19:04 ';

persistent mytitldone
if isempty(mytitldone),           
   mytitldone = titlfunction(mytitl);
end

global titls

if nargin < 3,
   savesetid2 = '';
end
if nargin < 4,
   savedir = '';
end
if nargin < 5,
   xr = [];
end
if nargin < 6,
   nfigs = [];
end

if isempty(savedir),
   savedir = '.';
end
if isempty(xr),
   xr = 400;
end
if isempty(nfigs),
   ch = get(0, 'Children');
   if isempty(ch),
      nfigs = 1;
   else
      nfigs = max(ch) + 1;
   end
end
if length(nfigs) == 1,
   nfigs = [0 : 9].' + nfigs;
end

if isempty(savesetid2),
   if length(nfigs) < 10,
      nfigs = [nfigs; NaN(10 - length(nfigs), 1)];
   end

   load(sprintf('%s/save.%s.mat', savedir, savesetid1), ...
        'priorsamples', 'postsamples', 'truesample', 'deaths', 'censored', 'synthetic');

   Npriorsamples = length(priorsamples);

   Npostsamples = length(postsamples);
   Nrunin = min(Npostsamples - 1, ceil(Npostsamples * 0.15));
   postsamples = postsamples(1 + Nrunin : Npostsamples);
   Npostsamples = length(postsamples);

   % First plot truth in comparison with prior samples.

   thistitles = {'Prior samples', 'Prior mean'; ...
                 'Truth among prior samples', 'Truth and prior mean'};
                 
   if synthetic,
      samples = [priorsamples; truesample];
      groups = [ones(Npriorsamples, 1); 2];
      colstrings = {'m', 'g'};
   else
      samples = [priorsamples];
      groups = [ones(Npriorsamples, 1)];
      colstrings = {'m'};
   end

   plot_survival_curve(samples, groups, xr, colstrings, nfigs(1 : 5), ...
                       thistitles{1 + synthetic, 1 + means}, means);

   if ~isnan(nfigs(4)),
      figure(nfigs(4)),
      title('Group 1 is prior samples, Group 2 is truth');
   end

   if ~isnan(nfigs(5)),
      figure(nfigs(5)),
      title('Group 1 is prior samples, Group 2 is truth');
   end

   % Then plot truth in comparison with posterior samples.

   thistitles = {'Posterior samples', 'Posterior mean'; ...
                 'Truth among posterior samples', 'Truth and posterior mean'};
                 
   if synthetic,
      samples = [postsamples; truesample];
      groups = [ones(Npostsamples, 1); 2];
      colstrings = {'b', 'g'};
   else
      samples = [postsamples];
      groups = [ones(Npostsamples, 1)];
      colstrings = {'b'};
   end

   plot_survival_curve(samples, groups, xr, colstrings, nfigs(6 : 10), ...
                       thistitles{1 + synthetic, 1 + means}, means);

   if ~isnan(nfigs(9)),
      figure(nfigs(9)),
      title('Group 1 is posterior samples, Group 2 is truth');
   end

   if ~isnan(nfigs(10)),
      figure(nfigs(10)),
      title('Group 1 is posterior samples, Group 2 is truth');
   end

   if ~isnan(nfigs(6)),
      figure(nfigs(6));
      hold on;
      plot_survival_data(deaths, censored, xr, 'k');
      hold off;
   end

elseif ~iscell(savesetid1) % i.e. if comparing two savesets   
   
   if length(nfigs) < 5,
      nfigs = [nfigs; NaN(5 - length(nfigs), 1)];
   end

   s1 = load(sprintf('%s/save.%s.mat', savedir, savesetid1), 'postsamples', 'deaths', 'censored');
   s2 = load(sprintf('%s/save.%s.mat', savedir, savesetid2), 'postsamples', 'deaths', 'censored');

   Npostsamples1 = length(s1.postsamples);
   Nrunin1 = min(Npostsamples1 - 1, ceil(Npostsamples1 * 0.15));
   s1.postsamples = s1.postsamples(1 + Nrunin1 : Npostsamples1);
   Npostsamples1 = length(s1.postsamples);

   Npostsamples2 = length(s2.postsamples);
   Nrunin2 = min(Npostsamples2 - 1, ceil(Npostsamples2 * 0.15));
   s2.postsamples = s2.postsamples(1 + Nrunin2 : Npostsamples2);
   Npostsamples2 = length(s2.postsamples);

   samples = [s1.postsamples; s2.postsamples];
   one = 1;
   two = 2;
   groups = [one(ones(Npostsamples1, 1)); two(ones(Npostsamples2, 1))];
   colstrings = {'r', 'g'};

   % Randomise the order so that one doesn't obscure the other.
   ind = randperm(Npostsamples1 + Npostsamples2);
   samples = samples(ind);
   groups = groups(ind);

   thistitle = 'Red is group 1 posterior, green is Group 2 posterior';
   plot_survival_curve(samples, groups, xr, colstrings, nfigs(1 : 5), thistitle, means);

   if ~isnan(nfigs(4)),
      figure(nfigs(4)),
      title('Probability that group 1 more likely to survive than group 2 at each time');
   end

   if ~isnan(nfigs(5)),
      figure(nfigs(5)),
      title('Probability that group 1 has higher hazard rate than group 2 at each time');
   end

   if ~isnan(nfigs(1)),
      figure(nfigs(1));
      hold on;
      plot_survival_data(s1.deaths, s1.censored, xr, 'k');
      plot_survival_data(s2.deaths, s2.censored, xr, 'k-.');
      hold off;
   end

elseif iscell(savesetid2),

   if length(nfigs) < 5,
      nfigs = [nfigs; NaN(5 - length(nfigs), 1)];
   end

   s1a = load(sprintf('%s/save.%s.mat', savedir, savesetid1{1}), 'postsamples', 'deaths', 'censored');
   s1b = load(sprintf('%s/save.%s.mat', savedir, savesetid1{2}), 'postsamples', 'deaths', 'censored');
   s2a = load(sprintf('%s/save.%s.mat', savedir, savesetid2{1}), 'postsamples', 'deaths', 'censored');
   s2b = load(sprintf('%s/save.%s.mat', savedir, savesetid2{2}), 'postsamples', 'deaths', 'censored');

   Npostsamples1 = length(s1a.postsamples);
   Npostsamples2 = length(s2a.postsamples);

   if Npostsamples1 ~= length(s1b.postsamples) || Npostsamples2 ~= length(s2b.postsamples),
      error('To make differences of corresponding samples both sets of samples should be of the same length');
   end

   Nrunin1 = min(Npostsamples1 - 1, ceil(Npostsamples1 * 0.15));
   s1a.postsamples = s1a.postsamples(1 + Nrunin1 : Npostsamples1);
   s1b.postsamples = s1b.postsamples(1 + Nrunin1 : Npostsamples1);
   Npostsamples1 = length(s1a.postsamples);

   Nrunin2 = min(Npostsamples2 - 1, ceil(Npostsamples2 * 0.15));
   s2a.postsamples = s2a.postsamples(1 + Nrunin2 : Npostsamples2);
   s2b.postsamples = s2b.postsamples(1 + Nrunin2 : Npostsamples2);
   Npostsamples2 = length(s2a.postsamples);

   samples = [s1a.postsamples, s1b.postsamples; s2a.postsamples, s2b.postsamples];
   one = 1;
   two = 2;
   groups = [one(ones(Npostsamples1, 1)); two(ones(Npostsamples2, 1))];
   colstrings = {'r', 'g'};

   % Randomise the order so that one doesn't obscure the other.
   ind = randperm(Npostsamples1 + Npostsamples2);
   samples = samples(ind, :);
   groups = groups(ind, :);

   thistitle = 'Red is group 1 posterior, green is Group 2 posterior';
   plot_survival_curve(samples, groups, xr, colstrings, nfigs(1 : 5), thistitle, means);

   if ~isnan(nfigs(4)),
      figure(nfigs(4)),
      title('Probability that group 1 survival difference greater than group 2 at each time');
   end

   if ~isnan(nfigs(5)),
      figure(nfigs(5)),
      title('Probability that group 1 hazard rate difference greater than group 2 at each time');
   end

else
   error('Both savesetids must be cells of length 2 if either is');
end

return;


% Local Variables: 
% indent-line-function: indent-relative
% eval: (auto-fill-mode 0)
% End:
