function MAIN(data, synthseed, analysisseed, savesetid, savedir);
% Usage: MAIN(data, synthseed, analysisseed, savesetid, savedir);
% data is a column vector of times of either observed deaths (if > 0)
%  or of censoring times (represented by -1 times their positive values).
% If deaths is of size [Npatients, 0] then synthetic data using that
%  number of patients is constructed.
% Either way the results are analysed and saved to the file with name
%  <savedir>/save.<savesetid>.mat; note that savesetid should be a character array.
% You can then use doplots.m to view the results, 
%  followed if you wish by sequence.m or sizzle.m .
% The model in use is described in ~rfs/ramakrishnan/modeldef/paper.pdf .
% In addition to this note that additional variables, known as
%  adherences, are assumed, which fix which mode of death is responsible
%  for each observed death. Samples are taken from the joint distribution 
%  of the survival distribution parameters, the adherences, and the 
%  times x at which the adherent and other modes might have killed the patient.
%  Then the x and adherences are effectively discarded and the survival
%  distributions and hazard rates plotted (in doplots.m).

% Change Log:
%
%     1.1          28:sep:19    jc2062   As first received from JC2062.
%     1.2          10:oct:19    jc2062   As received from jc2062.
%     1.3          11:oct:19    rfs34    Many trivial adjustments of a bookkeeping
%                                        nature, plus passing 1/r_r instead of r_r
%                                        to priorcoords. 
%     1.5          11:oct:19    rfs34    Survival curve right limit increased.
%     1.7          12:oct:19    rfs34    Loads of admin stuff added.
%     1.8          12:oct:19    rfs34    Rejigged initial setup of x and adherences.
%     1.9          12:oct:19    rfs34    Silly bug fixed.
%     1.15         14:oct:19    rfs34    Several trivial bugs fixed; appears to be working.
%     1.17         15:oct:19    rfs34    Made separate variable data for data in form input.
%     1.20         16:oct:19    rfs34    Resolving bugs caused by cheating.
%     1.21         16:oct:19    rfs34    Typo fixed.
%     1.24         17:oct:19    rfs34    Nsamples now 3000.
%     1.25         17:oct:19    rfs34    Prevented printout of Npriorsamples.
%     1.32         18:oct:19    rfs34    Now write protects save files.
%     1.42         22:oct:19    rfs34    Increased censoring of synthetic data to maximum.
%     1.43         23:oct:19    rfs34    Comments only changed.
%     1.46         05:nov:19    rfs34    Switched to priors agreed at meeting on 4/11/2019,
%                                        and limited samples in plots to 500, and now
%                                        runs 300 prior samples and 10000 posterior samples.

mytitl = ' /home/rfs/ramakrishnan/software/survival/jc2062/SCCS/s.MAIN.m 1.57 20/06/20 12:19:04 ';

persistent mytitldone
if isempty(mytitldone),           
   mytitldone = titlfunction(mytitl);
end

global titls

% RFS: I've deleted the two variants of MAIN.m from the set, as having
%      multiple variants increases risk of something not getting updated
%      in one of the variants the same as in one of the other.
%      Rather, one should have variables (or better, parameters) which
%      switch between the variants.

global synthstartstate analysisstartstate

if nargin < 1,
   data = [];
end
if nargin < 2,
   synthseed = [];
end
if nargin < 3,
   analysisseed = [];
end
if nargin < 4,
   savesetid = '';
end
if nargin < 5,
   savedir = '';
end

if isempty(savedir),
   savedir = '.';
end

N = size(data, 1);
synthetic = size(data, 2) == 0;

% Set up the parameters of the priors.
priorcoords = setpriorcoords;
[a_J, a_p, b_p, a_m, b_m, m_r, b_r, a_k, b_k, c_k, l_k] = priorcoords{:};
r_r = 1 / b_r;

if synthetic,

   % Start of synthesis.

   if ~isempty(synthseed),
      restorestate = rng(synthseed);
   end
   synthstartstate = rng;

   % Generate some sample data.

   y = generate_params_and_data(N,priorcoords, 1, {2}, {2}, {2}, {2}, {2});
   [true_J, true_p, true_m, true_r, true_k, true_adherences, true_x, deaths] = y{:};

   censored = true_adherences == 0;
   deaths(censored) = - deaths(censored);
   data = deaths;

   if ~isempty(synthseed),
      rng(restorestate);
   end

   % End of synthesis.

else % i.e. if using real data

   list = [{0}, repmat({zeros(1, 0)}, [1, 4]), {repmat(NaN, [N, 1]), zeros(N, 0)}];
   [true_J, true_p, true_m, true_r, true_k, true_adherences, true_x] ...
      = deal(list{:});
   % Note deaths omitted from this list.

end % if synthetic or not

% Start of analysis.

if ~isempty(analysisseed),
   restorestate = rng(analysisseed);
end
analysisstartstate = rng;

% We keep the variable data to record what was input.

deaths = data;

censored = deaths < 0;
deaths = abs(deaths);   

if any(deaths == 0),
   error('Not permitted to have deaths at exactly time zero');
end

Npriorsamples = 300;
psprior = cell(Npriorsamples, 1);
msprior = cell(Npriorsamples, 1);
rsprior = cell(Npriorsamples, 1);
ksprior = cell(Npriorsamples, 1);
Jsprior = cell(Npriorsamples, 1);
adhsprior = cell(Npriorsamples, 1);
xsprior = cell(Npriorsamples, 1);

% Make a pile of prior samples of the parameters to be saved
% with the rest of the output for later use in post-processing.

fprintf('Generating prior samples...\n');
for nsample = 1 : Npriorsamples,

   yp = generate_params_and_data(N,priorcoords, 0, {2}, {2}, {2}, {2}, {2}, 0, {2}, {2});
   [J, p, m, r, k, adherences, x] = yp{:};
   Jsprior{nsample} = J;
   psprior{nsample} = p;
   msprior{nsample} = m;
   rsprior{nsample} = r;
   ksprior{nsample} = k;
   adhsprior{nsample} = adherences;
   xsprior{nsample} = x;

   fprintf('\r...done %d of %d...', nsample, Npriorsamples);

end % for nsample = 1 : Npriorsamples
fprintf('\r...finished.                       \n');

priorsamples = struct('J', Jsprior, ...
                      'p', psprior, ...
                      'm', msprior, ...
                      'r', rsprior, ...
                      'k', ksprior, ...
                      'adh', adhsprior, ...
                      'x', xsprior);

% Set axes sampling or cheating.
% 2 means normal sampling, 
% 1 means cheating at the start then normal sampling, and
% 0 means fixed at the true value.

J_axis = 2;
p_axis = 2;
m_axis = 2;
r_axis = 2;
k_axis = 2;
x_axis = 2;
adherence_axis = 2;

if adherence_axis && ~x_axis,
   error('We are using an adherence resampling method which forces resampling x');
end
if J_axis && (~x_axis || ~adherence_axis),
   error('Cannot resample J without forcing resampling of x and adherence');
end

% Sample initial point for the Markov Chain.
% This now takes into account the known death and censoring times.

y2 = generate_params_and_data(N,priorcoords, 0, {J_axis, true_J}, {p_axis, true_p}, ...
                              {m_axis, true_m}, {r_axis, true_r}, {k_axis, true_k}, 0, ...
                              {adherence_axis, true_adherences}, {x_axis, true_x}, deaths, censored);
[J, p, m, r, k, adherences, x] = y2{:};

% Run palindromic sampling for Nsamples iterations.

Nsamples = 10000;

% RFS: I approve of using small letters to iterate up to big letters !
%      Even better to call them e.g. nsample = 1 : Nsamples, so that
%      you remember what each variable is iterating over.

% RFS: Moreover, using i as an iterator in Matlab is dangerous, as you
%      are changing the value of the complex basis vector !
%      The same applies to using eps as a variable, in case you are
%      ever tempted to - it is automatically global, and used by
%      loads of built-in functions ! (It took me AGES to realise what
%      was wrong when I first did this.)

% RFS: Better to allocate such things at the desired length in advance,
%      each time you extend an array you waste malloc and copying effort.
ps = cell(Nsamples, 1);
ms = cell(Nsamples, 1);
rs = cell(Nsamples, 1);
ks = cell(Nsamples, 1);
Js = cell(Nsamples, 1);
adhs = cell(Nsamples, 1);
xs = cell(Nsamples, 1);

% RFS: Make sure you record where you started !
rs{1} = r;
ms{1} = m;
ps{1} = p;
ks{1} = k;
Js{1} = J;
adhs{1} = adherences;
xs{1} = x;

fprintf('\nGenerating posterior samples...\n');
for nsample = 1 : Nsamples - 1,
   % RFS: Don't store samples during the potentially omitted parts of the sampling
   %      sequence, store them once at the end of each complete cycle. That way
   %      fixed values also get stored and can be displayed.

   if k_axis,
      k = k_sample(J, m, r, k, x, priorcoords);
   end

   if p_axis,
      p = p_sample(x, a_p, b_p);
   end

   if m_axis,
      m = m_sample(J, r, k, x, a_m, b_m);
   end

   if r_axis,
      r = r_sample(J, m, r, k, x, m_r, r_r);
   end

   % RFS: There's no point taking new x samples at this point, they're
   %      about to be discarded by adherence sample (unless adherence axis
   %      isn't being resampled).
   if x_axis && ~adherence_axis,
      x = x_sample(J, p, m, r, k, x, deaths, adherences);
   end

   if adherence_axis,
      y = adherence_sample(J, p, m, r, k, x, deaths, adherences);
      [x, adherences] = y{:};
   end
     
   if J_axis,
      y = J_sample(J, p, m, r, k, x, adherences, deaths, priorcoords);
      [J, p, m, r, k, x, adherences] = y{:};
   end

   % RFS: No point doing another J_sample here; palindromic only needs e.g.
   %      WXYZYX to be repeated (no repetition of Z or of W,
   %      the turn round points, though in theory at the end of the entire
   %      run one should do a final W, though it makes little difference in
   %      practice).
     
   if adherence_axis,
      y = adherence_sample(J, p, m, r, k, x, deaths, adherences);
      [x, adherences] = y{:};
   end

   if x_axis && ~adherence_axis,
      x = x_sample(J, p, m, r, k, x, deaths, adherences);
   end

   if r_axis,
      r = r_sample(J, m, r, k, x, m_r, r_r);
   end

   if m_axis,
      m = m_sample(J, r, k, x, a_m, b_m);
   end

   if p_axis,
      p = p_sample(x, a_p, b_p);
   end

   % RFS: No point doing another k sample here, as discussed above.

   rs{nsample + 1} = r;
   ms{nsample + 1} = m;
   ps{nsample + 1} = p;
   ks{nsample + 1} = k;
   Js{nsample + 1} = J;
   adhs{nsample + 1} = adherences;
   xs{nsample + 1} = x;

   fprintf('\r...done %d of %d...', nsample + 1, Nsamples);
end
fprintf('\r...finished.             \n');

postsamples = struct('J', Js, ...
                     'p', ps, ...
                     'm', ms, ...
                     'r', rs, ...
                     'k', ks, ...
                     'adh', adhs, ...
                     'x', xs);

truesample = struct('J', {true_J}, ...
                     'p', {true_p}, ...
                     'm', {true_m}, ...
                     'r', {true_r}, ...
                     'k', {true_k}, ...
                     'adh', {true_adherences}, ...
                     'x', {true_x});

% Save all variables and data under suitable conditions.

if ~isempty(analysisseed),
   rng(restorestate);
end

% End of analysis.

% Save results.
if isempty(savesetid),
   fprintf('Set savesetid and savedir now if you want to save results, then/else dbcont;');
   keyboard;
end

if ~isempty(savesetid),
   save(sprintf('%s/save.%s.mat', savedir, savesetid));
   if isunix,
      unix(sprintf('chmod a-w %s/save.%s.mat', savedir, savesetid));
   end
end

% Local Variables: 
% indent-line-function: indent-relative
% eval: (auto-fill-mode 0)
% End:
