function plot_survival_curve(samples, groups, ubound, colstrings, nfigs, thistitle, means);
% Usage: plot_survival_curve(samples, groups, ubound, colstrings, nfigs, thistitle, means);
% samples is a structure array of size [Nsamples, 1] or [Nsamples, 2] giving 
%  the parameters of the curves to be plotted.
% If size(samples, 2) == 2 then curves are made from from 
%  each pair of samples in a row of samples and subtracted (1 minus 2)
%  before being compared or plotted.
% groups is an array of size [Nsamples, 1] saying
%  which group each sample belongs to.
% colstrings is a cell array of size {Ngroups, 1} giving
%  the colours wanted for each group.
% ubound gives the largest time of interest in days.
% nfigs is an array of figure handles of size [3, 1]:
%  nfigs(1) is the figure number for survival fraction;
%  nfigs(2) is the figure number for lifetime distribution;
%  nfigs(3) is the figure number for hazard rate distribution;
%  nfigs(4) is the figure number for comparison probabilities of survival;
%  nfigs(5) is the figure number for comparison probabilities of hazard rate.
%  If any of these is NaN then that figure doesn't get plotted.
% means should be 1 to plot the mean things, and 0 to plot the samples.
% thistitle gives the title for any figures without one provided in this function.

% Change Log:
%
%     1.1          28:sep:19    jc2062   As first received from JC2062.
%     1.2          10:oct:19    jc2062   As received from jc2062.
%     1.7          12:oct:19    rfs34    Modified to take a structure input.
%     1.11         14:oct:19    rfs34    Modified to take multiple samples to 
%                                        be plotted and do various different plots.
%     1.14         14:oct:19    rfs34    Trivial bugs fixed.
%     1.15         14:oct:19    rfs34    Several trivial bugs fixed.
%     1.16         14:oct:19    rfs34    Added means option.
%     1.17         15:oct:19    rfs34    means added to lifetime distribution also;
%                                        equality now is half a count for comparison.
%     1.23         17:oct:19    rfs34    Removed spurious return parameter.
%     1.26         17:oct:19    rfs34    Added comparison probabilities of hazard rates also.
%     1.27         17:oct:19    rfs34    Trivial typo fixed.
%     1.28         17:oct:19    rfs34    Removed erroneous first point from hazard rate 
%                                        comparison probability plot.
%     1.29         17:oct:19    rfs34    Added progress reporting to comparison tasks.
%     1.30         17:oct:19    rfs34    Typos in above fixed.
%     1.32         18:oct:19    rfs34    Tried to speed up calculation of comparison probabilities.
%     1.33         18:oct:19    rfs34    Fixed missing parentheses.
%     1.43         23:oct:19    rfs34    Comments only changed.
%     1.46         05:nov:19    rfs34    Limits number of samples plotted to 500 without
%                                        limiting number of samples used to calculate means
%                                        or comparison probabilities.
%     1.50         13:nov:19    rfs34    Gap comparison added.
%     1.52         13:nov:19    rfs34    Left axes on auto when comparing gaps.
%     1.53         15:nov:19    rfs34    Added 0.025 and 0.975 quantiles to plots of means of
%                                        survival probability and hazard rate.

mytitl = ' /home/rfs/ramakrishnan/software/survival/jc2062/SCCS/s.plot_survival_curve.m 1.57 20/06/20 12:19:04 ';

persistent mytitldone
if isempty(mytitldone),           
   mytitldone = titlfunction(mytitl);
end

global titls

if ~iscell(colstrings),
   colstrings = {colstrings};
end

Nmaxsamplestoplot = 500;
lowquant = 0.025;
highquant = 0.975;

% If this is 1 then uses ndgrid to make comparisons.
% If 0, then each sample of group 2 is compared against all samples 
%  of group 1, and the comparisons summed.
vectorised = 0;

deltax = 0.1;
xx = [deltax : deltax : ubound];
Nxx = length(xx);

Nsamples = size(samples, 1);
dogaps = size(samples, 2) == 2;

if ~ismember(size(samples, 2), [1; 2]),
   error('Expecting only one column or two columns of samples');
end

if ~dogaps,

   pas = NaN(Nsamples, Nxx);

   for nsample = 1 : Nsamples,
      sample = samples(nsample);
   
      J = sample.J;
      p = sample.p;
      m = sample.m;
      r = sample.r;
      k = sample.k;
      
      pa = probability_alive(xx, J, p, m, r, k);
   
      pas(nsample, :) = pa;
   end

   haz = [NaN(Nsamples, 1), -diff(log(pas), [], 2)] / deltax;

else % i.e. if considering differences between savesets

   pas1 = NaN(Nsamples, Nxx);
   pas2 = NaN(Nsamples, Nxx);

   for nsample = 1 : Nsamples,
      sample1 = samples(nsample, 1);
      sample2 = samples(nsample, 2);
   
      J1 = sample1.J;
      p1 = sample1.p;
      m1 = sample1.m;
      r1 = sample1.r;
      k1 = sample1.k;
      
      J2 = sample2.J;
      p2 = sample2.p;
      m2 = sample2.m;
      r2 = sample2.r;
      k2 = sample2.k;
      
      pa1 = probability_alive(xx, J1, p1, m1, r1, k1);
      pa2 = probability_alive(xx, J2, p2, m2, r2, k2);
   
      pas1(nsample, :) = pa1;
      pas2(nsample, :) = pa2;
   end

   pas = pas1 - pas2;
   haz = ([NaN(Nsamples, 1), -diff(log(pas1), [], 2)] / deltax ...
          - [NaN(Nsamples, 1), -diff(log(pas2), [], 2)] / deltax);
end

Ngroups = length(colstrings);

if ~isnan(nfigs(1)) && means,
   meanpas = NaN(Ngroups, Nxx);
   lowpas = NaN(Ngroups, Nxx);
   highpas = NaN(Ngroups, Nxx);
   for ngroup = 1 : Ngroups,
      pasgroup = pas(groups == ngroup, :);
      meanpas(ngroup, :) = mean(pasgroup, 1);
      sortedpas = sort(pasgroup, 1, 'ascend');
      Npasgroup = size(pasgroup, 1);
      nlow = max(1, min(Npasgroup, round(lowquant * Npasgroup)));
      lowpas(ngroup, :) = sortedpas(nlow, :);
      nhigh = max(1, min(Npasgroup, round(highquant * Npasgroup)));
      highpas(ngroup, :) = sortedpas(nhigh, :);
   end
end

if ~isnan(nfigs(2)) && means,
   meanltd = NaN(Ngroups, Nxx);
   for ngroup = 1 : Ngroups,
      meanltd(ngroup, :) = mean([NaN(sum(groups == ngroup), 1), - diff(pas(groups == ngroup, :), [], 2)], 1) / deltax;
   end
end

if ~isnan(nfigs(3)) && means,
   meanhaz = NaN(Ngroups, Nxx);
   lowhaz = NaN(Ngroups, Nxx);
   highhaz = NaN(Ngroups, Nxx);
   for ngroup = 1 : Ngroups,
      hazgroup = haz(groups == ngroup, :);
      meanhaz(ngroup, :) = mean(hazgroup, 1);
      sortedhaz = sort(hazgroup, 1, 'ascend');
      Nhazgroup = size(hazgroup, 1);
      nlow = max(1, min(Nhazgroup, round(lowquant * Nhazgroup)));
      lowhaz(ngroup, :) = sortedhaz(nlow, :);
      nhigh = max(1, min(Nhazgroup, round(highquant * Nhazgroup)));
      highhaz(ngroup, :) = sortedhaz(nhigh, :);
   end
end

if ~dogaps,
   ylabels = {'Probability still alive', ...
              'Probability density of lifetime', ...
              'Hazard rate (days^{-1})', ...
              'P(group 1 survival > group 2 survival)', ...
              'P(group 1 hazard rate > group 2 hazard rate)'};
else
   ylabels = {'Difference in probability still alive', ...
              'Difference in probability density of lifetime', ...
              'Difference in hazard rate (days^{-1})', ...
              'P(diff in group 1 survival > diff in group 2 survival)', ...
              'P(diff in group 1 hazard rate > diff in group 2 hazard rate)'};
end

% Calculate the indices to plot.
% Always plot the last one, which may be the truth.
% Limit the number of samples to about Nmaxsamplestoplot.
gap = max(1, floor(Nsamples / Nmaxsamplestoplot));
Nnew = ceil(Nsamples / gap);
ind = [Nsamples - (Nnew - 1) * gap : gap : Nsamples];

for nf = 1 : 3,

   nfig = nfigs(nf);
   if ~isnan(nfig),
      figure(nfig);
      clf;
      hold on;

      switch nf,
         case 1,
            if ~means,
               for nsample = ind,
                  plot(xx, pas(nsample, :), colstrings{groups(nsample)});
               end
            else
               for ngroup = 1 : Ngroups,
                  plot(xx, meanpas(ngroup, :), colstrings{ngroup}, ...
                       xx, lowpas(ngroup, :), [colstrings{ngroup}(1), ':'], ...
                       xx, highpas(ngroup, :), [colstrings{ngroup}(1), ':']);
               end
            end
            if ~dogaps,
               axis([xx(1), xx(end), -0.01, 1.01]);
            end

         case 2,
            if ~means,
               for nsample = ind,
                  plot(xx, [NaN, - diff(pas(nsample, :), [], 2) ./ deltax], colstrings{groups(nsample)});
               end
            else
               for ngroup = 1 : Ngroups,
                  plot(xx, meanltd(ngroup, :), colstrings{ngroup});
               end
            end

         case 3,
            if ~means,
               for nsample = ind,
                  plot(xx, haz(nsample, :), colstrings{groups(nsample)});
               end
            else
               for ngroup = 1 : Ngroups,
                  plot(xx, meanhaz(ngroup, :), colstrings{ngroup}, ...
                       xx, lowhaz(ngroup, :), [colstrings{ngroup}(1), ':'], ...
                       xx, highhaz(ngroup, :), [colstrings{ngroup}(1), ':']);
               end
            end

         otherwise,
            error('Illegal value of nf');
      end

      hold off;
      xlabel('Time (days)');
      ylabel(ylabels{nf});
      title(thistitle);
   end
end

nfig = nfigs(4);
if ~isnan(nfig),
   figure(nfig);

   fprintf('Working on comparison probabilities for probability of survival...\n');
   pg = NaN(Nxx, 1);
   for nx = 1 : Nxx,

      if vectorised,
         [pas1, pas2] = ndgrid(pas(groups == 1, nx), pas(groups == 2, nx));
         padiff = pas1 - pas2;
         pg(nx) = (sum(padiff(:) > 0) + 0.5 * sum(padiff(:) == 0)) / numel(padiff);
      else
         pas1 = pas(groups == 1, nx);
         pas2 = pas(groups == 2, nx);
         tot = 0;
         for nsample = 1 : length(pas2),
            tot = tot + sum(pas1(:) > pas2(nsample)) + 0.5 * sum(pas1(:) == pas2(nsample));
         end
         pg(nx) = tot / (numel(pas1) * numel(pas2));
      end

      fprintf('\r...done %d of %d...', nx, Nxx);
   end
   fprintf('\r...finished.                  \n');

   plot(xx, pg, 'b');
   axis([xx(1), xx(end), -0.01, 1.01]);
   xlabel('Time (days)');
   if ~dogaps,
      ylabel('Probability group 1 more likely alive than group 2');
   else
      ylabel('Probability diff in group 1 survival prob greater than diff in group 2');
   end
end
   
nfig = nfigs(5);
if ~isnan(nfig),
   figure(nfig);

   fprintf('Working on comparison probabilities for hazard rate...\n');
   hg = NaN(Nxx, 1);
   for nx = 2 : Nxx, % Omit 1 as it is a comparison of NaN with NaN.

      if vectorised, % This is rather slow if Nsamples is large.
         [haz1, haz2] = ndgrid(haz(groups == 1, nx), haz(groups == 2, nx));
         hadiff = haz1 - haz2;
         hg(nx) = (sum(hadiff(:) > 0) + 0.5 * sum(hadiff(:) == 0)) / numel(hadiff);
      else
         haz1 = haz(groups == 1, nx);
         haz2 = haz(groups == 2, nx);
         tot = 0;
         for nsample = 1 : length(haz2),
            tot = tot + sum(haz1(:) > haz2(nsample)) + 0.5 * sum(haz1(:) == haz2(nsample));
         end
         hg(nx) = tot / (numel(haz1) * numel(haz2));
      end
       
      fprintf('\r...done %d of %d...', nx, Nxx);
   end
   fprintf('\r...finished.                  \n');

   plot(xx, hg, 'b');
   axis([xx(1), xx(end), -0.01, 1.01]);
   xlabel('Time (days)');
   if ~dogaps,
      ylabel('Probability group 1 has higher hazard rate than group 2');
   else
      ylabel('Probability diff in group 1 hazard rate greater than diff in group 2');
   end
end
   
return;

% Local Variables: 
% indent-line-function: indent-relative
% eval: (auto-fill-mode 0)
% End:
