import datetime
import os

p_material = '/Users/tstoeger/Dropbox/hostgenes/results/material'


def get_material_path(p, date=False):
    """
    Takes extension path p, and makes it as a subfile
    within material folder
    """

    p = os.path.join(
        p_material,
        p)

    if date:
        [fo, fn] = os.path.split(p)
        [fb, ext] = os.path.splitext(fn)
        dt = datetime.datetime.today().strftime('%y%m%d_%H%M')
        p = os.path.join(fo, fb + '_' + dt + ext)


    return p


def export_image(p, date=False):
    """
    Will export a current figure;
        - makes font edit-able in illustrator

    Input:
        date    optional; default True: will insert
                                date, hour, minute before file
                                extension
    """

    import matplotlib as mpl
    mpl.rcParams['pdf.fonttype'] = 42  # edit-able in illustrator

    p = os.path.join(
        p_material,
        p)

    if date:
        [fo, fn] = os.path.split(p)
        [fb, ext] = os.path.splitext(fn)
        dt = datetime.datetime.today().strftime('%y%m%d_%H%M')
        p = os.path.join(fo, fb + '_' + dt + ext)

    ensure_presence_of_directory(p)

    mpl.pyplot.savefig(p, bbox_inches='tight')


def export_raster_image(p, dpi, date=False):
    """
    Will export a current figure with 600 dpi

    Input:
        p       str path to file
        dpi     int dots per inch
        date    optional; default True: will insert
                                date, hour, minute before file
                                extension
    """

    import matplotlib as mpl
    mpl.rcParams['pdf.fonttype'] = 42  # edit-able in illustrator

    p = get_material_path(p, insert_date_time)
    ensure_presence_of_directory(p)

    mpl.pyplot.savefig(p, dpi=dpi, bbox_inches='tight')


def export_full_frame(p, df, date=False, save_index=False):
    """
    Will export a dataframe to materials
        - makes font edit-able in illustrator

    Input:
        p                   subpath within material folder
        df                  dataframe to export
        date                optional; default True: will insert
                                date, hour, minute before file
                                extension
        save_index          optional; default True: will also
                                export the index
    """

    p = os.path.join(
        p_material,
        p)

    if p.endswith('.csv.gz'):
        p = p[:-3]
        compress = True
        file_format = 'csv'
    elif p.endswith('.csv'):
        compress = False
        file_format = 'csv'
    elif p.endswith('.xlsx'):
        file_format = 'xlsx'
    else:
        raise EnvironmentError(
            'No support for preseent file type.')

    if date:
        [fo, fn] = os.path.split(p)
        [fb, ext] = os.path.splitext(fn)
        dt = datetime.datetime.today().strftime('%y%m%d_%H%M')
        p = os.path.join(fo, fb + '_' + dt + ext)

    ensure_presence_of_directory(p)

    if file_format == 'csv':
        if compress:
            p = p + '.gz'
            df.to_csv(p, compression='gzip', index=save_index)
        else:
            df.to_csv(p, index=save_index)
    elif file_format == 'xlsx':
        df.to_excel(p, index=save_index)


def ensure_presence_of_directory(directory_path=None, ):
    '''
    Ensure that the directory of exists. Creates dictionary with cognate
    name in case that directory does not exist. Anticipates that files have
    extensions separated by '.' symbol (can not create directory with . in
    its name); If file does not have an extension separated by '.' a folder
    will with its filname will be created, a behavior that can be avoided
    by calling os.path.dirname prior this function.

    Input:
        directory_path      str; Name of a directory or the full path of a file
    '''
    if directory_path is None:
        raise ValueError('No input specfied for ensure_presence_of_directory')

    directory_path_n, ext = os.path.split(directory_path)

    if '.' in ext:
        directory_path = directory_path_n

    if not os.path.exists(directory_path):
        os.makedirs(directory_path)
        