#include <TMB.hpp>
template<class Type>
Type objective_function<Type>::operator() ()
{
  using namespace R_inla;
  using namespace density;
  using namespace Eigen;
  
  // Input Data
  
  DATA_INTEGER(nHFs);
  DATA_INTEGER(nMonths);

  DATA_STRUCT(spde,spde_t); // INLA SPDE object (components of precision matrix)
  DATA_SPARSE_MATRIX(A); // INLA SPDE projection matrix: mesh to HFs [dim: nHFs x nMesh]
  
  DATA_MATRIX(HFcases_matrix); // [dim: nHFs x 72]  
  DATA_MATRIX(HFmissing_matrix); // [dim: nHFs x 72] 
  DATA_VECTOR(catchment_pop); // [length: nHFs]  
  DATA_MATRIX(temporal_spline_matrix); // [dim: 3 x 72]  
  
  DATA_MATRIX(intermonth_distmat); // [dim: 12 x 12]
  DATA_MATRIX(micfracs); // [dim: nHFs x 72]
  
  // Parameters
  
  PARAMETER(intercept_baseline);
  PARAMETER(log_range_baseline);
  PARAMETER(log_sd_baseline);
  PARAMETER(log_range_trend);
  PARAMETER(log_sd_trend);
  PARAMETER(log_ar_scale_trend);
  PARAMETER(logit_ar_param_trend);
  PARAMETER(log_range_seasonal);
  PARAMETER(log_sd_seasonal);
  PARAMETER(log_range_seasonaltime);
  PARAMETER(log_sd_seasonaltime);
  PARAMETER(log_overdispersion_sd);
  PARAMETER(mean_miceffect);
  PARAMETER(log_ar_scale_miceffect);
  PARAMETER(logit_ar_param_miceffect);
  PARAMETER_VECTOR(miceffect); // [length: 72]
  PARAMETER_VECTOR(field_baseline); // [length: nMesh]
  PARAMETER_ARRAY(field_trend); // [dim: nMesh x 3]
  PARAMETER_ARRAY(field_seasonal); // [dim: nMesh x 12]
  PARAMETER_VECTOR(log_overdispersion_factors); // [length: nHFs]
  
  // Parameter Transforms

  Type range_baseline = exp(log_range_baseline);
  Type kappa_baseline = 2.8284/range_baseline;
  Type sd_baseline = exp(log_sd_baseline);
  Type range_trend = exp(log_range_trend);
  Type kappa_trend = 2.8284/range_trend;
  Type sd_trend = exp(log_sd_trend);
  Type ar_scale_trend = exp(log_ar_scale_trend);
  Type ar_param_trend = invlogit(logit_ar_param_trend)*2.0-1.0;
  Type range_seasonal = exp(log_range_seasonal);
  Type kappa_seasonal = 2.8284/range_seasonal;
  Type sd_seasonal = exp(log_sd_seasonal);
  Type range_seasonaltime = exp(log_range_seasonaltime);
  Type var_seasonaltime = exp(log_sd_seasonaltime*2.0);
  Type overdispersion_sd = exp(log_overdispersion_sd);
  Type ar_scale_miceffect = exp(log_ar_scale_miceffect);
  Type ar_param_miceffect = invlogit(logit_ar_param_miceffect)*2.0-1.0;
  vector<Type> overdispersion_factors = exp(log_overdispersion_factors);
  
  // Priors
  
  Type nll = 0.0;
  
  nll -= dnorm(log_overdispersion_sd,Type(-1.0),Type(1.0));
  nll -= (dnorm(log_overdispersion_factors,Type(0),overdispersion_sd,true)).sum();
  
  nll -= dnorm(log_sd_baseline,Type(2),Type(1),true);
  nll -= dnorm(log_range_baseline,Type(-1),Type(1),true);
  SparseMatrix<Type> Q_baseline = Q_spde(spde,kappa_baseline);
  nll += SCALE(GMRF(Q_baseline),sd_baseline)(field_baseline);
  
  nll -= dnorm(log_sd_trend,Type(-1),Type(1),true);
  nll -= dnorm(log_range_trend,Type(1),Type(1),true);
  nll -= dnorm(log_ar_scale_trend,Type(-1),Type(1),true);
  nll -= dnorm(logit_ar_param_trend,Type(1),Type(1),true);
  SparseMatrix<Type> Q_trend = Q_spde(spde,kappa_trend);
  nll += SEPARABLE(SCALE(AR1(ar_param_trend),ar_scale_trend),SCALE(GMRF(Q_trend),sd_trend))(field_trend);
  
  nll -= dnorm(log_sd_seasonal,Type(-1),Type(1),true);
  nll -= dnorm(log_range_seasonal,Type(1),Type(1),true);
  nll -= dnorm(log_sd_seasonaltime,Type(-1),Type(1),true);
  nll -= dnorm(log_range_seasonaltime,Type(1),Type(1),true);
  SparseMatrix<Type> Q_seasonal = Q_spde(spde,kappa_seasonal);
  matrix<Type> tempcov(12,12);
  for (int i=0; i<12; i++) {
    for (int j=0; j<12; j++) {
      tempcov(i,j) = var_seasonaltime*exp(-intermonth_distmat(i,j)/range_seasonaltime);
    }
  }
  for (int i=0; i<12; i++) {
    tempcov(i,i) += 0.0001;
  }
  nll += SEPARABLE(MVNORM(tempcov),SCALE(GMRF(Q_seasonal),sd_seasonal))(field_seasonal);
  
  nll -= dnorm(log_ar_scale_miceffect,Type(-1.0),Type(1.0),true);
  nll -= dnorm(logit_ar_param_miceffect,Type(2.0),Type(1.0),true);
  nll -= dnorm(mean_miceffect,Type(0.0),Type(1.0),true);
  nll += SCALE(AR1(ar_param_miceffect),ar_scale_miceffect)(miceffect);
  
  // Algebra: Construct fields
   
  vector<Type> baseline_incidence_rate(nHFs);
  baseline_incidence_rate = A*field_baseline + intercept_baseline;
   
  matrix<Type> trend_incidence_rate(nHFs,nMonths);
  trend_incidence_rate = (A*field_trend.matrix())*temporal_spline_matrix;
  
  matrix<Type> seasonal_incidence_rate(nHFs,12);
  seasonal_incidence_rate = A*field_seasonal.matrix();
  
  matrix<Type> total_incidence_field(nHFs,nMonths);
  for (int i=0; i<nHFs; i++) {
    for (int j=0; j<nMonths; j++) {
      total_incidence_field(i,j) = exp((miceffect[j]+mean_miceffect)*micfracs(i,j) + baseline_incidence_rate[i] + trend_incidence_rate(i,j) + seasonal_incidence_rate(i,j % 12));
    }
  }
  
  matrix<Type> miceffect_matrix(nHFs,nMonths);
  for (int i=0; i<nHFs; i++) {
    for (int j=0; j<nMonths; j++) {
      miceffect_matrix(i,j) = exp((miceffect[j]+mean_miceffect)*micfracs(i,j));    
    }
  }
  
  // Likelihood

  for (int i=0; i<nHFs; i++) {
    for (int j=0; j<nMonths; j++) {
      if (HFmissing_matrix(i,j)==0) {
        nll -= dpois(HFcases_matrix(i,j),total_incidence_field(i,j)*catchment_pop[i]*overdispersion_factors[i],true);
      }
    }
  }
  
  // Reporting

  REPORT(baseline_incidence_rate);
  REPORT(trend_incidence_rate);
  REPORT(seasonal_incidence_rate);
  REPORT(total_incidence_field);
  REPORT(miceffect_matrix);
  
  return nll;
}
