#include <TMB.hpp>
template<class Type>
Type objective_function<Type>::operator() ()
{
  using namespace R_inla;
  using namespace density;
  using namespace Eigen;
  
  // Input Data
  
  DATA_INTEGER(bigN); // number of pixels in image plane
  DATA_MATRIX(static_covariate_matrix); // [dim: bigN x N_static_covariates]

  DATA_STRUCT(spde_fine,spde_t); // INLA SPDE object (components of precision matrix)
  DATA_SPARSE_MATRIX(A_fine); // INLA SPDE projection matrix: mesh to pixels [dim: bigN x nMesh]
  DATA_STRUCT(spde_coarse,spde_t); // INLA SPDE object (components of precision matrix)
  DATA_SPARSE_MATRIX(A_coarse); // INLA SPDE projection matrix: mesh to pixels [dim: bigN x nMesh]
  
  DATA_INTEGER(nHFs); // number of (aggregated) health facilities
  
  DATA_VECTOR(population); // [dim: bigN]

  DATA_VECTOR(HFcases); // [dim: nHFs]  

  DATA_INTEGER(Nunwrapped); // number of HF-labelled pixels with non-zero catchment population
  DATA_VECTOR(invdists); // [length: Nunwrapped]
  DATA_IVECTOR(hf_ids); // [length: Nunwrapped]
  DATA_IVECTOR(pixel_ids); // [length: Nunwrapped]
  
  DATA_VECTOR(treatment); // [length: bigN]
  // DATA_SCALAR(overdispersion_factor);
  
  // Parameters
  
  PARAMETER(intercept);
  
  PARAMETER_VECTOR(static_slopes); // [length: N_static_covariates]
  
  PARAMETER(log_range);
  PARAMETER(log_sd);
  PARAMETER_VECTOR(field); // [dim: nMesh]
  PARAMETER(log_range_slopes);
  PARAMETER(log_sd_slopes);
  PARAMETER_ARRAY(static_slopes_offsets); // [dim: nMesh x N_static_covariates]
  
  PARAMETER_VECTOR(log_masses); // [length: nHFs]
  
  //PARAMETER(log_overdispersion_sd);
  PARAMETER(log_overdispersion_scale);
  //PARAMETER_VECTOR(log_overdispersion_factors);
  
  // Parameter Transforms

  Type shrinkage_static_slopes = 1.0;
  Type range = exp(log_range);
  Type kappa = 2.8284/range;
  Type sd = exp(log_sd);
  Type range_slopes = exp(log_range_slopes);
  Type kappa_slopes = 2.8284/range_slopes;
  Type sd_slopes = exp(log_sd_slopes);
  vector<Type> masses = exp(log_masses);
  Type overdispersion_scale = exp(log_overdispersion_scale);
  // Type overdispersion_sd = exp(log_overdispersion_sd);
  // vector<Type> overdispersion_factors = exp(log_overdispersion_factors);

  // Priors
  
  Type nll = 0.0;
  
  nll -= (dnorm(static_slopes,Type(0.0),shrinkage_static_slopes,true)).sum();
  
  nll -= dnorm(log_sd,Type(-1),Type(0.5),true);
  nll -= dnorm(log_range,Type(1),Type(0.5),true);
  SparseMatrix<Type> Q = Q_spde(spde_fine,kappa);
  nll += SCALE(GMRF(Q),sd)(field);

  nll -= dnorm(log_sd_slopes,Type(-1),Type(0.5),true);
  nll -= dnorm(log_range_slopes,Type(1),Type(0.5),true);
  SparseMatrix<Type> Q_slopes = Q_spde(spde_coarse,kappa_slopes);
  for (int i=0; i<static_slopes.size();i++) {
    nll += SCALE(GMRF(Q_slopes),sd_slopes)(static_slopes_offsets.col(i));
  }

  nll -= (dnorm(log_masses,Type(0),Type(0.5),true)).sum();
  
  nll -= dnorm(log_overdispersion_scale,Type(-1),Type(1),true);
  // nll -= dnorm(log_overdispersion_sd,Type(-1),Type(0.5),true);
  // nll -= (dnorm(log_overdispersion_factors,Type(0),overdispersion_sd,true)).sum();
  
  // Algebra: Construct baseline field 
   
  vector<Type> baseline_field(bigN);
  baseline_field= A_fine*field;
  vector<Type> static_field(bigN);
  static_field = static_covariate_matrix*static_slopes;
  matrix<Type> offset_matrix(bigN,static_slopes.size());
  offset_matrix = A_coarse*static_slopes_offsets.matrix();
  vector<Type> static_field_offsets(bigN);
  for (int i=0; i<bigN; i++) {static_field_offsets[i] = ((static_covariate_matrix.row(i).array())*(offset_matrix.row(i).array())).sum();}

  matrix<Type> full_cov_preds(bigN,static_covariate_matrix.array().cols());
  for (int j=0; j<static_covariate_matrix.array().cols();j++) {
    full_cov_preds.col(j) = (static_covariate_matrix.col(j).array())*(static_slopes[j]+offset_matrix.col(j).array());
  }
  std::cout << full_cov_preds.array().cols() << '\n';
    
  vector<Type> predicted_surface_malaria(bigN);
  predicted_surface_malaria = intercept + static_field + baseline_field + static_field_offsets;

  vector<Type> predicted_surface_malaria_effrate(bigN);
  for (int i=0; i<bigN; i++) {
    predicted_surface_malaria_effrate[i] = exp(predicted_surface_malaria[i])*population[i]*treatment[i];
  }
  
  // Algebra: Catchments
  
  vector<Type> catchment_weights(Nunwrapped);
  vector<Type> catchment_totals(bigN);
  for (int i=0; i<Nunwrapped; i++) {
    catchment_weights[i] = invdists[i]*masses[hf_ids[i]-1];
  }
  for (int i=0; i<bigN; i++) {catchment_totals[i] = 0.0;}
  for (int i=0; i<Nunwrapped; i++) {
    catchment_totals[pixel_ids[i]-1] += catchment_weights[i];
  }
  for (int i=0; i<Nunwrapped; i++) {
    catchment_weights[i] = catchment_weights[i]/catchment_totals[pixel_ids[i]-1];
  }
  std::vector< Eigen::Triplet<Type> > catchment_vals;
  catchment_vals.reserve(Nunwrapped);
  for (int i=0; i<Nunwrapped; i++) {
    catchment_vals.push_back(Eigen::Triplet<Type>(hf_ids[i]-1,pixel_ids[i]-1,catchment_weights[i]));
  }
  SparseMatrix<Type> catchments(nHFs,bigN);
  catchments.setFromTriplets(catchment_vals.begin(),catchment_vals.end());
  
  // Algebra: Expected Cases

  vector<Type> predicted_cases(nHFs);
  predicted_cases = catchments*predicted_surface_malaria_effrate;

  // Likelihood

  for (int i=0; i<nHFs; i++) {
      //nll -= dpois(HFcases[i],predicted_cases[i]*overdispersion_factors[i],true);
      nll -= dnbinom2(HFcases[i],predicted_cases[i],predicted_cases[i]*(1.0+overdispersion_scale),true);
  }

  // Reporting

  REPORT(baseline_field);
  REPORT(static_field);
  REPORT(static_field_offsets);
  REPORT(predicted_surface_malaria);
  REPORT(predicted_cases);
  REPORT(catchments);
  REPORT(full_cov_preds);
  
  return nll;
}
