function gx = g_MCD_expEfficacy(x,P,u,in)
% observation function for MCD (with saturating type #2 effort efficacy)
% function gx = g_MCD(x,P,u,in)
% This function computes choice confidence, spreading-of-alternatives,
% response time and probability of deciding against prior values (i.e. of
% changing one's mind) as a function of 'choice ease' (i.e. value ratings'
% difference) and 'certainty' (i.e. average of value certainty ratings),
% per decision trial.
% All relevant mathematical details are described in:
% Lee, D., and Daunizeau, J. (2020). Trading Mental Effort for Confidence:
% The Metacognitive Control of Value-Based Decision-Making. BioRxiv 837054.
% IN:
%   - x: [useless]
%   - P: MCD parameters (P(1)=log(alpha), P(2)=log(beta), P(3)=log(gamma))
%   and rescaling parameters (see below)
%   - u: MCD inputs:
%       u(1)= choice ease
%       u(2)= certainty
%       u(3)= consequential decision (1 or 0)
%       u(4)= penalized choice (1 or 0)
%       u(5)= mean option values [useless here]
%   - in: [useless]
% OUT:
%   - gx: choice features:
%       gx(1)= choice confidence
%       gx(2)= spreading of alternatives
%       gx(3)= response time
%       gx(4)= probability of changing one's mind
%       gx(4)= subjetive effort rating
% Note: over-/under-confidence biases are included in the model, which can
% be made dependent upon the mean option values.

% inputs
ease = u(1); % choice ease (dVR0)
cert = u(2); % value certainty rating (VCR0)
consequential = u(3); % consequential choice (1 or 0)
penalty = u(4); % penalized choice (1 or 0)

% MCD parameters (including time/trial effects)
alpha = exp(P(1)+penalty*P(6)); % unitary effort cost
beta = exp(P(2)); % type #1 effort efficacy
gamma(1)=exp(P(3)); % type #2 effort efficacy (max var)
gamma(2)=exp(P(4)); % type #2 effort efficacy (half-life)
if isfield(in,'switchFlags') % switch MCD params off?
   alpha = in.switchFlags(1)*alpha;
   beta = in.switchFlags(2)*beta;
   gamma = in.switchFlags(3)*gamma;
end
R = exp(P(5)*consequential);

% prior value mean and variance
V = eps + ((1-cert)./2).^2; % prior value variance
dm = eps + ease; % prior value mean
v0 = V*[1,1];
m0 = [0,dm];

% derive EVC-optimal resource allocation
z = 0:1e-3:1; % control grid
Pc = getPc(z,beta,v0,m0,gamma,0); % expected confidence
evc = R.*Pc - alpha.*z; % ECVD
[~,imax] = max(evc); % find EVC optimizer
T = 1e-2*max(evc(imax)-evc); % numerical temperature (cf max[EVC])
pe = exp((evc-evc(imax))/T);
pe = pe./sum(pe);
zOpt = sum(z.*pe); % EVC softmax (keep gradient info)

% choice features
PcOpt = getPc(zOpt,beta,v0,m0,gamma,0); % EVC-optimal confidence
gaz = gamma(1)*(1-exp(-gamma(2)*zOpt)); % intermediary variable
Qcom = sig(-pi*abs(diff(m0))/sqrt(6*gaz)); % P(changing one's mind)
soa = 2*sqrt(gaz./pi).*exp(-abs(diff(m0)).^2./(4*gaz)); % value spreading
CC = 2*PcOpt-1; % choice confidence
diV = beta*zOpt; % precision gain

% choice features rescaling
np0 = 6;
soa = exp(P(np0+1))*soa + P(np0+2);
CC = exp(P(np0+3))*CC + P(np0+4);
ef = exp(P(np0+5))*zOpt + P(np0+6); % subjective effort rating
rt = exp(P(np0+7))*zOpt + P(np0+8); % response time
Qcom = exp(P(np0+9))*Qcom + P(np0+10);
dcert = sqrt(diV)*2;
dcert = exp(P(np0+11))*dcert + P(np0+12);

% wrap-up output
gx = [CC;soa;rt;Qcom;ef;dcert];


function Eax = Eabs2(mu,V)
% approximates E[|x|] from the 1st-order moment of the folded normal
t = mu./sqrt(V);
t(isnan(t)) = 0;
Eax = sqrt((2.*V)./pi).*exp(-0.5*t.^2) + mu.*(2*sig(pi*t./sqrt(3))-1);


function Pc = getPc(z,beta,v0,m0,gamma,bias)
% approximates expected confidence gain
v = 2./((1./mean(v0)) + beta.*z); % V[dv]
dm0 = -diff(m0); % initial dm
gaz = gamma(1)*(1-exp(-gamma(2)*z)); % intermediary variable
Eadm = Eabs2(dm0,2*gaz); % E[|dm|]
Vadm = abs(dm0).^2 + 2*gaz - Eadm.^2; % V[|dm|]
b = 3*v/pi^2;
a = 0.39; % !!! set to 0 to remove the variance term !!!
Pc = sig(bias+(Eadm./sqrt(b+a*Vadm)));


