% parameter recovery analysis for MCD

clear all
close all
clc

% Monte-Carlo simulations' variables
Nmc = 1e3; % # Monte-Carlo simulations
nx = 1e2; % nb of trials
verbose = 1;
sigma = 5; % simulation noise
thisdate = datestr(now,1);

% set dim and option structures
dim.n_theta = 0;
dim.n = 0;
dim.n_phi = 22;
dim.p = 6;
dim.n_t = nx; % nb of trials
options.dim = dim;
options.priors.SigmaPhi = 1e2.*eye(dim.n_phi);
options.DisplayWin = 0;
g_fname = @g_MCD;%@g_MCD_extended;% % this is where it all happens :)


P0 = NaN(dim.n_phi,Nmc/2); % simulated parameters
P = NaN(dim.n_phi,Nmc/2); % estimated parameters
in = []; % indices of non-problematic simulations




hw = waitbar(0,['run-recovery-VBA-MCD: ']);
fsn = ['MCD_recovery_V100_ws_sig',num2str(sigma),'_',thisdate,'.mat'];

for imc=1:Nmc
    
    waitbar(imc./Nmc,hw)
    
    % draw parameter samples
    dummyPs = randn(dim.n_phi,1);
    P0(:,imc) = dummyPs;
    
    % simulate nx trials
    ease = abs(randn(1,nx));
    cert = abs(rand(1,nx));
    cons = randn(1,nx)>1;
    pens = randn(1,nx)>1;
    sumv = abs(rand(1,nx));
    input = [ease;cert;cons;pens;sumv];
    [y,x,x0,eta,e] = simulateNLSS(dim.n_t,[],g_fname,[],dummyPs,input,[],Inf,options);
    data = zscore(y')';
    data = data + sigma*randn(size(data));
    data = zscore(data')';
    cy(:,imc) = diag(corr(y',data'));
    
    % perform model inversion if simulated data is not problematic
    if ~isweird(data)
        % store simulation index 
        in = [in;imc];
        % call inversion routine
        [posterior,out] = VBA_NLStateSpaceModel(data,input,[],g_fname,dim,options);
        % store estimated parameters
        P(:,imc) = posterior.muPhi;       
    end
    
    % store simulations up to now...
    save(fsn)
    
end

close(hw)




% display estimated P against simulated P and relative error
inP = 1:5;
for i=1:length(inP)
    xtl{i} = ['\phi',num2str(i)];
end
mile = Inf;
male = -Inf;
eP = abs((P(inP,in)-P0(inP,in))./P0(inP,in));
mile = min([mile,min(log(eP(:)))]);
male = max([male,max(log(eP(:)))]);

dle = (male-mile)/10;
hf0 = figure('name',['# valid simulations = ',num2str(length(in)),'/',num2str(Nmc)],'color',[1,1,1],'position',[210,80,680,510]);
% display estimated P against simulated P
ha = subplot(2,2,1,'parent',hf0,'nextplot','add');
sp0 = std(P0(inP,in),[],2);
plot(ha,P0(inP,in)'*diag(1./sp0),P(inP,in)'*diag(1./sp0),'.')
plot(ha,get(ha,'xlim'),get(ha,'xlim'),'k--')
xlabel(ha,'simulated')
ylabel(ha,'estimated')
legend(ha,xtl)
box(ha,'off')
% display relative error
[ne,xe] = hist(vec(log(eP)),mile:dle:male);
ha = subplot(2,2,2,'parent',hf0,'nextplot','add');
hb = bar(xe,100*ne./sum(ne),'parent',ha,'facecolor',0.8*[1 1 1]);
xlabel(ha,'log(relative error)')
ylabel(ha,'Monte-Carlo histogram (%)')
box(ha,'off')
set(ha,'xlim',[mile-1,male+1],'ylim',[0,100])

% perform regression Pj = sum_i w_ij*P0_i
[pv,stat,df,all] = GLM_contrast(zscore(P0(inP,in)'),zscore(P(inP,in)'),eye(length(inP)),'F',1);
ha = subplot(2,2,3,'parent',hf0);
hi = imagesc(all.b'.^2,'parent',ha);
ytl = xtl(inP);
set(ha,'clim',[0;1],'xtick',1:length(xtl),'ytick',1:1:length(ytl),'xticklabel',xtl,'yticklabel',ytl)
colormap(ha,'pink')
colorbar
axis(ha,'square')
xlabel(ha,'simulated')
ylabel(ha,'estimated')
title(ha,'recovery matrix')



