% This function combines most stimulus-related functions into one file.

function sphereStimulus

  display('--- Verify version of sphereStimulusMask! ---')
  makevideo = 'yes';
  % makevideo = 'no';
  % cutinhalf = 'yes';  % save only forward half-periods to save disk space?
  cutinhalf = 'no';

  tic
  period = 10; % in seconds
  framerate = 20; % in Hz
  parameter.maxvelocity = 12.5;
  parameter.diskdiameter = 40;
  % parameter.spatialfrequency_deg = 22/360;

  folderpath = uigetdir('//Users/florian/Desktop/', ...
                        'Select destination folder');

  switch cutinhalf
    case 'no'
      filepath = [folderpath, ...
                  '/stimV_',num2str(period),'sec', ...
                  ... %'_',num2str(parameter.barnumber),'bar', ...
                  '_',num2str(parameter.diskdiameter),'deg', ...
                  '_',num2str(parameter.maxvelocity),'degps', ...
                  '_',num2str(framerate),'fps', ...
                  '.mat'];
    case 'yes'
      filepath = [folderpath, ...
                  '/stimV_',num2str(period),'sec', ...
                  ... %'_',num2str(parameter.barnumber),'bar', ...
                  '_',num2str(parameter.diskdiameter),'deg', ...
                  '_',num2str(parameter.maxvelocity),'degps', ...
                  '_',num2str(framerate),'fps', ...
                  '_halfperiods.mat'];
  end


  parameter.spatialfrequency_deg = [0.0181070, ...
                                    0.0271605, ...
                                    0.0407407, ...
                                    0.0611111, ...
                                    0.0916667, ...
                                    0.1375000, ...
                                    0.2062500];

  parameter.barnumber = parameter.spatialfrequency_deg*360;
  parameter.barwidth_deg = 360./(2*parameter.barnumber);

  spatial = parameter.spatialfrequency_deg;
  diam    = parameter.diskdiameter;


  parameter.dark = {'single disk', 0, 0, 1/1000, 1/1000};

  parameter.velPos1left = {'single disk',   70,  15, spatial(1), diam; ...
                           'single disk',   70,  15, spatial(2), diam; ...
                           'single disk',   70,  15, spatial(3), diam; ...
                           'single disk',   70,  15, spatial(4), diam; ...
                           'single disk',   70,  15, spatial(5), diam; ...
                           'single disk',   70,  15, spatial(6), diam; ...
                           'single disk',   70,  15, spatial(7), diam};

  parameter.velPos2left = {'single disk',  110, -15, spatial(1), diam; ...
                           'single disk',  110, -15, spatial(2), diam; ...
                           'single disk',  110, -15, spatial(3), diam; ...
                           'single disk',  110, -15, spatial(4), diam; ...
                           'single disk',  110, -15, spatial(5), diam; ...
                           'single disk',  110, -15, spatial(6), diam; ...
                           'single disk',  110, -15, spatial(7), diam};

  parameter.velPos3left = {'single disk',   28,  15, spatial(1), diam; ...
                           'single disk',   28,  15, spatial(2), diam; ...
                           'single disk',   28,  15, spatial(3), diam; ...
                           'single disk',   28,  15, spatial(4), diam; ...
                           'single disk',   28,  15, spatial(5), diam; ...
                           'single disk',   28,  15, spatial(6), diam; ...
                           'single disk',   28,  15, spatial(7), diam};

  parameter.velPos1right = {'single disk',  -70,  15, spatial(1), diam; ...
                            'single disk',  -70,  15, spatial(2), diam; ...
                            'single disk',  -70,  15, spatial(3), diam; ...
                            'single disk',  -70,  15, spatial(4), diam; ...
                            'single disk',  -70,  15, spatial(5), diam; ...
                            'single disk',  -70,  15, spatial(6), diam; ...
                            'single disk',  -70,  15, spatial(7), diam};

  parameter.velPos2right = {'single disk', -110, -15, spatial(1), diam; ...
                            'single disk', -110, -15, spatial(2), diam; ...
                            'single disk', -110, -15, spatial(3), diam; ...
                            'single disk', -110, -15, spatial(4), diam; ...
                            'single disk', -110, -15, spatial(5), diam; ...
                            'single disk', -110, -15, spatial(6), diam; ...
                            'single disk', -110, -15, spatial(7), diam};

  parameter.velPos3right = {'single disk',  -28,  15, spatial(1), diam; ...
                            'single disk',  -28,  15, spatial(2), diam; ...
                            'single disk',  -28,  15, spatial(3), diam; ...
                            'single disk',  -28,  15, spatial(4), diam; ...
                            'single disk',  -28,  15, spatial(5), diam; ...
                            'single disk',  -28,  15, spatial(6), diam; ...
                            'single disk',  -28,  15, spatial(7), diam};

  parameter.joint = [parameter.dark; ...
                     parameter.velPos1left; ...
                     parameter.velPos2left; ...
                     parameter.velPos3left; ...
                     parameter.velPos1right; ...
                     parameter.velPos2right; ...
                     parameter.velPos3right];
                   
% 	parameter.joint = {parameter.velPos1left{4,:}} % HACK TO ISOLATE A SINGLE TYPE

  numstimulus = size(parameter.joint,1);

  numframe = 360;
%   % numframe = round(period/2 * framerate);
%   numframe = round(period * framerate);

  parameter.barwidth = 1/2 * 1./cell2mat(parameter.joint(:,4));

  % step 1: create the raw, unmasked, oscillating-bar stimulus
  multigrating = zeros(180,360,numframe,numstimulus);
  for phase = 1:numstimulus

    display(['Generating basic stimulus pattern ',num2str(phase),' of ', ...
             num2str(numstimulus)'.'])

    if phase <= 8

      multigrating(:,:,:,phase) = stimulusBar(framerate, ...
                                          'velocityprofile','sinusoidal', ...
                                          'maxvelocity',parameter.maxvelocity, ...
                                          'barwidth',parameter.barwidth(phase), ...
                                          'numframe',numframe, ...
                                          'direction',0, ...
                                          'showplot','no');

    else

      multigrating(:,:,:,phase) = multigrating(:,:,:,phase-7);

    end

    toc

  end

  % display('Replicating pattern for each frame.')
  % multigrating = repmat(grating, [1 1 1 numstimulus]);
  % toc

  % step 2: create a mask for each stimulus
  multimask = [];
  for phase = 1:numstimulus

    display(['Generating mask ',num2str(phase),' of ', ...
             num2str(numstimulus)'.'])

    mask = sphereStimulusMask('masktype',parameter.joint{phase,1}, ...
                              'yaw',parameter.joint{phase,2}, ...
                              'pitch',parameter.joint{phase,3}, ...
                              'diameter',parameter.joint{phase,5});

    multimask = cat(4, multimask, mask);

  end
  toc

  % step 3: apply each mask to one copy of the basic oscillating-bar stimulus
  display('Replicating masks for each frame.')
  multimask = repmat(multimask,[1 1 numframe 1]);
  pattern = multigrating .* multimask;
  numphase = size(pattern,4);
  toc

  time = clock;
  timestring = ['_', ...
                num2str(time(1),'%.4u'), ...
                num2str(time(2),'%.2u'), ...
                num2str(time(3),'%.2u'), ...
                '_', ...
                num2str(time(4),'%.2u'), ...
                num2str(time(5),'%.2u'), ...
                num2str(time(6),'%02.0f')];

%   switch makevideo
%     case 'no'
%       video = [];
%     case 'yes'
%       switch cutinhalf
%         case 'no'
%           video = VideoWriter(['stim',num2str(framerate),'fps',timestring,'.avi']);
%         case 'yes'
%           video = VideoWriter(['stim',num2str(framerate),'fps_halfperiods',timestring,'.avi']);
%       end
%   %     set(video,'Quality',30);
%       open(video);
%   end

  multistatusmap = [];
  for phase = 8:numphase %HACK!!!

    display(['Generating LED status map ',num2str(phase),' of ', ...
             num2str(numphase),'.'])
           
    switch makevideo
      case 'no'
        video = [];
      case 'yes'
        switch cutinhalf
          case 'no'
            videostring = ['type',num2str(phase-1),'_',num2str(framerate),'fps',timestring,'.avi'];
          case 'yes'
            videostring = ['type',num2str(phase-1),'_',num2str(framerate),'fps_halfperiods',timestring,'.avi'];
        end
        display(videostring)
        video = VideoWriter(videostring);
  %       set(video,'Quality',30);
        open(video);
    end

    [statusmap,~] = sphereEachLed(phase,video,'stimulus',pattern(:,:,:,phase));
    multistatusmap = cat(4, multistatusmap, statusmap);
    
    close(video)
    toc

  end

  display('Splitting into separate arrays for left and right hemisphere.')

  lefthemisphere  = NaN(64,120,numframe,numphase);
  righthemisphere = NaN(64,120,numframe,numphase);
  % for k = 1:numphase
    lefthemisphere  = multistatusmap(:,  1:120,:,:);
    righthemisphere = multistatusmap(:,121:240,:,:);
  % end
  toc

  switch cutinhalf
    case 'yes'
      lefthemisphere  = lefthemisphere (:,:,1:ceil(end/2),:);
      righthemisphere = righthemisphere(:,:,1:ceil(end/2),:);
  end

  display('Saving both arrays to a single file.')

  % filepath = '//Users/florian/Desktop/stim30fps.mat';
  % % filepath = '//Users/florian/Desktop/stim20170519a_fivesizes.mat';
  save(filepath,'lefthemisphere','righthemisphere')

  toc
  display('--- DONE. ---')
  % 
  %   if savevideo
%       close(video)
  %   end

  % Parameters are spatial frequency, max. velocity, temp. freq., disk angle.

  
end



function [ledstatusmap,ledstatus] = sphereEachLed(phasenum,video,varargin)
%SPHEREEACHLED Individual LED positions and activity in spherical arena.
%
%   LEDSTATUSMAP = SPHEREEACHLED computes the positions of each individual 
%   LED in a spherical stimulus arena (i.e. 64 individual LEDs per LED 
%   tile). It then compares these positions to a given stimulus pattern, 
%   to provide the on/off status of each individual LED over time.
%
%   LEDSTATUSMAP is a 3D numerical array. Its first two dimensions taken
%   together represent the status of all individual LEDs during one
%   stimulus frame; it is worth noting that the position of LEDs in this
%   two-dimensional matrix is shuffled in a semi-nonsensical way to meet 
%   the requirements of the code driving the stimulus arena (see below).
%   The third dimension of LEDSTATUSMAP represents one frame at a time.
%
%   [~,LEDSTATUS] = SPHEREEACHLED returns a 3D numerical array. Its first 
%   two dimensions taken together represent the status of all individual 
%   LEDs during one stimulus frame. Here, columns represent the 64 
%   individual LEDs on one tile, and rows represent one LED from each of 
%   the (e.g., 236) tiles. The third dimension of LEDSTATUS represents one 
%   stimulus frame at a time.
%
%   There are no required input arguments, but a number of optional ones.
%
%   LEDSTATUSMAP = SPHEREEACHLED(..., 'PARAM1',val1, 'PARAM2',val2, ...)
%   specifies optional parameter name/value pairs to adapt mask shapes and
%   positions, and to control or suppress figure creation. Parameters are:
%   
%          'lid' - Is there a lid covering the top or bottom of the sphere?
%                  Valid options are 'none' (default), 'top only' and
%                  'bottom only'.
%      'pattern' - Custom stimulus pattern. Must be a 2D or 3D numerical
%                  array, where the first dimension represents elevation,
%                  the second represent azimuth, and the third represents 
%                  one stimulus frame at a time.
%     'showinfo' - Whether to display debugging messages.
%                  Options are 'yes' (default) and 'no'.
%
%   Which of these parameter/value pairs are specified and which ones are
%   left at default values is completely up to the user. Also, they can be
%   specified in any order, as long as they are specified as pairs.
%
%   --------
%
%   Code example 0: To display this help file from the command window, call
%     help sphereEachLed
%
%   Code example 1: To obtain LED status for the default stimulus, and 
%     without a lid covering the sphere arena, call
%     ledstatusmap = sphereEachLed;
%
%   Code example 2: To obtain LED status for a custom stimulus, call
%     ledstatusmap = sphereEachLed('stimulus',mystimulus);
%
%   --------
%
%   This function requires other custom functions. These may be included
%   in this file (scroll down to verify), or included as separate .m files.
%   When split into individual .m files, functions should be named thus:
%
%     sphereShape.m            -  computes overall sphere shape
%     sphereTilePosition.m     -  computes LED tile positions
%     sphereLedPosition.m      -  computes individual LED positions
%     spherePlotLedPosition.m  -  optional, shows individual LED positions
%     sphereLedStatus.m        -  computes when each LED must be on or off
%     sphereFakeCylinder.m     -  arranges tiles by ID number to match code
%     spherePlotStatus.m       -  optional, shows individual LED activity
%
%   Version number and credits apply to all of these functions as a whole.
%
%   (In addition, the function STIMULUSBAR is called to create the default
%   stimulus pattern. If a custom pattern is provided to SPHEREEACHLED by 
%   specifying the 'pattern' parameter, that pattern is used instead.
% 
%     sphereStimulusPattern.m  -  optional, creates default stimulus
%
%   This function STIMULUSBAR is a standalone function and comes with its 
%   own documentation. Thus, it is always contained in a separate file.
%   Compatibility has been tested for STIMULUS of version 2016-08-25a.)
%
%   --------
%
%   This is version 2017-11-22a.
%
%   Created by Florian Alexander Dehmelt, U Tuebingen, 7 August 2016.
%   Based on an earlier set of functions by Julian Hinz, U Tuebingen, with 
%   contributions by Kun Wang, U Tuebingen. If you require any changes, 
%   let me know: florian.dehmelt@uni-tuebingen.de



  % PARSE VARIABLE INPUT ARGUMENTS
  %
  % Check which optional input arguments were provided, and whether their
  % values were provided in the correct format, e.g. a real number, a
  % positive real number, a character array, etc.; if an argument was not
  % provided, assign a default value instead.
  
  % Note: by default, a standard stimulus pattern is created. If a custom 
  % stimulus pattern is provided by specifying the 'pattern' input
  % parameter of SPHEREEACHLED, that custom pattern is used instead.

  close all
  p = inputParser;
  
  validstimulus = @(x) isnumeric(x) && numel(size(x))==3;
  validshowinfo = @(x) strcmp(x,'yes') || strcmp(x,'no');
  validlid   = @(x) strcmp(x,'none') || ...
                    strcmp(x,'top only') || ...
                    strcmp(x,'bottom only');                

  default.lid = 'none';
  addOptional(p,'lid',default.lid,validlid);

% %   default.stimulus = stimulusBar('barwidth',30,'direction',0, ...
% %                                  'showplot','no','numframe',360);
%   default.stimulus = stimulusBar('barwidth',30,'direction',0, ...
%                                  'showplot','no','numframe',10);
%   default.stimulus = stimulusBar(1,'barwidth',30,'direction',0, ...
%                                           'showplot','no','numframe',10);
  default.stimulus = stimulusBar(25,'barwidth',180/22,'direction',0, ...
                                 'showplot','no','numframe',360);
  default.stimulus = cat(3,default.stimulus,default.stimulus);
  addOptional(p,'stimulus',default.stimulus,validstimulus);

  default.showinfo = 'yes';
  addOptional(p,'showinfo',default.showinfo,validshowinfo);

  parse(p,varargin{:});
  
  lid      = p.Results.lid;
  stimulus = p.Results.stimulus;
  showinfo = p.Results.showinfo;
 
  

  % DISPLAY WELCOME MESSAGE
  %
  % Display a welcome message on the command line, containing some basic
  % instructions for the user.
  
  if strcmp(showinfo,'yes')
    display(['To display the help file for this function, enter the ', ...
             'following command in the MATLAB command window: ', ...
             'help sphereEachLed'])
  end

  
  
  % COMPUTE SPHERE SHAPE
  %
  % Compute the overall shape of each hemisphere, its structural ribs, and 
  % the ribbons of LED tiles in between the ribs. This generates a sphere
  % with a default radius, which can be altered inside the function
  % sphereShape. Once sharing this code, different radii could be offered.
  
  [sphereradius, ribbonradius, ribbonangle, ribangle] = sphereShape(lid);

  
  
  % COMPUTE TILE POSITIONS
  %
  % Manually set the number of tiles per ribbon (top to bottom, meridian to
  % side - i.e., decreasing elevation from +90, increasing azimuth from 0),
  % for one hemisphere of the spherical arena. This computation could
  % easily be automated before sharing the code; as long as it is used in
  % conjunction with our arena, there is no need to do so. Afterwards,
  % compute the geographic coordinates of all tile centres.
  
  % Set number of tiles in each ribbon (top to bottom, meridian to side).
  switch lid
    case 'none'
      numtile = [5 9 11 13 14 15 14 13 11 8 5];
    case 'top only'
      numtile = [2 5 9 11 13 14 15 14 13 11 8 5];
    case 'bottom only'
      numtile = [5 9 11 13 14 15 14 13 11 8 5 2];
    otherwise
      error(['-- How many lids are covering the poles of the sphere? ', ...
             'Three scenarios are possible: ''none'', ''top only'', ', ...
             'and ''bottom only''. Please select one of them. --']);
  end
  
  % Compute the position of LED tiles (holding 64 LEDs each),
  % for one hemisphere of the spherical arena.
  tilepos = sphereTilePosition(ribbonradius, ribbonangle, ribangle, ...
                               numtile, lid);

  % Note: "position" refers to the actual geographic coordinates in actual 
  % space, not the sequential ID numbers (e.g. 1 to 128) by which the tiles
  % are called. These will be set and assigned further below.
  
  
  
  % COMPUTE LED POSITIONS
  %
  % Compute the positions of all individual LEDs (64 per tile),
  % for one hemisphere of the spherical arena.
  
  % Given the position of the tiles, and under the assumption that all
  % tiles are perpendicular to the sphere surface, compute LED positions 
  % in both cartesian and geographic coordinates.
  [ledposcartesian,ledposgeographic] = ...
    sphereLedPosition(tilepos,sphereradius);
  
  % Replicate the second hemisphere by creating a mirror-symmetric image.
  % Do so for individual LED positions expressed both in cartesian...
  ledposcartesian2 = ledposcartesian .* ...
                     repmat([1;-1;1], [1, ...
                                       size(ledposcartesian,2), ...
                                       size(ledposcartesian,3), ...
                                       size(ledposcartesian,4)]);
 
  % This is a hack. Clean up in the near future.
  % Actually, it's not a hack, but an important fix not included in some
  % more recent versions. Remember to include it there, too!
  ledposcartesian2 = flipdim(ledposcartesian2,2);
  
  
  ledposcartesian = cat(4, ledposcartesian, ledposcartesian2);
  
  % ...and in geographic coordinates.
  ledposgeographic2 = ledposgeographic .* ...
                      repmat([1;-1], [1, ...
                                      size(ledposgeographic,2), ...
                                      size(ledposgeographic,3), ...
                                      size(ledposgeographic,4)]);
 
  % This is a hack. Clean up in the near future.
  % Actually, it's not a hack, but an important fix not included in some
  % more recent versions. Remember to include it there, too!
  ledposgeographic2 = flipdim(ledposgeographic2,2);
  
  
  ledposgeographic = cat(4, ledposgeographic, ledposgeographic2);
 
  
  % DISPLAY LED POSITIONS
  %
  % Display the position of each individual LED in both cartesian and
  % geographic coordinate systems to verify their proper arrangement.
  
  spherePlotLedPosition(ledposcartesian,ledposgeographic)
  
  
  
%   % COMPUTE LED ACTIVITY STATUS
%   %
%   % Compare the chosen stimulus pattern to the positions of individual LEDs
%   % to find out which one should be active at what time.
%   
  ledstatus = sphereLedStatus(stimulus,ledposgeographic);
% 
% %   % This is a hack. Clean up in the near future.
% %   ledstatus = permute(ledstatus,[2 1 3 4]);
%   
% %   % Plot the result (slow, for debugging only).
  savevideo = ~isempty(video);
%   savevideo = 0
  showvideo = 1;
  if savevideo || showvideo
    spherePlotStatus(ledstatus,ledposcartesian,savevideo,phasenum,video)    
  end
  
   
  % REARRANGE LED ACTIVITY MAP TO ACCOMODATE EXISTING HARDWARE CONTROL CODE
  %
  % As the existing code driving our stimulus arena expects LED tiles to be
  % arranged on the surface of a cylinder (or on a flat rectangular
  % surface), we need to rearrange our spherical distribution of LED tiles
  % into such a rectangular array. The resulting position of certain tiles
  % may seem nonsensical (a tile from the top ending up in the centre of
  % the rectangle, its neighbour up in the bottom right corner etc.), but
  % this new rearrangement is only virtual and has no deeper meaning 
  % besides getting the code to work properly. Don't worry about it.
  
  % Based on the tile ID number displayed by the spherical arena, arrange
  % LED status information in different parts of a rectangular array.
  ledstatusmap = sphereFakeCylinder(ledstatus,lid);   
  
end

% The main function ends here. Below are all(!) required custom functions
% called by the main function. These can be moved to appropriately named,
% separate .m files if desired, but doing so may lead to version conflicts.






%% Function to compute the physical parameters of the basic sphere.
function [sphereradius, ...
          ribbonradius, ribbonangle, ribangle] = sphereShape(lid)

  numribbon = 11;
  tilewidth = 21; % Here, used only to compute elevation of ribs/ribbons.
  ribwidth = 2.1; % Here, used only to compute elevation of ribs/ribbons.
  % Elsewhere, tilewidth is 20 (which is correct). Suggestion: Set ribwidth
  % to 3.1 instead, and use tilewidth = 20 throughout.
  
  % To minimize the gap near the poles of the sphere, determine the radius
  % for which the sum of the elevation angles covered by LED tiles, covered
  % by structural ribs, and covered by the desired holes at the top and
  % bottom equals approaches 180 degrees. The following equation is a cost
  % function penalising any deviation from this optimal radius.
  eqn = @(radius) abs((numribbon)   * 2*asind((tilewidth/2)/radius) + ...
                      (numribbon+1) * 2*asind((ribwidth/2)/radius) + ...
                      1             * 2*asind(60.1/(2*radius)) - 180);
  
  % Numerically solve the equation to find the optimal sphere radius.
  sphereradius = fminsearch(eqn,50);
  
  % Next, Julian decided to deviate from the optimum for practical reasons.
  % The original code did NOT work for any stretchfactors other than 1.05,
  % but this problem has been fixed since. All factors > 1.05 should work.
  stretchfactor = 1.05;
%   stretchfactor = 1;
  sphereradius = sphereradius * stretchfactor;
  
  % Here, "sphereradius" refers to the radius of the sphere on which the
  % centres of all LED tiles are located. The following "outerradius" is
  % the radius of the sphere on which the inner edges of tiles are located.
  outersphereradius = sqrt(sphereradius^2+(tilewidth/2)^2);
  
  % (Re-)Compute the elevation angles covered by a tile, and by a rib.
  ribbonangle = 2*asind((tilewidth/2)/sphereradius);
  ribangle    = 2*asind((ribwidth/2)/sphereradius);
  
  % Compute the radii of perfectly horizontal planes containing one rib
  % each. This computation assumes an odd number of ribbons, i.e. the
  % presence of an equatorial ribbon, rather than an equatorial rib.
  % Because of symmetry, only the unique radii are computed, then
  % replicated once for their mirror-symmetric counterpart.
  
  % First, compute the number of unique ribs.
  numuniquerib = (numribbon-1)/2+1;
  
  % Second, compute the unique circular radii.
  ribbonradius = NaN(numuniquerib,1);
%   Position_connector = NaN((numribbon-1)/2+1,1);

  for k = 1:((numribbon-1)/2+1);
    ribbonradius(k) = outersphereradius * cosd((k-1/2)*ribbonangle + ...
                                                    (k-1)*ribangle);
%     Position_connector(i) = outerradius * sind(ribbonangle/2 + ...
%                             (i-1) * ribbonangle + (i-1) * ribangle);
  end

  % Third, replicate the mirror-symmetric copies.
  ribbonradius = [fliplr(ribbonradius(2:numuniquerib)'), ...
                       ribbonradius'];
                     
	% Fourth, if a lid is present, add an additional circular radius.
  if ~strcmp(lid,'none')
    
    lidradius = outersphereradius * ...
                cosd((numuniquerib+.5)*ribbonangle + numuniquerib*ribangle);

    % Depending on where the lid is located, place its radius
    % at the top or at the bottom of the list of circular radii.
    switch lid
      case 'top only'
        ribbonradius = [lidradius ribbonradius];
      case 'bottom only'
        ribbonradius = [ribbonradius lidradius];
      otherwise
        error(['-- How many lids are covering the poles of the ', ...
               'sphere? Three scenarios are possible: ''none'', ', ...
               '''top only'', and ''bottom only''. Please select ', ...
               'one of them.--']);
    end
    
  end
  
  % The following line looks useless.
%   Value_for_triangle = sind(Theta_1/2)*6;

end





%% Function to compute position of LED tiles from basic sphere shape.
function tilepos = sphereTilePosition(ribbonradius, ribbonangle, ...
                                      ribangle, numtileperribbon, lid)

  ribbonbeta = (ribbonangle+ribangle) * (5:-1:-5);

  switch lid
    case 'none'
      % Relax. Do nothing.
    case 'top'
      ribbonbeta = [ribbonangle*6 + ribangle*6, ribbonbeta];
    case 'bottom'
      ribbonbeta = [ribbonbeta, -(ribbonangle*6 + ribangle*6)]; 
    otherwise
      error(['-- How many lids are covering the poles of the sphere? ', ...
             'Three scenarios are possible: ''none'', ''top only'', ', ...
             'and ''bottom only''. Please select one of them. --']);   
  end


  tilepos = NaN(sum(numtileperribbon),5);
  tilepos(:,1) = 1; % Orientation 1 = Upside down, 0 = Normal
  counter = 0;

  for a = 1:length(ribbonradius)%go through all rows
    for b = 1:numtileperribbon(a)%go through all elements of each row
      
      % Compute the azimuth of each tile centre,
      % taking into account the 6mm wide meridian "keel" or "spine",
      % and the 20mm width of each tile.
      keelwidth = 6;
      tilewidth = 20;

      % Compute the azimuth covered by the keel, and by each tile.
      % These numbers are exact for the keel, as well as for tiles along
      % the equatorial ribbon. For all other ribbons, they are ONLY
      % APPROXIMATE, because these tiles are perpendicular to the
      % sphere, but not perpendicular to the circle formed by the ribbon
      % (i.e., they are not perfectly vertical). Their true azimuth spread
      % would be slightly larger.
      keelangle = 2*asind((keelwidth/2)/ribbonradius(a));
      tileangle = 2*asind((tilewidth/2)/ribbonradius(a));
  
      % Compute the azimuth of the centre of this tile,
      % considering the azimuth offset and the tiles already placed.
      thistilealpha = keelangle + (b-1/2)*tileangle;
 
      % Counter to save in the desired order.
      counter = counter + 1; 
      tilepos(counter,3) = thistilealpha;
      tilepos(counter,4) = ribbonbeta(a);
      tilepos(counter,5) = ribbonradius(a);
      
    end

  end

end





%% Function to compute individual LED positions from tile positions.
function [ledposcartesian, ...
          ledposgeographic] = sphereLedPosition(tilepos,sphereradius)

  ledseparation = 2.48;
  numtile = size(tilepos,1);
  tileisflipped = tilepos(:,1);
  
  % Pre-allocate variable size to speed up computation.
  ledpos           = NaN(length(tilepos)*64,5);  
  ledposcartesian  = NaN(3,8,8,numtile);
  ledposgeographic = NaN(2,8,8,numtile);
  
  for tile = 1:numtile

    % Read out the position of the tile centre in geographic coordinates.
    tilealpha  = tilepos(tile,3);
    tilebeta   = tilepos(tile,4);
    tileradius = sphereradius;
    
    % Convert the position of the tile centre into cartesian coordinates.
    tilecentre = tileradius * [cosd(tilebeta)*cosd(tilealpha); ...
                               cosd(tilebeta)*sind(tilealpha); ...
                               sind(tilebeta)];

    % Compute the "unit" vector in the beta direction (still cartesian).
    betaunitvector = [-sind(tilebeta)*cosd(tilealpha); ...
                      -sind(tilebeta)*sind(tilealpha); ...
                       cosd(tilebeta)];
                     
    % Normalise the vector to make sure it is a unit vector.
    betaunitvector = betaunitvector/norm(betaunitvector);

    % Compute the "unit" vector in the alpha direction  (still cartesian).
    alphaunitvector = [-cosd(tilebeta)*sind(tilealpha); ...
                        cosd(tilebeta)*cosd(tilealpha); ...
                        0];
                      
    % Normalise the vector to make sure it is a unit vector.
    alphaunitvector = alphaunitvector/norm(alphaunitvector);

    
    % Now, place 64 individual LEDs around the tile centre.
    for row = 1:8    % go through columns (!)
      for col = 1:8  % go through rows (!)

        % Compute LED position in cartesian coordinates.
        b = ledseparation * (row - 4.5);
        a = ledseparation * (col - 4.5);
        rled = tilecentre + b*betaunitvector + a*alphaunitvector;
        
        % Save them for later.
        ledposcartesian(:,col,row,tile) = rled;

        % Convert LED position from cartesian to geographic coordinates.
        beta  = asind(rled(3)/norm(rled));
        alpha = atand(rled(2)/rled(1));

        % Constrain geographic coordinate values to standard range.
        % beta = mod(beta,180)-90; % Not needed.
        alpha = mod(alpha,180);
        
        % Save them for later.
        ledposgeographic(:,col,row,tile) = [beta,alpha];

        % Remember which tile this LED is on (i.e., its ID number).
        id = (tile-1)*64 + (row-1)*8 + col;
        
        ledpos(id,2) = tilepos(tile,2); % Remember the tile ID number.

      end
    end
  end
  
  allisflipped = floor(sum(tileisflipped)/numel(tileisflipped));
  
  if allisflipped
    
    % Were all tiles accidentally flipped upside down during construction?
    % If so, flip them back the way they belong.
%     ledposcartesian = flipdim(flipdim(ledposcartesian, 2), 3);
%     ledposgeographic = flipdim(flipdim(ledposgeographic, 2), 3);
%% REVISIT THIS!

  else
    
    % Were individual tile flipped upside down during construction?
    % Or, more precisely, rotated 180 degrees around its centre point?
    % If so, flip the LED coordinates back the way they belong.
    
    for tile = 1:numtile
      if tileisflipped(tile)

        ledposcartesian(:,:,:,tile) = ...
          flipdim(flipdim(ledposcartesian(:,:,:,tile), 2), 3);

        ledposgeographic(:,:,:,tile) = ...
          flipdim(flipdim(ledposgeographic(:,:,:,tile), 2), 3);

      end
    end
    
  end
  
end





%% Function to assign custom tile numbers to tile positions.
function rearranged = sphereFakeCylinder(original,lid)
  
  % FIRST, REORDER TILES BY TILE ID NUMBER
  %
  % Rearrange information on the activity of each individual LED based on
  % the ID number of the tile upon which they are located. These tile
  % numbers (ranging from 1 to 240) are displayed on the spherical arena.

  % List the ID numbers of all tiles in one hemisphere, top-to-bottom and
  % meridian-to-side (i.e., going through one horiz. row after another).
  hemisphere1 = ...
    [115, 116,  65,  45,  47, ...
     113, 114,  80,  78,  66,  46, 48,  4,  2, ...
     119, 120, 118,  79,  77,  67, 56, 15, 14, 16,  3, ...
     112, 111, 110, 109, 117,  61, 68, 54, 38, 37, 12, 13,  1, ...
     108, 107, 106, 105,  85,  86, 63, 52, 55, 42, 39, 11, 24,  6, ...
      84,  83,  82,  81,  88,  87, 64, 51, 53, 43, 40,  9, 10, 23, 5, ...
     104, 103, 102, 101,  74,  76, 62, 50, 44, 41, 25, 18, 17, 22, ...
     100,  98,  97,  90,  73,  75, 49, 28, 27, 26, 20, 19, 21, ...
      99,  92,  89,  72,  71,  60, 58, 33, 34, 30, ...
      29,  94,  91,  70,  69,  59, 36, 35, 31, ...
      95,  96,  93,  57,  32];
  
  % List the ID numbers of the tiles on the lid of the same hemisphere.
  lid1 = [7, 8];
  
%   % THIS IS A HACK FOR DEBUGGING. REMOVE.
% %   zebra = repmat([1 1 0 0 1 0 1 0],[8 1]);
%   zebra = [0 0 0 0 1 1 0 1; ...
%            0 0 0 1 1 1 1 1; ...
%            0 0 0 0 1 1 0 1; ...
%            0 0 0 0 0 0 0 0; ...
%            0 0 1 1 0 1 0 0; ...
%            0 1 1 1 1 1 0 0; ...
%            0 0 1 1 0 1 0 0; ...
%            0 0 0 0 0 0 0 0];  
%   one = [0 0 0 0 0 0 0 0; ...
%          0 0 0 0 0 1 0 0; ...
%          0 0 0 0 1 1 0 0; ...
%          0 0 0 1 0 1 0 0; ...
%          0 0 0 0 0 1 0 0; ...
%          0 0 0 0 0 1 0 0; ...
%          0 0 0 0 0 1 0 0; ...
%          0 0 0 0 0 1 0 0];
%   two = [0 0 0 0 0 0 0 0; ...
%          0 0 1 1 1 0 0 0; ...
%          0 1 0 0 0 1 0 0; ...
%          0 0 0 0 0 1 0 0; ...
%          0 0 0 0 1 0 0 0; ...
%          0 0 0 1 0 0 0 0; ...
%          0 0 1 0 0 0 0 0; ...
%          0 1 1 1 1 1 0 0];
%   three = [0 0 0 0 0 0 0 0; ...
%            0 0 1 1 1 0 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 0 0 0 0 1 0 0; ...
%            0 0 0 1 1 0 0 0; ...
%            0 0 0 0 0 1 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 0 1 1 1 0 0 0];
%   four = [0 0 0 0 0 0 0 0; ...
%           0 1 0 0 0 1 0 0; ...
%           0 1 0 0 0 1 0 0; ...
%           0 1 0 0 0 1 0 0; ...
%           0 1 1 1 1 1 0 0; ...
%           0 0 0 0 0 1 0 0; ...
%           0 0 0 0 0 1 0 0; ...
%           0 0 0 0 0 1 0 0];
%   five = [0 0 0 0 0 0 0 0; ...
%           0 1 1 1 1 1 0 0; ...
%           0 1 0 0 0 0 0 0; ...
%           0 1 0 0 0 0 0 0; ...
%           0 1 1 1 1 0 0 0; ...
%           0 0 0 0 0 1 0 0; ...
%           0 0 0 0 0 1 0 0; ...
%           0 1 1 1 1 0 0 0];
%   six = [0 0 0 0 0 0 0 0; ...
%          0 0 1 1 1 0 0 0; ...
%          0 1 0 0 0 0 0 0; ...
%          0 1 0 0 0 0 0 0; ...
%          0 1 1 1 1 0 0 0; ...
%          0 1 0 0 0 1 0 0; ...
%          0 1 0 0 0 1 0 0; ...
%          0 0 1 1 1 0 0 0];
%   seven = [0 0 0 0 0 0 0 0; ...
%            0 1 1 1 1 1 0 0; ...
%            0 0 0 0 0 1 0 0; ...
%            0 0 0 0 1 0 0 0; ...
%            0 0 0 0 1 0 0 0; ...
%            0 0 0 1 0 0 0 0; ...
%            0 0 0 1 0 0 0 0; ...
%            0 0 0 1 0 0 0 0];
%   eight = [0 0 0 0 0 0 0 0; ...
%            0 0 1 1 1 0 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 0 1 1 1 0 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 1 0 0 0 1 0 0; ...
%            0 0 1 1 1 0 0 0];
%   numframe = size(original,3);
%   original(:,:,:,1) = repmat(one,[1 1 numframe 1]);
%   original(:,:,:,2) = repmat(two,[1 1 numframe 1]);
%   original(:,:,:,3) = repmat(three,[1 1 numframe 1]);
%   original(:,:,:,4) = repmat(four,[1 1 numframe 1]);
%   original(:,:,:,5) = repmat(five,[1 1 numframe 1]);
%   original(:,:,:,6) = repmat(six,[1 1 numframe 1]);
%   original(:,:,:,7) = repmat(seven,[1 1 numframe 1]);
%   original(:,:,:,8) = repmat(eight,[1 1 numframe 1]);
  
%   % The following are dummy IDs created for the second hemisphere and the
%   % second half of the lid. They must be replaced with the true IDs there
%   % as soon as Kun Wang has made these available.
%   hemisphere2 = hemisphere1 + 120;
%   lid2        = lid1        + 120;

  hemisphere2 = 120 + ...
    [100, 71, 72, 32, 28, ...
      99, 69, 70, 29, 30, 31, 26,  6,  8, ...
      98, 97, 88, 87, 68, 22, 23, 24, 25, 27,  7, ...
      91, 92, 85, 86, 66, 67, 21, 19, 20, 16, 12, 11,  5, ...
      89, 90, 95, 93, 82, 80, 65, 17, 18, 13, 14, 15, 10,  9, ...
     104,103,111,107,108, 84, 78, 77, 79, 58, 47, 48, 46, 45, 49, ...
     102,101,110, 94, 73, 81, 83, 36, 57, 42, 40, 38, 56, 51, ...
     116,115,112, 96, 75, 76, 33, 59, 43, 44, 39, 55, 50, ...
     113,114,109,  4, 74, 34, 60, 41, 37, 53, 52,...
     120,118,  3,  1, 35, 61, 63, 54, ...
     119,117,  2, 64, 62];
   
   lid2 = [105, 106]; % These two scalars may be flipped. Verify.

  % Aggregate the tile ID numbers of all tiles in the spherical arena in
  % the correct order, taking into account where exactly the lid is placed.
  switch lid
    case 'none'
      neworder = [hemisphere1,hemisphere2];
    case 'top only'
      neworder = [lid1,hemisphere1,lid2,hemisphere2];
    case 'bottom only'
      neworder = [hemisphere1,lid1,hemisphere2,lid2];
  end
  
  % The following line ensures that unassigned tile numbers are padded with
  % NaNs. If you remove the line, they will be padded with zeros instead -
  % or skipped entirely if there are no higher, actually assigned numbers.
  % To safely pad the array with zeros, replace NaN(...) with zeros(...).
  numframe = size(original,3);
  reordered = NaN(8,8,numframe,240);
  
  % Rearrange the fourth dimension of the arrays containing LED positions.
  % Remember that dimension 1 are the actual coordinates (e.g., azimuth and
  % elevation), dimensions 2 and 3 cluster the 8*8 individual LED on each
  % tile, and dimension 4 goes through all tiles in the setup. The old
  % order went through all tiles top-to-bottom, meridian-to-side; the new
  % order goes through all tiles from the tile with ID no. 1 to the tile
  % with ID no. 236 (or whatever else the maximum is).
  numtile = numel(neworder);
%   numtile = size(original,4);
  oldorder = 1:numtile;
  reordered(:,:,:,neworder) = original(:,:,:,oldorder);
  
  % Display how many tile IDs were found, and how many more could be used.
%   unassigned = numel(find(isnan(reordered)))/(64*size(original,1));
  unassigned = 240 - numtile;
  display(['-- Out of 240 supported tiles, ',num2str(numtile), ...
           ' tile IDs were assigned; ',num2str(unassigned), ...
           ' were left unassigned. --'])

  

  % SECOND, REARRANGE TILES INTO A VIRTUAL, RECTANGULAR PATTERN
  %
  % As the existing code driving our stimulus arena expects LED tiles to be
  % arranged on the surface of a cylinder (or on a flat rectangular
  % surface), we need to rearrange our distribution of LED tiles into such 
  % a rectangular array. The resulting position of certain tiles may seem
  % nonsensical (a tile from the top ending up in the centre of the
  % rectangle, its neighbour up in the bottom right corner etc.), but this
  % new rearrangement is only virtual and has no deeper meaning besides
  % getting the code to work properly. Don't worry too much about it.

  % Arrange increasing tile ID top-to-bottom, then left-to-right, in
  % vertical columns of 8. The total number of columns is 15 for 120 tiles,
  % 30 for 240 tiles.

  rearranged = NaN(64,240,numframe);
  
  % This is a hack. Clean up in the near future.
  reordered = permute(reordered,[2 1 3 4]);
  
  % This is not a hack. The following loop must always run to 240.
%   for tileID = 1:numtile
  for tileID = 1:240
    
    xshift = 8*floor((tileID-1)/8);
    yshift = 8*mod(tileID-1,8);
%     display([tileID xshift/8 yshift/8])
   
%     if tileID <= 5
%         % THIS IS A HACK FOR DEBUGGING. REMOVE.
%       rearranged((1:8)+56-yshift,(1:8)+xshift,:)  ...
%         = repmat(zebra,[1 1 numframe 1]);
%     else
      rearranged((1:8)+56-yshift,(1:8)+xshift,:) = reordered(:,:,:,tileID);
%     end
    
  end
  
  
end





%% Function to display the positions of all individual LEDs.
function spherePlotLedPosition(ledposcartesian,ledposgeographic)


  % Part 1/2: plot LED positions in cartesian coordinates
  
  figure(44)
%   clf
  set(gcf,'Color',[1 1 1])
  
  rledplot = reshape(ledposcartesian,[3 numel(ledposcartesian)/3]);
  numtile = size(ledposcartesian,4);

  group1 = 1:4:numtile;
  group2 = 2:4:numtile;
  group3 = 3:4:numtile;
  group4 = 4:4:numtile;

  index1 = repmat(1:64,[1 numel(group1)]) + ...
           reshape(64*ones(64,1)*(group1-1), [1 64*numel(group1)]);
  index2 = repmat(1:64,[1 numel(group2)]) + ...
           reshape(64*ones(64,1)*(group2-1), [1 64*numel(group2)]);
  index3 = repmat(1:64,[1 numel(group3)]) + ...
           reshape(64*ones(64,1)*(group3-1), [1 64*numel(group3)]);
  index4 = repmat(1:64,[1 numel(group4)]) + ...
           reshape(64*ones(64,1)*(group4-1), [1 64*numel(group4)]);

  hold on
  s1 = scatter3(rledplot(1,index1),rledplot(2,index1),rledplot(3,index1));
  s2 = scatter3(rledplot(1,index2),rledplot(2,index2),rledplot(3,index2));
  s3 = scatter3(rledplot(1,index3),rledplot(2,index3),rledplot(3,index3));
  s4 = scatter3(rledplot(1,index4),rledplot(2,index4),rledplot(3,index4));
  hold off

  axis equal
  colour = [[.8 .4 .2]; [.2 .4 .8]; [.2 .8 .6]; .2*[1 1 1]];
  set(s1,'MarkerEdgeColor',colour(1,:),'MarkerFaceColor',colour(1,:))
  set(s2,'MarkerEdgeColor',colour(2,:),'MarkerFaceColor',colour(2,:))
  set(s3,'MarkerEdgeColor',colour(3,:),'MarkerFaceColor',colour(3,:))
  set(s4,'MarkerEdgeColor',colour(4,:),'MarkerFaceColor',colour(4,:))
  set([s1,s2,s3,s4],'SizeData',1.7)
  
  
  % Part 2/2: plot LED positions in geographic coordinates
  
  figure(45)
%   clf
  set(gcf,'Color',[1 1 1])

  axis([-180 180 -90 90])
  box on
  xlabel('Azimuth')
  ylabel('Elevation')
  plot(ledposgeographic(2,:), ledposgeographic(1,:),'ko','MarkerSize', 2)

end





%% Function to compute when each LED should be on or off.
function ledstatus = sphereLedStatus(pattern,ledposgeographic)
  
  % Create a discrete grid in geographic coordinates. This grid must have
  % the same resolution as the stimulus pattern.
  betagridstep  = 180/(size(pattern,1)-1);
  alphagridstep = 360/(size(pattern,2)-1);
  betagrid  = (-90:betagridstep:+90)';
  alphagrid = (-180:alphagridstep:+180)';
          
  % How many individual LEDs are on each tile, how many tiles are there?
%   numtile = 120;
  numtile = 240;
  numframe = size(pattern,3);

%   ledstatus = NaN(numledpertile,numtile,numframe);
  ledstatus = NaN(8,8,numframe,numtile);
  
  for tile = 1:size(ledposgeographic,4)
    for row = 1:size(ledposgeographic,3)
      for col = 1:size(ledposgeographic,2)
    
        % Find the geographic grid point closest to the exact LED position.
        beta  = ledposgeographic(1,row,col,tile);
        alpha = ledposgeographic(2,row,col,tile);
        [~ , bestbeta]  = min(abs(beta-betagrid));
        [~ , bestalpha] = min(abs(alpha-alphagrid));
        
        % The LED status is the value of this grid point.
        ledstatus(row,col,:,tile) = pattern(bestbeta,bestalpha,:);
        
      end
    end
  end
    
end





%% Function to display the activity of all individual LEDs (optional)
function spherePlotStatus(ledstatus,ledposcartesian,savevideo,phasenum,video)
  
  figure(46)
  set(gcf,'Color',[1 1 1],'Position',[200 0 480 480])
  
%   if savevideo
%     video = VideoWriter(['type',num2str(phasenum),'.avi']);
%     set(video,'Quality',100);
%     open(video);
%   end

  xled = squeeze(ledposcartesian(1,:,:,:));
  yled = squeeze(ledposcartesian(2,:,:,:));
  zled = squeeze(ledposcartesian(3,:,:,:));
  
  numframe = size(ledstatus,3);
  [xball,yball,zball] = sphere;
  ballradius = .95 * norm(ledposcartesian(:,1,1,1));
  axes('Position',[-.3 -.3 1.6 1.6])
  axis square

  for k = 1:numframe
    
    ongroup  = squeeze(ledstatus(:,:,k,:)==1);
    offgroup = squeeze(ledstatus(:,:,k,:)==0);
             
    scatter3(xled(ongroup),yled(ongroup),zled(ongroup), ...
             'MarkerFaceColor',[.4 1 .6], ...
             'MarkerEdgeColor',[.2 .2 .2], ...
             'SizeData',30)
    set(gca,'XLim',[-120 120],'YLim',[-120 120],'ZLim',[-120 120])
    hold on
%     scatter3(xled(offgroup),yled(offgroup),zled(offgroup), ...
%              'MarkerFaceColor',[.2 .2 .2], ...
%              'MarkerEdgeColor',[.2 .2 .2], ...
%              'SizeData',35)
%     set(gca,'XLim',[-120 120],'YLim',[-120 120],'ZLim',[-120 120])
    surf(ballradius*xball,ballradius*yball,ballradius*zball, ...
         'EdgeColor','none','FaceColor',[1 1 1],'FaceAlpha',.4)
    text(113,610,['frame ',num2str(k)],'FontSize',22,'Units','pixels')
%     text(363,610,['stimulus index ',num2str(phasenum-1)],'FontSize',22, ...
%                   'Units','pixels')
% %     text(463,610,['stimulus ',num2str(ceil(k/360))],'FontSize',22, ...
% %                   'Units','pixels')
    colour.eye = 0*[1 1 1];
    colour.body = .4*[1 1 1];
    plot3dFish(10,180,0,0,colour);
    hold off
    
    axis square
    axis off
    
    if savevideo
      frame = getframe(gcf);
      writeVideo(video,frame)
    else
      drawnow
    end
%     pause(.01)
    
  end
  
  if savevideo
    close(video)
  end
  
  axis square
  
end



function [lefteyehandle,righteyehandle,bodyhandle] = ...
  plot3dFish(scale,yaw,pitch,roll,colour)

  % This corresponds to version 20170505a, a.k.a. plot3dFish20170505a.m.

  set(gcf,'Color',[1 1 1])

  resolution = 10;
%   colour.eye = .1*[1 1 1];
%   colour.body = .7*[1 1 1];

  eye.x = scale * (cos(-pi:pi/resolution:3*pi-pi/resolution)');
  eye.y = scale * (0.7*sin(0:pi/resolution:4*pi-pi/resolution)');
  eye.z = scale * ([.7*ones(2*resolution,1); ...
                    1*ones(2*resolution,1)]);

  N = 2*resolution;

  lefteye.vertices = [eye.x, eye.y, eye.z];
  righteye.vertices = [eye.x, -eye.y, eye.z];

  lefteye.faces = [[1:2*resolution 1]; ...
                  [1+2*resolution:4*resolution 1+2*resolution]];


  for k = 1:N

    lefteye.faces = [lefteye.faces; ...
                    [mod([k-1 k],N)+1, ...
                     mod([k k-1],N)+1+N, ...
                     mod(k-1,N)+1, ...
                     NaN(1,N-4)]];

  end

  lefteye.facevertexcdata = repmat(colour.eye,[size(lefteye.faces,1) 1]);

  righteye.faces = lefteye.faces;
  righteye.facevertexcdata = lefteye.facevertexcdata;

  lefteyehandle  = patch(lefteye);
  righteyehandle = patch(righteye);
  set([lefteyehandle,righteyehandle],'FaceColor','flat','EdgeColor','none')

  mainbody.x = scale * (-.1 + [1.0*cos(-pi:pi/resolution:pi-pi/resolution), ...
                               0.4*cos(-pi:pi/resolution:pi-pi/resolution)]');
  mainbody.y = scale * ([0.7*sin(0:pi/resolution:2*pi-pi/resolution), ...
                         0.2*sin(0:pi/resolution:2*pi-pi/resolution)]');
  mainbody.z = scale * ([1.1*ones(2*resolution,1); ...
                        -7*ones(2*resolution,1)]);

  N = 2*resolution;

  body.vertices = [mainbody.x, mainbody.y, mainbody.z];

  body.faces = [[1:2*resolution 1]; ...
                [1+2*resolution:4*resolution 1+2*resolution]];

  for k = 1:N

    body.faces = [body.faces; ...
                  [mod([k-1 k],N)+1, ...
                   mod([k k-1],N)+1+N, ...
                   mod(k-1,N)+1, ...
                   NaN(1,N-4)]];

  end

  body.facevertexcdata = repmat(colour.body,[size(body.faces,1) 1]);

  bodyhandle = patch(body);
  set(bodyhandle,'FaceColor','flat','EdgeColor','none')

  rotate(lefteyehandle,[1 0 0],80)
  rotate(lefteyehandle,[0 0 1],-10)
  rotate(righteyehandle,[1 0 0],-80)
  rotate(righteyehandle,[0 0 1],10)
  rotate(bodyhandle,[0 1 0],-90)
  
  rotate(lefteyehandle,[0 0 1],yaw)
  rotate(righteyehandle,[0 0 1],yaw)
  rotate(bodyhandle,[0 0 1],yaw)
  
  rotate(lefteyehandle,[0 1 0],pitch)
  rotate(righteyehandle,[0 1 0],pitch)
  rotate(bodyhandle,[0 1 0],pitch)
  
  rotate(lefteyehandle,[1 0 0],roll)
  rotate(righteyehandle,[1 0 0],roll)
  rotate(bodyhandle,[1 0 0],roll)

end




function stimulus = stimulusBar(framerate,varargin)

%STIMULUSBAR Moving-bar stimulus with custom direction, speed & frequency.
%
%   STIMULUS = STIMULUSBAR computes a set of parallel bars to be used as 
%   a visual stimulus. Their orientation and spatial frequency can be 
%   chosen at will, and they will move in the direction perpendicular to 
%   their orientation according to the desired velocity profile.
%
%   STIMULUS is a 3D numerical array where the first direction represents y 
%   (or beta/elevation in geographic coordinates), the second represent x 
%   (or alpha/azimuth in geographic coordinates), and the third contains 
%   one stimulus frame each. Values are either 1 (on) or 0 (off).
%
%   There are no required input arguments, but a number of optional ones.
%
%   STIMULUS = STIMULUSBAR(..., 'PARAM1',val1, 'PARAM2',val2, ...)
%   specifies optional parameter name/value pairs to adapt mask shapes and
%   positions, and to control or suppress figure creation. Parameters are:
%   
%        'barwidth' - Angular size in degrees of each bar. This number is
%                     independent of bar orientation, as it always refers
%                     to the shortest angular extent (i.e., the intuitive
%                     bar width). Any(!) positive real number can be
%                     provided; the default value is 30 degrees.
%       'direction' - The direction of motion. Because motion is always
%                     perpendicular to the orientation of the bars, this
%                     parameter also defines orientation. Any(!) real
%                     number can be chosen; the default is 30 degrees.
%                     Directions are measured counterclockwise starting
%                     with rightward motion: 0 degrees moves to the right,
%                     45 degrees to the top right, -135 degrees to the
%                     bottom left, 900 degrees to the left, etc.
% 'velocityprofile' - The desired velocity profile of motion. At the
%                     moment, options are "constant", "sinusoidal" 
%                     (default) and "random" (i.e., exponentially filtered 
%                     Brownian motion).
%        'velocity' - Maximum velocity in degrees per second.
%      'xpixelsize' - The resolution in the x (i.e., azimuth) direction.
%                     Measured in degrees per pixel. Any(!) positive real
%                     number can be chosen, the default value is 1.
%      'ypixelsize' - The resolution in the y (i.e., elevation) direction.
%                     Measured in degrees per pixel. Any(!) positive real
%                     number can be chosen, the default value is 1.
%        'numframe' - The number of frames to be generated. Any positive
%                     integer can be chosen, the default value is 400.
%        'showplot' - Whether or not to create figures. Options are: 'no' 
%                     to suppress figures, 'yes' (default) to allow them.
%
%   Which of these parameter/value pairs are specified and which ones are
%   left at default values is completely up to the user. Also, they can be
%   specified in any order, as long as they are specified as pairs.
%
%   Note: several input arguments only make sense for certain types of
%   masks. This does not cause any crashes, as they will simply be ignored.
%   To avoid confusing users, however, this should be addressed in the
%   future.
%
%   --------
%
%   Code example 1: To compute the default visual stimulus, call
%     stimulus = stimulusBar;
%
%   Code example 2: To compute a stimulus with really narrow bars, call
%     stimulus = stimulusBar('barwidth',5);
%
%   Code example 3: To compute this stimulus without displaying it, call
%     stimulus = stimulusBar('barwidth',5,'showplot','no');
%
%   Code example 4: To compute a stimulus moving to the top left with a
%   random velocity profile, call
%     stimulus = stimulusBar('direction',110.2,'velocityprofile','random')
%
%   --------
%
%   This is version 2016-05-19a.
%
%   Created by Florian Alexander Dehmelt, U Tuebingen, 24 August 2016. If
%   you require any changes, let me know: florian.dehmelt@uni-tuebingen.de



  % PARSE VARIABLE INPUT ARGUMENTS
  %
  % Check which optional input arguments were provided, and whether their
  % values were provided in the correct format, e.g. a real number, a
  % positive real number, a character array, etc.; if an argument was not
  % provided, assign a default value instead.
  
  p = inputParser;
  
  realnumber = @(x) isnumeric(x) && isscalar(x) && isreal(x);
	positiverealnumber = @(x) realnumber(x) && (x>0);
  positiveinteger = @(x) (x==round(x)) && (x>0);
  validshowplot = @(x) strcmp(x,'yes') || strcmp(x,'no');
  validvelocityprofile = @(x) strcmp(x,'constant') || ...
                              strcmp(x,'sinusoidal') || ...
                              strcmp(x,'random');
                
  default.barwidth = 30;
  addOptional(p,'barwidth',default.barwidth,positiverealnumber);

  default.direction = 30;
  addOptional(p,'direction',default.direction,realnumber);

  default.velocityprofile = 'sinusoidal';
  addOptional(p,'velocityprofile',default.velocityprofile, ...
              validvelocityprofile);
            
  default.maxvelocity = 12.6;
  addOptional(p,'maxvelocity',default.maxvelocity,positiverealnumber);

  default.xpixelsize = 1;
  addOptional(p,'xpixelsize',default.xpixelsize,positiverealnumber);

  default.ypixelsize = 1;
  addOptional(p,'ypixelsize',default.ypixelsize,positiverealnumber);

  default.numframe = 400;
  addOptional(p,'numframe',default.numframe,positiveinteger);

  default.showplot = 'yes';
  addOptional(p,'showplot',default.showplot,validshowplot);

  parse(p,varargin{:});
  
  barwidth        = p.Results.barwidth;
  direction       = p.Results.direction;
  velocityprofile = p.Results.velocityprofile;
  maxvelocity     = p.Results.maxvelocity;
  xpixelsize      = p.Results.xpixelsize;
  ypixelsize      = p.Results.ypixelsize;
  numframe        = p.Results.numframe;
  showplot        = p.Results.showplot;
  

  
  % GENERATE COORDINATE GRID AND DETERMINE BAR PROPERTIES
  %
  % Generate coordinate grid, ready-made for subsequent matrix operations.
  % Then, compute all required properties of the moving bars (e.g., how
  % many bars to render). The number of bars along with the x and y
  % resolution determines the size of the resulting 3D arrays. Their first
  % dimension represents y, the second x, and the third one bar at a time.
  % When identifying pixels that are part of a bar (in the loop further
  % below), this array is compared to the array of raw x and y coordinates
  % of the screen as a whole to determine which pixels are close enough.
  
  xvalue = -180+xpixelsize:xpixelsize:180;
  yvalue = -90+ypixelsize:ypixelsize:90;
  numxpixel = numel(xvalue);
  numypixel = numel(yvalue);
  maxbar = 1/cosd(45);
  numbar = ceil(maxbar*360/(2*barwidth)) * 10;
  xgrid = repmat(xvalue,[numypixel 1 numbar]);
  ygrid = flipdim(repmat(yvalue',[1 numxpixel numbar]), 1);

  orientation = direction + 90;
  slope = tand(orientation);

  
  
  % PREPARE MAIN FIGURE
  %
  % Generate the main figure displaying the visual stimulus. Set the colour
  % map so the bars will appear green in front of a black background.
  
  if strcmp(showplot,'yes')
    
    close all
    figure(11)
    set(gcf,'Color',[1 1 1], ...
            'Position',[0 250 800 400])
    colormap([0 0 0; .4 1 .5])
    
  end
  
  
  
  % MOVE BARS
  %
  % Generate stationary bars in a different position for each frame. The
  % pixels making up the bars are computed at every iteration, based on
  % their Euclidean distance from the "true", or ideal centre line of each
  % bar. These centre lines are shifted at each iteration, moving the bars.
    
  switch velocityprofile
    case 'constant'
      speed = maxvelocity * ones(numframe,1);
    case 'sinusoidal'
      speed = maxvelocity * sind((0:360/numframe:360-1/numframe));
    case 'random'
      speed = maxvelocity * randn(numframe,1);
      speed = conv(speed,.5*exp(-(1:numframe)/20));
      speed = speed(1:numframe);
  end
  
  % The initial position of the bars can be shifted in the direction of
  % motion, if desired. This number will evolve between frames.
  shift = 0;
  
  % Prepare the 3D output array before running the loop.
  stimulus = NaN(numypixel,numxpixel,numframe);

  % Create one stimulus frame after another. In principle, this could be
  % written using 4D arrays instead, but it runs fast enough for now.
  for k = 1:numframe

    % Shift position of stationary bars incrementally at each iteration.
    shift = shift + speed(k)/framerate;
    xshift = mod(-shift*sind(orientation),2*barwidth*sind(orientation));
    yshift = mod(-shift*cosd(orientation),2*barwidth*cosd(orientation));

    % Generate stationary bars with the desired orientation and position.
    % This position includes the incremental shift, thus moving the bars.
    
    if ~mod(direction,180)

      % If the motion is perfectly horizontal, all bars are vertical.
      barseparation = 2*barwidth*(-numbar/2:numbar/2-1)/sind(orientation);
      xoffset = reshape(barseparation,[1,1,numbar]);
      xoffset = repmat(xoffset,[numypixel numxpixel 1]);
      xbar = xoffset - xshift;
      
      % The stimulus is made up of pixels close enough to the centre lines.
      singlebar = abs(xgrid-xbar) < barwidth/2;

    else

      % If the motion is not perfectly horizontal,
      % each bar is instead described by its vertical offset and its slope.
      barseparation = 2*barwidth*(-numbar/2:numbar/2-1)/cosd(orientation);
      yoffset = reshape(barseparation,[1,1,numbar]);
      yoffset = repmat(yoffset,[numypixel numxpixel 1]);
      ybar = slope*(xgrid + xshift) + yoffset + yshift;
      
      % The stimulus is made up of pixels close enough to the centre lines.
      singlebar = abs((ygrid-ybar)*cosd(orientation)) < barwidth/2;
      
    end

    % Combine the individually computed bars into one joint pattern.
    stimulus(:,:,k) = sum(singlebar,3);

    if strcmp(showplot,'yes')
      
      % Include a pause so the live video displays smoothly. Remove this
      % pause to save time when precise live video is not important.
      pause(.01)

      % Display current appearance of the pattern (i.e. stimulus frame).
      % Setting the axis to 'equal' ensures that e.g. 45 degree motion 
      % actually appears as such to the viewer. This is cosmetic only, 
      % however, and does not affect the stimulus generated.
      imagesc(xvalue,yvalue,stimulus(:,:,k))
      axis equal
      set(gca, 'XLim',        [-180 180], ...
               'YLim',        [-90 90], ...
               'XTick',       -180:90:180, ...
               'YTick',       -90:45:45, ...
               'YTickLabel',  90:-45:-45, ...
               'TickLength',  [0 0])

      % It is possible to include axis labels while running the loop, but 
      % doing so significantly slows it down (> 40% longer). Comment out the 
      % following lines to save a lot of time at the expense of beauty.
      xlabel('azimuth')
      ylabel('elevation')
      title('visual stimulus')
      
    end
 
    % Save the accumulated displacement for debugging plots outside loop.
    displacement(k) = shift; %#ok<AGROW>
    
  end

  
  
  % FINAL PLOTS
  %
  % Wrap up the video display, and create other figures for debugging.
  % Specifically, it may be important to check whether the velocity profile
  % worked as intended and created the desired displacement at all times.
  
  if strcmp(showplot,'yes')

    % If axis labels were not included during the loop, add them at the
    % end so the final frame of the stimulus can be exported with labels.
    figure(11)
    xlabel('azimuth')
    ylabel('elevation')
    title('visual stimulus')
  
    % Debugging plot, showing the displacement in the direction of motion
    % as a function of time.
    figure(12)
    set(gcf,'Color',[1 1 1],'Position',[820 250 400 400])
    p = plot(displacement);
    xlabel('frame no.')
    ylabel('accumulated displacement in the direction of motion')
    set(gca,'Layer','top', ...
            'YGrid','on')
    set(p,'Color',[.2 .8 .4], ...
          'LineWidth',2)
        
  end
  
end



function mask = sphereStimulusMask20171011a(varargin)
%SPHERESTIMULUSMASK Masks for visual stimulus presentation. Now in CCW!
%
%   MASK = SPHERESTIMULUSMASK computes 12 circular masks in geographic
%   coordinates, each centred on a different node of a regular icosahedron.
%   It also creates one figure each to check their appearance.
%
%   There are no required input arguments, but a number of optional ones.
%
%   [...] = SPHERESTIMULUSMASK(..., 'PARAM1',val1, 'PARAM2',val2, ...)
%   specifies optional parameter name/value pairs to adapt mask shapes and
%   positions, and to control or suppress figure creation. Parameters are:
%   
%        'plotflag' - Whether or not to create the figures. Options are:
%                     0 (default) to suppress figures, 1 to allow them
%        'diameter' - Custom disk diameter in degrees (default = 70)
%       'pixelsize' - Display resolution, i.e. degrees/pixel of the LED
%                     array (default = 1). Low values mean high resolution.
%             'yaw' - Rotate disk or whole icosahedron "in alpha (azimuth)
%                     direction" by a number of degrees (default = 0).
%                     Positive values rotate pattern to the right.
%           'pitch' - Rotate disk or whole icosahedron "in beta (elevation)
%                     direction" by a number of degrees (default = 0).
%                     Positive values rotate pattern upwards.
%    'sphereradius' - Radius of the sphere supported by the icosahedron
%                     (default = 1).
%         'numbars' - Number of bars making up the sample stimulus pattern
%                     in the background (default = 18).
%        'masktype' - Type of mask to be used. Options are "whole field",
%                     "front hemisphere", "rear hemisphere", 
%                     "left hemisphere",  "right hemisphere",
%                     "upper hemisphere", "lower hemisphere"
%                     and "icosahedron disks". Less commonly used options
%                     are "front tritosphere", "front tetartosphere",
%                     "front hectosphere" and "front ogdoosphere".
%
%   Which of these parameter/value pairs are specified and which ones are
%   left at default values is completely up to the user. Also, they can be
%   specified in any order, as long as they are specified as pairs.
%
%   Note: several input arguments only make sense for certain types of
%   masks. This does not cause any crashes, as they will simply be ignored.
%   To avoid confusing users, however, this should be addressed in the
%   future.
%
%   --------
%
%   Code example 1: To compute a rotated icosahedral distribution of small
%   disk-shaped masks, call
%   MASK = SPHERESTIMULUSMASK('yaw',45,'diameter',10,'masktype', ...
%                             'icosahedron disks');
%
%   Code example 2: To compute a mask covering the entire upper hemisphere,
%   while suppressing the generation of all preview figures, call
%   MASK = SPHERESTIMULUSMASK('masktype','upper hemisphere','plotflag',0);
%
%   Code example 3: To compute a single sphere-shaped mask, call
%   MASK = SPHERESTIMULUSMASK('masktype','single disk','diameter',10, ...
%                             'yaw',90,'pitch',0)
%
%   --------
%
%   This is version 2017-10-11a.
%
%   Created by Florian Alexander Dehmelt, U Tuebingen, 29 July 2016.
%   Based on an earlier script by Aristides Arrenberg, U Tuebingen,
%   with contributions by Rebecca Meier, U Tuebingen.
%
%   If you want any changes, let me know: florian.dehmelt@uni-tuebingen.de
%
%   --------
%
%   The following are Ari's original comments:
%   
%   160722 sphere LED arena stimuli masks
%   This code can be used to generate a stimulation mask for the spherical
%   LED arena. It generates stimulus patterns that are restricted in space
%   to a certain circular disk on the sphere that covers approximately one
%   steradian and is positioned in one of the 12 corners of an icosahedron.
%   The stimulus mask is programmed in a matrix that is in polar coordinate
%   space. For further explanations see document "160719Stimulus Muster
%   fuer Rebeccas Kugelarena_4.docx"
%
%                 - Aristides Arrenberg, University of Tuebingen, July 2016

  

  % PARSE VARIABLE INPUT ARGUMENTS
  %
  % Check which optional input arguments were provided, and whether their
  % values were provided in the correct format, e.g. a real number, a
  % positive real number, a character array, etc.; if an argument was not
  % provided, assign a default value instead.
  
  p = inputParser;
  
  realnumber = @(x) isnumeric(x) && isscalar(x) && isreal(x);
	positiverealnumber = @(x) realnumber(x) && (x>0);

  default.plotflag = 0;
  addOptional(p,'plotflag',default.plotflag,@(x)or(x==1,x==0));
  
  default.pixelsize = 1;
  addOptional(p,'pixelsize',default.pixelsize,positiverealnumber);
    
  default.diameter = 68;
  addOptional(p,'diameter',default.diameter,positiverealnumber);
  
  default.yaw = 0;
  addOptional(p,'yaw',default.yaw,realnumber);
  
  default.pitch = 0;
  addOptional(p,'pitch',default.pitch,realnumber);
  
  default.sphereradius = 1;
  addOptional(p,'sphereradius',default.sphereradius,positiverealnumber);
  
  default.numbars = 18;
  addOptional(p,'numbars',default.numbars,positiverealnumber);

  default.masktype = 'whole field';
  addOptional(p,'masktype',default.masktype,@ischar);

  parse(p,varargin{:});
  
  plotflag     = p.Results.plotflag;
  diameter     = p.Results.diameter;
  pixelsize    = p.Results.pixelsize;
  yaw          = -p.Results.yaw; % minus sign = CCW convention
  pitch        = p.Results.pitch;
  sphereradius = p.Results.sphereradius;
  numbars      = p.Results.numbars;
  masktype     = p.Results.masktype;
 
  

  % SET FIXED PARAMETERS
  % Alpha and beta are the azimuth and elevation of the visual field,
  % respectively. They cover the entire field, and hence the entire sphere.

  alphalimit = [-180,180];
  betalimit  = [ -90, 90];
  
  
  
  % DEFINE THE PIXEL GRID

  % Create an empty alpha/beta grid on which the masks will be defined.
  pixelgrid = zeros(abs(diff(betalimit))  / pixelsize, ...
                    abs(diff(alphalimit)) / pixelsize);

  % List all alpha and beta values that correspond to these pixels.
  alphapixel = alphalimit(1) : pixelsize : alphalimit(2)-pixelsize;
  betapixel  = betalimit(1)  : pixelsize : betalimit(2)-pixelsize;

  % Assign these values as geographic coordinates of all pixel grid points.
  geocoord(:,:,1) = repmat(betapixel',[1 size(pixelgrid,2)]);
  geocoord(:,:,2) = repmat(alphapixel,[size(pixelgrid,1) 1]);

  % For all pixel grid points, convert geographic to cartesian coordinates.
  X = sphereradius * cos(geocoord(:,:,1)/360*2*pi) ...
                  .* cos(geocoord(:,:,2)/360*2*pi);
  Y = sphereradius * cos(geocoord(:,:,1)/360*2*pi) ...
                  .* sin(geocoord(:,:,2)/360*2*pi);
  Z = sphereradius * sin(geocoord(:,:,1)/360*2*pi);

  
  
  % GENERATE A TOY STIMULUS PATTERN FOR ILLUSTRATION AND DEBUGGING
  % Generate a stimulus grating to illustrate the effect of the masks.
  % Here, the construction using "pixelsize" was included to round the
  % exact "barwidth" to the nearest actually existing pixel.

  barwidth = floor(360/(2*numbars)/pixelsize);
  pattern = NaN(size(pixelgrid));
  for k = 0:numbars-1
    pattern(:, k*2*barwidth + (1:barwidth))            = 1;
    pattern(:, k*2*barwidth + (barwidth+1:2*barwidth)) = 2;
  end

  

  % GENERATE MASK(S)
  % According to the type of mask chosen, generate one or several masks of
  % different shape and position. Save them as a stack of binary matrices.

  switch masktype
    
    case 'whole field'
      
      mask = ones(size(pixelgrid));
    
      
    case 'single disk'
      
      % Define centre point of a single disk in geographic coordinates,
      % [elevation,azimuth], with elevation -90 to 90, azimuth -180 to 180.
      node = [0,0];

      % Move this node to a different position, if desired.      
      % Begin by rotating around the left-right axis to adjust pitch, i.e.
      % the axis running from -90 to +90 degrees azimuth at zero elevation.
      
      % Rotation in geographic (or any other spherical) coordinates is an
      % affine transformation, and thus a little nastier than in cartesian
      % coordinates. To avoid this, we can convert all vectors to cartesian
      % coordinates, perform the rotation, and convert back to geographic.
      icocartesian = sphereradius * [cosd(node(:,1)).*cosd(node(:,2)), ...
                                     cosd(node(:,1)).*sind(node(:,2)), ...
                                     sind(node(:,1))];
      
      % Split each vector into one component unaffected by this rotation
      % (i.e., a vector along the rotation axis), and a second component
      % which is affected. All of this, including axis definition, is still
      % cartesian.
      rotationaxis = [0 1 0];
      unaffected = (rotationaxis' * (rotationaxis*icocartesian'))';
      affected = icocartesian - unaffected;
      
      % Rotate the affected component, in the style of Olinde Rodrigues.
      % Still cartesian.
      repeatedaxis = repmat(rotationaxis,size(affected,1),1);
      icocartesian = unaffected + ...
                     cosd(-pitch)*affected + ...
                     sind(-pitch)*cross(repeatedaxis',affected')';
      
      % Recombine the rotated and unaffected components, then convert the
      % resulting rotated vectors back to geographic coordinates.
      node = [asind(icocartesian(:,3)./sphereradius), ...
              atan2d(icocartesian(:,2),icocartesian(:,1))];
      
      % The pitch rotation is done.
      % Now we can apply the desired overall yaw rotation, if any.
      node(:,2) = node(:,2) + yaw;
      
      % If some centre points crossed the "poles" and/or "date line", make
      % sure they are reinterpreted as within the valid range of angles.      
      node(:,1) = asind(sind(node(:,1)));
      node(:,2) = alphalimit(1) + ...
                  mod(node(:,2)-alphalimit(1), diff(alphalimit));
            
      % These ideal icosahedron nodes won't usually fall on available grid
      % points. For each disk, compute next best points on a discrete grid:
      subscript = NaN(size(node,1),2);
      
      for k = 1:size(node,1)
        
        % Distance of ALL possible grid points to the ideal centre point:
        distance = sqrt((geocoord(:,:,1)-node(k,1)).^2 + ...
                        (geocoord(:,:,2)-node(k,2)).^2);
        
        % Find the closest such grid point one for each centre point.
        [nodealpha,nodebeta] = find(distance==min(min(distance)));
        
        % Remember the geographic coordinates of this one grid point.
        subscript(k,1:2) = [nodealpha(1) nodebeta(1)];
        
      end
      
      % Next, identify pixels to be associated with each disk, based on the
      % position of the centre point and the desired diameter of the
      % stimulus disk. The following matrix operations accomplish this very
      % efficiently, although they may be a bit hard to decipher.
      
      % Cartesian coordinates of all disk centre points, as row vectors:
      centreindex = sub2ind(size(X),subscript(:,1),subscript(:,2));
      centre = [X(centreindex), Y(centreindex), Z(centreindex)];
      
      % All points of the cartesian pixel grid, arranged as column vectors:
      cartesiangrid = [X(:)'; Y(:)'; Z(:)'];
      
      % The angles between grid vectors and centre point vectors:
      angle = acos(centre*cartesiangrid/sphereradius^2);
      
      % Note: The norms of grid vectors, as well as the norms of centre
      % point vectors should ALWAYS be equal to the sphere radius, because
      % they all live happily on that same sphere; there is no need to
      % compute them explicitly, so they don't appear above. --FD
      
      % Use only grid points within a certain angle, i.e. chosen disk size.
      rawmask = angle/(2*pi*sphereradius)*360 <= (diameter/2);
      mask = reshape(permute(rawmask,[2 1]),[size(X) size(centreindex,1)]);
      
      
    case 'icosahedron disks'
      
      % Define 12 corner points of an icosahedron in geographic coordinates
      % Replacing this list with a longer or shorter one (e.g., for
      % different platonic bodies) should work, but I have not tested
      % this extensively.
      ico = [ 90,      0;
              26.5,    0;
              26.5,   72;
              26.5,  144;
              26.5, -144;
              26.5,  -72;
             -26.5,   36;
             -26.5,  108;
             -26.5,  180; 
             -26.5, -108; 
             -26.5,  -36; 
             -90,      0];

      % Rotate the distribution of icosahedron nodes, if desired.      
      % Begin by rotating the distribution around the left-right axis, i.e.
      % the axis running from -90 to +90 degrees azimuth at zero elevation.
      
      % Rotation in geographic (or any other spherical) coordinates is an
      % affine transformation, and thus a little nastier than in cartesian
      % coordinates. To avoid this, we can convert all vectors to cartesian
      % coordinates, perform the rotation, and convert back to geographic.
      icocartesian = sphereradius * [cosd(ico(:,1)).*cosd(ico(:,2)), ...
                                     cosd(ico(:,1)).*sind(ico(:,2)), ...
                                     sind(ico(:,1))];
      
      % Split each vector into one component unaffected by this rotation
      % (i.e., a vector along the rotation axis), and a second component
      % which is affected. All of this, including axis definition, is still
      % cartesian.
      rotationaxis = [0 1 0];
      unaffected = (rotationaxis' * (rotationaxis*icocartesian'))';
      affected = icocartesian - unaffected;
      
      % Rotate the affected component, in the style of Olinde Rodrigues.
      % Still cartesian.
      repeatedaxis = repmat(rotationaxis,size(affected,1),1);
      icocartesian = unaffected + ...
                     cosd(-pitch)*affected + ...
                     sind(-pitch)*cross(repeatedaxis',affected')';
      
      % Recombine the rotated and unaffected components, then convert the
      % resulting rotated vectors back to geographic coordinates.
      ico = [asind(icocartesian(:,3)./sphereradius), ...
             atan2d(icocartesian(:,2),icocartesian(:,1))];
      
      % The pitch rotation is done.
      % Now we can apply the desired overall yaw rotation, if any.
      ico(:,2) = ico(:,2) + yaw;
      
      % If some centre points crossed the "poles" and/or "date line", make
      % sure they are reinterpreted as within the valid range of angles.      
      ico(:,1) = asind(sind(ico(:,1)));
      ico(:,2) = alphalimit(1) + ...
                 mod(ico(:,2)-alphalimit(1), diff(alphalimit));
            
      % These ideal icosahedron nodes won't usually fall on available grid
      % points. For each disk, compute next best points on a discrete grid:
      subscript = NaN(size(ico,1),2);
      
      for k = 1:size(ico,1)
        
        % Distance of ALL possible grid points to the ideal centre point:
        distance = sqrt((geocoord(:,:,1)-ico(k,1)).^2 + ...
                        (geocoord(:,:,2)-ico(k,2)).^2);
        
        % Find the closest such grid point one for each centre point.
        [nodealpha,nodebeta] = find(distance==min(min(distance)));
        
        % Remember the geographic coordinates of this one grid point.
        subscript(k,1:2) = [nodealpha(1) nodebeta(1)];
        
      end
      
      % Next, identify pixels to be associated with each disk, based on the
      % position of the centre point and the desired diameter of the
      % stimulus disk. The following matrix operations accomplish this very
      % efficiently, although they may be a bit hard to decipher.
      
      % Cartesian coordinates of all disk centre points, as row vectors:
      centreindex = sub2ind(size(X),subscript(:,1),subscript(:,2));
      centre = [X(centreindex), Y(centreindex), Z(centreindex)];
      
      % All points of the cartesian pixel grid, arranged as column vectors:
      cartesiangrid = [X(:)'; Y(:)'; Z(:)'];
      
      % The angles between grid vectors and centre point vectors:
      angle = acos(centre*cartesiangrid/sphereradius^2);
      
      % Note: The norms of grid vectors, as well as the norms of centre
      % point vectors should ALWAYS be equal to the sphere radius, because
      % they all live happily on that same sphere; there is no need to
      % compute them explicitly, so they don't appear above. --FD
      
      % Use only grid points within a certain angle, i.e. chosen disk size.
      rawmask = angle/(2*pi*sphereradius)*360 <= (diameter/2);
      mask = reshape(permute(rawmask,[2 1]),[size(X) size(centreindex,1)]);
      
      
    case {'left hemisphere',  'right hemisphere', ...
          'upper hemisphere', 'lower hemisphere', ...
          'front hemisphere', ...
          'front tritosphere', 'front tetartosphere', ...
          'front hectosphere', 'front ogdoosphere'}
        
      % Most hemisphere-shaped masks can be computed the same way, with
      % only slightly different parameters in each case. The more complex
      % ones (e.g., 'rear hemisphere') are a separate case treated below.
      
      switch masktype

        case 'front hemisphere'

          alpharange = [ -90  90];
          betarange  = [ -90  90];  
          
        case 'left hemisphere'

          alpharange = [-180   0];
          betarange  = [ -90  90];  

        case 'right hemisphere'

          alpharange = [   0 180];
          betarange  = [ -90  90];  

        case 'upper hemisphere'

          alpharange = [-180 180];
          betarange  = [   0  90];  

        case 'lower hemisphere'

          alpharange = [-180 180];
          betarange  = [ -90   0];
          
        % Here come the crazy ones. Feel free to add your own.
        % If you want, I can create series of e.g. six complementary
        % hectospheres to be present in a random sequence. Let me know!
        case 'front tritosphere' 

          alpharange = [ -60  60];
          betarange  = [ -90  90]; 
          
        case 'front tetartosphere' 

          alpharange = [ -45  45];
          betarange  = [ -90  90]; 
          
        case 'front hectosphere' 

          alpharange = [ -30  30];
          betarange  = [ -90  90]; 
        
        case 'front ogdoosphere' 

          alpharange = [ -22.5  22.5];
          betarange  = [ -90    90]; 
                
      end

      alphastart = (alpharange(1)-min(alphalimit)) ...
                   /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop  = (alpharange(2)-min(alphalimit)) ...
                   /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphaindex = floor(alphastart):round(alphastop);
            
      betastart = (betarange(1)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1) + 1;
      betastop  = (betarange(2)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1);
      betaindex = round(betastart):round(betastop);
      
      % The mask should be zero everywhere, except within the area
      % delimited by the indices above. Flipping the matrix upside down
      % ensures that its top-left corner corresponds to the top-left corner
      % of the screen (i.e., -180/+90, not -180/-90 as by default):
      
      mask = zeros(size(pixelgrid));
      mask(betaindex,alphaindex) = 1;
%       mask = flipud(mask);

      
    case 'rear hemisphere'
      
      % The rear hemisphere is slightly more complex to compute than most
      % others, as it crosses the "date line" of our sphere.
      
      alpharange1 = [-180 -90];
      alpharange2 = [  90 180];
      betarange   = [ -90  90];
      
      alphastart1 = (alpharange1(1)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop1  = (alpharange1(2)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphastart2 = (alpharange2(1)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop2  = (alpharange2(2)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphaindex = [floor(alphastart1):round(alphastop1), ...
                    floor(alphastart2):round(alphastop2)];
            
      betastart = (betarange(1)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1) + 1;
      betastop  = (betarange(2)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1);
      betaindex = round(betastart):round(betastop);      
      
      % Again, the mask should be zero everywhere, except within the area
      % delimited by the indices above. Flipping the matrix upside down
      % ensures that its top-left corner corresponds to the top-left corner
      % of the screen (i.e., -180/+90, not -180/-90 as by default):
      
      mask = zeros(size(pixelgrid));
      mask(betaindex,alphaindex) = 1;
%       mask = flipud(mask);
      
      
    otherwise

      error(['-- Unknown kind of ''masktype''. ', ...
             'Available types include ''whole field'', ', ...
             '''front hemisphere'', ''rear hemisphere'', ', ...
             '''left hemisphere'', ''right hemisphere'', ', ...
             '''upper hemisphere'', ''lower hemisphere'', ', ...
             'and ''icosahedron disks''. ', ...
             'All of these type names are case sensitive. ', ...
             'Read in-file documentation for more. --'])
      
      
  end
  
%   mask = flipud(mask);
  mask = fliplr(mask);
%   mask = rot90(mask,2);

  
  % PLOTS FOR DEBUGGING
  % These plots display all the masks created individually, as well as
  % their overlap. They serve no other purpose than verification and
  % debugging, and can safely be deactivated in routine use by calling the
  % function with PLOTFLAG/0 as an optional parameter/value input pair.

  if plotflag
    
    close all
    imagehandle = [];

    grating = repmat(pattern,[1 1 size(mask,3)]) .* mask;
    
    for k=1:size(mask,3)
      
      figure('NumberTitle','off', ...
             'Name',['Mask no. ',num2str(k)], ...
             'Position',[10+250*mod(k-1,4), 510-250*floor((k-1)/4), 250, 200], ...
             'Color',[1 1 1]);
%       grating(:,:,k)=pattern.*mask(:,:,k)*k;
%       imagesc(grating(:,:,k))
%       imagesc(flipud(grating(:,:,k)))
%       imagesc(fliplr(grating(:,:,k)))
      imagesc(rot90(grating(:,:,k),2))
      colormap(gray)
      xlabel('azimuth')
      ylabel('elevation')
      imagehandle = [imagehandle,gca]; %#ok<AGROW>
      
    end
    
    figure('NumberTitle','off', ...
           'Name','Composite image of all masks', ...
           'Color',[1 1 1]);
%     imagesc(sum(grating,3))
%     imagesc(flipud(sum(grating,3)))
%     imagesc(fliplr(sum(grating,3)))
    imagesc(rot90(sum(grating,3),2))
    colormap(hot)
    hold on
    plot(get(gca,'XLim'), 15/180*size(grating,1)*[1 1],'--w','LineWidth',3)
    plot(get(gca,'XLim'),165/180*size(grating,1)*[1 1],'--w','LineWidth',3)
    hold off
    xlabel('azimuth')
    ylabel('elevation')
    imagehandle = [imagehandle,gca];

    imagestyle = {'XTick',ceil((0:90:360)/pixelsize), ...
                  'YTick',ceil((0:45:135)/pixelsize), ...
                  'XTickLabel',fliplr(-180:90:180), ...
                  'YTickLabel',90:-45:-45, ...
                  'TickLength',[0 0], ...
                  'XLim',[0 round(360/pixelsize)], ...
                  'YLim',[0 round(180/pixelsize)]};

    imagestylename  = imagestyle(1:2:end);
    imagestylevalue = imagestyle(2:2:end);
        
    set(imagehandle,imagestylename,imagestylevalue)

  end

end




function mask = sphereStimulusMask(varargin)
%SPHERESTIMULUSMASK Masks for visual stimulus presentation. Now in CCW!
%
%   MASK = SPHERESTIMULUSMASK computes 12 circular masks in geographic
%   coordinates, each centred on a different node of a regular icosahedron.
%   It also creates one figure each to check their appearance.
%
%   There are no required input arguments, but a number of optional ones.
%
%   [...] = SPHERESTIMULUSMASK(..., 'PARAM1',val1, 'PARAM2',val2, ...)
%   specifies optional parameter name/value pairs to adapt mask shapes and
%   positions, and to control or suppress figure creation. Parameters are:
%   
%        'plotflag' - Whether or not to create the figures. Options are:
%                     0 (default) to suppress figures, 1 to allow them
%        'diameter' - Custom disk diameter in degrees (default = 70)
%       'pixelsize' - Display resolution, i.e. degrees/pixel of the LED
%                     array (default = 1). Low values mean high resolution.
%             'yaw' - Rotate disk or whole icosahedron "in alpha (azimuth)
%                     direction" by a number of degrees (default = 0).
%                     Positive values rotate pattern to the right.
%           'pitch' - Rotate disk or whole icosahedron "in beta (elevation)
%                     direction" by a number of degrees (default = 0).
%                     Positive values rotate pattern upwards.
%    'sphereradius' - Radius of the sphere supported by the icosahedron
%                     (default = 1).
%         'numbars' - Number of bars making up the sample stimulus pattern
%                     in the background (default = 18).
%        'masktype' - Type of mask to be used. Options are "whole field",
%                     "front hemisphere", "rear hemisphere", 
%                     "left hemisphere",  "right hemisphere",
%                     "upper hemisphere", "lower hemisphere"
%                     and "icosahedron disks". Less commonly used options
%                     are "front tritosphere", "front tetartosphere",
%                     "front hectosphere" and "front ogdoosphere".
%
%   Which of these parameter/value pairs are specified and which ones are
%   left at default values is completely up to the user. Also, they can be
%   specified in any order, as long as they are specified as pairs.
%
%   Note: several input arguments only make sense for certain types of
%   masks. This does not cause any crashes, as they will simply be ignored.
%   To avoid confusing users, however, this should be addressed in the
%   future.
%
%   --------
%
%   Code example 1: To compute a rotated icosahedral distribution of small
%   disk-shaped masks, call
%   MASK = SPHERESTIMULUSMASK('yaw',45,'diameter',10,'masktype', ...
%                             'icosahedron disks');
%
%   Code example 2: To compute a mask covering the entire upper hemisphere,
%   while suppressing the generation of all preview figures, call
%   MASK = SPHERESTIMULUSMASK('masktype','upper hemisphere','plotflag',0);
%
%   Code example 3: To compute a single sphere-shaped mask, call
%   MASK = SPHERESTIMULUSMASK('masktype','single disk','diameter',10, ...
%                             'yaw',90,'pitch',0)
%
%   --------
%
%   This is version 2017-10-11a.
%
%   Created by Florian Alexander Dehmelt, U Tuebingen, 29 July 2016.
%   Based on an earlier script by Aristides Arrenberg, U Tuebingen,
%   with contributions by Rebecca Meier, U Tuebingen.
%
%   If you want any changes, let me know: florian.dehmelt@uni-tuebingen.de
%
%   --------
%
%   The following are Ari's original comments:
%   
%   160722 sphere LED arena stimuli masks
%   This code can be used to generate a stimulation mask for the spherical
%   LED arena. It generates stimulus patterns that are restricted in space
%   to a certain circular disk on the sphere that covers approximately one
%   steradian and is positioned in one of the 12 corners of an icosahedron.
%   The stimulus mask is programmed in a matrix that is in polar coordinate
%   space. For further explanations see document "160719Stimulus Muster
%   fuer Rebeccas Kugelarena_4.docx"
%
%                 - Aristides Arrenberg, University of Tuebingen, July 2016

  

  % PARSE VARIABLE INPUT ARGUMENTS
  %
  % Check which optional input arguments were provided, and whether their
  % values were provided in the correct format, e.g. a real number, a
  % positive real number, a character array, etc.; if an argument was not
  % provided, assign a default value instead.
  
  p = inputParser;
  
  realnumber = @(x) isnumeric(x) && isscalar(x) && isreal(x);
	positiverealnumber = @(x) realnumber(x) && (x>0);

  default.plotflag = 0;
  addOptional(p,'plotflag',default.plotflag,@(x)or(x==1,x==0));
  
  default.pixelsize = 1;
  addOptional(p,'pixelsize',default.pixelsize,positiverealnumber);
    
  default.diameter = 68;
  addOptional(p,'diameter',default.diameter,positiverealnumber);
  
  default.yaw = 0;
  addOptional(p,'yaw',default.yaw,realnumber);
  
  default.pitch = 0;
  addOptional(p,'pitch',default.pitch,realnumber);
  
  default.sphereradius = 1;
  addOptional(p,'sphereradius',default.sphereradius,positiverealnumber);
  
  default.numbars = 18;
  addOptional(p,'numbars',default.numbars,positiverealnumber);

  default.masktype = 'whole field';
  addOptional(p,'masktype',default.masktype,@ischar);

  parse(p,varargin{:});
  
  plotflag     = p.Results.plotflag;
  diameter     = p.Results.diameter;
  pixelsize    = p.Results.pixelsize;
  yaw          = -p.Results.yaw; % minus sign = CCW convention
  pitch        = p.Results.pitch;
  sphereradius = p.Results.sphereradius;
  numbars      = p.Results.numbars;
  masktype     = p.Results.masktype;
 
  

  % SET FIXED PARAMETERS
  % Alpha and beta are the azimuth and elevation of the visual field,
  % respectively. They cover the entire field, and hence the entire sphere.

  alphalimit = [-180,180];
  betalimit  = [ -90, 90];
  
  
  
  % DEFINE THE PIXEL GRID

  % Create an empty alpha/beta grid on which the masks will be defined.
  pixelgrid = zeros(abs(diff(betalimit))  / pixelsize, ...
                    abs(diff(alphalimit)) / pixelsize);

  % List all alpha and beta values that correspond to these pixels.
  alphapixel = alphalimit(1) : pixelsize : alphalimit(2)-pixelsize;
  betapixel  = betalimit(1)  : pixelsize : betalimit(2)-pixelsize;

  % Assign these values as geographic coordinates of all pixel grid points.
  geocoord(:,:,1) = repmat(betapixel',[1 size(pixelgrid,2)]);
  geocoord(:,:,2) = repmat(alphapixel,[size(pixelgrid,1) 1]);

  % For all pixel grid points, convert geographic to cartesian coordinates.
  X = sphereradius * cos(geocoord(:,:,1)/360*2*pi) ...
                  .* cos(geocoord(:,:,2)/360*2*pi);
  Y = sphereradius * cos(geocoord(:,:,1)/360*2*pi) ...
                  .* sin(geocoord(:,:,2)/360*2*pi);
  Z = sphereradius * sin(geocoord(:,:,1)/360*2*pi);

  
  
  % GENERATE A TOY STIMULUS PATTERN FOR ILLUSTRATION AND DEBUGGING
  % Generate a stimulus grating to illustrate the effect of the masks.
  % Here, the construction using "pixelsize" was included to round the
  % exact "barwidth" to the nearest actually existing pixel.

  barwidth = floor(360/(2*numbars)/pixelsize);
  pattern = NaN(size(pixelgrid));
  for k = 0:numbars-1
    pattern(:, k*2*barwidth + (1:barwidth))            = 1;
    pattern(:, k*2*barwidth + (barwidth+1:2*barwidth)) = 2;
  end

  

  % GENERATE MASK(S)
  % According to the type of mask chosen, generate one or several masks of
  % different shape and position. Save them as a stack of binary matrices.

  switch masktype
    
    case 'whole field'
      
      mask = ones(size(pixelgrid));
    
      
    case 'single disk'
      
      % Define centre point of a single disk in geographic coordinates,
      % [elevation,azimuth], with elevation -90 to 90, azimuth -180 to 180.
      node = [0,0];

      % Move this node to a different position, if desired.      
      % Begin by rotating around the left-right axis to adjust pitch, i.e.
      % the axis running from -90 to +90 degrees azimuth at zero elevation.
      
      % Rotation in geographic (or any other spherical) coordinates is an
      % affine transformation, and thus a little nastier than in cartesian
      % coordinates. To avoid this, we can convert all vectors to cartesian
      % coordinates, perform the rotation, and convert back to geographic.
      icocartesian = sphereradius * [cosd(node(:,1)).*cosd(node(:,2)), ...
                                     cosd(node(:,1)).*sind(node(:,2)), ...
                                     sind(node(:,1))];
      
      % Split each vector into one component unaffected by this rotation
      % (i.e., a vector along the rotation axis), and a second component
      % which is affected. All of this, including axis definition, is still
      % cartesian.
      rotationaxis = [0 1 0];
      unaffected = (rotationaxis' * (rotationaxis*icocartesian'))';
      affected = icocartesian - unaffected;
      
      % Rotate the affected component, in the style of Olinde Rodrigues.
      % Still cartesian.
      repeatedaxis = repmat(rotationaxis,size(affected,1),1);
      icocartesian = unaffected + ...
                     cosd(-pitch)*affected + ...
                     sind(-pitch)*cross(repeatedaxis',affected')';
      
      % Recombine the rotated and unaffected components, then convert the
      % resulting rotated vectors back to geographic coordinates.
      node = [asind(icocartesian(:,3)./sphereradius), ...
              atan2d(icocartesian(:,2),icocartesian(:,1))];
      
      % The pitch rotation is done.
      % Now we can apply the desired overall yaw rotation, if any.
      node(:,2) = node(:,2) + yaw;
      
      % If some centre points crossed the "poles" and/or "date line", make
      % sure they are reinterpreted as within the valid range of angles.      
      node(:,1) = asind(sind(node(:,1)));
      node(:,2) = alphalimit(1) + ...
                  mod(node(:,2)-alphalimit(1), diff(alphalimit));
            
      % These ideal icosahedron nodes won't usually fall on available grid
      % points. For each disk, compute next best points on a discrete grid:
      subscript = NaN(size(node,1),2);
      
      for k = 1:size(node,1)
        
        % Distance of ALL possible grid points to the ideal centre point:
        distance = sqrt((geocoord(:,:,1)-node(k,1)).^2 + ...
                        (geocoord(:,:,2)-node(k,2)).^2);
        
        % Find the closest such grid point one for each centre point.
        [nodealpha,nodebeta] = find(distance==min(min(distance)));
        
        % Remember the geographic coordinates of this one grid point.
        subscript(k,1:2) = [nodealpha(1) nodebeta(1)];
        
      end
      
      % Next, identify pixels to be associated with each disk, based on the
      % position of the centre point and the desired diameter of the
      % stimulus disk. The following matrix operations accomplish this very
      % efficiently, although they may be a bit hard to decipher.
      
      % Cartesian coordinates of all disk centre points, as row vectors:
      centreindex = sub2ind(size(X),subscript(:,1),subscript(:,2));
      centre = [X(centreindex), Y(centreindex), Z(centreindex)];
      
      % All points of the cartesian pixel grid, arranged as column vectors:
      cartesiangrid = [X(:)'; Y(:)'; Z(:)'];
      
      % The angles between grid vectors and centre point vectors:
      angle = acos(centre*cartesiangrid/sphereradius^2);
      
      % Note: The norms of grid vectors, as well as the norms of centre
      % point vectors should ALWAYS be equal to the sphere radius, because
      % they all live happily on that same sphere; there is no need to
      % compute them explicitly, so they don't appear above. --FD
      
      % Use only grid points within a certain angle, i.e. chosen disk size.
      rawmask = angle/(2*pi*sphereradius)*360 <= (diameter/2);
      mask = reshape(permute(rawmask,[2 1]),[size(X) size(centreindex,1)]);
      
      
    case 'icosahedron disks'
      
      % Define 12 corner points of an icosahedron in geographic coordinates
      % Replacing this list with a longer or shorter one (e.g., for
      % different platonic bodies) should work, but I have not tested
      % this extensively.
      ico = [ 90,      0;
              26.5,    0;
              26.5,   72;
              26.5,  144;
              26.5, -144;
              26.5,  -72;
             -26.5,   36;
             -26.5,  108;
             -26.5,  180; 
             -26.5, -108; 
             -26.5,  -36; 
             -90,      0];

      % Rotate the distribution of icosahedron nodes, if desired.      
      % Begin by rotating the distribution around the left-right axis, i.e.
      % the axis running from -90 to +90 degrees azimuth at zero elevation.
      
      % Rotation in geographic (or any other spherical) coordinates is an
      % affine transformation, and thus a little nastier than in cartesian
      % coordinates. To avoid this, we can convert all vectors to cartesian
      % coordinates, perform the rotation, and convert back to geographic.
      icocartesian = sphereradius * [cosd(ico(:,1)).*cosd(ico(:,2)), ...
                                     cosd(ico(:,1)).*sind(ico(:,2)), ...
                                     sind(ico(:,1))];
      
      % Split each vector into one component unaffected by this rotation
      % (i.e., a vector along the rotation axis), and a second component
      % which is affected. All of this, including axis definition, is still
      % cartesian.
      rotationaxis = [0 1 0];
      unaffected = (rotationaxis' * (rotationaxis*icocartesian'))';
      affected = icocartesian - unaffected;
      
      % Rotate the affected component, in the style of Olinde Rodrigues.
      % Still cartesian.
      repeatedaxis = repmat(rotationaxis,size(affected,1),1);
      icocartesian = unaffected + ...
                     cosd(-pitch)*affected + ...
                     sind(-pitch)*cross(repeatedaxis',affected')';
      
      % Recombine the rotated and unaffected components, then convert the
      % resulting rotated vectors back to geographic coordinates.
      ico = [asind(icocartesian(:,3)./sphereradius), ...
             atan2d(icocartesian(:,2),icocartesian(:,1))];
      
      % The pitch rotation is done.
      % Now we can apply the desired overall yaw rotation, if any.
      ico(:,2) = ico(:,2) + yaw;
      
      % If some centre points crossed the "poles" and/or "date line", make
      % sure they are reinterpreted as within the valid range of angles.      
      ico(:,1) = asind(sind(ico(:,1)));
      ico(:,2) = alphalimit(1) + ...
                 mod(ico(:,2)-alphalimit(1), diff(alphalimit));
            
      % These ideal icosahedron nodes won't usually fall on available grid
      % points. For each disk, compute next best points on a discrete grid:
      subscript = NaN(size(ico,1),2);
      
      for k = 1:size(ico,1)
        
        % Distance of ALL possible grid points to the ideal centre point:
        distance = sqrt((geocoord(:,:,1)-ico(k,1)).^2 + ...
                        (geocoord(:,:,2)-ico(k,2)).^2);
        
        % Find the closest such grid point one for each centre point.
        [nodealpha,nodebeta] = find(distance==min(min(distance)));
        
        % Remember the geographic coordinates of this one grid point.
        subscript(k,1:2) = [nodealpha(1) nodebeta(1)];
        
      end
      
      % Next, identify pixels to be associated with each disk, based on the
      % position of the centre point and the desired diameter of the
      % stimulus disk. The following matrix operations accomplish this very
      % efficiently, although they may be a bit hard to decipher.
      
      % Cartesian coordinates of all disk centre points, as row vectors:
      centreindex = sub2ind(size(X),subscript(:,1),subscript(:,2));
      centre = [X(centreindex), Y(centreindex), Z(centreindex)];
      
      % All points of the cartesian pixel grid, arranged as column vectors:
      cartesiangrid = [X(:)'; Y(:)'; Z(:)'];
      
      % The angles between grid vectors and centre point vectors:
      angle = acos(centre*cartesiangrid/sphereradius^2);
      
      % Note: The norms of grid vectors, as well as the norms of centre
      % point vectors should ALWAYS be equal to the sphere radius, because
      % they all live happily on that same sphere; there is no need to
      % compute them explicitly, so they don't appear above. --FD
      
      % Use only grid points within a certain angle, i.e. chosen disk size.
      rawmask = angle/(2*pi*sphereradius)*360 <= (diameter/2);
      mask = reshape(permute(rawmask,[2 1]),[size(X) size(centreindex,1)]);
      
      
    case {'left hemisphere',  'right hemisphere', ...
          'upper hemisphere', 'lower hemisphere', ...
          'front hemisphere', ...
          'front tritosphere', 'front tetartosphere', ...
          'front hectosphere', 'front ogdoosphere'}
        
      % Most hemisphere-shaped masks can be computed the same way, with
      % only slightly different parameters in each case. The more complex
      % ones (e.g., 'rear hemisphere') are a separate case treated below.
      
      switch masktype

        case 'front hemisphere'

          alpharange = [ -90  90];
          betarange  = [ -90  90];  
          
        case 'left hemisphere'

          alpharange = [-180   0];
          betarange  = [ -90  90];  

        case 'right hemisphere'

          alpharange = [   0 180];
          betarange  = [ -90  90];  

        case 'upper hemisphere'

          alpharange = [-180 180];
          betarange  = [   0  90];  

        case 'lower hemisphere'

          alpharange = [-180 180];
          betarange  = [ -90   0];
          
        % Here come the crazy ones. Feel free to add your own.
        % If you want, I can create series of e.g. six complementary
        % hectospheres to be present in a random sequence. Let me know!
        case 'front tritosphere' 

          alpharange = [ -60  60];
          betarange  = [ -90  90]; 
          
        case 'front tetartosphere' 

          alpharange = [ -45  45];
          betarange  = [ -90  90]; 
          
        case 'front hectosphere' 

          alpharange = [ -30  30];
          betarange  = [ -90  90]; 
        
        case 'front ogdoosphere' 

          alpharange = [ -22.5  22.5];
          betarange  = [ -90    90]; 
                
      end

      alphastart = (alpharange(1)-min(alphalimit)) ...
                   /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop  = (alpharange(2)-min(alphalimit)) ...
                   /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphaindex = floor(alphastart):round(alphastop);
            
      betastart = (betarange(1)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1) + 1;
      betastop  = (betarange(2)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1);
      betaindex = round(betastart):round(betastop);
      
      % The mask should be zero everywhere, except within the area
      % delimited by the indices above. Flipping the matrix upside down
      % ensures that its top-left corner corresponds to the top-left corner
      % of the screen (i.e., -180/+90, not -180/-90 as by default):
      
      mask = zeros(size(pixelgrid));
      mask(betaindex,alphaindex) = 1;
%       mask = flipud(mask);

      
    case 'rear hemisphere'
      
      % The rear hemisphere is slightly more complex to compute than most
      % others, as it crosses the "date line" of our sphere.
      
      alpharange1 = [-180 -90];
      alpharange2 = [  90 180];
      betarange   = [ -90  90];
      
      alphastart1 = (alpharange1(1)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop1  = (alpharange1(2)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphastart2 = (alpharange2(1)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2) + 1;
      alphastop2  = (alpharange2(2)-min(alphalimit)) ...
                    /abs(diff(alphalimit)) * size(pixelgrid,2);
      alphaindex = [floor(alphastart1):round(alphastop1), ...
                    floor(alphastart2):round(alphastop2)];
            
      betastart = (betarange(1)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1) + 1;
      betastop  = (betarange(2)-min(betalimit)) ...
                   /abs(diff(betalimit)) * size(pixelgrid,1);
      betaindex = round(betastart):round(betastop);      
      
      % Again, the mask should be zero everywhere, except within the area
      % delimited by the indices above. Flipping the matrix upside down
      % ensures that its top-left corner corresponds to the top-left corner
      % of the screen (i.e., -180/+90, not -180/-90 as by default):
      
      mask = zeros(size(pixelgrid));
      mask(betaindex,alphaindex) = 1;
%       mask = flipud(mask);
      
      
    otherwise

      error(['-- Unknown kind of ''masktype''. ', ...
             'Available types include ''whole field'', ', ...
             '''front hemisphere'', ''rear hemisphere'', ', ...
             '''left hemisphere'', ''right hemisphere'', ', ...
             '''upper hemisphere'', ''lower hemisphere'', ', ...
             'and ''icosahedron disks''. ', ...
             'All of these type names are case sensitive. ', ...
             'Read in-file documentation for more. --'])
      
      
  end
  
%   mask = flipud(mask);
%   mask = fliplr(mask);
%   mask = rot90(mask,2);

  
  % PLOTS FOR DEBUGGING
  % These plots display all the masks created individually, as well as
  % their overlap. They serve no other purpose than verification and
  % debugging, and can safely be deactivated in routine use by calling the
  % function with PLOTFLAG/0 as an optional parameter/value input pair.

  if plotflag
    
    close all
    imagehandle = [];

    grating = repmat(pattern,[1 1 size(mask,3)]) .* mask;
    
    for k=1:size(mask,3)
      
      figure('NumberTitle','off', ...
             'Name',['Mask no. ',num2str(k)], ...
             'Position',[10+250*mod(k-1,4), 510-250*floor((k-1)/4), 250, 200], ...
             'Color',[1 1 1]);
%       grating(:,:,k)=pattern.*mask(:,:,k)*k;
%       imagesc(grating(:,:,k))
%       imagesc(flipud(grating(:,:,k)))
%       imagesc(fliplr(grating(:,:,k)))
      imagesc(rot90(grating(:,:,k),2))
      colormap(gray)
      xlabel('azimuth')
      ylabel('elevation')
      imagehandle = [imagehandle,gca]; %#ok<AGROW>
      
    end
    
    figure('NumberTitle','off', ...
           'Name','Composite image of all masks', ...
           'Color',[1 1 1]);
%     imagesc(sum(grating,3))
%     imagesc(flipud(sum(grating,3)))
%     imagesc(fliplr(sum(grating,3)))
    imagesc(rot90(sum(grating,3),2))
    colormap(hot)
    hold on
    plot(get(gca,'XLim'), 15/180*size(grating,1)*[1 1],'--w','LineWidth',3)
    plot(get(gca,'XLim'),165/180*size(grating,1)*[1 1],'--w','LineWidth',3)
    hold off
    xlabel('azimuth')
    ylabel('elevation')
    imagehandle = [imagehandle,gca];

    imagestyle = {'XTick',ceil((0:90:360)/pixelsize), ...
                  'YTick',ceil((0:45:135)/pixelsize), ...
                  'XTickLabel',fliplr(-180:90:180), ...
                  'YTickLabel',90:-45:-45, ...
                  'TickLength',[0 0], ...
                  'XLim',[0 round(360/pixelsize)], ...
                  'YLim',[0 round(180/pixelsize)]};

    imagestylename  = imagestyle(1:2:end);
    imagestylevalue = imagestyle(2:2:end);
        
    set(imagehandle,imagestylename,imagestylevalue)

  end

end