function [position,lasttrial,laststrength] = coverSphere(varargin)
% FuNElemtion to distribute points on the 2D surface of a 3D sphere (only!).
% Optionally, the positions on opposite hemispheres can be coupled for
% mirror symmetry. Currently implemented for left/right hemispheres only.
%
% Written by Florian Alexander Dehmelt, Tuebingen University Hospital.
% Based on an earlier function by FAD, coversphereshell.m.
% This is version 20171008a.
  
  p = inputParser;
  
  realnumber = @(x) isnumeric(x) && isscalar(x) && isreal(x);
	positiverealnumber = @(x) realnumber(x) && (x>0);
  positiveinteger = @(x) (x==round(x)) && (x>0);
  validsymmetry = @(x) strcmp(x,'none')      || ...
                       strcmp(x,'mirror x')  || ...
                       strcmp(x,'mirror y')  || ...
                       strcmp(x,'mirror z')  || ...
                       strcmp(x,'mirror xy') || ...
                       strcmp(x,'mirror yz') || ...
                       strcmp(x,'mirror xz') || ...
                       strcmp(x,'point xyz') || ...
                       strcmp(x,'point xy')  || ...
                       strcmp(x,'special');

  default.quantity = 36;
  addOptional(p,'quantity',default.quantity,positiveinteger);

  default.strength = 1e-2;
  addOptional(p,'strength',default.strength,positiverealnumber);
  
  default.symmetry = 'none';
  addOptional(p,'symmetry',default.symmetry,validsymmetry);

  parse(p,varargin{:});  
  strength = p.Results.strength;
  NElemrange  = p.Results.quantity;
  symmetry = p.Results.symmetry;


  tic
    
  position = cell(numel(NElemrange),1);   % input weights chosen
  lasttrial    = NaN(numel(NElemrange),1);   % no. of trials needed
  laststrength = NaN(numel(NElemrange),1);   % repulsion strength needed
  
  disp(' ')
  
  for k = 1:numel(NElemrange)

    disp(['Now computing set ',num2str(k), ...
             ' of ',num2str(numel(NElemrange)),':'])

    ND = 3;
    NElem = NElemrange(k);

    NBelt = [];
    [weightsample,finalstep,pos,rep,NBelt] = ...
      surfaceRepulsion(NElem,ND,strength,symmetry);
    plotSurfaceEvolution(pos,rep,finalstep,symmetry,NBelt)

    % output all weights
    position{k} = weightsample;
    disp(['--- Done distributing ',num2str(NElem), ...
             ' points. Symmetry: ',symmetry,'. ---'])
    toc
    disp(' ')

  end
  
  disp('--- All done. ---')
  
end



function [finalpos,finalstep,pos,rep,NBelt] = surfaceRepulsion(NElem,ND,initialstrength,symmetry)

  NS = 1e5;
  
  pos = NaN(NElem,ND,NS);
%   pos(:,:,1) = ones(NElem,ND,1) + 1e-1*rand(NElem,ND,1);

  NBelt = NaN;

  switch symmetry
    
    case 'none'
      
      pos(:,:,1) = rand(NElem,ND,1) - .5;
      % "-.5" shifts interval to -.5:.5, and to -1:1 after normalisation.
      
    case {'mirror x',  'mirror y',  'mirror z', ...
          'point xy',  'point xyz'}
      
      firsthalf = rand(floor(NElem/2),ND,1) - .5;
      secondhalf = firsthalf;                         % same as first...
      
      switch symmetry                                 % ...but mirrored...
        case 'mirror x'
          secondhalf(:,1,:)   = -secondhalf(:,1,:);   % ...in x.     
        case 'mirror y'
          secondhalf(:,2,:)   = -secondhalf(:,2,:);   % ...in y.
        case 'mirror z'
          secondhalf(:,3,:)   = -secondhalf(:,3,:);   % ...in z.
        case 'point xyz'
          secondhalf(:,:,:)   = -secondhalf(:,:,:);   % Or point-symmetric.
        case 'point xy'
          secondhalf(:,1:2,:) = -secondhalf(:,1:2,:); % Or in plane.
      end
      
      if mod(NElem,2) == 0 % NElem is even number: create symmetric groups.
        
        pos(1:NElem/2,:,1)       = firsthalf;
        pos((NElem/2+1):end,:,1) = secondhalf;
        
      else % NElem is odd number: add single point (no. 1) along the midline.
        
        switch symmetry
          case {'point xy', 'point xyz'}
            error('Strict point symmetry only works for even numbers.')
        end
        
        pos(1+(1:floor(NElem/2)),:,1)   = firsthalf;
        pos((2+floor(NElem/2)):end,:,1) = secondhalf; 
        pos(1,:,1) = rand(1,ND,1); % Create random point no. 1, ...
        pos(1,2,1) = 0;            % ...then move it to the midline (y=0).
        
      end
      
    case {'mirror xy', 'mirror yz', 'mirror xz'}
      
      if mod(NElem,4) ~= 0
        error('Two-fold mirror symmetry only works for multiples of four.')
      end
        
      firstquarter = rand(NElem/4,ND,1) - .5;
      [secondquarter,thirdquarter,fourthquarter] = deal(firstquarter);
      
      switch symmetry
        
        case 'mirror xy'
          
          secondquarter(:,1,:)    = -secondquarter(:,1,:);
          thirdquarter(:,[1 2],:) = -thirdquarter(:,[1 2],:);
          fourthquarter(:,2,:)    = -fourthquarter(:,2,:);
        
        case 'mirror yz'
          
          secondquarter(:,2,:)    = -secondquarter(:,2,:);
          thirdquarter(:,[2 3],:) = -thirdquarter(:,[2 3],:);
          fourthquarter(:,3,:)    = -fourthquarter(:,3,:);
        
        case 'mirror xz'
          
          secondquarter(:,1,:)    = -secondquarter(:,1,:);
          thirdquarter(:,[1 3],:) = -thirdquarter(:,[1 3],:);
          fourthquarter(:,3,:)    = -fourthquarter(:,3,:);
          
      end
      
      pos((1:NElem/4) + 0*NElem/4,:,1) = firstquarter;
      pos((1:NElem/4) + 1*NElem/4,:,1) = secondquarter;
      pos((1:NElem/4) + 2*NElem/4,:,1) = thirdquarter;
      pos((1:NElem/4) + 3*NElem/4,:,1) = fourthquarter;
      
    case 'special'
      
      NBelt = ceil(sqrt(NElem));  % Take first guess at number of elements 
      NRest = NElem - NBelt;      % to include in "belt", and "rest".
      
      switch mod(NRest,4)         % Make sure NRest is a multiple of four.
        case 1
          NRest = NRest - 1;
          NBelt = NBelt + 1;
        case 2
          NRest = NRest - 2;
          NBelt = NBelt + 2;          
        case 3
          NRest = NRest + 1;
          NBelt = NBelt - 1;
      end
      
      firstquarter = rand(NRest/4,ND,1) - .5;
      [secondquarter,thirdquarter,fourthquarter] = deal(firstquarter);
      
      % Now, as in case "mirror yz":
      secondquarter(:,2,:)    = -secondquarter(:,2,:);
      thirdquarter(:,[2 3],:) = -thirdquarter(:,[2 3],:);
      fourthquarter(:,3,:)    = -fourthquarter(:,3,:);
      
      firstbelthalf = [cos((.5:floor(NBelt/2)-.5)*pi/floor(NBelt/2))', ...
                       sin((.5:floor(NBelt/2)-.5)*pi/floor(NBelt/2))', ...
                       zeros(floor(NBelt/2),1)];

      % Now, as in case "mirror y":
      secondbelthalf = firstbelthalf;
      secondbelthalf(:,2,:) = -secondbelthalf(:,2,:); 
      
      if mod(NBelt,2) == 0 % even number
        
        pos(1:NBelt/2,:,1)         = firstbelthalf;
        pos((1+NBelt/2):NBelt,:,1) = secondbelthalf; 
      
      else % odd number
        
        pos(1+(1:floor(NBelt/2)),:,1)         = firstbelthalf;
        pos((2+floor(NBelt/2)):(1+NBelt),:,1) = secondbelthalf; 
        pos(1,:,1) = rand(1,ND,1); % Create random point no. 1, ...
        pos(1,[2 3],1) = 0;        % ...then move it to the front (y,z=0).
        
      end
      
      pos((NBelt+1+0*NRest/4:NBelt+1*NRest/4),:,1) = firstquarter;
      pos((NBelt+1+1*NRest/4:NBelt+2*NRest/4),:,1) = secondquarter;
      pos((NBelt+1+2*NRest/4:NBelt+3*NRest/4),:,1) = thirdquarter;
      pos((NBelt+1+3*NRest/4:NBelt+4*NRest/4),:,1) = fourthquarter;
      
  end
  
  pos(:,:,1) = pos(:,:,1)./repmat(sqrt(sum(pos(:,:,1).^2,2)),[1 ND 1]);
  rep = NaN(NElem,ND,NS);
  
  strength = initialstrength;   % initialise repulsion strength
  flipcounter = 1;              % initialise counter for changes to strength
  convergence = 0;              % initialise convergence flag as false
  checkpoint = 35;       % when to check whether variaNEleme has begun to decrease noticeably
%   checkpointConv = 20;          % when to check whether variaNEleme has begun to approach zero
  checkrange = 5;               % how many steps each are used to compute early/later variaNEleme
  
  % pos = position of weights over time, in NElem x ND x NS
  % distaNEleme = distaNEleme vectors between all pairs of points, in NElem x ND x NElem
  % euclidean = abs. value of distaNElemes betw. pairs, in NElem x ND x NElem (for further computation)
  % geodesic = inner product between all pairs of points, NElem x ND x NElem
  % direction = unit distaNEleme vectors, i.e. normalised "distaNEleme", NElem x ND x NElem
  % repulsion = sum of distaNEleme vectors, each weighted by euclidean, in NElem x ND
  % definition of euclidean calls "min" to eliminate "Inf" contribution from self-paibelt

  while ~convergence
    
    for t = 1:NS

  %       pos(:,:,t) = max(pos(:,:,t),1e-6*ones(NElem,ND));

      % (2) randomize slightly, then normalize length to force weights onto unit hypersphere
      pos(:,:,t) = pos(:,:,t) + 0e-6*abs(rand(NElem,ND,1));
      pos(:,:,t) = pos(:,:,t) ./ repmat(sqrt(sum(pos(:,:,t).^2,2)),[1 ND]);

      % (3b) compute distaNEleme measures
      distaNEleme  = repmat(permute(pos(:,:,t),[3 2 1]),[NElem 1 1]) - repmat(pos(:,:,t),[1 1 NElem]);
      euclidean = repmat(sqrt(sum(distaNEleme.^2,2)),[1 ND 1]);
      geodesic  = repmat(permute(real(acos(pos(:,:,t)*pos(:,:,t)')),[1 3 2]),[1 ND 1]);
      direction = distaNEleme ./ repmat(sqrt(sum(distaNEleme.^2,2)),[1 ND 1]);

      % (3b) sanitize distaNEleme measures: minimally positive distaNEleme between a point and itself
      [euclidean(euclidean==0), geodesic(geodesic==0)] = deal(1e10*rand(1));
      direction(isnan(direction)) = deal(0);

      % (4) compute repulsion, impose upper bound for stability, update positions
      power = -1;
      rep(:,:,t) = strength * squeeze(sum((geodesic.^power).*direction,1))';
      switch symmetry
        
        case 'special'
          
          % Hold belt in position, flat in x/y plane.
          pos(1:NBelt,:,t+1) = pos(1:NBelt,:,t);
          
          % Update all other points as computed before.
          pos(NBelt+1:end,:,t+1) = pos(NBelt+1:end,:,t) + rep(NBelt+1:end,:,t);
          
        otherwise
          
          if mod(NElem,2) == 0 % NElem is even number: create symmetric groups.
            pos(:,:,t+1) = pos(:,:,t) + rep(:,:,t);
          else % NElem is odd number: hold point no. 1 on the midline.
            pos(1,:,t+1) = pos(1,:,t) + [0 0 1].*rep(1,:,t);
            pos(2:end,:,t+1) = pos(2:end,:,t) + rep(2:end,:,t);
          end
          
      end

      % check if the paramaters are way off, i.e., repulsion much too weak
      if t == checkpoint 
        earlymaxvar = max(sqrt(sum(var(pos(:,:,1:checkrange),[],3).^2,2)));
        latermaxvar = max(sqrt(sum(var(pos(:,:,checkpoint+1-checkrange:checkpoint),[],3).^2,2)));
        tooslow = latermaxvar > .9*earlymaxvar;
        if tooslow
          disp('Too slow. Adapting parameters...')
          strength = strength * 10^flipcounter;
          flipcounter = -sign(flipcounter) * (abs(flipcounter)+1);    % => *1e1,*1e-2,*1e3,*1e-4...
          break
        else
          disp('Looking good so far, keeping parameters.')
        end

      end

      % check for convergence, and quit happily if encountered; else continue up to max. NS steps
      if ~mod(t,20) && t >= 40
        finalmaxvar = max(sqrt(sum(var(pos(:,:,t+(1-checkrange:0)),[],3).^2,2)));
        slowenough = finalmaxvar < .0001*latermaxvar;
        closeenough = sqrt(sum(mean(pos(:,:,t),1).^2,2)) < .0001;
        convergence = slowenough && closeenough;
        if convergence
          disp(['Good enough convergence after ',num2str(t),' steps.'])
          disp(['Repulsion strength was ',num2str(strength),'.'])
          break
        end
      end

      finalstep = t;

    end

    if ~convergence && ~tooslow
      disp(['No convergence after ',num2str(NS),' steps. Adapting parameters...'])
      strength = strength * 10^flipcounter;
      iNElemrement = .5;
      flipcounter = -sign(flipcounter) * (abs(flipcounter)+iNElemrement);
    end
    
  end
  
  pos = pos(:,:,1:finalstep);
  finalpos = squeeze(pos(:,:,end));

end



function plotSurfaceEvolution(pos,rep,finalstep,symmetry,NBelt)
  
  figure(1)
  set(gcf,'Color',[1 1 1],'Position',[50,240,748,714])
  clf
  
  NElem = size(pos,1);
  ND = size(pos,2);
  NS = size(pos,3);

  % plot unit circle
  x = -1:.01:1;
  ypos = sqrt(1-x.^2);
  yneg = -ypos;

  plotstep = 1;
  logstep = [floor(logspace(0,log10(finalstep),250)), finalstep];
  tracecolour = winter(1+ceil(1.2*numel(logstep)/plotstep));
  colourcounter = 0;
  greycolour = .75*[1 1 1];
    
  if ND == 2
    
    for t = 1:plotstep:NS

      plot(x,ypos,'Color',.6*[1 1 1])
      xlabel('dimension 1','FontSize',12)
      ylabel('dimension 2','FontSize',12)
      hold on
      plot(x,yneg,'Color',greycolour)
      set(gca,'XLim',[-1.5 1.5],'YLim',[-1.5 1.5],'XTick',[-1 0 1],'YTick',[-1 0 1],'LineWidth',2)
      p = scatter(pos(:,1,t),pos(:,2,t),'o');
      for j = 1:NElem
        r(j) = plot([pos(j,1,t),pos(j,1,t)+rep(j,1,t)], ...
                    [pos(j,2,t),pos(j,2,t)+rep(j,2,t)]);
        n(j) = plot([pos(j,1,t)+rep(j,1,t),0], ...
                    [pos(j,2,t)+rep(j,2,t),0],'--','Color',greycolour);
      end
      hold off

      colourcounter = colourcounter + 1;
      colour = tracecolour(1+mod(colourcounter,size(tracecolour,1)+1),:);
      set(p,'MarkerEdgeColor',colour)
      drawnow
      pause(.01)

    end
  
  elseif ND == 3
    
    for t = logstep
%     for t = 1:plotstep:NS

      % plot unit sphere
      [x,y,z] = sphere(200);
      s = surfl(x,y,z);
      colormap([1 1 1])
      set(s,'FaceAlpha',.7)
      shading flat
            
      col.eye = .1*[1 1 1];
      col.body = .7*[1 1 1];
      
      xlabel('dimension 1','FontSize',12)
      ylabel('dimension 2','FontSize',12)
      zlabel('dimension 3','FontSize',12)
      hold on
      plot3dFish(.1,-180,0,0,col);
      axis equal
      set(gca,'XLim',[-1.2 1.2],'YLim',[-1.2 1.2],'ZLim',[-1.2 1.2], ...
              'XTick',[-1 0 1],'YTick',[-1 0 1],'ZTick',[-1 0 1],'LineWidth',2)
      
      switch symmetry
        
        case 'special'
          
          p1 = scatter3(pos(1:NBelt,1,t),pos(1:NBelt,2,t),pos(1:NBelt,3,t),'o');
          p2 = scatter3(pos(NBelt+1:NElem,1,t),pos(NBelt+1:NElem,2,t),pos(NBelt+1:NElem,3,t),'o');
          [xx1,yy1,zz1,xx2,yy2,zz2] = deal([]);
          for j = 1:NBelt
            xx1 = [xx1, pos(j,1,t),pos(j,1,t)+rep(j,1,t), NaN];
            yy1 = [yy1, pos(j,2,t),pos(j,2,t)+rep(j,2,t), NaN];
            zz1 = [zz1, pos(j,3,t),pos(j,3,t)+rep(j,3,t), NaN];
          end
          for j = NBelt+1:NElem
            xx2 = [xx2, pos(j,1,t),pos(j,1,t)+rep(j,1,t), NaN];
            yy2 = [yy2, pos(j,2,t),pos(j,2,t)+rep(j,2,t), NaN];
            zz2 = [zz2, pos(j,3,t),pos(j,3,t)+rep(j,3,t), NaN];
          end
          r1 = plot3(xx1,yy1,zz1);
          r2 = plot3(xx2,yy2,zz2);
          
        otherwise
          
          p = scatter3(pos(:,1,t),pos(:,2,t),pos(:,3,t),'o');
          xx = [];
          yy = [];
          zz = [];
          for j = 1:NElem
            xx = [xx, pos(j,1,t),pos(j,1,t)+rep(j,1,t), NaN];
            yy = [yy, pos(j,2,t),pos(j,2,t)+rep(j,2,t), NaN];
            zz = [zz, pos(j,3,t),pos(j,3,t)+rep(j,3,t), NaN];
          end
          r = plot3(xx,yy,zz);
          
      end

      hold off

      colourcounter = colourcounter + 1;
      colour = tracecolour(1+mod(colourcounter,size(tracecolour,1)+1),:);
      switch symmetry
        case 'special'
          set(p1,'MarkerEdgeColor','none','MarkerFaceColor','r','SizeData',70)
          set(p2,'MarkerEdgeColor','none','MarkerFaceColor',colour,'SizeData',70)
          set(r1,'Color','r')
          set(r2,'Color',colour)
        otherwise
          set(p,'MarkerEdgeColor','none','MarkerFaceColor',colour,'SizeData',70)
          set(r,'Color',colour)
      end
      drawnow
      pause(.01)

    end
    
  else
    
    axis off
    tt = text(.5,.5,'Weight evolution will only be displayed in 2D and 3D.');
    set(tt,'HorizontalAlignment','center')
    
  end

end



function [lefteyehandle,righteyehandle,bodyhandle] = ...
  plot3dFish(scale,yaw,pitch,roll,colour)

  % This corresponds to version 20170505a, a.k.a. plot3dFish20170505a.m.

  set(gcf,'Color',[1 1 1])

  resolution = 10;

  eye.x = scale * (cos(-pi:pi/resolution:3*pi-pi/resolution)');
  eye.y = scale * (0.7*sin(0:pi/resolution:4*pi-pi/resolution)');
  eye.z = scale * ([.7*ones(2*resolution,1); ...
                    1*ones(2*resolution,1)]);

  N = 2*resolution;

  lefteye.vertices = [eye.x, eye.y, eye.z];
  righteye.vertices = [eye.x, -eye.y, eye.z];

  lefteye.faces = [[1:2*resolution 1]; ...
                  [1+2*resolution:4*resolution 1+2*resolution]];


  for k = 1:N

    lefteye.faces = [lefteye.faces; ...
                    [mod([k-1 k],N)+1, ...
                     mod([k k-1],N)+1+N, ...
                     mod(k-1,N)+1, ...
                     NaN(1,N-4)]];

  end

  lefteye.facevertexcdata = repmat(colour.eye,[size(lefteye.faces,1) 1]);

  righteye.faces = lefteye.faces;
  righteye.facevertexcdata = lefteye.facevertexcdata;

  lefteyehandle  = patch(lefteye);
  righteyehandle = patch(righteye);
  set([lefteyehandle,righteyehandle],'FaceColor','flat','EdgeColor','none')

  mainbody.x = scale * (-.1 + [1.0*cos(-pi:pi/resolution:pi-pi/resolution), ...
                               0.4*cos(-pi:pi/resolution:pi-pi/resolution)]');
  mainbody.y = scale * ([0.7*sin(0:pi/resolution:2*pi-pi/resolution), ...
                         0.2*sin(0:pi/resolution:2*pi-pi/resolution)]');
  mainbody.z = scale * ([1.1*ones(2*resolution,1); ...
                        -7*ones(2*resolution,1)]);

  N = 2*resolution;

  body.vertices = [mainbody.x, mainbody.y, mainbody.z];

  body.faces = [[1:2*resolution 1]; ...
                [1+2*resolution:4*resolution 1+2*resolution]];

  for k = 1:N

    body.faces = [body.faces; ...
                  [mod([k-1 k],N)+1, ...
                   mod([k k-1],N)+1+N, ...
                   mod(k-1,N)+1, ...
                   NaN(1,N-4)]];

  end

  body.facevertexcdata = repmat(colour.body,[size(body.faces,1) 1]);

  bodyhandle = patch(body);
  set(bodyhandle,'FaceColor','flat','EdgeColor','none')

  rotate(lefteyehandle,[1 0 0],80)
  rotate(lefteyehandle,[0 0 1],-10)
  rotate(righteyehandle,[1 0 0],-80)
  rotate(righteyehandle,[0 0 1],10)
  rotate(bodyhandle,[0 1 0],-90)
  
  rotate(lefteyehandle,[0 0 1],yaw)
  rotate(righteyehandle,[0 0 1],yaw)
  rotate(bodyhandle,[0 0 1],yaw)
  
  rotate(lefteyehandle,[0 1 0],pitch)
  rotate(righteyehandle,[0 1 0],pitch)
  rotate(bodyhandle,[0 1 0],pitch)
  
  rotate(lefteyehandle,[1 0 0],roll)
  rotate(righteyehandle,[1 0 0],roll)
  rotate(bodyhandle,[1 0 0],roll)

end
