% The null hypothesis is that two distinct experimental conditions have no
% effect on the location of maximum OKR gain as determined by a fit to data.
% The present test is meant to reject or not reject this null hypothesis.
% 
% florian.dehmelt@gmail.com, 25 August 2019, U Tuebingen, Germany.

function pvalue = permutationTest
  
  % Choose azimuth sign convention: 'CW' for clockwise, 'CCW' counter-clockwise
  % This has to be a global variable because nlinfit does not accept additional
  % inputs to the modelfun function handle (which nlinfit receives as input).
  global convention
  convention = 'CW';
  
  % The following is the desired number of perturbations to be conducted.
  % If you wish to quickly test the code, reduce this to 100 or less.
  numiteration = 10000;
  
  % Locate data files to be compared. By default, these paths refer to locations
  % on our CIN server. If you cannot access it, just create your own local copy
  % and update the following paths accordingly. Make sure to save both datasets
  % to separate folders (as is the case on the server), because they contain
  % identically named files (DiscardTrial.xlsx, Stimulus.xlsx, etc.).
  % Loading data directly from the server can be an order of magnitude slower
  % than loading it from a local copy - but it is also easier/safer.
  toplevel = ['\\172.25.250.112\arrenberg_data\shared\MS_SphereArena\', ...
              'Resources\ContributionFlorian\Analysis pipeline\'];
  dataset1.file = 'result_20180110_125537.mat';
  dataset1.folder = [toplevel,'Data D Series (neue Experimente)\'];
  dataset2.file = 'result_20180913_151415.mat';
  dataset2.folder = [toplevel,'Control Data Upside Down (Kontrolle ventral)\'];
  
  % Importantly, we must manually confirm which Excel rows the disk stimuli are
  % located on! For historical reasons, this changes between data sets.
  datarange1 = 8:45; % These are default Excel row numbers for disk stimuli.
  datarange2 = 2:37; % The older row numbers for disk stimuli from RM's thesis.
  
  
  % Import data from these files.
  disp('Loading raw data...')
  dataset1.structure = load([dataset1.folder,dataset1.file],'result');
  dataset2.structure = load([dataset2.folder,dataset2.file],'result');
  dataset1.result = dataset1.structure.result;
  dataset2.result = dataset2.structure.result;
  disp('Processing raw data...')
  
  % Discard unwanted trials, based on the manual assessment in DiscardTrial.xlsx.
  dataset1.result = discardTrial(dataset1.result,dataset1.folder);
  dataset2.result = discardTrial(dataset2.result,dataset2.folder);
  
%   % If applicable, also set manually identified trials from ZeroTrial.xlsx to zero.
%   dataset1.result = zeroTrial(dataset1.result,dataset1.folder);
%   dataset2.result = zeroTrial(dataset2.result,dataset2.folder);

  % Identify each label with real-world stimulus centre coordinates.
  % "datarange...+1" simply takes the Excel header row into account.
  disp('Looking up stimulus locations...')
  stimcentre1 = readStimulus(dataset1.folder,datarange1+1);
  stimcentre2 = readStimulus(dataset2.folder,datarange2+1);
  
  % Convert body-centred to world-centred coordinates.
  stimcentre2 = -stimcentre2;
  
  % If stimuli used for the two data sets were not identical, we need to
  % identify the closest possible matches between them.
  stimulusmatch = matchStimulus(stimcentre1,stimcentre2,convention);
  
  % Pool data across trials and fish (among other options). In field names such
  % as "FR" pr "FER", F refers to pooling across fish, R refers to pooling across 
  % repetitions of stimulus presentations, E across left/right eyes.
  % We will specifically be using "FEstimR" further below. Pooling across eyes 
  % means pooling the gain values of both eyes for the same stimulus - it does not
  % refer to pooling stimuli on opposite sides of the fish. "Estim" as in 
  % "FEstimR" indicates that only gains of directly stimulated eyes are pooled.
  dataset1.gain = poolGainData(dataset1.result(:,:,datarange1,:),stimcentre1,convention);
  dataset2.gain = poolGainData(dataset2.result(:,:,datarange2,:),stimcentre2,convention);
  
  % Select original data sets for the two experimental conditions to be compared.
  % These are called "real" because they were actually observed, whereas
  % subsequent permuted sets were not, and are thus called "fake" for short.
  realset1 = dataset1.gain.mean.FEstimR;
  realset2 = dataset2.gain.mean.FEstimR;
  numstim1 = numel(datarange1);
  numstim2 = numel(datarange2);
    
  % Fit von-Mises-Fisher-like functions to each original data set.
  disp('Computing fits to observed data...')
  realfit1 = bestFit(stimcentre1,realset1,convention);
  realfit2 = bestFit(stimcentre2,realset2,convention);
  disp('Parameters of the best fits to observed data are:')
  disp([realfit1;realfit2])
  disp('Confirm whether these are plausible; debug otherwise.')
  
  % Compute the actual test statistic.
  realstatistic = [mean(realfit1([2 4])), mean(realfit2([2 4]))];
  % This is the mean of the differences in elevation of the peak of the fit on
  % each side (L/R). We take their absolute value so as to be more conservative,
  % since this avoids accidentally averaging out positive/negative differences.
  
  % You may see numerous warning during fitting that matrix ranks are deficient.
  % This is indicative of an unreliable fit - but for randomised data, that is
  % of course what we would expect. I have thus included the following code to
  % suppress these warnings even before their first occurrence. Comment out 
  % these lines to re-activate individual warnings for every occurrence.
  warning('off','MATLAB:rankDeficientMatrix')
  warning('off','stats:nlinfit:ModelConstantWRTParam')
  warning('off','stats:nlinfit:IllConditionedJacobian')
  
  % Pool original data from both sets, then randomly re-order it back into two 
  % subsets. This random assignment is equivalent to a permutation of all labels.
  % Compute the test statistic obtained from each such permutation. If the two
  % original set had different numbers of labels, so will the permuted ones.
  pooledset = [realset1; realset2];
  fakestatistic = NaN(numiteration,2);
  
  try % Use parallel computing if available and properly configured.
    
    parfor k = 1
      % Do nothing. This is just to trigger an error if parfor is unavailable.
      % In this case, execution of the code will resume at "catch" below.
    end
    disp('Computing fits to permuted data using parallel nodes...')
    parfor pariteration = 1:numiteration
      disp(pariteration)
      doswap = logical(randi(2,numstim1,1)-1);
      donotswap = ~doswap;
      fakeset1 = NaN(size(realset1));
      fakeset1(donotswap) = realset1(donotswap);
      fakeset2 = realset2; % For now, at least. Is updated below.
      for k = 1:numstim1
        if doswap(k)
          randommatch = stimulusmatch(k,randi(size(stimulusmatch,2),1,1)); %#ok<PFBNS>
          fakeset1(k) = realset2(randommatch);
          fakeset2(randommatch) = realset1(k);
        end
      end
      fakefit1 = bestFit(stimcentre1,fakeset1,convention); %#ok<PFGV>
      fakefit2 = bestFit(stimcentre2,fakeset2,convention);
      fakestatistic(pariteration) = mean(fakefit2([2 4]));
    end

  catch % Otherwise, just compute each iteration serially.
    
    disp(['There was an issue with using parallel computing on your system. ', ...
          'Computing serially.'])
    for iteration = 1:numiteration
      disp(iteration)
      doswap = logical(randi(2,numstim1,1)-1);
      donotswap = ~doswap;
      fakeset1(donotswap,:) = realset1(donotswap);
      fakeset2 = realset2; % For now, at least. Is updated below.
      for k = 1:numstim1
        if doswap(k)
          randommatch = stimulusmatch(k,randi(size(stimulusmatch,2),1,1));
          fakeset1(k) = realset2(randommatch);
          fakeset2(randommatch) = realset1(k);
        end
      end
      fakefit1 = bestFit(stimcentre1,fakeset1,convention);
      fakefit2 = bestFit(stimcentre2,fakeset2,convention);
      fakestatistic(iteration,:) = [mean(fakefit1([2 4])), mean(fakefit2([2 4]))];
    end
    
  end
    
  % The p value indicates how often results would be more extreme than those
  % actually observed, under the assumption that the null hypothesis is true.
  % To assess the significance of up/down differences between experimental 
  % conditions, we are looking for two joint single-tailed events, one regarding
  % the pseudo-"upright" permuted data, and one for pseudo-"upside down". Please
  % note that coordinates here are fish-centred, so positive elevations are
  % "dorsal", not "upwards"/"away from Earth". All of this could also be
  % phrased with "...==sign(realstatistic(:,N)" instead of comparing to 0.
  regbelowequator = fakestatistic(:,1) < 0;
  regaboveequator = fakestatistic(:,1) >= 0;
  invbelowequator = fakestatistic(:,2) < 0;
  invaboveequator = fakestatistic(:,2) >= 0;
  consistentevent = regaboveequator .* invaboveequator; % H0 = both dorsal
%   consistentevent = regaboveequator .* invaboveequator + ...
%                     regbelowequator .* invbelowequator; % H0 = both same
  pvalue = 1 - sum(consistentevent)/numiteration;
  
  save('permutationResult.mat')
  
  % Clear the sole global variable as a matter of housekeeping.
  clearvars -global convention
  
end



function peaklocation = bestFit(stimcentre,dataset,convention)
    
  % Compute best fit to data.
  resolution = 100;
  [x,y,z] = sphere(resolution);
  [~,fitparam] = vonMisesFisherFit(stimcentre,dataset,x,y,z,convention);
  
  % Collect relevant output. Indices correspond to those of "initialparam".
  leftazimuth    = fitparam(3);
  leftelevation  = fitparam(4);
  rightazimuth   = fitparam(7);
  rightelevation = fitparam(8);
  
  peaklocation = [leftazimuth,leftelevation,rightazimuth,rightelevation];
  
end



function [fitresult,fitparam] = vonMisesFisherFit(stimcentre,response,x,y,z,convention)

  predictor = stimcentre;

  % Initialise fit parameters.
  offset = 0;
  amplitude(1) = 1;
  amplitude(2) = 1;
  meanazimuth(1)   =  70;
  meanelevation(1) =   0;
  meanazimuth(2)   = -70;
  meanelevation(2) =   0;
  concentration(1) = 2.5;
  concentration(2) = 2.5;
  
  if strcmp(convention,'CW')
    meanazimuth = -meanazimuth;
  end

  initialparam = [offset,        ...
                  amplitude(1),     ...
                  meanazimuth(1),   ...
                  meanelevation(1), ...
                  concentration(1), ...
                  amplitude(2),     ...
                  meanazimuth(2),   ...
                  meanelevation(2), ...
                  concentration(2)];

  % Fit.
  option.MaxIter = 1e4;
  datafit.param = nlinfit(predictor,response,@vonmisesfisher9param,initialparam,option);
  datafit.value = vonmisesfisher9param(datafit.param,predictor);

  % Identify centres.
  azimuth(1)   = datafit.param(3);
  elevation(1) = datafit.param(4);
  azimuth(2)   = datafit.param(7);
  elevation(2) = datafit.param(8);
  
  for side = 1:2

    uniqueazimuth(side)    = mod(azimuth(side)+180,360) - 180; %#ok<AGROW>
    uniqueelevation(side)  = mod(elevation(side)+90,180) - 90; %#ok<AGROW>
    datafit.centre(:,side) = [uniqueazimuth(side), uniqueelevation(side)];

  end

  % Compute geographic coordinates of sphere, and its colour data values.
  [azimuth,elevation,radius] = car2geo(x,y,z,convention);
  predictor = [azimuth(:),elevation(:),radius(:)];
  fitresult = reshape(vonmisesfisher9param(datafit.param,predictor),size(radius));
  fitparam = datafit.param;

end



function combinedvalue = vonmisesfisher9param(param,predictor)
%VONMISESFISHER9PARAM Evaluate two overlapping von Mises-Fisher distributions.
%
%   This function is designed in such a way as to be usable as an input argument
%   to the built-in solver nlinfit. Modify it at your own risk! :)
%
%   VALUE = VONMISESFISHER9PARAM computes the value of two(!) overlapping, 
%   spherical (p=3) von Mises-Fisher distributions at the chosen location.
%
%   It is formatted to match the input/output structure required by the
%   built-in non-linear regression function nlinfit.

  global convention

  dimension = 3;
  radius = 1;
  if size(predictor,2) == 2
    predictor(:,3) = radius*ones(size(predictor(:,1)));
  end
  [x,y,z] = geo2car(predictor(:,1),predictor(:,2),predictor(:,3),convention);
  direction = [x,y,z];
  
  numberofdistributions = 2;
  value = NaN(size(predictor,1),numberofdistributions);
  
  for k = 1:numberofdistributions
    
    offset        = param(1);
    amplitude     = param(4*k-2);
    meanazimuth   = param(4*k-1);
    meanelevation = param(4*k);
    concentration = param(4*k+1);
    
    [meanx,meany,meanz] = geo2car(meanazimuth,meanelevation,radius,convention);
    meandirection = [meanx,meany,meanz];

    bessel = concentration / ...
             (2*pi * (exp(concentration) - exp(-concentration)));

    normalisation = concentration^(dimension/2 - 1) / ...
                    ((2*pi)^(dimension/2) * bessel);
                
    value(:,k) = normalisation * exp(concentration*meandirection*direction')';
    value(:,k) = amplitude * value(:,k) + offset;
    
    % Artificially constrain values to improves convergence.
    if abs(meanazimuth) > 180  || ...
       abs(meanelevation) > 90 || ...
       amplitude < 0           || ...
       concentration < 0
     
      value(:,k) = 1e10*ones(size(value(:,k)));
      
    end

  end
  
  combinedvalue = sum(value,2);
  
end



function [x,y,z] = geo2car(azimuth,elevation,radius,convention)
 
  switch convention
    case 'CCW'
      x = radius .* cosd(elevation).*cosd(azimuth);
      y = radius .* cosd(elevation).*sind(azimuth);
      z = radius .* sind(elevation);
    case 'CW'
      x = radius .* cosd(elevation).*cosd(azimuth);
      y = radius .* cosd(elevation).*-sind(azimuth);
      z = radius .* sind(elevation);
    otherwise
      error('Convention not supported.')
  end
  
end



function [azimuth,elevation,radius] = car2geo(x,y,z,convention)
  
  azimuth = atand(y./x) + 180*((x<0).*(y>=0) - (x<0).*(y<0));
        
  switch convention
    case 'CCW'
    case 'CW'
      azimuth = -azimuth;
    otherwise
      error('Convention not supported.')
  end
  
  radius    = sqrt(x.^2+y.^2+z.^2);
  elevation = asind(z./radius);
    
end



function gain = poolGainData(result,varargin)
%POOLGAINDATA converts a raw result structure into plottable data by
%pooling OKR gain data in various ways, and storing those results in a new
%structure called "gain".

  % Pool data in various ways (e.g., .FR = pooled across all [F]ish and
  % all [R]epetitions, but with a separate array for each [E]ye).
  for stimtype = 1:size(result,3)

    gainpool(stimtype).FER = [result(:,:,stimtype,:).gain];

    for eye = 1:size(result,2)
      gainpool(stimtype).FR{eye} = [result(:,eye,stimtype,:).gain];
    end

    for fish = 1:size(result,1)
      gainpool(stimtype).ER{fish} = [result(fish,:,stimtype,:).gain];
    end

    for repetition = 1:size(result,4)
      gainpool(stimtype).FE{repetition} = [result(:,:,stimtype,repetition).gain];
    end

    for fish = 1:size(result,1)
      for eye = 1:size(result,2)
        gainpool(stimtype).R{fish,eye} = [result(fish,eye,stimtype,:).gain];
      end
    end

    
    % The following code pools data across fish and repetitions, and from the
    % directly stimulated eye for each stimulus; data from the non-stimulated eye
    % are discarded.
    if ~isempty(varargin)
      stimcentre = varargin{1};
      convention = varargin{2};
    end

    switch convention
      case 'CCW'
        if stimcentre(stimtype,1) > 0
          stimulatedeye = 1; % 1 = left eye, 2 = right eye
        elseif stimcentre(stimtype,1) < 0
          stimulatedeye = 2;
        else
          stimulatedeye = []; % for stimuli along the prime meridian
        end
      case 'CW'
        if stimcentre(stimtype,1) < 0
          stimulatedeye = 1; % 1 = left eye, 2 = right eye
        elseif stimcentre(stimtype,1) > 0
          stimulatedeye = 2;
        else
          stimulatedeye = []; % for stimuli along the prime meridian
        end      
    end
  
    gainpool(stimtype).FEstimR = [result(:,stimulatedeye,stimtype,:).gain];
    % Generally, the right hand side of the statement above will contain but a
    % scalar. Only for stimuli along the prime meridian will there be a
    % two-element vector, in which case we will take the mean of both elements.
    
  end

  
  % Make sure that in following section, if a stimulus type is not present,
  % the array contain a NaN at that position, instead of a zero.
  nantemplate = NaN(numel(gainpool),1);

  gain.mean  .FER = nantemplate;
  gain.median.FER = nantemplate;
  gain.std   .FER = nantemplate;
  gain.sem   .FER = nantemplate;

  for eye = 1:size(result,2)
    gain.mean  .FR{eye} = nantemplate;
    gain.median.FR{eye} = nantemplate;
    gain.std   .FR{eye} = nantemplate;
    gain.sem   .FR{eye} = nantemplate;
  end

  for fish = 1:size(result,1)
    gain.mean  .ER{fish} = nantemplate;
    gain.median.ER{fish} = nantemplate;
    gain.std   .ER{fish} = nantemplate;
    gain.sem   .ER{fish} = nantemplate;
  end

  for fish = 1:size(result,1)
    for eye = 1:size(result,2)
      gain.mean  .R{fish,eye} = nantemplate;
      gain.median.R{fish,eye} = nantemplate;
      gain.std   .R{fish,eye} = nantemplate;
      gain.sem   .R{fish,eye} = nantemplate;
    end
  end
  
  gain.mean  .FEstimR = nantemplate;
  gain.median.FEstimR = nantemplate;
  gain.std   .FEstimR = nantemplate;
  gain.sem   .FEstimR = nantemplate;


  % mean, median, stdev, sem across all pooled data, one per stimulus type
  for stimtype = 1:numel(gainpool)

    selection = gainpool(stimtype).FER;
    gain.mean  .FER(stimtype) = mean(selection);
    gain.median.FER(stimtype) = median(selection);
    gain.std   .FER(stimtype) = std(selection);
    gain.sem   .FER(stimtype) = std(selection)/sqrt(numel(selection));

    for eye = 1:size(result,2)
      selection = gainpool(stimtype).FR{eye};
      gain.mean  .FR{eye}(stimtype) = mean(selection);
      gain.median.FR{eye}(stimtype) = median(selection);
      gain.std   .FR{eye}(stimtype) = std(selection);
      gain.sem   .FR{eye}(stimtype) = std(selection)/sqrt(numel(selection));
    end

    for fish = 1:size(result,1)
      selection = gainpool(stimtype).ER{fish};
      gain.mean  .ER{fish}(stimtype) = mean(selection);
      gain.median.ER{fish}(stimtype) = median(selection);
      gain.std   .ER{fish}(stimtype) = std(selection);
      gain.sem   .ER{fish}(stimtype) = std(selection)/sqrt(numel(selection));
    end

    for fish = 1:size(result,1)
      for eye = 1:size(result,2)
        selection = gainpool(stimtype).R{fish,eye};
        gain.mean  .R{fish,eye}(stimtype) = mean(selection);
        gain.median.R{fish,eye}(stimtype) = median(selection);
        gain.std   .R{fish,eye}(stimtype) = std(selection);
        gain.sem   .R{fish,eye}(stimtype) = std(selection)/sqrt(numel(selection));
      end
    end
    
    selection = gainpool(stimtype).FEstimR;
    gain.mean  .FEstimR(stimtype) = mean(selection);
    gain.median.FEstimR(stimtype) = median(selection);
    gain.std   .FEstimR(stimtype) = std(selection);
    gain.sem   .FEstimR(stimtype) = std(selection)/sqrt(numel(selection));
    
  end

end



function [retained,original] = discardTrial(original,folderstring)
% DISCARDTRIAL removes data obtained from trials consider "poorly fit" etc.
%
% This function takes a "result" structure as its only input argument, and
% requires an XLSX file containing the indices of trials to be discarded.
% It then removes those bad trials, and returns a reduced structure as its
% primary output argument, "retained". For easy access, it also returns its
% original input.
%
% This function can safely be executed multiple times, as "discarded"
% trials are in fact just set to empty, so their array indices are not 
% reassigned to other trials.
%
% Written by Florian Alexander Dehmelt, Tuebingen University, in 2018.


  % Identify and load an Excel file containing the indices of all unwanted
  % trials to be discarded. This file must contain four columns of numbers
  % (from left to right: fish no., eye no., stimulus type, and repetition.
  % It may contain as many text comments as desired; these will be ignored.
  folder.badtrial = folderstring;
%   folder.badtrial = 'C:\Users\fdehmelt\Dropbox\ThinkPad Dropbox\MATLAB\201808 SphereAnalysis\Sample Data\D series\';
  regularset.badtrial = 'DiscardTrial.xlsx';
  badtrial = xlsread([folder.badtrial,regularset.badtrial]);
  
  % The "retained" structure is the same as the "original" one...
  retained = original;
  
  % ...except that unwanted trials are overwritten by empty values.
  % They thus appear the same way as non-existent trials already did.
  for trial = 1:size(badtrial,1)
    
    % Find the position of the bad trial within the structure.
    fish       = badtrial(trial,1);
    eye        = badtrial(trial,2);
    stimtype   = badtrial(trial,3) + 1; % The +1 converts index >=0 to >=1.
    repetition = badtrial(trial,4);
    
    % Overwrite all field entries with "[]".
    fieldname = fieldnames(retained);
    for field = 1:numel(fieldname)
      retained(fish,eye,stimtype,repetition).(fieldname{field}) = [];
    end
  
  end
  
end



function [altered,original] = zeroTrial(original,folderstring)
% ZEROTRIAL sets gains to zero based on visual inspection of trials.
%
% This function takes a "result" structure as its only input argument, and
% requires an XLSX file containing the indices of trials to be set to zero.
% It then adjusts those gains, and returns an altered structure as its
% primary output argument, "retained". For easy access, it also returns its
% original input.
%
% This function can safely be executed multiple times, as "altered"
% trials are in fact just set to zero, so their array indices are not 
% reassigned to other trials.
%
% Written by Florian Alexander Dehmelt, Tuebingen University, in 2018.


  % Identify and load an Excel file containing the indices of all unwanted
  % trials to be discarded. This file must contain four columns of numbers
  % (from left to right: fish no., eye no., stimulus type, and repetition.
  % It may contain as many text comments as desired; these will be ignored.
  folder.badtrial = folderstring;
%   folder.badtrial = 'C:\Users\fdehmelt\Dropbox\ThinkPad Dropbox\MATLAB\201808 SphereAnalysis\Sample Data\D series\';
  regularset.badtrial = 'ZeroTrial.xlsx';
  badtrial = xlsread([folder.badtrial,regularset.badtrial]);
  
  % The "altered" structure is the same as the "original" one...
  altered = original;
  
  % ...except that gains of visually identified trials are set to zero.
  % They thus appear the same way as non-existent trials already did.
  for trial = 1:size(badtrial,1)
    
    % Find the position of the bad trial within the structure.
    fish       = badtrial(trial,1);
    eye        = badtrial(trial,2);
    stimtype   = badtrial(trial,3) + 1; % The +1 converts index >=0 to >=1.
    repetition = badtrial(trial,4);
    
    % Overwrite the gain field entries with "0", retain all other values.
    altered(fish,eye,stimtype,repetition).gain = 0;
  
  end
  
end




function stimcentre = readStimulus(datafolder,datarange)

  stimulus = 'Stimulus.xlsx';
  [~,tableconvention,~] = xlsread([datafolder,stimulus],'C1:C1');
  azimuth = xlsread([datafolder,stimulus],['C',num2str(datarange(1)),':C',num2str(datarange(end))]);
  elevation = xlsread([datafolder,stimulus],['D',num2str(datarange(1)),':D',num2str(datarange(end))]);

  % Stimulus tables may use CW convention; default code uses CCW.
  switch tableconvention{:}
    case 'azimuth (CCW)'
    case 'azimuth (CW)'
      azimuth = -azimuth;
    otherwise
      error(['Field C1 of Stimulus.xlsx should contain either ' ...
             'string ''azimuth (CCW)'' or string ''azimuth (CW)''.'])
  end

  stimcentre = [azimuth,elevation];

end



function [possiblematch1,possiblematch2] = matchStimulus(stimcentre1,stimcentre2,convention)

  numcandidate = 3; % How many nearest neighbours to list as alternatives.

  numstim1 = size(stimcentre1,1);
  numstim2 = size(stimcentre2,1);

  newazimuth = stimcentre1(:,1);
  newelevation = stimcentre1(:,2);
  newradius = ones(size(newazimuth));
  [x1,y1,z1] = geo2car(newazimuth,newelevation,newradius,convention);
  oldazimuth = stimcentre2(:,1);
  oldelevation = stimcentre2(:,2);
  oldradius = ones(size(oldazimuth));
  [x2,y2,z2] = geo2car(oldazimuth,oldelevation,oldradius,convention);

  euclidean = NaN(numstim1,numstim2);
  for newstimtype = 1:numstim1
    for oldstimtype = 1:numstim2

      % Nearest neighbours by angle are the same as nearest neighbours by
      % Euclidean distance, so choose the latter for robustness.
      euclidean(newstimtype,oldstimtype) = ...
        sqrt(sum([x1(newstimtype)-x2(oldstimtype), ...
                  y1(newstimtype)-y2(oldstimtype), ...
                  z1(newstimtype)-z2(oldstimtype)].^2));

    end
  end

  [~,candidate1] = sort(euclidean,2);
  [~,candidate2] = sort(euclidean,1);
  possiblematch1 = candidate1(:,1:numcandidate);
  possiblematch2 = candidate2(:,1:numcandidate);

end