

###################
# Set environment #
###################

# load libraries and initialize environment
source("set_env.R")


############################################
# Load global sensitivity analysis results #
############################################

# This file is generated by script "1-Model_construction.R".
setwd(results_dir)
load("mc_results_100.RData")


###################
# Testing model 4 #
###################

setwd(results_dir)
pdf(file="Figure 3.pdf", width = 7, height = 9)
par(mfrow=c(4,3))

# Pulse experiment (Enjalbert et al., 2017)
###########################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model.cps")
setwd(results_dir)

# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)
# simulate response to acetate pulse
res_pulse <- runTimeCourse()
# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]

# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
res_nopulse <- runTimeCourse()
# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]

run_sa <- function(){
  # set biomass concentration at which the pulse is performed
  setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)

  # simulate response to acetate pulse
  res_pulse <- runTimeCourse()
  
  # get simulation results
  id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
  id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
  
  t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
  ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
  glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]
  
  # set biomass concentration at which the pulse is performed
  setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
  
  # simulate response to water pulse (control experiment)
  res_nopulse <- runTimeCourse()
  
  # get simulation results
  ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
  glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]
  
  return(list(ace_conc_pulse=ace_conc_pulse, glc_conc_pulse=glc_conc_pulse, ace_conc_nopulse=ace_conc_nopulse, glc_conc_nopulse=glc_conc_nopulse, t_pulse=t_pulse))
}

# create progress bar
pb <- txtProgressBar(min=0, max=ncol(fit_results$res_par)-1, style=3)

# sensitivity analysis
# run simulations for all sets of parameters
res <- array(NA, dim=c(ncol(fit_results$res_par)-1, 4, length(t_pulse)), dimnames=list(iter=NULL, var=c("ace_conc_pulse", "glc_conc_pulse", "ace_conc_nopulse", "glc_conc_nopulse"), time=NULL))
for (i in seq(ncol(fit_results$res_par)-1)){
  rp <- c(fit_results$res_par[,i+1])
  names(rp) <- fit_results$res_par[,"parameter"]
  model <- update_params(getCurrentModel(), rp)
  rsa <- run_sa()
  for (j in names(rsa)){
    if (j != "t_pulse"){
      res[i,j,] <- rsa[[j]]
    }
  }
  # update the progress bar
  setTxtProgressBar(pb, i)
}

# close progress bar
close(pb)

# measurements
# sampling time
time_meas <- seq(0,8)/60
# acetate pulse experiment
glc <- c(0, -0.06931183, -0.151415145, -0.189227994, -0.269451057, -0.290764495, -0.230785281, -0.464084162, -0.551878527)
sd_glc <- c(0, 0.032476344, 0.073133915, 0.113018846, 0.049485284, 0.005325541, 0.163377704, 0.034786419, 0.048477157)
ace <- c(0, -0.027907926, -0.078000853, -0.155334163, -0.165031608, -0.111098424, -0.182877548, -0.237262298, -0.276903255)
sd_ace <- c(0, 0.002740145, 0.025693594, 0.053641876, 0.089975321, 0.005875669, 0.085604161, 0.061930626, 0.099140975)
# control experiment (pulse without acetate)
glc_nopulse <- c(0, -0.195637774, -0.325512845, -0.433785334, -0.628518958, -0.726913548, -0.892908748, -1.071230579, -1.16285575)
sd_glc_nopulse <- c(0, 0.058023617, 0.109115511, 0.047090371, 0.052331837, 0.065619906, 0.133896355, 0.16828754, 0.03515465)
ace_nopulse <- c(0, 0.01012067, 0.05974009, 0.086787283, 0.086690257, 0.104329693, 0.128507087, 0.130026354, 0.187336965)
sd_ace_nopulse <- c(0, 0.005117257, 0.022507218, 0.040319045, 0.037548873, 0.037235285, 0.044494365, 0.045029023, 0.023982374)

# plot simulations vs measurements
plot_with_ci_2(t_pulse, ace_conc_pulse, res[,"ace_conc_pulse",], time_meas, ace, sd_ace, col="#D6685C", h=0, ylim=c(-0.4,0.3), las=1, main="Enjalbert_2017 (pulse 30mM ace)", ylab="change of [Ace]", xlab="time (h)", lwd=1.2)
plot_with_ci_2(t_pulse, glc_conc_pulse, res[,"glc_conc_pulse",], time_meas, glc, sd_glc, col="#70AD47", h=NULL, ylim=c(-1.4,0), las=1, main="Enjalbert_2017 (pulse 30mM ace)", ylab="change of [Glc]", xlab="time (h)", lwd=1.2)
plot_with_ci_2(t_pulse, ace_conc_nopulse, res[,"ace_conc_nopulse",], time_meas, ace_nopulse, sd_ace_nopulse, col="#D6685C", h=0, ylim=c(-0.4,0.3), las=1, main="Enjalbert_2017 (control)", ylab="change of [Ace]", xlab="time (h)", lwd=1.2)
plot_with_ci_2(t_pulse, glc_conc_nopulse, res[,"glc_conc_nopulse",], time_meas, glc_nopulse, sd_glc_nopulse, col="#70AD47", h=NULL, ylim=c(-1.4,0), las=1, main="Enjalbert_2017 (control)", ylab="change of [Glc]", xlab="time (h)", lwd=1.2)

# Chemostat experiment (Renilla et al., 2012)
#############################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model.cps")
setwd(results_dir)

# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)

n_step <- 50
dilution_rates <- seq(0.1, 0.5, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
  
# sensitivity analysis

# create progress bar
pb <- txtProgressBar(min=0, max=ncol(fit_results$res_par)-1, style=3)

# run simulations for all sets of parameters
res <- array(NA, dim=c(ncol(fit_results$res_par)-1, n_step, length(fluxes)+1), dimnames=list(iter=NULL, dilution_rate=NULL, specie=c("dilution_rate", "Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")))
for (i in seq(ncol(fit_results$res_par)-1)){
  rp <- c(fit_results$res_par[,i+1])
  names(rp) <- fit_results$res_par[,"parameter"]
  model <- update_params(getCurrentModel(), rp)
  res_chemostat <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("dilution_rate", fluxes)))
  for (j in seq(n_step)){
    setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rates[j], model=model)
    res_ss <- runSteadyState(model=model)
    res_chemostat[j,] <- c(dilution_rates[j], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
  }
  res[i,,] <- res_chemostat
  # update the progress bar
  setTxtProgressBar(pb, i)
}

# close progress bar
close(pb)

# acetate flux as function of dilution rate
dilution_rate <- c(0.09586056644880175, 0.20043572984749464, 0.2997821350762528, 0.3468409586056645, 0.39912854030501094, 0.44618736383442276, 0.5002178649237474)
q_ace <- c(0.40472342596168076, 0.7396156614846294, 1.477019302736899, 1.2154669794005626, 1.96934635755591, 2.0929143843289824, 2.006569318707304)
q_ace_sd <- c(0.120440687823878, 0.15316333195082, 0.04595662724122, 0.0274214232252601, 0.28885933172199, 0.0777639526513201, 0.28664731148963)

plot_with_ci_3(res, "dilution_rate", "Values[v_ace_net]", "#D6685C", las=1, xlim=c(0,0.6), ylim=c(0,4), main="Renilla_2012", lwd=1.2, xlab="dilution rate (h-1)", ylab="ace flux (mmol/gDW/h)")
plot_points(dilution_rate, q_ace, q_ace_sd, offset=0.01, col="#D6685C")

# acetate flux as function of glc uptake
q_glc <- c(1.43478260869565, 2.7391304347826, 4.59130434782608, 4.69565217391304, 5.7391304347826, 5.92173913043478, 6.10434782608695, 7.12173913043477, 8.34782608695652)
q_ace <- c(0.381358340437624, 0.762906128635029, 1.40509614473808, 1.17836506583309, 1.99602159704461, 2.08290233967983, 2.01292033721701, 2.44717249218528, 2.5848252344416)
q_ace_sd <- c(0.139547219854126, 0.139471440750213, 0.15697641375391, 0.1045751633987, 0.19179691200152, 0.0174291938997801, 0.24397082504499, 0.52295159609737, 0.33130624230368)
q_glc_sd <- c(0.20234760540444, 0.58899555210581, 0.36827270271402, 0.14607356337583, 0.630432119199, 0.13069807968161, 0.47630364392495, 0.99361847036576, 0.66185122193432)

plot_with_ci_3(res, "Values[v_glc_uptake]", "Values[v_ace_net]", col="#D6685C", las=1, xlim=c(0,10), ylim=c(0,4), main="Renilla_2012", lwd=1.2, xlab="glc uptake (mmol/gDW/h)", ylab="ace flux (mmol/gDW/h)")
plot_points(q_glc, q_ace, q_ace_sd, offset=0.2, col="#D6685C")
plot_points(q_glc, q_ace, q_glc_sd, offset=0.08, mode="h", col="#D6685C")

# Steady-state fluxes for acetate concentration between 0.1 and 100mM (Enjalbert et al., 2017; Pinhal et al., 2019)
###################################################################################################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model.cps")
setwd(results_dir)

# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")

# run simulations
n_step <- 50
ace_concentration <- 10**seq(-1, 2, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")

# sensitivity analysis
# run simulations for all sets of parameters
res_e2 <- array(NA, dim=c(ncol(fit_results$res_par)-1, n_step, length(fluxes)+1), dimnames=list(iter=NULL, ace_concentration=NULL, specie=c("ace_concentration", fluxes)))

# create progress bar
pb <- txtProgressBar(min=0, max=ncol(fit_results$res_par)-1, style=3)

for (i in seq(ncol(fit_results$res_par)-1)){
  rp <- c(fit_results$res_par[,i+1])
  names(rp) <- fit_results$res_par[,"parameter"]
  model <- update_params(getCurrentModel(), rp)
  
  res_ace_range <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("ace_concentration", fluxes)))
  for (j in seq(n_step)){
    setSpecies(key="Ace_out", initial_concentration=ace_concentration[j])
    applyInitialState()
    res_ss <- runSteadyState()
    res_ace_range[j,] <- c(ace_concentration[j], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
  }
  res_e2[i,,] <- res_ace_range
  # update the progress bar
  setTxtProgressBar(pb, i)
}

# close progress bar
close(pb)

xlab_main <- c(0.1, 1, 10, 100)
xlab_sec <- c(seq(0.2, 0.9, by=0.1), seq(2, 9, by=1), seq(20, 90, by=10))

# growth rate as function of acetate concentration
growth_rates <- c(0.521128511, 0.611148842625582, 0.613161998174498, 0.502533817, 0.496290415, 0.488201506, 0.547635665, 0.499830448, 0.474554197, 0.425356578, 0.377534684, 0.645724326, 0.618475601, 0.554887936, 0.564811523, 0.527571192, 0.434972836, 0.3824734, 0.583623355, 0.620905534, 0.564259247, 0.532148135, 0.483885215, 0.557074418, 0.630654409249223)
sd_growth_rates <- c(0.001793104, 0.00204807928657914, 0.00219396182484705, 0.001709207, 0.001846205, 0.001757403, 0.001821375, 0.001025702, 0.001940912, 0.001204707, 0.001999188, 0.001418374, 0.001932601, 0.001455791, 0.001574234, 0.001206265, 0.001292476, 0.001068259, 0.001804648, 0.001643459, 0.001598405, 0.001121218, 0.000912408, 0.00194896, 0.00203369597574686)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)

plot_with_ci_3(res_e2, "ace_concentration", "Values[v_growth_rate]", col="#2E75B6", xaxt='n', las=1, lwd=1.2, xlim=c(0.1,100), ylim=c(0,0.8), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="growth rate (h-1)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, growth_rates, sd_growth_rates, offset=0.01, col="#2E75B6")
plot_points(ace_conc, growth_rates, sd_ace_conc, offset=0.02, mode="h", col="#2E75B6")

# glc uptake as function of acetate concentration
glc_upt <- c(8.654860167, 8.36127542981722, 7.98010111285252, 9.236935826, 8.274418986, 7.560431219, 7.339194455, 5.775312502, 6.423391263, 5.1544758, 3.938631573, 8.115447647, 9.28067031, 6.737153424, 7.172748804, 5.884186033, 5.684201497, 4.811576974, 9.632702365, 8.055042777, 9.708342814, 7.100081588, 5.505759496, 9.242859752, 8.18621623190759)
sd_glc_upt <- c(0.337812425, 0.38531328268303, 0.373770045721031, 0.356787032, 0.334672954, 0.317509322, 0.288025925, 0.16053276, 0.375934255, 0.293148172, 0.359225607, 0.197331684, 0.360984112, 0.229372278, 0.241820396, 0.20450532, 0.260869273, 0.216134352, 0.34289286, 0.350305744, 0.293144783, 0.220135755, 0.153471508, 0.25245346, 0.396815184905029)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)

plot_with_ci_3(res_e2, "ace_concentration", "Values[v_glc_uptake]", col="#70AD47", xaxt='n', las=1, lwd=1.2, xlim=c(0.1,100), ylim=c(0,10), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="glc uptake (mmol/gDW/h)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, glc_upt, sd_glc_upt, offset=0.02, col="#70AD47")
plot_points(ace_conc, glc_upt, sd_ace_conc, offset=0.2, mode="h", col="#70AD47")

# ace flux as function of acetate concentration
ace_flx <- c(3.5, -2.7, 1.516999356, 1.26845123082679, 0.775821380507016, 0.678877137, 0.017366464, -0.991478151, -1.286687213, -2.078474994, -1.530841439, -1.525342269, -1.253581266, 1.984679487, 0.546462624, -0.136780389, -0.393883917, -0.610240984, -1.120767885, -1.277455315, 2.574285211, 2.051935093, 1.828415596, -1.262442483, -1.317987733, 2.333568565, 1.85234639824858)
sd_ace_flx <- c(0.35, 0.27, 0.316118066, 0.388752117258161, 0.40715278851436, 0.33357638, 0.37333751, 0.347029894, 0.280501612, 0.195031303, 0.376252463, 0.226182385, 0.303661317, 0.253610517, 0.385450715, 0.243880325, 0.30665695, 0.257983739, 0.23844407, 0.198458448, 0.299832036, 0.334956504, 0.317134334, 0.263807154, 0.195219648, 0.016120887, 0.386174129654754)
ace_conc <- c(0.2, 100, 0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.05, 10, 0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot_with_ci_3(res_e2, "ace_concentration", "Values[v_ace_net]", col="#D6685C", xaxt='n', las=1, lwd=1.2, xlim=c(0.1,100), ylim=c(-4,4), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="ace flux (mmol/gDW/h)")
abline(h=0)
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, ace_flx, sd_ace_flx, offset=0.04, col="#D6685C")
plot_points(ace_conc, ace_flx, sd_ace_conc, offset=0.2, mode="h", col="#D6685C")

dev.off()


##############################
# Testing alternative models #
##############################

###########
# Model 1 #
###########

setwd(results_dir)
pdf(file="Figure 3-figure supplement 1.pdf", width = 7, height = 9)
par(mfrow=c(4,3))

# Pulse experiment (Enjalbert et al., 2017)
###########################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")

# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)

# simulate response to acetate pulse
res_pulse <- runTimeCourse()

# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1

t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]

# measurements
time_meas <- seq(0,8)/60
glc <- c(0, -0.06931183, -0.151415145, -0.189227994, -0.269451057, -0.290764495, -0.230785281, -0.464084162, -0.551878527)
sd_glc <- c(0, 0.032476344, 0.073133915, 0.113018846, 0.049485284, 0.005325541, 0.163377704, 0.034786419, 0.048477157)
ace <- c(0, -0.027907926, -0.078000853, -0.155334163, -0.165031608, -0.111098424, -0.182877548, -0.237262298, -0.276903255)
sd_ace <- c(0, 0.002740145, 0.025693594, 0.053641876, 0.089975321, 0.005875669, 0.085604161, 0.061930626, 0.099140975)

# plot simulations vs measurements
plot(t_pulse, ace_conc_pulse, type="l", ylim=c(-0.4,0), las=1, main="Enjalbert_2017 (pulse 30mM ace)", ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace, sd_ace, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_pulse, type="l", main="Enjalbert_2017 (pulse 30mM ace)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc, sd_glc, offset=0.002, col="#70AD47")

# simulate control experiment (i.e. no acetate pulse)
deleteEvent(getEvents()$key)

res_nopulse <- runTimeCourse()

# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]

# measurements
glc_nopulse <- c(0, -0.195637774, -0.325512845, -0.433785334, -0.628518958, -0.726913548, -0.892908748, -1.071230579, -1.16285575)
sd_glc_nopulse <- c(0, 0.058023617, 0.109115511, 0.047090371, 0.052331837, 0.065619906, 0.133896355, 0.16828754, 0.03515465)
ace_nopulse <- c(0, 0.01012067, 0.05974009, 0.086787283, 0.086690257, 0.104329693, 0.128507087, 0.130026354, 0.187336965)
sd_ace_nopulse <- c(0, 0.005117257, 0.022507218, 0.040319045, 0.037548873, 0.037235285, 0.044494365, 0.045029023, 0.023982374)

# plot simulations vs measurements
plot(t_pulse, ace_conc_nopulse, type="l", ylim=c(0,0.3), main="Enjalbert_2017 (control)", las=1, ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace_nopulse, sd_ace_nopulse, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_nopulse, type="l", ylim=c(-1.4,0), main="Enjalbert_2017 (control)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc_nopulse, sd_glc_nopulse, offset=0.002, col="#70AD47")

# Chemostat experiment (Renilla et al., 2012)
#############################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")

# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)

n_step <- 50
dilution_rates <- seq(0.1, 0.5, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_chemostat <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("dilution_rate", fluxes)))
for (i in seq(n_step)){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rates[i])
  res_ss <- runSteadyState()
  res_chemostat[i,] <- c(dilution_rates[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

# acetate flux as function of dilution rate
dilution_rate <- c(0.09586056644880175, 0.20043572984749464, 0.2997821350762528, 0.3468409586056645, 0.39912854030501094, 0.44618736383442276, 0.5002178649237474)
q_ace <- c(0.40472342596168076, 0.7396156614846294, 1.477019302736899, 1.2154669794005626, 1.96934635755591, 2.0929143843289824, 2.006569318707304)
q_ace_sd <- c(0.120440687823878, 0.15316333195082, 0.04595662724122, 0.0274214232252601, 0.28885933172199, 0.0777639526513201, 0.28664731148963)

plot(res_chemostat[,"dilution_rate"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,0.6), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="dilution rate (h-1)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(dilution_rate, q_ace, q_ace_sd, offset=0.01, col="#D6685C")

# acetate flux as function of glc uptake
q_glc <- c(1.43478260869565, 2.7391304347826, 4.59130434782608, 4.69565217391304, 5.7391304347826, 5.92173913043478, 6.10434782608695, 7.12173913043477, 8.34782608695652)
q_ace <- c(0.381358340437624, 0.762906128635029, 1.40509614473808, 1.17836506583309, 1.99602159704461, 2.08290233967983, 2.01292033721701, 2.44717249218528, 2.5848252344416)
q_ace_sd <- c(0.139547219854126, 0.139471440750213, 0.15697641375391, 0.1045751633987, 0.19179691200152, 0.0174291938997801, 0.24397082504499, 0.52295159609737, 0.33130624230368)
q_glc_sd <- c(0.20234760540444, 0.58899555210581, 0.36827270271402, 0.14607356337583, 0.630432119199, 0.13069807968161, 0.47630364392495, 0.99361847036576, 0.66185122193432)

plot(res_chemostat[,"Values[v_glc_uptake]"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,10), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="glc uptake (mmol/gDW/h)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(q_glc, q_ace, q_ace_sd, offset=0.2, col="#D6685C")
plot_points(q_glc, q_ace, q_glc_sd, offset=0.08, mode="h", col="#D6685C")

# Steady-state fluxes for acetate concentration between 0.1 and 100mM (Enjalbert et al., 2017; Pinhal et al., 2019)
##############################################################################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")

# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")

# run simulations
n_step <- 50
ace_concentration <- 10**seq(-1, 2, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_ace_range <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("ace_concentration", fluxes)))
for (i in seq(n_step)){
  setSpecies(key="Ace_out", initial_concentration=ace_concentration[i])
  applyInitialState()
  res_ss <- runSteadyState()
  res_ace_range[i,] <- c(ace_concentration[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

xlab_main <- c(0.1, 1, 10, 100)
xlab_sec <- c(seq(0.2, 0.9, by=0.1), seq(2, 9, by=1), seq(20, 90, by=10))

# growth rate as function of acetate concentration
growth_rates <- c(0.521128511, 0.611148842625582, 0.613161998174498, 0.502533817, 0.496290415, 0.488201506, 0.547635665, 0.499830448, 0.474554197, 0.425356578, 0.377534684, 0.645724326, 0.618475601, 0.554887936, 0.564811523, 0.527571192, 0.434972836, 0.3824734, 0.583623355, 0.620905534, 0.564259247, 0.532148135, 0.483885215, 0.557074418, 0.630654409249223)
sd_growth_rates <- c(0.001793104, 0.00204807928657914, 0.00219396182484705, 0.001709207, 0.001846205, 0.001757403, 0.001821375, 0.001025702, 0.001940912, 0.001204707, 0.001999188, 0.001418374, 0.001932601, 0.001455791, 0.001574234, 0.001206265, 0.001292476, 0.001068259, 0.001804648, 0.001643459, 0.001598405, 0.001121218, 0.000912408, 0.00194896, 0.00203369597574686)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], xaxt='n', res_ace_range[,"Values[v_growth_rate]"], las=1, col="#2E75B6", lwd=2, type="l", xlim=c(0.1,100), ylim=c(0,0.8), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="growth rate (h-1)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, growth_rates, sd_growth_rates, offset=0.01, col="#2E75B6")
plot_points(ace_conc, growth_rates, sd_ace_conc, offset=0.02, mode="h", col="#2E75B6")

# glc uptake as function of acetate concentration
glc_upt <- c(8.654860167, 8.36127542981722, 7.98010111285252, 9.236935826, 8.274418986, 7.560431219, 7.339194455, 5.775312502, 6.423391263, 5.1544758, 3.938631573, 8.115447647, 9.28067031, 6.737153424, 7.172748804, 5.884186033, 5.684201497, 4.811576974, 9.632702365, 8.055042777, 9.708342814, 7.100081588, 5.505759496, 9.242859752, 8.18621623190759)
sd_glc_upt <- c(0.337812425, 0.38531328268303, 0.373770045721031, 0.356787032, 0.334672954, 0.317509322, 0.288025925, 0.16053276, 0.375934255, 0.293148172, 0.359225607, 0.197331684, 0.360984112, 0.229372278, 0.241820396, 0.20450532, 0.260869273, 0.216134352, 0.34289286, 0.350305744, 0.293144783, 0.220135755, 0.153471508, 0.25245346, 0.396815184905029)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_glc_uptake]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), col="#70AD47", ylim=c(0,10), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="glc uptake (mmol/gDW/h)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, glc_upt, sd_glc_upt, offset=0.02, col="#70AD47")
plot_points(ace_conc, glc_upt, sd_ace_conc, offset=0.2, mode="h", col="#70AD47")

# ace flux as function of acetate concentration
ace_flx <- c(3.5, -2.7, 1.516999356, 1.26845123082679, 0.775821380507016, 0.678877137, 0.017366464, -0.991478151, -1.286687213, -2.078474994, -1.530841439, -1.525342269, -1.253581266, 1.984679487, 0.546462624, -0.136780389, -0.393883917, -0.610240984, -1.120767885, -1.277455315, 2.574285211, 2.051935093, 1.828415596, -1.262442483, -1.317987733, 2.333568565, 1.85234639824858)
sd_ace_flx <- c(0.35, 0.27, 0.316118066, 0.388752117258161, 0.40715278851436, 0.33357638, 0.37333751, 0.347029894, 0.280501612, 0.195031303, 0.376252463, 0.226182385, 0.303661317, 0.253610517, 0.385450715, 0.243880325, 0.30665695, 0.257983739, 0.23844407, 0.198458448, 0.299832036, 0.334956504, 0.317134334, 0.263807154, 0.195219648, 0.016120887, 0.386174129654754)
ace_conc <- c(0.2, 100, 0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.05, 10, 0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_ace_net]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), ylim=c(-4,4), log="x", main="Enjalbert_2017, Pinhal_2019", col="#D6685C", xlab="[acetate] (mM)", ylab="ace flux (mmol/gDW/h)")
abline(h=0)
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, ace_flx, sd_ace_flx, offset=0.04, col="#D6685C")
plot_points(ace_conc, ace_flx, sd_ace_conc, offset=0.2, mode="h", col="#D6685C")

dev.off()

###########
# Model 2 #
###########

setwd(results_dir)
pdf(file="Figure 3-figure supplement 2.pdf", width = 7, height = 9)
par(mfrow=c(4,3))

# Pulse experiment (Enjalbert et al., 2017)
###########################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")

# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)

# simulate response to acetate pulse
res_pulse <- runTimeCourse()

# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1

t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]

# measurements
time_meas <- seq(0,8)/60
glc <- c(0, -0.06931183, -0.151415145, -0.189227994, -0.269451057, -0.290764495, -0.230785281, -0.464084162, -0.551878527)
sd_glc <- c(0, 0.032476344, 0.073133915, 0.113018846, 0.049485284, 0.005325541, 0.163377704, 0.034786419, 0.048477157)
ace <- c(0, -0.027907926, -0.078000853, -0.155334163, -0.165031608, -0.111098424, -0.182877548, -0.237262298, -0.276903255)
sd_ace <- c(0, 0.002740145, 0.025693594, 0.053641876, 0.089975321, 0.005875669, 0.085604161, 0.061930626, 0.099140975)

# plot simulations vs measurements
plot(t_pulse, ace_conc_pulse, type="l", ylim=c(-0.4,0), las=1, main="Enjalbert_2017 (pulse 30mM ace)", ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace, sd_ace, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_pulse, type="l", main="Enjalbert_2017 (pulse 30mM ace)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc, sd_glc, offset=0.002, col="#70AD47")

# simulate control experiment (i.e. no acetate pulse)
deleteEvent(getEvents()$key)

res_nopulse <- runTimeCourse()

# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]

# measurements
glc_nopulse <- c(0, -0.195637774, -0.325512845, -0.433785334, -0.628518958, -0.726913548, -0.892908748, -1.071230579, -1.16285575)
sd_glc_nopulse <- c(0, 0.058023617, 0.109115511, 0.047090371, 0.052331837, 0.065619906, 0.133896355, 0.16828754, 0.03515465)
ace_nopulse <- c(0, 0.01012067, 0.05974009, 0.086787283, 0.086690257, 0.104329693, 0.128507087, 0.130026354, 0.187336965)
sd_ace_nopulse <- c(0, 0.005117257, 0.022507218, 0.040319045, 0.037548873, 0.037235285, 0.044494365, 0.045029023, 0.023982374)

# plot simulations vs measurements
plot(t_pulse, ace_conc_nopulse, type="l", ylim=c(0,0.3), main="Enjalbert_2017 (control)", las=1, ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace_nopulse, sd_ace_nopulse, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_nopulse, type="l", ylim=c(-1.4,0), main="Enjalbert_2017 (control)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc_nopulse, sd_glc_nopulse, offset=0.002, col="#70AD47")

# Chemostat experiment (Renilla et al., 2012)
#############################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")

# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)

n_step <- 50
dilution_rates <- seq(0.1, 0.5, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_chemostat <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("dilution_rate", fluxes)))
for (i in seq(n_step)){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rates[i])
  res_ss <- runSteadyState()
  res_chemostat[i,] <- c(dilution_rates[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

# acetate flux as function of dilution rate
dilution_rate <- c(0.09586056644880175, 0.20043572984749464, 0.2997821350762528, 0.3468409586056645, 0.39912854030501094, 0.44618736383442276, 0.5002178649237474)
q_ace <- c(0.40472342596168076, 0.7396156614846294, 1.477019302736899, 1.2154669794005626, 1.96934635755591, 2.0929143843289824, 2.006569318707304)
q_ace_sd <- c(0.120440687823878, 0.15316333195082, 0.04595662724122, 0.0274214232252601, 0.28885933172199, 0.0777639526513201, 0.28664731148963)

plot(res_chemostat[,"dilution_rate"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,0.6), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="dilution rate (h-1)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(dilution_rate, q_ace, q_ace_sd, offset=0.01, col="#D6685C")

# acetate flux as function of glc uptake
q_glc <- c(1.43478260869565, 2.7391304347826, 4.59130434782608, 4.69565217391304, 5.7391304347826, 5.92173913043478, 6.10434782608695, 7.12173913043477, 8.34782608695652)
q_ace <- c(0.381358340437624, 0.762906128635029, 1.40509614473808, 1.17836506583309, 1.99602159704461, 2.08290233967983, 2.01292033721701, 2.44717249218528, 2.5848252344416)
q_ace_sd <- c(0.139547219854126, 0.139471440750213, 0.15697641375391, 0.1045751633987, 0.19179691200152, 0.0174291938997801, 0.24397082504499, 0.52295159609737, 0.33130624230368)
q_glc_sd <- c(0.20234760540444, 0.58899555210581, 0.36827270271402, 0.14607356337583, 0.630432119199, 0.13069807968161, 0.47630364392495, 0.99361847036576, 0.66185122193432)

plot(res_chemostat[,"Values[v_glc_uptake]"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,10), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="glc uptake (mmol/gDW/h)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(q_glc, q_ace, q_ace_sd, offset=0.2, col="#D6685C")
plot_points(q_glc, q_ace, q_glc_sd, offset=0.08, mode="h", col="#D6685C")

# Steady-state fluxes for acetate concentration between 0.1 and 100mM (Enjalbert et al., 2017; Pinhal et al., 2019)
##############################################################################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")

# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")

# run simulations
n_step <- 50
ace_concentration <- 10**seq(-1, 2, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_ace_range <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("ace_concentration", fluxes)))
for (i in seq(n_step)){
  setSpecies(key="Ace_out", initial_concentration=ace_concentration[i])
  applyInitialState()
  res_ss <- runSteadyState()
  res_ace_range[i,] <- c(ace_concentration[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

xlab_main <- c(0.1, 1, 10, 100)
xlab_sec <- c(seq(0.2, 0.9, by=0.1), seq(2, 9, by=1), seq(20, 90, by=10))

# growth rate as function of acetate concentration
growth_rates <- c(0.521128511, 0.611148842625582, 0.613161998174498, 0.502533817, 0.496290415, 0.488201506, 0.547635665, 0.499830448, 0.474554197, 0.425356578, 0.377534684, 0.645724326, 0.618475601, 0.554887936, 0.564811523, 0.527571192, 0.434972836, 0.3824734, 0.583623355, 0.620905534, 0.564259247, 0.532148135, 0.483885215, 0.557074418, 0.630654409249223)
sd_growth_rates <- c(0.001793104, 0.00204807928657914, 0.00219396182484705, 0.001709207, 0.001846205, 0.001757403, 0.001821375, 0.001025702, 0.001940912, 0.001204707, 0.001999188, 0.001418374, 0.001932601, 0.001455791, 0.001574234, 0.001206265, 0.001292476, 0.001068259, 0.001804648, 0.001643459, 0.001598405, 0.001121218, 0.000912408, 0.00194896, 0.00203369597574686)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], xaxt='n', res_ace_range[,"Values[v_growth_rate]"], las=1, col="#2E75B6", lwd=2, type="l", xlim=c(0.1,100), ylim=c(0,0.8), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="growth rate (h-1)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, growth_rates, sd_growth_rates, offset=0.01, col="#2E75B6")
plot_points(ace_conc, growth_rates, sd_ace_conc, offset=0.02, mode="h", col="#2E75B6")

# glc uptake as function of acetate concentration
glc_upt <- c(8.654860167, 8.36127542981722, 7.98010111285252, 9.236935826, 8.274418986, 7.560431219, 7.339194455, 5.775312502, 6.423391263, 5.1544758, 3.938631573, 8.115447647, 9.28067031, 6.737153424, 7.172748804, 5.884186033, 5.684201497, 4.811576974, 9.632702365, 8.055042777, 9.708342814, 7.100081588, 5.505759496, 9.242859752, 8.18621623190759)
sd_glc_upt <- c(0.337812425, 0.38531328268303, 0.373770045721031, 0.356787032, 0.334672954, 0.317509322, 0.288025925, 0.16053276, 0.375934255, 0.293148172, 0.359225607, 0.197331684, 0.360984112, 0.229372278, 0.241820396, 0.20450532, 0.260869273, 0.216134352, 0.34289286, 0.350305744, 0.293144783, 0.220135755, 0.153471508, 0.25245346, 0.396815184905029)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_glc_uptake]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), col="#70AD47", ylim=c(0,10), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="glc uptake (mmol/gDW/h)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, glc_upt, sd_glc_upt, offset=0.02, col="#70AD47")
plot_points(ace_conc, glc_upt, sd_ace_conc, offset=0.2, mode="h", col="#70AD47")

# ace flux as function of acetate concentration
ace_flx <- c(3.5, -2.7, 1.516999356, 1.26845123082679, 0.775821380507016, 0.678877137, 0.017366464, -0.991478151, -1.286687213, -2.078474994, -1.530841439, -1.525342269, -1.253581266, 1.984679487, 0.546462624, -0.136780389, -0.393883917, -0.610240984, -1.120767885, -1.277455315, 2.574285211, 2.051935093, 1.828415596, -1.262442483, -1.317987733, 2.333568565, 1.85234639824858)
sd_ace_flx <- c(0.35, 0.27, 0.316118066, 0.388752117258161, 0.40715278851436, 0.33357638, 0.37333751, 0.347029894, 0.280501612, 0.195031303, 0.376252463, 0.226182385, 0.303661317, 0.253610517, 0.385450715, 0.243880325, 0.30665695, 0.257983739, 0.23844407, 0.198458448, 0.299832036, 0.334956504, 0.317134334, 0.263807154, 0.195219648, 0.016120887, 0.386174129654754)
ace_conc <- c(0.2, 100, 0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.05, 10, 0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_ace_net]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), ylim=c(-4,4), log="x", main="Enjalbert_2017, Pinhal_2019", col="#D6685C", xlab="[acetate] (mM)", ylab="ace flux (mmol/gDW/h)")
abline(h=0)
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, ace_flx, sd_ace_flx, offset=0.04, col="#D6685C")
plot_points(ace_conc, ace_flx, sd_ace_conc, offset=0.2, mode="h", col="#D6685C")

dev.off()

###########
# Model 3 #
###########

setwd(results_dir)
pdf(file="Figure 3-figure supplement 3.pdf", width = 7, height = 9)
par(mfrow=c(4,3))

# Pulse experiment (Enjalbert et al., 2017)
###########################################

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")

# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)

# simulate response to acetate pulse
res_pulse <- runTimeCourse()

# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1

t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]

# measurements
time_meas <- seq(0,8)/60
glc <- c(0, -0.06931183, -0.151415145, -0.189227994, -0.269451057, -0.290764495, -0.230785281, -0.464084162, -0.551878527)
sd_glc <- c(0, 0.032476344, 0.073133915, 0.113018846, 0.049485284, 0.005325541, 0.163377704, 0.034786419, 0.048477157)
ace <- c(0, -0.027907926, -0.078000853, -0.155334163, -0.165031608, -0.111098424, -0.182877548, -0.237262298, -0.276903255)
sd_ace <- c(0, 0.002740145, 0.025693594, 0.053641876, 0.089975321, 0.005875669, 0.085604161, 0.061930626, 0.099140975)

# plot simulations vs measurements
plot(t_pulse, ace_conc_pulse, type="l", ylim=c(-0.4,0), las=1, main="Enjalbert_2017 (pulse 30mM ace)", ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace, sd_ace, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_pulse, type="l", main="Enjalbert_2017 (pulse 30mM ace)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc, sd_glc, offset=0.002, col="#70AD47")

# simulate control experiment (i.e. no acetate pulse)
deleteEvent(getEvents()$key)

res_nopulse <- runTimeCourse()

# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]

# measurements
glc_nopulse <- c(0, -0.195637774, -0.325512845, -0.433785334, -0.628518958, -0.726913548, -0.892908748, -1.071230579, -1.16285575)
sd_glc_nopulse <- c(0, 0.058023617, 0.109115511, 0.047090371, 0.052331837, 0.065619906, 0.133896355, 0.16828754, 0.03515465)
ace_nopulse <- c(0, 0.01012067, 0.05974009, 0.086787283, 0.086690257, 0.104329693, 0.128507087, 0.130026354, 0.187336965)
sd_ace_nopulse <- c(0, 0.005117257, 0.022507218, 0.040319045, 0.037548873, 0.037235285, 0.044494365, 0.045029023, 0.023982374)

# plot simulations vs measurements
plot(t_pulse, ace_conc_nopulse, type="l", ylim=c(0,0.3), main="Enjalbert_2017 (control)", las=1, ylab="change of [Ace]", xlab="time (h)", col="#D6685C", lwd=2)
plot_points(time_meas, ace_nopulse, sd_ace_nopulse, offset=0.002, col="#D6685C")
plot(t_pulse, glc_conc_nopulse, type="l", ylim=c(-1.4,0), main="Enjalbert_2017 (control)", las=1, ylab="change of [Glc]", xlab="time (h)", col="#70AD47", lwd=2)
plot_points(time_meas, glc_nopulse, sd_glc_nopulse, offset=0.002, col="#70AD47")

# Chemostat experiment (Renilla et al., 2012)
#############################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")

# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)

n_step <- 50
dilution_rates <- seq(0.1, 0.5, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_chemostat <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("dilution_rate", fluxes)))
for (i in seq(n_step)){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rates[i])
  res_ss <- runSteadyState()
  res_chemostat[i,] <- c(dilution_rates[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

# acetate flux as function of dilution rate
dilution_rate <- c(0.09586056644880175, 0.20043572984749464, 0.2997821350762528, 0.3468409586056645, 0.39912854030501094, 0.44618736383442276, 0.5002178649237474)
q_ace <- c(0.40472342596168076, 0.7396156614846294, 1.477019302736899, 1.2154669794005626, 1.96934635755591, 2.0929143843289824, 2.006569318707304)
q_ace_sd <- c(0.120440687823878, 0.15316333195082, 0.04595662724122, 0.0274214232252601, 0.28885933172199, 0.0777639526513201, 0.28664731148963)

plot(res_chemostat[,"dilution_rate"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,0.6), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="dilution rate (h-1)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(dilution_rate, q_ace, q_ace_sd, offset=0.01, col="#D6685C")

# acetate flux as function of glc uptake
q_glc <- c(1.43478260869565, 2.7391304347826, 4.59130434782608, 4.69565217391304, 5.7391304347826, 5.92173913043478, 6.10434782608695, 7.12173913043477, 8.34782608695652)
q_ace <- c(0.381358340437624, 0.762906128635029, 1.40509614473808, 1.17836506583309, 1.99602159704461, 2.08290233967983, 2.01292033721701, 2.44717249218528, 2.5848252344416)
q_ace_sd <- c(0.139547219854126, 0.139471440750213, 0.15697641375391, 0.1045751633987, 0.19179691200152, 0.0174291938997801, 0.24397082504499, 0.52295159609737, 0.33130624230368)
q_glc_sd <- c(0.20234760540444, 0.58899555210581, 0.36827270271402, 0.14607356337583, 0.630432119199, 0.13069807968161, 0.47630364392495, 0.99361847036576, 0.66185122193432)

plot(res_chemostat[,"Values[v_glc_uptake]"], res_chemostat[,"Values[v_ace_net]"], type="l", las=1, xlim=c(0,10), ylim=c(0,4), main="Renilla_2012", lwd=2, xlab="glc uptake (mmol/gDW/h)", ylab="ace flux (mmol/gDW/h)", col="#D6685C")
plot_points(q_glc, q_ace, q_ace_sd, offset=0.2, col="#D6685C")
plot_points(q_glc, q_ace, q_glc_sd, offset=0.08, mode="h", col="#D6685C")

# Steady-state fluxes for acetate concentration between 0.1 and 100mM (Enjalbert et al., 2017; Pinhal et al., 2019)
##############################################################################################

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")

# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")

# run simulations
n_step <- 50
ace_concentration <- 10**seq(-1, 2, length.out = n_step)
fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")
res_ace_range <- matrix(NA, nrow=n_step, ncol=length(fluxes)+1, dimnames=list(r=NULL, c=c("ace_concentration", fluxes)))
for (i in seq(n_step)){
  setSpecies(key="Ace_out", initial_concentration=ace_concentration[i])
  applyInitialState()
  res_ss <- runSteadyState()
  res_ace_range[i,] <- c(ace_concentration[i], unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"]))
}

xlab_main <- c(0.1, 1, 10, 100)
xlab_sec <- c(seq(0.2, 0.9, by=0.1), seq(2, 9, by=1), seq(20, 90, by=10))

# growth rate as function of acetate concentration
growth_rates <- c(0.521128511, 0.611148842625582, 0.613161998174498, 0.502533817, 0.496290415, 0.488201506, 0.547635665, 0.499830448, 0.474554197, 0.425356578, 0.377534684, 0.645724326, 0.618475601, 0.554887936, 0.564811523, 0.527571192, 0.434972836, 0.3824734, 0.583623355, 0.620905534, 0.564259247, 0.532148135, 0.483885215, 0.557074418, 0.630654409249223)
sd_growth_rates <- c(0.001793104, 0.00204807928657914, 0.00219396182484705, 0.001709207, 0.001846205, 0.001757403, 0.001821375, 0.001025702, 0.001940912, 0.001204707, 0.001999188, 0.001418374, 0.001932601, 0.001455791, 0.001574234, 0.001206265, 0.001292476, 0.001068259, 0.001804648, 0.001643459, 0.001598405, 0.001121218, 0.000912408, 0.00194896, 0.00203369597574686)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], xaxt='n', res_ace_range[,"Values[v_growth_rate]"], las=1, col="#2E75B6", lwd=2, type="l", xlim=c(0.1,100), ylim=c(0,0.8), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="growth rate (h-1)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, growth_rates, sd_growth_rates, offset=0.01, col="#2E75B6")
plot_points(ace_conc, growth_rates, sd_ace_conc, offset=0.02, mode="h", col="#2E75B6")

# glc uptake as function of acetate concentration
glc_upt <- c(8.654860167, 8.36127542981722, 7.98010111285252, 9.236935826, 8.274418986, 7.560431219, 7.339194455, 5.775312502, 6.423391263, 5.1544758, 3.938631573, 8.115447647, 9.28067031, 6.737153424, 7.172748804, 5.884186033, 5.684201497, 4.811576974, 9.632702365, 8.055042777, 9.708342814, 7.100081588, 5.505759496, 9.242859752, 8.18621623190759)
sd_glc_upt <- c(0.337812425, 0.38531328268303, 0.373770045721031, 0.356787032, 0.334672954, 0.317509322, 0.288025925, 0.16053276, 0.375934255, 0.293148172, 0.359225607, 0.197331684, 0.360984112, 0.229372278, 0.241820396, 0.20450532, 0.260869273, 0.216134352, 0.34289286, 0.350305744, 0.293144783, 0.220135755, 0.153471508, 0.25245346, 0.396815184905029)
ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_glc_uptake]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), col="#70AD47", ylim=c(0,10), log="x", main="Enjalbert_2017", xlab="[acetate] (mM)", ylab="glc uptake (mmol/gDW/h)")
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, glc_upt, sd_glc_upt, offset=0.02, col="#70AD47")
plot_points(ace_conc, glc_upt, sd_ace_conc, offset=0.2, mode="h", col="#70AD47")

# ace flux as function of acetate concentration
ace_flx <- c(3.5, -2.7, 1.516999356, 1.26845123082679, 0.775821380507016, 0.678877137, 0.017366464, -0.991478151, -1.286687213, -2.078474994, -1.530841439, -1.525342269, -1.253581266, 1.984679487, 0.546462624, -0.136780389, -0.393883917, -0.610240984, -1.120767885, -1.277455315, 2.574285211, 2.051935093, 1.828415596, -1.262442483, -1.317987733, 2.333568565, 1.85234639824858)
sd_ace_flx <- c(0.35, 0.27, 0.316118066, 0.388752117258161, 0.40715278851436, 0.33357638, 0.37333751, 0.347029894, 0.280501612, 0.195031303, 0.376252463, 0.226182385, 0.303661317, 0.253610517, 0.385450715, 0.243880325, 0.30665695, 0.257983739, 0.23844407, 0.198458448, 0.299832036, 0.334956504, 0.317134334, 0.263807154, 0.195219648, 0.016120887, 0.386174129654754)
ace_conc <- c(0.2, 100, 0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)
sd_ace_conc <- c(0.05, 10, 0.262318006, 0.281361538208953, 0.289527601555302, 0.302351163, 0.327705513, 0.330782201, 0.277259011, 0.233956798, 0.31883929, 0.300578057, 0.313371784, 0.231202155, 0.264687437, 0.243480317, 0.289821733, 0.263808862, 0.289478134, 0.264861034, 0.22461248, 0.229031308, 0.241718918, 0.254111384, 0.187394292, 0.011107606, 0.290519090995892)
plot(res_ace_range[,"ace_concentration"], res_ace_range[,"Values[v_ace_net]"], xaxt='n', las=1, type="l", lwd=2, xlim=c(0.1,100), ylim=c(-4,4), log="x", main="Enjalbert_2017, Pinhal_2019", col="#D6685C", xlab="[acetate] (mM)", ylab="ace flux (mmol/gDW/h)")
abline(h=0)
axis(side = 1, at = xlab_main, labels = TRUE)
axis(side = 1, at = xlab_sec, labels = FALSE, tcl=-0.3)
plot_points(ace_conc, ace_flx, sd_ace_flx, offset=0.04, col="#D6685C")
plot_points(ace_conc, ace_flx, sd_ace_conc, offset=0.2, mode="h", col="#D6685C")

dev.off()


##################################################
# Calculate variance weighted SSR for all models #
##################################################

# Pulse experiment (Enjalbert et al., 2017)
###########################################

# measurements
# sampling time
time_meas <- seq(0,8)/60
# acetate pulse experiment
glc <- c(0, -0.06931183, -0.151415145, -0.189227994, -0.269451057, -0.290764495, -0.230785281, -0.464084162, -0.551878527)
sd_glc <- c(0.1, 0.032476344, 0.073133915, 0.113018846, 0.049485284, 0.005325541, 0.163377704, 0.034786419, 0.048477157)
ace <- c(0, -0.027907926, -0.078000853, -0.155334163, -0.165031608, -0.111098424, -0.182877548, -0.237262298, -0.276903255)
sd_ace <- c(0.01, 0.002740145, 0.025693594, 0.053641876, 0.089975321, 0.005875669, 0.085604161, 0.061930626, 0.099140975)
# control experiment (pulse without acetate)
glc_nopulse <- c(0, -0.195637774, -0.325512845, -0.433785334, -0.628518958, -0.726913548, -0.892908748, -1.071230579, -1.16285575)
sd_glc_nopulse <- c(0.01, 0.058023617, 0.109115511, 0.047090371, 0.052331837, 0.065619906, 0.133896355, 0.16828754, 0.03515465)
ace_nopulse <- c(0, 0.01012067, 0.05974009, 0.086787283, 0.086690257, 0.104329693, 0.128507087, 0.130026354, 0.187336965)
sd_ace_nopulse <- c(0.01, 0.005117257, 0.022507218, 0.040319045, 0.037548873, 0.037235285, 0.044494365, 0.045029023, 0.023982374)

setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_4.cps")
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)
# simulate response to acetate pulse
res_pulse <- runTimeCourse(intervals=10000)
# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
res_nopulse <- runTimeCourse(intervals=10000)
# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]
# get simulation times
time_sim <- res_nopulse$result$Time[seq(id_start, id_end)] - res_nopulse$result$Time[id_start]
idx_sim <- get_index_closest(time_meas, time_sim)
# calculate variance weighted SSR
SSR_model_4 <- sum((glc-glc_conc_pulse[idx_sim])**2/sd_glc**2, (ace-ace_conc_pulse[idx_sim])**2/sd_ace**2, (glc_nopulse-glc_conc_nopulse[idx_sim])**2/sd_glc_nopulse**2, (ace_nopulse-ace_conc_nopulse[idx_sim])**2/sd_ace_nopulse**2 )

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)
# simulate response to acetate pulse
res_pulse <- runTimeCourse(intervals=10000)
# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
res_nopulse <- runTimeCourse(intervals=10000)
# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]
# get simulation times
time_sim <- res_nopulse$result$Time[seq(id_start, id_end)] - res_nopulse$result$Time[id_start]
idx_sim <- get_index_closest(time_meas, time_sim)
# calculate variance weighted SSR
SSR_model_1 <- sum((glc-glc_conc_pulse[idx_sim])**2/sd_glc**2, (ace-ace_conc_pulse[idx_sim])**2/sd_ace**2, (glc_nopulse-glc_conc_nopulse[idx_sim])**2/sd_glc_nopulse**2, (ace_nopulse-ace_conc_nopulse[idx_sim])**2/sd_ace_nopulse**2 )

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)
# simulate response to acetate pulse
res_pulse <- runTimeCourse(intervals=10000)
# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
res_nopulse <- runTimeCourse(intervals=10000)
# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]
# get simulation times
time_sim <- res_nopulse$result$Time[seq(id_start, id_end)] - res_nopulse$result$Time[id_start]
idx_sim <- get_index_closest(time_meas, time_sim)
# calculate variance weighted SSR
SSR_model_2 <- sum((glc-glc_conc_pulse[idx_sim])**2/sd_glc**2, (ace-ace_conc_pulse[idx_sim])**2/sd_ace**2, (glc_nopulse-glc_conc_nopulse[idx_sim])**2/sd_glc_nopulse**2, (ace_nopulse-ace_conc_nopulse[idx_sim])**2/sd_ace_nopulse**2 )

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 0.9)
# simulate response to acetate pulse
res_pulse <- runTimeCourse(intervals=10000)
# get simulation results
id_start <- which(res_pulse$result$Ace_out >= 30)[1]+1
id_end <- which(res_pulse$result$Time >= (res_pulse$result$Time[id_start]+8/60))[1]+1
t_pulse <- (res_pulse$result$Time[seq(id_start, id_end)] - res_pulse$result$Time[id_start])
ace_conc_pulse <- res_pulse$result$Ace_out[seq(id_start, id_end)] - res_pulse$result$Ace_out[id_start]
glc_conc_pulse <- res_pulse$result$Glc[seq(id_start, id_end)] - res_pulse$result$Glc[id_start]
# set biomass concentration at which the pulse is performed
setGlobalQuantities(key = "_X_conc_pulse", initial_value = 1000)
res_nopulse <- runTimeCourse(intervals=10000)
# get simulation results
ace_conc_nopulse <- res_nopulse$result$Ace_out[seq(id_start, id_end)] - res_nopulse$result$Ace_out[id_start]
glc_conc_nopulse <- res_nopulse$result$Glc[seq(id_start, id_end)] - res_nopulse$result$Glc[id_start]
# get simulation times
time_sim <- res_nopulse$result$Time[seq(id_start, id_end)] - res_nopulse$result$Time[id_start]
idx_sim <- get_index_closest(time_meas, time_sim)
# calculate variance weighted SSR
SSR_model_3 <- sum((glc-glc_conc_pulse[idx_sim])**2/sd_glc**2, (ace-ace_conc_pulse[idx_sim])**2/sd_ace**2, (glc_nopulse-glc_conc_nopulse[idx_sim])**2/sd_glc_nopulse**2, (ace_nopulse-ace_conc_nopulse[idx_sim])**2/sd_ace_nopulse**2 )


# Steady-state fluxes for acetate concentration between 0.1 and 100mM (Enjalbert et al., 2017; Pinhal et al., 2019)
##############################################################################################

growth_rates <- c(0.521128511, 0.611148842625582, 0.613161998174498, 0.502533817, 0.496290415, 0.488201506, 0.547635665, 0.499830448, 0.474554197, 0.425356578, 0.377534684, 0.645724326, 0.618475601, 0.554887936, 0.564811523, 0.527571192, 0.434972836, 0.3824734, 0.583623355, 0.620905534, 0.564259247, 0.532148135, 0.483885215, 0.557074418, 0.630654409249223)
sd_growth_rates <- c(0.001793104, 0.00204807928657914, 0.00219396182484705, 0.001709207, 0.001846205, 0.001757403, 0.001821375, 0.001025702, 0.001940912, 0.001204707, 0.001999188, 0.001418374, 0.001932601, 0.001455791, 0.001574234, 0.001206265, 0.001292476, 0.001068259, 0.001804648, 0.001643459, 0.001598405, 0.001121218, 0.000912408, 0.00194896, 0.00203369597574686)
sd_growth_rates <- sapply(sd_growth_rates, FUN=function(x) max(x,0.01))
glc_upt <- c(8.654860167, 8.36127542981722, 7.98010111285252, 9.236935826, 8.274418986, 7.560431219, 7.339194455, 5.775312502, 6.423391263, 5.1544758, 3.938631573, 8.115447647, 9.28067031, 6.737153424, 7.172748804, 5.884186033, 5.684201497, 4.811576974, 9.632702365, 8.055042777, 9.708342814, 7.100081588, 5.505759496, 9.242859752, 8.18621623190759)
sd_glc_upt <- c(0.337812425, 0.38531328268303, 0.373770045721031, 0.356787032, 0.334672954, 0.317509322, 0.288025925, 0.16053276, 0.375934255, 0.293148172, 0.359225607, 0.197331684, 0.360984112, 0.229372278, 0.241820396, 0.20450532, 0.260869273, 0.216134352, 0.34289286, 0.350305744, 0.293144783, 0.220135755, 0.153471508, 0.25245346, 0.396815184905029)
sd_glc_upt <- sapply(sd_glc_upt, FUN=function(x) max(x,0.1))
ace_flx <- c(3.5, -2.7, 1.516999356, 1.26845123082679, 0.775821380507016, 0.678877137, 0.017366464, -0.991478151, -1.286687213, -2.078474994, -1.530841439, -1.525342269, -1.253581266, 1.984679487, 0.546462624, -0.136780389, -0.393883917, -0.610240984, -1.120767885, -1.277455315, 2.574285211, 2.051935093, 1.828415596, -1.262442483, -1.317987733, 2.333568565, 1.85234639824858)
sd_ace_flx <- c(0.35, 0.27, 0.316118066, 0.388752117258161, 0.40715278851436, 0.33357638, 0.37333751, 0.347029894, 0.280501612, 0.195031303, 0.376252463, 0.226182385, 0.303661317, 0.253610517, 0.385450715, 0.243880325, 0.30665695, 0.257983739, 0.23844407, 0.198458448, 0.299832036, 0.334956504, 0.317134334, 0.263807154, 0.195219648, 0.016120887, 0.386174129654754)
sd_ace_flx <- sapply(sd_ace_flx, FUN=function(x) max(x,0.1))

ace_conc <- c(0.451964624, 1.11600286648471, 2.04718732931708, 2.86252145, 5.285907977, 9.106164204, 16.67476528, 16.76626787, 17.00218707, 30.8667961, 57.92292091, 0.478352574, 4.55673229, 8.10163028, 8.22100734, 7.829591756, 33.53244905, 66.20361403, 0.436956014, 1.28468189, 1.555875222, 12.02564968, 30.24566673, 0.548011282, 2.29562227069566)

fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")
# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")
Enj_SS_SSR_model_1 <- c()
for (j in seq(length(ace_conc))){
  setSpecies(key="Ace_out", initial_concentration=ace_conc[j])
  applyInitialState()
  res_ss <- runSteadyState()
  res_flx <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Enj_SS_SSR_model_1 <- c(Enj_SS_SSR_model_1, (res_flx[1]-growth_rates[j])**2/sd_growth_rates[j]**2 + (res_flx[2]-glc_upt[j])**2/sd_glc_upt[j]**2 + (res_flx[3]-ace_flx[j])**2/sd_ace_flx[j]**2)
}
Enj_SS_SSR_model_1 <- sum(Enj_SS_SSR_model_1)

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")
# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")
Enj_SS_SSR_model_2 <- c()
for (j in seq(length(ace_conc))){
  setSpecies(key="Ace_out", initial_concentration=ace_conc[j])
  applyInitialState()
  res_ss <- runSteadyState()
  res_flx <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Enj_SS_SSR_model_2 <- c(Enj_SS_SSR_model_2, (res_flx[1]-growth_rates[j])**2/sd_growth_rates[j]**2 + (res_flx[2]-glc_upt[j])**2/sd_glc_upt[j]**2 + (res_flx[3]-ace_flx[j])**2/sd_ace_flx[j]**2)
}
Enj_SS_SSR_model_2 <- sum(Enj_SS_SSR_model_2)

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")
# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")
Enj_SS_SSR_model_3 <- c()
for (j in seq(length(ace_conc))){
  setSpecies(key="Ace_out", initial_concentration=ace_conc[j])
  applyInitialState()
  res_ss <- runSteadyState()
  res_flx <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Enj_SS_SSR_model_3 <- c(Enj_SS_SSR_model_3, (res_flx[1]-growth_rates[j])**2/sd_growth_rates[j]**2 + (res_flx[2]-glc_upt[j])**2/sd_glc_upt[j]**2 + (res_flx[3]-ace_flx[j])**2/sd_ace_flx[j]**2)
}
Enj_SS_SSR_model_3 <- sum(Enj_SS_SSR_model_3)

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_4.cps")
# remove events and fix concentrations of actate, glucose and biomass
deleteEvent(getEvents()$key)
setSpecies(key="Ace_out", type="fixed")
setSpecies(key="Glc", type="fixed")
setSpecies(key="X", type="fixed")
Enj_SS_SSR_model_4 <- c()
for (j in seq(length(ace_conc))){
  setSpecies(key="Ace_out", initial_concentration=ace_conc[j])
  applyInitialState()
  res_ss <- runSteadyState()
  res_flx <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Enj_SS_SSR_model_4 <- c(Enj_SS_SSR_model_4, (res_flx[1]-growth_rates[j])**2/sd_growth_rates[j]**2 + (res_flx[2]-glc_upt[j])**2/sd_glc_upt[j]**2 + (res_flx[3]-ace_flx[j])**2/sd_ace_flx[j]**2)
}
Enj_SS_SSR_model_4 <- sum(Enj_SS_SSR_model_4)


# Chemostat experiment (Renilla et al., 2012)
#############################################

fluxes <- c("Values[v_growth_rate]", "Values[v_glc_uptake]", "Values[v_ace_net]")

# acetate flux as function of dilution rate
dilution_rate <- c(0.09586056644880175, 0.20043572984749464, 0.2997821350762528, 0.3468409586056645, 0.39912854030501094, 0.44618736383442276, 0.5002178649237474)
# acetate flux as function of glc uptake
q_glc <- c(1.43478260869565, 2.7391304347826, 4.59130434782608, 4.69565217391304, 5.7391304347826, 5.92173913043478, 6.10434782608695, 7.12173913043477, 8.34782608695652)
q_ace <- c(0.381358340437624, 0.762906128635029, 1.40509614473808, 1.17836506583309, 1.99602159704461, 2.08290233967983, 2.01292033721701, 2.44717249218528, 2.5848252344416)
q_ace_sd <- c(0.139547219854126, 0.139471440750213, 0.15697641375391, 0.1045751633987, 0.19179691200152, 0.0174291938997801, 0.24397082504499, 0.52295159609737, 0.33130624230368)
q_glc_sd <- c(0.20234760540444, 0.58899555210581, 0.36827270271402, 0.14607356337583, 0.630432119199, 0.13069807968161, 0.47630364392495, 0.99361847036576, 0.66185122193432)
q_ace_sd <- sapply(q_ace_sd, FUN=function(x) max(x,0.1))
q_glc_sd <- sapply(q_glc_sd, FUN=function(x) max(x,0.1))


setwd(model_dir)
loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_1.cps")
# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)
Ren_SSR_model_1 <- c()
for (j in seq(length(dilution_rate))){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rate[j], model=getCurrentModel())
  applyInitialState(model=getCurrentModel())
  res_ss <- runSteadyState(model=getCurrentModel())
  val <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Ren_SSR_model_1 <- c(Ren_SSR_model_1,
                          (val[2]-q_glc[j])**2/q_glc_sd[j]**2 + (val[3]-q_ace[j])**2/q_ace_sd[j]**2)
}
Ren_SSR_model_1 <- sum(Ren_SSR_model_1)

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_2.cps")
# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)
Ren_SSR_model_2 <- c()
for (j in seq(length(dilution_rate))){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rate[j])
  applyInitialState()
  res_ss <- runSteadyState()
  val <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Ren_SSR_model_2 <- c(Ren_SSR_model_2,
                          (val[2]-q_glc[j])**2/q_glc_sd[j]**2 + (val[3]-q_ace[j])**2/q_ace_sd[j]**2)
}
Ren_SSR_model_2 <- sum(Ren_SSR_model_2)

loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_3.cps")
# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)
Ren_SSR_model_3 <- c()
for (j in seq(length(dilution_rate))){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rate[j], model=getCurrentModel())
  applyInitialState(model=getCurrentModel())
  res_ss <- runSteadyState(model=getCurrentModel())
  val <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Ren_SSR_model_3 <- c(Ren_SSR_model_3,
                          (val[2]-q_glc[j])**2/q_glc_sd[j]**2 + (val[3]-q_ace[j])**2/q_ace_sd[j]**2)
}
Ren_SSR_model_3 <- sum(Ren_SSR_model_3)


loadModel("Millard2020_Ecoli_glc_ace_kinetic_model_4.cps")
# delete events and set glc feed to 1
deleteEvent(getEvents()$key)
setGlobalQuantities(key = "_feed", initial_value = 1)
Ren_SSR_model_4 <- c()
for (j in seq(length(dilution_rate))){
  setGlobalQuantities(key = "_dilution_rate", initial_value = dilution_rate[j], model=getCurrentModel())
  applyInitialState(model=getCurrentModel())
  res_ss <- runSteadyState(model=getCurrentModel())
  val <- unlist(res_ss$global_quantities[res_ss$global_quantities$key %in% fluxes, "value"])
  Ren_SSR_model_4 <- c(Ren_SSR_model_4,
                          (val[2]-q_glc[j])**2/q_glc_sd[j]**2 + (val[3]-q_ace[j])**2/q_ace_sd[j]**2)
}
Ren_SSR_model_4 <- sum(Ren_SSR_model_4)


# Plot results
setwd(results_dir)
pdf(file="Figure 3-figure supplement 4.pdf", width = 5, height = 5)

data <- c(SSR_model_1, SSR_model_2, SSR_model_3, SSR_model_4)
xx <- barplot(data, las=1, ylab="variance-weighted SSR", log="y", names.arg=c("model_1", "model_2", "model_3", "model_4"))
text(x = xx, y = round(data), label = round(data), pos = c(1,3,1,3))

data <- c(Enj_SS_SSR_model_1, Enj_SS_SSR_model_2, Enj_SS_SSR_model_3, Enj_SS_SSR_model_4)
xx <- barplot(data, las=1, ylab="variance-weighted SSR (steady-state data from Enjalbert 2017)", log="y", names.arg=c("model_1", "model_2", "model_3", "model_4"))
text(x = xx, y = round(data), label = round(data), pos = c(1,3,1,3))

data <- c(Ren_SSR_model_1, Ren_SSR_model_2, Ren_SSR_model_3, Ren_SSR_model_4)
xx <- barplot(data, las=1, ylab="variance-weighted SSR", names.arg=c("model_1", "model_2", "model_3", "model_4"))
text(x = xx, y = round(data), label = round(data), pos = 1)

dev.off()
