import numpy as np

nbframes = 200 # saving "nbframes" frames

dx = 0.001 # in mm
dt = 0.02 # in minutes

L = 2.5 # size of spatial domain, i.e. [0,L]
T = 600 # time of simulation

b0 = 0.01 # oxygen consumption rate per cell
chi = 0.0006 # proportionality constant of aerotactic strength
Drho = 3/(10**5) # diffusivity of cells
Doxy = 0.12 # diffusivity of oxygen
r = np.log(2)/440 # cell division rate
Ninit = 21 # initial oxygen concentration
Nt2 = 0.7 # first oxygen threshold C_0
Nt1 = 0.1 # first oxygen threshold C_0

N = np.load("N_initial.npy") # oxygen field
rho = np.load("rho_initial.npy") # cell density field

nbsteps = int(np.floor(T/dt))
I = int(np.floor(L/dx))
everynthframe = int(np.ceil(nbsteps/nbframes))
countframe = 0

N_data = np.zeros((nbframes,I)) # saves oxygen field after each everynthframe iterations
rho_data = np.zeros((nbframes,I)) # saves cell density field after each everynthframe iterations

R = dx+np.linspace(0,L,I) # distance from center

def chi_array(I, N, Nt1, Nt2, chi, dx): # returns the field a(x) as an array
    res = np.zeros(I)
    GN = np.zeros(I-1)
    GN = (N[1:]-N[:I-1])/dx
    for i in range(I-1):
        res[i] = chi*GN[i]/(1+np.exp((N[i]-0.7)/0.2))
    return(res)

def r_array(I, N, Nt1, Nt2, r): # returns the field r(x) as an array
    res = np.zeros(I)
    for i in range(I):
        Naux = N[i]
        if Nt2<=Naux:
            res[i] = r
    return(res)

def thomas_tridiag(a, b, c, d, I): #Solves the system Mx=d where M=tridiag(a,b,c)
    res = np.zeros(I)
    cp = np.zeros(I)
    dp = np.zeros(I)
    cp[0] = c[0]/b[0]
    for i in range(1,I-1):
        cp[i] = c[i]/(b[i]-a[i]*cp[i-1])
    dp[0] = d[0]/b[0]
    for i in range(1,I):
        dp[i] = (d[i]-a[i]*dp[i-1])/(b[i]-a[i]*cp[i-1])
    res[I-1] = dp[I-1]
    for i in range(I-2,-1,-1):
        res[i] = dp[i]-cp[i]*res[i+1]
    return(res)

for t in range(nbsteps):
    # Saving data
    if t%everynthframe==0: 
        N_data[countframe,:] = N
        rho_data[countframe,:] = rho
        countframe += 1
    # Scheme for rho
    chi_field = chi_array(I, N, Nt1, Nt2, chi, dx)
    r_field = r_array(I, N, Nt1, Nt2, r)
    a = np.zeros(I)
    b = np.zeros(I)
    c = np.zeros(I)
    i = 0
    a[i] = 0
    b[i] = 1+Drho*dt/(dx*dx)-dt*r_field[i] # homogeneous Neumann BC
    c[i] = -Drho*dt/(dx*dx)
    for i in range(1,I-1):
        a[i] = -Drho*dt/(dx*dx)*(R[i]-dx/2)/R[i]-dt/dx*chi_field[i-1]*(1-dx/R[i])
        b[i] = 1+2*Drho*dt/(dx*dx)-dt*r_field[i]+dt/dx*chi_field[i]
        c[i] = -Drho*dt/(dx*dx)*(R[i]+dx/2)/R[i]
    i = I-1
    a[i] = -Drho*dt/(dx*dx)*(R[i]-dx/2)/R[i]-dt/dx*chi_field[i-1]*(1-dx/R[i])
    b[i] = 1+Drho*dt/(dx*dx)-dt*r_field[i]+dt/dx*chi_field[i] # homogeneous Neumann BC
    c[i] = 0
    rho = thomas_tridiag(a, b, c, rho, I)
    # scheme for N
    a = np.zeros(I)
    b = np.zeros(I)
    c = np.zeros(I)
    e = np.zeros(I)
    a = -Doxy*dt/(dx*dx)*(R-dx/2)/R
    b[1:] = (1+2*Doxy*dt/(dx*dx))*np.ones(I-1)+dt*b0*rho[1:]*(N[1:]<Nt1)*N[1:]/Nt1
    c[1:] = -dt*Doxy/(dx*dx)*(R[1:]+dx/2)/R[1:]
    b[0] = 1+Doxy*dt/(dx*dx)+dt*b0*rho[0]*(N[0]<Nt1)*N[0]/Nt1 # homogeneous Neumann BC   
    c[0] = -dt*Doxy/(dx*dx)
    e += -dt*b0*rho*(N>Nt1)
    e[-1] += Doxy*dt/(dx*dx)*Ninit # Dirichlet BC N=Ninit
    N = thomas_tridiag(a, b, c, N+e, I)

# Saving data

# Loading data.npz
data_dic = dict(np.load('../../data.npz'))
# Replacing data
data_dic['N_data_AB'] = N_data
data_dic['rho_data_AB'] = rho_data
# Saving present data.npz
np.savez('../../data.npz',**data_dic)

    
    

    
    
    
