import numpy as np

precision = 10**(-11) # precision of binary search algorithm

b0 = 0.01 # oxygen consumption rate per cell
Drho = 3/(10**5) # diffusivity of cells
Doxy = 0.12 # diffusivity of oxygen
Ninit = 21 # initial oxygen concentration
Nt2 = 0.7 # first oxygen threshold C_0
Nt1 = 0.1 # first oxygen threshold C_0

doubling_time = np.arange(2,24,0.01)
r = np.log(2)/(doubling_time*60)  # cell division rate in 1/min
a = np.arange(0.1,2,0.01)/1000 # aerotactic advection speed in mm/min

A,R = np.meshgrid(a,r)

#compute SIGMA

def compute_equality(sigma,a0,r):
    # computes the constraint given by EQMM10
    mu = (sigma+np.sqrt(sigma**2-4*Drho*r))/(2*Drho)
    B = (sigma-mu*Drho)/(sigma-a0) #E
    E = (mu*Drho-a0)/(sigma-a0) #F
    exponential = a0*(sigma-mu*Drho)/(sigma*(a0-mu*Drho)) #e
    h = Drho*np.log(exponential)/(sigma-a0)
    F = b0*(B+E*exponential)/sigma #kt
    H = b0*B/sigma #gt
    I = b0*E/(sigma*(sigma-a0)/Drho-Doxy*(sigma-a0)**2/(Drho**2)) #ht
    L = b0/(mu*(Doxy*mu-sigma)) #bt
    
    a = H+L*(mu-sigma/Drho)-(sigma-a0)/Drho*I
    b = -sigma/Doxy
    c = H-F-I*exponential*(sigma-a0)/Drho #c
    d = -sigma/Doxy*np.exp(sigma*h/Doxy)
    f = sigma/Doxy*(Ninit-Nt2)
    
    Delta = a*d-b*c
    numA = d*f
    numK = -c*f
    
    A = numA/Delta
    K = numK/Delta #J
    
    G = Nt1+F*A*h
    M = Nt2-Ninit-L*A
    J = Nt2-I*A-K
    
    Left = Nt1
    Right = -H*A*h+I*A*exponential+J+K*np.exp(sigma*h/Doxy)
    return(Left-Right)

def find_sigma(a0,r):
    # implementation of a binary search algorithm
    sigma_fisher = 2*np.sqrt(r*Drho)
    if a0<=sigma_fisher/2:
        # in this case sigma = sigma_fisher
        return(sigma_fisher)
    if a0+r*Drho/a0-sigma_fisher<precision:
        # in this case, sigma is in (sigma_fisher,sigma_fisher+precision)
        return(sigma_fisher)
    sigma_left = np.maximum(a0,sigma_fisher)+precision
    c = compute_equality(sigma_left,a0,r)
    is_positive_left = ( c >0)
    sigma_right = a0+r*Drho/a0-precision
    c = compute_equality(sigma_right,a0,r)
    is_positive_right = ( c >0)
    dx = sigma_right-sigma_left
    if is_positive_right == is_positive_left:
        # in this case, sigma is in (a0+r*Drho/a0-precision,a0+r*Drho/a0)
        return(a0+r*Drho/a0)
    else:
        while dx>precision:
            sigma = (sigma_right+sigma_left)/2
            dx /= 2
            is_positive = (compute_equality(sigma,a0,r)>0)
            if is_positive == is_positive_left:
                sigma_left = sigma
            else:
                sigma_right = sigma
    return(sigma)

find_sigma_vectorize = np.vectorize(find_sigma)

SIGMA = find_sigma_vectorize(A,R)

# Saving Data

np.savez("../data.npz", a=a,doubling_time=doubling_time,R=R,SIGMA=SIGMA)


