%% Code associated with Figure 5 in Abrams et al. single fly tracking figures 

% Set file path below or place excel spreadsheet in working directory
masterFilePath = 'Drosophila data.xlsx';

%% Display Parameters
colorPalDec = [31 119 180;
    255 127 14;
    44 160 44;
    214 39 40]/255;
pctDisplayLowerBound = -40;
micronDisplayUpperBound = 700;

%% Read table from file

warning('off','MATLAB:table:ModifiedAndSavedVarnames')
T = readtable(masterFilePath);

%% Loop through flies by ID and extract length data and labels 
eachFlyID = unique(T.FlyID);
eachFlyID = eachFlyID((~cellfun(@isempty,eachFlyID)));
flyNum = length(eachFlyID);
eachFlyPctChange = zeros(1,flyNum);
isMale = zeros(1,flyNum);
isControl = zeros(1,flyNum);
for ii = 1:flyNum
    % For each fly, extract magnification-corrected length against days
    thisFlyRows = find(contains(T.FlyID,eachFlyID{ii}));
    if any(strcmpi(T.Treatment(thisFlyRows),'Control'))
        isControl(ii) = 1;
    end
    
    if any(strcmpi(T.Sex(thisFlyRows),'M'))
        isMale(ii) = 1;
    end
    
    allDays = T.DPA(thisFlyRows);
    allLen = T.Length_pixel_(thisFlyRows)./T.FrameMag(thisFlyRows)*100/214*8;
    dataDays = allDays(~isnan(allLen));
    dataLens = allLen(~isnan(allLen));
    uniqueDays = unique(dataDays);
    uniqueLens = zeros(length(uniqueDays),1);
    for jj = 1:length(uniqueDays)
        uniqueLens(jj) = mean(dataLens(dataDays==uniqueDays(jj)));
    end
    eachFlyLen{ii} = [uniqueDays uniqueLens];
    if ~isempty(uniqueLens)
        eachFlyPctChange(ii) = (uniqueLens(end)-uniqueLens(1))./uniqueLens(1)*100;
    end
end

%% Collect data and statistics
isValid = ~isnan(eachFlyPctChange);
treatedData = eachFlyPctChange(~isControl & isValid);
controlData = eachFlyPctChange(isControl & isValid);
kwlabels =[ones(size(treatedData)) zeros(size(controlData))];
p_treat = kruskalwallis([treatedData controlData],kwlabels,'off');
p_mvf = kruskalwallis(isMale(~isControl&isValid), eachFlyPctChange(~isControl&isValid),'off');

highCI = mean(eachFlyPctChange(isControl&isValid))+2*std(eachFlyPctChange(isControl&isValid));
lowCI = mean(eachFlyPctChange(isControl&isValid))-2*std(eachFlyPctChange(isControl&isValid));

[sortedControl] = sort(controlData,'descend');
[sortedTreat] = sort(treatedData,'descend');

%% Figure: Histogram and Single-Fly Bars
figure('Position',[1 53 1440 752]);

subplot(2,1,1)
histogram(controlData,-100:5:100,'FaceColor',colorPalDec(1,:), 'normalization','probability','FaceAlpha',1);
legend(sprintf('Control (n = %i)',length(controlData)))
axis square
set(gca,'FontSize',16)
xlabel('\bfPct Change in Length')
ylabel('\bfFraction of Flies')
axis([pctDisplayLowerBound 100 0 .75])
title(sprintf('Control'),'FontSize',24)

subplot(2,1,2)
histogram(treatedData,-100:5:100,'FaceColor',colorPalDec(2,:), 'normalization','probability','FaceAlpha',1);
hold on
line([highCI highCI],[0 1],'Color','k','LineStyle','--')
line([lowCI lowCI],[0 1],'Color','k','LineStyle','--')
legend(sprintf('Treated (n = %i)',length(treatedData)))
axis square
set(gca,'FontSize',16)
xlabel('\bfPct Change in Length')
ylabel('\bfFraction of Flies')
axis([pctDisplayLowerBound 100 0 .75])
title(sprintf('Treated (p = %0.d, Kruskal-Wallis)',p_treat),'FontSize',24)

h4 = gcf;
set(h4,'Units','Inches');
pos = get(h4,'Position');
set(h4,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(h4,'FlyLengthSummary','-painters','-depsc','-r0')

%% Figure: Male vs Female Box, Whiskers, and Dots

figure('Position',[1 53 1440 752]);
boxplot(eachFlyPctChange(~isControl&isValid),double(isMale(~isControl&isValid)),'symbol','')
hold on;
plot(1*ones(size(isMale(isMale==0&~isControl&isValid))),eachFlyPctChange(isMale==0&~isControl&isValid),'.','Color',colorPalDec(3,:));
plot(2*ones(size(isMale(isMale==1&~isControl&isValid))),eachFlyPctChange(isMale==1&~isControl&isValid),'.','Color',colorPalDec(4,:))

h_box = findobj(gca,'Tag','Box');
patch(get(h_box(2),'XData'),get(h_box(2),'YData'),colorPalDec(3,:),'FaceAlpha',.85);
patch(get(h_box(1),'XData'),get(h_box(1),'YData'),colorPalDec(4,:),'FaceAlpha',.85);

axis square
set(gca,'xticklabel',{'F','M'})
set(gca,'FontSize',16)
ylabel('\bfPct Change in Length')
title(sprintf('Male vs Female (p = %0.2f, Kruskal-Wallis)',p_mvf))

h4 = gcf;
set(h4,'Units','Inches');
pos = get(h4,'Position');
set(h4,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(h4,'FlyLengthMvF','-painters','-depsc','-r0')


%% Figure: Pct Change Across Time
figure('Position',[1 53 1440 752]);
for ii = 1:flyNum
    if ~isValid(ii) || eachFlyLen{ii}(1,1)~=0
        continue
    end
    pctChangeTime = (eachFlyLen{ii}(:,2)-eachFlyLen{ii}(1,2))/eachFlyLen{ii}(1,2)*100;
    if isControl(ii)
        subplot(211)
        thisCol = colorPalDec(1,:);
    else
        subplot(212)
        thisCol = colorPalDec(2,:);
    end
    plot(eachFlyLen{ii}(:,1),pctChangeTime,'Color',[thisCol,1],'LineWidth',.1)
    hold on
    axis square
end
subplot(211)
axis([0 22 pctDisplayLowerBound 100])
xlabel('Time (days)')
ylabel('Percent Change')
set(gca,'FontSize',16)

subplot(212)
axis([0 22 pctDisplayLowerBound 100])
xlabel('Time (days)')
ylabel('Percent Change')
set(gca,'FontSize',16)

h4 = gcf;
set(h4,'Units','Inches');
pos = get(h4,'Position');
set(h4,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(h4,'FlyLengthThroughTime','-painters','-depsc','-r0')

%% Figure: Starting Length vs Pct Change
figure('Position',[1 53 1440 752]);
for ii = 1:flyNum
    if ~isValid(ii) || eachFlyLen{ii}(1,1)~=0
        continue
    end
    % The first column of eachFlyLen is the day, second column is the lengths
    pctChangeTime = (eachFlyLen{ii}(:,2)-eachFlyLen{ii}(1,2))/eachFlyLen{ii}(1,2)*100;
    startingVal = (eachFlyLen{ii}(1,2));

    if isControl(ii)
        subplot(211)
        thisCol = colorPalDec(1,:);
    else
        subplot(212)
        thisCol = colorPalDec(2,:);
    end
    if abs(max(pctChangeTime))>abs(min(pctChangeTime))
        pctPlot = max(pctChangeTime);
    else
        pctPlot = min(pctChangeTime);
    end
    plot(startingVal,pctPlot,'.','MarkerFaceColor',thisCol,'Color',[thisCol,1],'LineWidth',1)
    hold on
end
subplot(211)
line([0 micronDisplayUpperBound],[lowCI lowCI],'Color','k','LineStyle','--')
line([0 micronDisplayUpperBound],[highCI highCI],'Color','k','LineStyle','--')
axis square
xlabel('Starting Value (micrometers)')
ylabel('Percent Change')
set(gca,'FontSize',16)
axis([0 micronDisplayUpperBound pctDisplayLowerBound 100])

subplot(212)
line([0 micronDisplayUpperBound],[lowCI lowCI],'Color','k','LineStyle','--')
line([0 micronDisplayUpperBound],[highCI highCI],'Color','k','LineStyle','--')
axis([0 micronDisplayUpperBound pctDisplayLowerBound 100])
axis square
xlabel('Starting Value (micrometers)')
ylabel('Percent Change')
set(gca,'FontSize',16)

h4 = gcf;
set(h4,'Units','Inches');
pos = get(h4,'Position');
set(h4,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(h4,'FlyLengthStartingVal','-painters','-depsc','-r0')
