function [] = figure1_supplement2()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%% Neville & Cressman analysis %%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% load the data
dd = load('data-neville-cressman.mat');
iNSm = dd.iNSm; iNSs = dd.iNSs;
iSm = dd.iSm; iSs = dd.iSs;
tNSm = dd.tNSm; tNSs = dd.tNSs;
tSm = dd.tSm; tSs = dd.tSs;

% convert to SD
iNSs = iNSs * sqrt(10);
iSs = iSs * sqrt(10);

% the retention factor
a = 0.9853^3; % 9 trials during the exclusion period

% the proportionality constant
pi = @(a,b) b / (1 - a + b);

% predict asymptote
asymptote = @(a,b,r,t) pi(a,b) * (r-t) / (1 - pi(a,b));

% the cost function
costfun = @(a,b,r,t,i) sum((i - asymptote(a,b,r,t)).^2);

% iterate through each strap
Nstraps = 10000;
r = [20,20,40,40,60,60];
V = zeros(Nstraps,6);
for n = 1 : Nstraps
    % sample data
    i20NS = normrnd(iNSm(1),iNSs(1),10,1);
    i40NS = normrnd(iNSm(2),iNSs(2),10,1);
    i60NS = normrnd(iNSm(3),iNSs(3),10,1);
    i20S = normrnd(iSm(1),iSs(1),10,1);
    i40S = normrnd(iSm(2),iSs(2),10,1);
    i60S = normrnd(iSm(3),iSs(3),10,1);
    t20NS = normrnd(tNSm(1),tNSs(1),10,1);
    t40NS = normrnd(tNSm(2),tNSs(2),10,1);
    t60NS = normrnd(tNSm(3),tNSs(3),10,1);
    t20S = normrnd(tSm(1),tSs(1),10,1);
    t40S = normrnd(tSm(2),tSs(2),10,1);
    t60S = normrnd(tSm(3),tSs(3),10,1);
    
    % calculate the mean
    i20NS = mean(i20NS);
    i40NS = mean(i40NS);
    i60NS = mean(i60NS);
    i20S = mean(i20S);
    i40S = mean(i40S);
    i60S = mean(i60S);
    t20NS = mean(t20NS);
    t40NS = mean(t40NS);
    t60NS = mean(t60NS);
    t20S = mean(t20S);
    t40S = mean(t40S);
    t60S = mean(t60S);   
    
    % wrap up for fit
    t = [t20NS,t20S,t40NS,t40S,t60NS,t60S];
    i = [i20NS,i20S,i40NS,i40S,i60NS,i60S];
    
    % get the optimal b
    b = fminbnd(@(b)costfun(a,b,r,t,i),0,1);
    
    % get the model prediction
    im = asymptote(a,b,r,t);
    V(n,:) = im;
end

% plot the results
figure
hold on
for n = 1 : 3
    bar(n,iNSm(n),'FaceColor','k','EdgeColor','k')
    plot([n,n],[iNSm(n)-iNSs(n)/sqrt(10),iNSm(n)+iNSs(n)/sqrt(10)],'-k')
end

m = mean(V);
s = std(V);
for n = 1 : 3
    ii = 2*n-1;
    bar(n+4,m(ii),'FaceColor',[200,200,200]/255,'EdgeColor','k')
    plot([n+4,n+4],[m(ii)-s(ii),m(ii)+s(ii)],'-k')
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% Experiment 1 Analysis %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% load the data
dd = load('data-experiment-1.mat');
meanA = dd.meanA;
meanS = dd.meanS;
totalA = dd.totalA;
totalS = dd.totalS;

% assign variables
iA = meanA;
iS = meanS;
tA = totalA;
tS = totalS;

% the rotations
rA = 60*ones(size(iA));
rS = repmat([15;30;45;60],1,size(iS,2));

% the driving force (for the re-cast competition model)
dA = rA - tA;
dS = rS - tS;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%% fit the competition model %%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% get the scaling factor (b/1-a+b)
fun = @(im,dr,p) sum((im - p*dr/(1-p)).^2);

% randomly samples participants
iiA = nanmean(iA,2);
ddA = nanmean(dA,2);
iiS = nanmean(iS,2);
ddS = nanmean(dS,2);

% fit the competition model
i = [iiA(4,:) ; iiS];
d = [ddA(4,:) ; ddS];
p = fminbnd(@(p)fun(i,d,p),0,1);
iCompS = dS * p / (1-p);

for b = 1 : 4
    ii = iS(b,:);
    m = nanmean(ii);
    s = nanstd(ii) / sqrt(length(ii));
    bar(b+12,m,'FaceColor','k','EdgeColor','k')
    plot([b+12,b+12],[m-s,m+s],'-k')
    
    ii = iCompS(b,:);
    m = nanmean(ii);
    s = nanstd(ii) / sqrt(length(ii));
    bar(b+18,m,'FaceColor',[200,200,200]/255,'EdgeColor','k')
    plot([b+18,b+18],[m-s,m+s],'-k')
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% Tsay et al analysis %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% load the data
dd = load('data-tsay.mat');
i15 = dd.i15; i30 = dd.i30; i60 = dd.i60; i90 = dd.i90;
t15 = dd.t15; t30 = dd.t30; t60 = dd.t60; t90 = dd.t90;
angles15 = dd.angles15; angles30 = dd.angles30; angles60 = dd.angles60; angles90 = dd.angles90;

% no feedback period
y15 = nanmean(angles15(157:176,:),2); %#ok<*NANMEAN>
y30 = nanmean(angles30(157:176,:),2);
y60 = nanmean(angles60(157:176,:),2);
y90 = nanmean(angles90(157:176,:),2);

lb = [-60,0];
ub = [60,1];
fdecay = @(y0,a,tt) y0 * (a*ones(size(tt))) .^ tt;
costfun = @(y0,a,tt,y) sum((y - fdecay(y0,a,tt)).^2);
pars15 = fmincon(@(xx)costfun(xx(1),xx(2),0:19,y15'),[15,0.8],[],[],[],[],lb,ub);
pars30 = fmincon(@(xx)costfun(xx(1),xx(2),0:19,y30'),[15,0.8],[],[],[],[],lb,ub);
pars60 = fmincon(@(xx)costfun(xx(1),xx(2),0:19,y60'),[15,0.8],[],[],[],[],lb,ub);
pars90 = fmincon(@(xx)costfun(xx(1),xx(2),0:19,y90'),[15,0.8],[],[],[],[],lb,ub);

% mean retention
a = pars15(2) + pars30(2) + pars60(2) + pars90(2);
a = a / 4;

% get the proportionality constant
pi = @(a,b) b / (1 - a + b);

% predict the u-shape phenomena
fpredict = @(r,t,a,b) pi(a,b) * (r-t) / (1 - pi(a,b));

% the cost function
costfun = @(r,t,a,b,i) sum((i - fpredict(r,t,a,b)).^2);

% get the average e in each group
t = [nanmean(t15),nanmean(t30),nanmean(t60),nanmean(t90)];
r = [15,30,60,90];
i = [nanmean(i15),nanmean(i30),nanmean(i60),nanmean(i90)];

% search for optimal b
b = fminbnd(@(xx)costfun(r,t,a,xx,i),0,1);

% the optimal p
p = b / (1 - a + b);

% predict implicit in each group
i15model = p * (15 - t15) / (1 - p);
i30model = p * (30 - t30) / (1 - p);
i60model = p * (60 - t60) / (1 - p);
i90model = p * (90 - t90) / (1 - p);

% the offset x-axis
offs = 28;

% the predicted values
m = nanmean(i15);
s = nanstd(i15) / sqrt(length(i15)); %#ok<*NANSTD>
bar(1+offs,m,'k','EdgeColor','k')
plot([1+offs,1+offs],[m-s,m+s],'-k')
m = nanmean(i30);
s = nanstd(i30) / sqrt(length(i30));
bar(2+offs,m,'k','EdgeColor','k')
plot([2+offs,2+offs],[m-s,m+s],'-k')
m = nanmean(i60);
s = nanstd(i60) / sqrt(length(i60));
bar(3+offs,m,'k','EdgeColor','k')
plot([3+offs,3+offs],[m-s,m+s],'-k')
m = nanmean(i90);
s = nanstd(i90) / sqrt(length(i90));
bar(4+offs,m,'k','EdgeColor','k')
plot([4+offs,4+offs],[m-s,m+s],'-k')

m = nanmean(i15model);
s = nanstd(i15model) / sqrt(length(i15model));
bar(6+offs,m,'FaceColor',[200,200,200]/255,'EdgeColor','k')
plot([6+offs,6+offs],[m-s,m+s],'-k')
m = nanmean(i30model);
s = nanstd(i30model) / sqrt(length(i30model));
bar(7+offs,m,'FaceColor',[200,200,200]/255,'EdgeColor','k')
plot([7+offs,7+offs],[m-s,m+s],'-k')
m = nanmean(i60model);
s = nanstd(i60model) / sqrt(length(i60model));
bar(8+offs,m,'FaceColor',[200,200,200]/255,'EdgeColor','k')
plot([8+offs,8+offs],[m-s,m+s],'-k')
m = nanmean(i90model);
s = nanstd(i90model) / sqrt(length(i90model));
bar(9+offs,m,'FaceColor',[200,200,200]/255,'EdgeColor','k')
plot([9+offs,9+offs],[m-s,m+s],'-k')

ax = gca;
ax.TickDir = 'out';
ylim([0,25])
ax.YTick = 0 : 5 : 25;
text(0,15,'Neville & Cressman')
text(15,24,'Experiment 1')
text(30,22,'Tsay et al')
ylabel('implicit aftereffect (deg)')
title('implicit data and competition model-2')