%% HYPNOS 11
%
% version 11.0, updated 12/13/21
% written by Nicholas Stavropoulos
%
%% get path and filename(s), set valid monitors and channels
clear
disp('HYPNOS 11');
disp('');
ValidMonitors=(1:120);
ValidChannels=(1:32);
yes={'y','Y','yes','Yes'};
[filenames,pathname]=uigetfile('*.*', 'Pick files to analyze','Multiselect','on');

% if one data file selected, make filename a cell array
if iscell(filenames)<1
    filenames={filenames};
end

% find excel spreadsheet if present and separate from other filenames
filenameExcel=[];
for x=1:length(filenames)
    if ~isempty(xlsfinfo(fullfile(pathname,filenames{x})))
        filenameExcel=filenames{x};
    end
end
filenames=setdiff(filenames,filenameExcel);

%% prompt for default settings
disp(' ')
disp('DEFAULT SETTINGS: ')
disp(' ')
disp('Filter dead animals:   Y')
disp('Analyze an integral number of days:   Y')
disp(' ')
Defaults=input('Use these default settings? Y/N [Y]: ', 's');

% default settings
if isempty(Defaults) || ismember(Defaults,yes)
    DeadFilter='y';
    TrimDataEven='y';
else
    % if defaults are rejected, prompt for settings
    TrimDataEven=input('Analyze an integral number of days? Y/N [Y]: ', 's');
    if isempty(TrimDataEven) || ismember(TrimDataEven,yes)
        TrimDataEven='y';
    else
        disp '** DATA WILL NOT BE TRIMMED TO AN EVEN NUMBER OF DAYS **'
        TrimDataEven='n';
    end
    
    % dead animal filtering
    DeadFilter=input('Filter dead animals? Y/N [Y]: ', 's');
    if isempty(DeadFilter)
        DeadFilter='y';
    else 
        disp 'FILTERING IS OFF'
        DeadFilter='n';
    end
end
disp ' '

%% data import and trimming
disp 'Importing data...'
n=length(filenames);
header=cell(1,n);
for x=1:n
    % get path, filenames, headers, and file header in 1st line
    fname=fullfile(pathname,filenames{x});
    fileID=fopen(fname);
    header{x}=textscan(fileID,'%s%d%s%d',1);
    fclose(fileID);
    
    % preallocate matrices
    if x==1
        ReadingIntervalInMin=zeros(1,n);
        ExpDateAndTimeStart=zeros(1,n);
        FileStartDateAndTimeVec=zeros(n,6);
        ExpDateAndTimeEnd=zeros(1,n);
        FileStartTime=zeros(1,n);
        NumberReadsPerDay=zeros(1,n);
        NumberReadingsInFile=zeros(1,n);
        NumberDaysInFile=zeros(1,n);
        NumberDays=zeros(1,n);
    end
    
    % get row 1 - number of readings
    % get row 2 - reading interval MM(SS)
    % get row 3 - first reading time HHMM(SS), 0 = 0000 = HHMM = 12 AM
    NumberReadingsInFile(x)=dlmread(fname,'',[1 0 1 0]);
    ReadingIntervalInMin(x)=dlmread(fname,'',[2 0 2 0]);
    FileStartTime(x)=dlmread(fname,'',[3 0 3 0]);
    
    % calculate number of days and mins recorded and readings per day
    NumberReadsPerDay(x)=1440/ReadingIntervalInMin(x);
    NumberDaysInFile(x)=NumberReadingsInFile(x)/NumberReadsPerDay(x);
    NumberDays(x)=NumberDaysInFile(x);

    % trim data to an integral number of days, rounding down
    if TrimDataEven=='y'
        NumberDays=fix(NumberDaysInFile);
    end
    
    % number of readings and minutes
    NumberReadings=NumberDays.*NumberReadsPerDay;
    NumberMinutes=NumberReadings.*ReadingIntervalInMin;
    
    % get data
    if x==1
        data=zeros(NumberReadings(x),n);
        ExpDatenumbers=zeros(NumberReadings(x),n);       
    end
    data(:,x)=dlmread(fname,'',[4,0,3+NumberReadings(x),0]);
   
    % get file dates and times
    ExpDateAndTimeStart(x)=datenum([num2str(header{x}{2}) ' ' char(header{x}{3}) ' ' num2str(header{x}{4}),...
        num2str(dlmread(fname,'',[3 0 3 0]))],'ddmmmyyyyHHMM');
    ExpDateAndTimeEnd(x)=datenum(ExpDateAndTimeStart(x)+(ReadingIntervalInMin(x)*NumberReadings(x)-1)/1440);
    ExpDatenumbers(:,x)=ExpDateAndTimeStart(x):ReadingIntervalInMin(x)/1440:ExpDateAndTimeEnd(x);
    ExpDateAndTimeVec=datevec(ExpDateAndTimeEnd(x));
end

%% verify that files have the same start/end dates and times and 1 min readings

if numel(unique(ExpDateAndTimeStart))==1 && ...
        numel(unique(ExpDateAndTimeEnd))==1 && ...
        sum(ReadingIntervalInMin-1)==0
    disp(['Files contain ' num2str(NumberDaysInFile(1)) ' days of data. Analyzing the first ' num2str(NumberDays(1)) ' days...' ]);
else
    if numel(unique(ExpDateAndTimeStart))>1
        beep
        disp '*** Files do not start at the same date and time ***'
        disp 'Files start at:'
        disp(datestr(unique(ExpDateAndTimeStart)));
        disp 'Files end at:';
        disp(datestr(unique(ExpDateAndTimeEnd)));
    end
    if numel(unique(ReadingIntervalInMin))
        beep
        disp '*** File reading intervals are not all 1 min ***'
        disp(['Files have readings at:  ' num2str(unique(ReadingIntervalInMin)) ' min intervals']);
    end
end

%% find environmental monitor and import data

% find zero channels and those for humidity, light, temperature
Channels_Zero=find(sum(data)<1);
Channels_TempHumLight=find(sum(data)./NumberMinutes>100);
EnvMonPutativeFilenames=cell(1,length(Channels_TempHumLight));
EnvMonPutativeNumber=zeros(1,length(Channels_TempHumLight));
EnvMonPutativeChannels=zeros(1,length(Channels_TempHumLight));

for r=1:length(Channels_TempHumLight)
    EnvMonPutativeFilenames(r)=regexp(filenames{Channels_TempHumLight(r)},'M(\d{3})C(\d{2})','tokens');
    EnvMonPutativeNumber(r)=str2double(char(EnvMonPutativeFilenames{r}{1}));
    EnvMonPutativeChannels(r)=str2double(char(EnvMonPutativeFilenames{r}{2}));
end

if length(unique(EnvMonPutativeNumber))==1 &&...
        length(intersect(EnvMonPutativeChannels,(7:10)))==4
    EnvMonitor=unique(EnvMonPutativeNumber);
    disp(['Monitor ' num2str(EnvMonitor) ' was identified as an Environmental Monitor.']);
    IndicesEnvMonitor=find(~cellfun(@isempty,regexp(filenames,[num2str(EnvMonitor) 'C\d{2}'])));
    EnvMonitorPresent='y';
else
    EnvMonitor=[];
    disp('No Environmental Monitor was detected.');
    IndicesEnvMonitor=[];
    EnvMonitorPresent='n';
end

% separate environment monitor files
filenamesEnvMonitor=filenames(IndicesEnvMonitor);
EnvMonitorNumberDays=NumberDays(IndicesEnvMonitor);
EnvMonitorNumberMinutes=NumberMinutes(IndicesEnvMonitor);
EnvMonitorNumberReadings=NumberReadings(IndicesEnvMonitor);
EnvMonitorNumberReadsPerDay=NumberReadsPerDay(IndicesEnvMonitor);

% separate DAM monitors, DAM channels, and reset n
IndicesDAMs=setdiff(1:n,IndicesEnvMonitor);
n=length(IndicesDAMs);
filenamesDAMs=filenames(IndicesDAMs);
NumberDays=NumberDays(IndicesDAMs);
NumberMinutes=NumberMinutes(IndicesDAMs);
NumberReadings=NumberReadings(IndicesDAMs);
NumberReadsPerDay=NumberReadsPerDay(IndicesDAMs);

% monitor info
MonitorNumbers=regexp(filenames,'M(\d{3})C\d{2}','tokens');
Monitors=zeros(1,length(MonitorNumbers));
for w=1:length(MonitorNumbers)
    Monitors(w)=str2double(MonitorNumbers{w}{1});
end
Monitors=unique(Monitors);
Monitors=setdiff(Monitors,EnvMonitor);

% get environmental data
if EnvMonitorPresent=='y'
    LightInLux=dlmread([pathname,filenames{IndicesEnvMonitor(2)}],'',[4 0 3+EnvMonitorNumberReadings(2) 0]);
    Temperature=dlmread([pathname,filenames{IndicesEnvMonitor(7)}],'',[4 0 3+EnvMonitorNumberReadings(7) 0]);
    Humidity=dlmread([pathname,filenames{IndicesEnvMonitor(12)}],'',[4 0 3+EnvMonitorNumberReadings(12) 0]);
    Light=+logical(LightInLux);
    LightsOnTimes=find(diff(Light)>0)+1;
    LightsOffTimes=find(diff(Light)<0)+1;
end

% feedback for number of data files
if n==1
    disp('Anayzing 1 file...')
else
    disp(['Analyzing ' num2str(n) ' files...']);
end

%% score locomotion and sleep
Activity=data(1:NumberReadings(1),IndicesDAMs);
Active=+logical(Activity);

% preallocate matrices
Sleep=zeros(size(Activity));
LocomotorStops=cell(1,n);
LocomotorStarts=cell(1,n);
LocomotorTransitions=cell(1,n);
LocomotorBoutLengths=cell(1,n);
LocomotorBoutLengthsActive=cell(1,n);
LocomotorBoutLengthsInactive=cell(1,n);
SleepStops=cell(1,n);
SleepStarts=cell(1,n);
SleepTransitions=cell(1,n);
SleepBoutLengths=cell(1,n);
WakeBoutLengths=cell(1,n);
SleepLatency=zeros(NumberDays(1),n);

for x=1:n
    % score locomotor stops, starts, and bouts
    LocomotorStops{1,x}=find(diff(Active(:,x))<0)+1;
    LocomotorStarts{1,x}=find(diff(Active(:,x))>0)+1;
    LocomotorTransitions{1,x}=sort([1 ; LocomotorStops{1,x} ; LocomotorStarts{1,x} ; 1+size(Active(:,x),1)]);
    LocomotorBoutLengths=diff(LocomotorTransitions{1,x});
    % bout lengths - note that first and last are truncated arbitrarily by file start/end
    if Active(1,x)>0
        LocomotorBoutLengthsActive=LocomotorBoutLengths(1:2:end);
        LocomotorBoutLengthsInactive=LocomotorBoutLengths(2:2:end);
    elseif Active(1,x)<1
        LocomotorBoutLengthsInactive=LocomotorBoutLengths(1:2:end);
        LocomotorBoutLengthsActive=LocomotorBoutLengths(2:2:end);
    end
    clear LocomotorBoutLengths
    
    % score sleep
    for y=1:length(Active(:,x))
        if Active(y,x)>0
        Sleep(y,x)=0;
        elseif Active(y,x)<1
            % find locomotor start and stop flanking minute being scored
            LocomotorStopPreceding=LocomotorStops{1,x}(find(LocomotorStops{1,x}<=y,1,'last'));
            LocomotorStartFollowing=LocomotorStarts{1,x}(find(LocomotorStarts{1,x}>y,1,'first'));
            % score sleep at start of file
            if isempty(LocomotorStopPreceding)
                if LocomotorStartFollowing>5
                    Sleep(y,x)=1;
                end
            end
            % score sleep in core of file
            if ~isempty(LocomotorStartFollowing)
                if LocomotorStartFollowing-LocomotorStopPreceding>4
                    Sleep(y,x)=1;
                end
            end
            % score sleep at end of file
            if isempty(LocomotorStartFollowing)
                if length(Active(:,x))-LocomotorStopPreceding>4
                    Sleep(y,x)=1;
                end
            end
        end
    end
       
    % score sleep stops, starts, and bouts
    SleepStops{1,x}=find(diff(Sleep(:,x))<0)+1;
    SleepStarts{1,x}=find(diff(Sleep(:,x))>0)+1;    
    SleepTransitions{1,x}=sort([1 ; SleepStops{1,x} ; SleepStarts{1,x} ; 1+size(Sleep(:,x),1)]);
    SleepWakeBoutLengths=diff(SleepTransitions{1,x});
    % sleep bout lengths - note that first and last bouts are truncated arbitrarily by file start/end
    if Sleep(1,x)>0
        SleepBoutLengths{1,x}=SleepWakeBoutLengths(1:2:end);
        WakeBoutLengths{1,x}=SleepWakeBoutLengths(2:2:end);
    elseif Sleep(1,x)<1
        WakeBoutLengths{1,x}=SleepWakeBoutLengths(1:2:end);
        SleepBoutLengths{1,x}=SleepWakeBoutLengths(2:2:end);
    end
    clear SleepWakeBoutLengths

    % sleep latency - scored only for LD transitions where animal is awake before lights off
    if EnvMonitorPresent=='y'
        for y=1:numel(LightsOffTimes)
            if Sleep(LightsOffTimes(y)-1,x)<1 &&...
                    ~isempty(find(Sleep(LightsOffTimes(y):end,x),1))
                SleepLatency(y,x)=find(Sleep(LightsOffTimes(y):end,x),1);
            else
                SleepLatency(y,x)=NaN;
            end
        end
    end
    
end
Awake=1-Sleep;

% sleep in light and dark, if light information is present and read interval is same for EnvMonitor and DAMs
if exist('Light','var')>0 && unique(EnvMonitorNumberReadings)==unique(NumberReadings)    
    SleepLight=+(Sleep&Light);
    SleepDark=Sleep-SleepLight;
    AwakeLight=+(Awake&Light);
    AwakeDark=Awake-AwakeLight;
    TotalMinutesLight=sum(Light);
    TotalMinutesDark=length(Light)-TotalMinutesLight;
end

%% bin single animal locomotor activity and sleep
% set up bin indices
BinIndices5m(:,1)=1+5*(0:size(Activity,1)/5-1);
BinIndices5m(:,2)=5*(1:size(Activity,1)/5);
BinIndices1h(:,1)=1+60*(0:size(Sleep,1)/60-1);
BinIndices1h(:,2)=60*(1:size(Sleep,1)/60);
BinIndices12h(:,1)=1+720*(0:size(Sleep,1)/720-1);
BinIndices12h(:,2)=720*(1:size(Sleep,1)/720);
BinIndices24h(:,1)=1+1440*(0:size(Sleep,1)/1440-1);
BinIndices24h(:,2)=1440*(1:size(Sleep,1)/1440);

% preallocate
ActivityBinned5m=zeros(size(Activity,1)/5,size(Activity,2));
SleepBinned1h=zeros(size(Sleep,1)/60,size(Sleep,2));
ActivityBinned1h=SleepBinned1h;
SleepBinned12h=zeros(size(Sleep,1)/720,size(Sleep,2));
ActivityBinned12h=SleepBinned12h;
SleepBinned24h=zeros(size(Sleep,1)/1440,size(Sleep,2));
ActivityBinned24h=SleepBinned24h;

% bin single animal activity and sleep into 5m, 1h, 12h, and 24h bins
for tt=1:size(ActivityBinned5m,1)
    ActivityBinned5m(tt,:)=sum(Activity(BinIndices5m(tt,1):BinIndices5m(tt,2),:));
end
for tt=1:size(SleepBinned1h,1)
    SleepBinned1h(tt,:)=sum(Sleep(BinIndices1h(tt,1):BinIndices1h(tt,2),:));
    ActivityBinned1h(tt,:)=sum(Activity(BinIndices1h(tt,1):BinIndices1h(tt,2),:));
end
for tt=1:size(SleepBinned12h,1)
    SleepBinned12h(tt,:)=sum(Sleep(BinIndices12h(tt,1):BinIndices12h(tt,2),:));
    ActivityBinned12h(tt,:)=sum(Activity(BinIndices12h(tt,1):BinIndices12h(tt,2),:));    
end
for tt=1:size(SleepBinned24h,1)
    SleepBinned24h(tt,:)=sum(Sleep(BinIndices24h(tt,1):BinIndices24h(tt,2),:));
    ActivityBinned24h(tt,:)=sum(Activity(BinIndices24h(tt,1):BinIndices24h(tt,2),:));
end

% avg day across experiment for individual animal activity and sleep for 1h bins
AvgActivityBinned1hAvgDay=squeeze(mean(reshape(ActivityBinned1h,24,size(ActivityBinned1h,1)/24,[]),2));
AvgSleepBinned1hAvgDay=squeeze(mean(reshape(SleepBinned1h,24,size(SleepBinned1h,1)/24,[]),2));

% daily sleep differences
SleepBinned24h_diff=diff(SleepBinned24h);

%% dead animal filtering
% to be considered alive, animals must have >1 count each 24h
Alive=ones(1,n);
if DeadFilter=='y'
    [DeadDays,DeadAnimals]=find(ActivityBinned24h<1.1);
    DeadAnimals=unique(DeadAnimals);
    Alive(DeadAnimals)=0;
end
AliveIndices=find(Alive);

%% parameters for individual alive animals (dead animals are set to 0)
% preallocate variables
TotalMinutesSleep=zeros(1,n);
TotalMinutesAwake=zeros(1,n);
TotalMinutesActive=zeros(1,n);
TotalActivity=zeros(1,n);
AvgSleepPerDay=zeros(1,n);
AvgSleepPerDayLight=zeros(1,n);
AvgSleepPerDayDark=zeros(1,n);
AvgAwakePerDay=zeros(1,n);
AvgAwakePerDayLight=zeros(1,n);
AvgAwakePerDayDark=zeros(1,n);
AvgActivityPerDay=zeros(1,n);
AvgActivityPerMin=zeros(1,n);
AvgActivityPerAwakeMin=zeros(1,n);
AvgActivityPerActiveMin=zeros(1,n);
AvgSleepLatency=zeros(1,n);

% totals across experiment
TotalMinutesSleep(AliveIndices)=sum(Sleep(:,AliveIndices));
TotalMinutesAwake(AliveIndices)=sum(Awake(:,AliveIndices));
TotalMinutesActive(AliveIndices)=sum(Active(:,AliveIndices));
TotalActivity(AliveIndices)=sum(Activity(:,AliveIndices));

% averages for individual animals
AvgSleepPerDay(AliveIndices)=sum(Sleep(:,AliveIndices))./NumberDays(AliveIndices);
AvgAwakePerDay(AliveIndices)=sum(Awake(:,AliveIndices))./NumberDays(AliveIndices);
AvgActivityPerDay(AliveIndices)=TotalActivity(:,AliveIndices)./NumberDays(AliveIndices);
AvgActivityPerMin(AliveIndices)=TotalActivity(:,AliveIndices)./NumberMinutes(AliveIndices);
AvgActivityPerAwakeMin(AliveIndices)=TotalActivity(:,AliveIndices)./TotalMinutesAwake(AliveIndices);
AvgActivityPerActiveMin(AliveIndices)=TotalActivity(:,AliveIndices)./TotalMinutesActive(AliveIndices);
if EnvMonitorPresent=='y'
    AvgSleepPerDayLight(AliveIndices)=sum(SleepLight(:,AliveIndices))./NumberDays(AliveIndices);
    AvgSleepPerDayDark(AliveIndices)=sum(SleepDark(:,AliveIndices))./NumberDays(AliveIndices);
    AvgAwakePerDayLight(AliveIndices)=sum(AwakeLight(:,AliveIndices))./NumberDays(AliveIndices);
    AvgAwakePerDayDark(AliveIndices)=sum(AwakeDark(:,AliveIndices))./NumberDays(AliveIndices);
    AvgSleepLatency(AliveIndices)=mean(SleepLatency(:,AliveIndices),'omitnan');
end

% preallocate matrices for bouts
AvgSleepBoutLength=zeros(1,n);
AvgAwakeBoutLength=zeros(1,n);
MedianSleepBoutLength=zeros(1,n);
MedianAwakeBoutLength=zeros(1,n);
NumberSleepBouts=zeros(1,n);
NumberAwakeBouts=zeros(1,n);
AvgSleepBoutsPerDay=zeros(1,n);
AvgAwakeBoutsPerDay=zeros(1,n);

% individual animal bout lengths and numbers
for x=1:n
    if Alive(x)>0
        AvgSleepBoutLength(x)=mean(SleepBoutLengths{x});
        AvgAwakeBoutLength(x)=mean(WakeBoutLengths{x});
        MedianSleepBoutLength(x)=median(SleepBoutLengths{x});        
        MedianAwakeBoutLength(x)=median(WakeBoutLengths{x});
        % total number sleep and wake bouts
        NumberSleepBouts(x)=numel(SleepBoutLengths{x});
        NumberAwakeBouts(x)=numel(WakeBoutLengths{x});        
        % note number of sleep bouts includes first and last bouts
        AvgSleepBoutsPerDay(x)=NumberSleepBouts(x)/NumberDays(x);
        AvgAwakeBoutsPerDay(x)=NumberAwakeBouts(x)/NumberDays(x);
    end
end
    
%% get line indices for averaging animals into lines
% detect excel spreadsheet or prompt for averaging
if ~isempty(filenameExcel)
    AveragingMode='excel';
    pathnameExcel=pathname;
else
    AveragingMode=input('Would you like to average individual animals into lines? Y/N [Y]: ', 's');
    if ismember(AveragingMode,yes) || isempty(AveragingMode)
        AveragingMode=input('Excel spreadsheet (E) / Manual entry (M) / Same # per line (S)?  E/M/S [E]: ', 's');
        if isempty(AveragingMode) || AveragingMode=='e'
            AveragingMode='excel';
        elseif AveragingMode=='m'
            AveragingMode='manual';
        elseif AveragingMode=='s'
            AveragingMode='same';
        elseif ismember(AveragingMode,no) || isempty(AveragingMode) || ~ismember(AveragingMode,no)
            AveragingMode='none';
        end
    else
        AveragingMode='none';
    end
end
    
% excel spreadsheet averaging mode - find line indices
if strcmp(AveragingMode,'excel')
    if isempty(filenameExcel)
        [filenameExcel,pathnameExcel]=uigetfile('*.*', 'Select Excel file with genotypes',pathname);
    end
    [num,txt,raw] = xlsread(fullfile(pathnameExcel,filenameExcel));
    genotypes=raw(2:end,1);
    % ensure that pure numeric titles become strings
    for gg=1:length(genotypes)
        if isnumeric(genotypes{gg})>0
            genotypes{gg}=num2str(genotypes{gg});
        end
    end
    
    LineMonitorFirst=xlsread(fullfile(pathnameExcel,filenameExcel),'B:B');
    LineCellFirst=xlsread(fullfile(pathnameExcel,filenameExcel),'C:C');
    NumberLines=length(genotypes);
    LineIndicesFirst=zeros(NumberLines,1);
    LineIndicesLast=zeros(NumberLines,1);
    
    % check that all monitors specified in excel are present in files
    if isequal(sort(Monitors),unique(LineMonitorFirst)')
        
        % check that monitor range is 1-120 and channel range is 1-32
        if all(ismember(LineMonitorFirst,ValidMonitors)) && ...
                all(ismember(LineCellFirst,ValidChannels)) ...
                
            % asssemble search strings and search selected files
            LineIndicesFirstFilenames=strcat('M',num2str(LineMonitorFirst,'%03.0f'),'C',num2str(LineCellFirst,'%02.0f'));
            for u=1:NumberLines
                if cellfun('isempty',strfind(filenames,LineIndicesFirstFilenames(u,:)))>0
                    beep
                    disp '** Monitors specified in Excel file are absent from selected files **'
                    disp 'LINES NOT AVERAGED'
                    AveragingMode='none';
                else
                    LineIndicesFirst(u)=find(~cellfun('isempty',strfind(filenamesDAMs,LineIndicesFirstFilenames(u,:)))>0);
                    for u=1:NumberLines-1
                        LineIndicesLast(u)=LineIndicesFirst(u+1)-1;
                    end
                    LineIndicesLast(NumberLines)=n;
                end
            end
        else
            beep
            disp '** Monitor numbers must be integers between 1 and 120 **'
            disp '** Channel numbers must be integers between 1 and 32 **'
            disp 'LINES NOT AVERAGED'
            AveragingMode='none';
        end
    else
        beep
        disp '** Mismatch between monitors in selected files and Excel spreadsheet **'
        MonitorsMissing=setxor(Monitors,unique(LineMonitorFirst));
        MonitorsMissingFromExcel=intersect(MonitorsMissing,Monitors);
        MonitorsMissingFromFiles=intersect(MonitorsMissing,LineMonitorFirst);
        if isempty(MonitorsMissingFromExcel)<1
            disp (['** Monitor(s) ' num2str(MonitorsMissingFromExcel') ' appear to be missing from the Excel spreadsheet'])
        end
        if isempty(MonitorsMissingFromFiles)<1
            disp (['** Monitor(s) ' num2str(MonitorsMissingFromFiles') '  appear to be missing from the selected files'])
        end
        disp 'LINES NOT AVERAGED'
        AveragingMode='none';
    end
end

% manual averaging mode - find line indices
if strcmp(AveragingMode,'manual')
    proceed='n';
    while proceed=='n'
        Line=input('Enter # of animals for each line, separated by commas (e.g. 5,6,7): \n','s');
        Line=str2num(Line); %#ok<ST2NM>
        if sum(Line)>length(filenamesDAMs)
            beep
            disp '***Number of indicated animals exceeds number of input files.***'
            proceed='n';
        else
            disp ([num2str(length(Line)) ' line(s) and ' num2str(sum(Line)) ' animals indicated.'])
            proceed=input('Correct? Y/N [Y]: ','s');
        end
    end
    
    % setup indices of first and last animals in each line
    NumberLines=length(Line);
    LineIndicesLast=cumsum(Line);
    LineIndicesFirst=[1,1+LineIndicesLast(1:length(LineIndicesLast)-1)];
    genotypes=cell(NumberLines,1);
    LineNames=input('Enter line names? Y/N [Y]: ','s');
    if isempty(LineNames) || ismember(LineNames,yes)
        LineNames='y';
        for x=1:NumberLines
            genotypes{x}=input(['Enter name of line ' num2str(x) ' (n=' num2str(Line(x)) ') : '],'s');
        end
    end
end

% same # of animals in each line averaging mode - find line indices
% ensure # animals per line is integer and round down
if strcmp(AveragingMode,'same')
    AnimalsPerLine=input('How many animals per line? : ');
    AnimalsPerLine=fix(AnimalsPerLine);
    NumberLines=n/AnimalsPerLine;
    genotypes=cell(NumberLines,1);
    
    % setup indices of first and last animals in each line
    LineIndicesLast=AnimalsPerLine*(1:NumberLines);
    LineIndicesFirst=LineIndicesLast-(AnimalsPerLine-1);
    
    % no input in # animals per line
    if isempty(AnimalsPerLine)>0
        AveragingMode='none';
    end
    
    % ensure animals is divisible by # lines
    if ~strcmp(AveragingMode,'none')
        if NumberLines==fix(NumberLines)<1
            AveragingMode='none';
            beep
            disp '** Number of animals not evenly divisible **'
            disp 'LINES NOT AVERAGED'
        end        
        % ensure line data consistent with # input animals
        if max(LineIndicesLast)>n
            AveragingMode='none';
            beep
            disp '** Number of animals specified exceeds number of analyzed files **'
            disp 'LINES NOT AVERAGED'
        end
    end
end

%% average variables by line
if ~strcmp(AveragingMode,'none')
    
    % preallocate variables
    AliveNumberInLine=zeros(1,NumberLines);
    AliveIndicesByLine=cell(NumberLines,1);
    
    AvgActivityPerAwakeMinByLine=cell(1,NumberLines);
    AvgSleepPerDayByLine=cell(1,NumberLines);
    AvgSleepPerDayDarkByLine=cell(1,NumberLines);
    AvgSleepPerDayLightByLine=cell(1,NumberLines);
    AvgSleepBoutLengthByLine=cell(1,NumberLines);
    AvgSleepBoutsPerDayByLine=cell(1,NumberLines);
    AvgSleepLatencyByLine=cell(1,NumberLines);
    
    LineAvgActivityPerAwakeMin=zeros(1,NumberLines);
    LineAvgAwakePerDay=zeros(1,NumberLines);
    LineAvgAwakePerDayDark=zeros(1,NumberLines);
    LineAvgAwakePerDayLight=zeros(1,NumberLines);
    LineAvgAwakeBoutLength=zeros(1,NumberLines);
    LineAvgAwakeBoutsPerDay=zeros(1,NumberLines);
    LineAvgSleepPerDay=zeros(1,NumberLines);
    LineAvgSleepPerDayDark=zeros(1,NumberLines);
    LineAvgSleepPerDayLight=zeros(1,NumberLines);
    LineAvgSleepBoutLength=zeros(1,NumberLines);
    LineAvgSleepBoutsPerDay=zeros(1,NumberLines);
    LineAvgSleepLatency=zeros(1,NumberLines);
    
    LineStdevActivityPerAwakeMin=zeros(1,NumberLines);    
    LineStdevAwakePerDay=zeros(1,NumberLines);
    LineStdevAwakePerDayDark=zeros(1,NumberLines);
    LineStdevAwakePerDayLight=zeros(1,NumberLines);
    LineStdevAwakeBoutLength=zeros(1,NumberLines);
    LineStdevAwakeBoutsPerDay=zeros(1,NumberLines);
    LineStdevSleepPerDay=zeros(1,NumberLines);  
    LineStdevSleepPerDayDark=zeros(1,NumberLines);
    LineStdevSleepPerDayLight=zeros(1,NumberLines);
    LineStdevSleepBoutLength=zeros(1,NumberLines);
    LineStdevSleepBoutsPerDay=zeros(1,NumberLines);
    LineStdevSleepLatency=zeros(1,NumberLines);
    
    LineSEMActivityPerAwakeMin=zeros(1,NumberLines);
    LineSEMAwakePerDay=zeros(1,NumberLines);
    LineSEMAwakePerDayDark=zeros(1,NumberLines);
    LineSEMAwakePerDayLight=zeros(1,NumberLines);
    LineSEMAwakeBoutLength=zeros(1,NumberLines);
    LineSEMAwakeBoutsPerDay=zeros(1,NumberLines);
    LineSEMSleepPerDay=zeros(1,NumberLines);
    LineSEMSleepPerDayDark=zeros(1,NumberLines);
    LineSEMSleepPerDayLight=zeros(1,NumberLines);
    LineSEMSleepBoutLength=zeros(1,NumberLines);
    LineSEMSleepBoutsPerDay=zeros(1,NumberLines);
    LineSEMSleepLatency=zeros(1,NumberLines);
  
    LineSleepPercentile05=zeros(1,NumberLines);
    LineSleepPercentile25=zeros(1,NumberLines);
    LineSleepPercentile50=zeros(1,NumberLines);
    LineSleepPercentile75=zeros(1,NumberLines);
    LineSleepPercentile95=zeros(1,NumberLines);
        
    LineAvgActivity=zeros(length(Activity),NumberLines);
    LineStdevActivity=zeros(length(Activity),NumberLines);
    LineSEMActivity=zeros(length(Activity),NumberLines);
    
    LineAvgSleep=zeros(length(Sleep),NumberLines);
    LineStdevSleep=zeros(length(Sleep),NumberLines);
    LineSEMSleep=zeros(length(Sleep),NumberLines);
    
    LineAvgSleepBinned1hAvgDay=zeros(24,NumberLines);
    LineSEMSleepBinned1hAvgDay=zeros(24,NumberLines);
    LineAvgActivityBinned1hAvgDay=zeros(24,NumberLines);
    LineSEMActivityBinned1hAvgDay=zeros(24,NumberLines);
    
    disp 'Averaging animals for each line...'
    if isempty(NumberLines)<1
        for y=1:NumberLines
            AliveIndicesByLine{y}=intersect(LineIndicesFirst(y):LineIndicesLast(y),AliveIndices);
            AliveNumberInLine(y)=numel(AliveIndicesByLine{y});
            
            % cluster variables by line
            AvgActivityPerAwakeMinByLine{y}=AvgActivityPerAwakeMin(AliveIndicesByLine{y});
            AvgSleepPerDayByLine{y}=AvgSleepPerDay(AliveIndicesByLine{y});
            AvgSleepPerDayDarkByLine{y}=AvgSleepPerDayDark(AliveIndicesByLine{y});
            AvgSleepPerDayLightByLine{y}=AvgSleepPerDayLight(AliveIndicesByLine{y});
            AvgSleepBoutLengthByLine{y}=AvgSleepBoutLength(AliveIndicesByLine{y});
            AvgSleepBoutsPerDayByLine{y}=AvgSleepBoutsPerDay(AliveIndicesByLine{y});
            AvgSleepLatencyByLine{y}=AvgSleepLatency(AliveIndicesByLine{y});

            % line avg, stdev, SEM
            LineAvgActivityPerAwakeMin(y)=mean(AvgActivityPerAwakeMin(AliveIndicesByLine{y}));
            LineAvgAwakePerDay(y)=mean(AvgAwakePerDay(AliveIndicesByLine{y}));
            LineAvgAwakeBoutLength(y)=mean(AvgAwakeBoutLength(AliveIndicesByLine{y}));
            LineAvgAwakeBoutsPerDay(y)=mean(AvgAwakeBoutsPerDay(AliveIndicesByLine{y}));
            LineAvgSleepPerDay(y)=mean(AvgSleepPerDay(AliveIndicesByLine{y}));
            LineAvgSleepBoutLength(y)=mean(AvgSleepBoutLength(AliveIndicesByLine{y}));
            LineAvgSleepBoutsPerDay(y)=mean(AvgSleepBoutsPerDay(AliveIndicesByLine{y}));
            
            LineStdevActivityPerAwakeMin(y)=std(AvgActivityPerAwakeMin(AliveIndicesByLine{y}));
            LineStdevAwakePerDay(y)=std(AvgAwakePerDay(AliveIndicesByLine{y}));
            LineStdevAwakeBoutLength(y)=std(AvgAwakeBoutLength(AliveIndicesByLine{y}));
            LineStdevAwakeBoutsPerDay(y)=std(AvgAwakeBoutsPerDay(AliveIndicesByLine{y}));
            LineStdevSleepPerDay(y)=std(AvgSleepPerDay(AliveIndicesByLine{y}));
            LineStdevSleepBoutLength(y)=std(AvgSleepBoutLength(AliveIndicesByLine{y}));
            LineStdevSleepBoutsPerDay(y)=std(AvgSleepBoutsPerDay(AliveIndicesByLine{y}));
            
            LineSEMActivityPerAwakeMin(y)=LineStdevActivityPerAwakeMin(y)/sqrt(AliveNumberInLine(y));
            LineSEMAwakePerDay(y)=LineStdevAwakePerDay(y)/sqrt(AliveNumberInLine(y));
            LineSEMAwakeBoutLength(y)=LineStdevAwakeBoutLength(y)/sqrt(AliveNumberInLine(y));
            LineSEMAwakeBoutsPerDay(y)=LineStdevAwakeBoutsPerDay(y)/sqrt(AliveNumberInLine(y));
            LineSEMSleepPerDay(y)=LineStdevSleepPerDay(y)/sqrt(AliveNumberInLine(y));
            LineSEMSleepBoutLength(y)=LineStdevSleepBoutLength(y)/sqrt(AliveNumberInLine(y));
            LineSEMSleepBoutsPerDay(y)=LineStdevSleepBoutsPerDay(y)/sqrt(AliveNumberInLine(y));
                        
            if EnvMonitorPresent=='y'
                LineAvgAwakePerDayDark(y)=mean(AvgAwakePerDayDark(AliveIndicesByLine{y}));
                LineAvgAwakePerDayLight(y)=mean(AvgAwakePerDayLight(AliveIndicesByLine{y}));
                LineAvgSleepPerDayDark(y)=mean(AvgSleepPerDayDark(AliveIndicesByLine{y}));
                LineAvgSleepPerDayLight(y)=mean(AvgSleepPerDayLight(AliveIndicesByLine{y}));
                LineAvgSleepLatency(y)=mean(AvgSleepLatency(AliveIndicesByLine{y}),'omitnan');
                
                LineStdevAwakePerDayDark(y)=std(AvgAwakePerDayDark(AliveIndicesByLine{y}));
                LineStdevAwakePerDayLight(y)=std(AvgAwakePerDayLight(AliveIndicesByLine{y}));
                LineStdevSleepPerDayDark(y)=std(AvgSleepPerDayDark(AliveIndicesByLine{y}));
                LineStdevSleepPerDayLight(y)=std(AvgSleepPerDayLight(AliveIndicesByLine{y}));
                LineStdevSleepLatency(y)=std(AvgSleepLatency(AliveIndicesByLine{y}),'omitnan');
                
                LineSEMAwakePerDayDark(y)=LineStdevAwakePerDayDark(y)/sqrt(AliveNumberInLine(y));
                LineSEMAwakePerDayLight(y)=LineStdevAwakePerDayLight(y)/sqrt(AliveNumberInLine(y));
                LineSEMSleepPerDayDark(y)=LineStdevSleepPerDayDark(y)/sqrt(AliveNumberInLine(y));
                LineSEMSleepPerDayLight(y)= LineStdevSleepPerDayLight(y)/sqrt(AliveNumberInLine(y));
                LineSEMSleepLatency(y)=LineStdevSleepLatency(y)/sqrt(sum(~isnan(AvgSleepLatency(AliveIndicesByLine{y}))));
            end
            
            % average activity and sleep profiles
            LineAvgActivity(:,y)=mean(Activity(:,AliveIndicesByLine{y}),2);
            LineStdevActivity(:,y)=std(Activity(:,AliveIndicesByLine{y}),0,2);
            LineSEMActivity(:,y)=LineStdevActivity(:,y)/sqrt(length(AliveIndicesByLine{y}));           
            LineAvgSleep(:,y)=mean(Sleep(:,AliveIndicesByLine{y}),2);
            LineStdevSleep(:,y)=std(Sleep(:,AliveIndicesByLine{y}),0,2);
            LineSEMSleep(:,y)=LineStdevSleep(:,y)/sqrt(length(AliveIndicesByLine{y}));
                        
            % median and percentiles
            LineSleepPercentile05(y)=prctile(AvgSleepPerDay(AliveIndicesByLine{y}),5,'Method','Exact');
            LineSleepPercentile25(y)=prctile(AvgSleepPerDay(AliveIndicesByLine{y}),25,'Method','Exact');
            LineSleepPercentile50(y)=median(AvgSleepPerDay(AliveIndicesByLine{y}));
            LineSleepPercentile75(y)=prctile(AvgSleepPerDay(AliveIndicesByLine{y}),75,'Method','Exact');
            LineSleepPercentile95(y)=prctile(AvgSleepPerDay(AliveIndicesByLine{y}),95,'Method','Exact');          
        end
    end
    
    % preallocate binned variables
    LineAvgActivityBinned1h=zeros(size(ActivityBinned1h,1),NumberLines);
    LineAvgActivityBinned12h=zeros(size(ActivityBinned12h,1),NumberLines);
    LineAvgActivityBinned24h=zeros(size(ActivityBinned24h,1),NumberLines);
    LineAvgSleepBinned1h=zeros(size(SleepBinned1h,1),NumberLines);
    LineAvgSleepBinned12h=zeros(size(SleepBinned12h,1),NumberLines);
    LineAvgSleepBinned24h=zeros(size(SleepBinned24h,1),NumberLines);
    
    LineStdevSleepBinned1h=zeros(size(SleepBinned1h,1),NumberLines);
    LineStdevSleepBinned12h=zeros(size(ActivityBinned12h,1),NumberLines);
    LineStdevSleepBinned24h=zeros(size(SleepBinned24h,1),NumberLines);
    LineStdevActivityBinned1h=zeros(size(ActivityBinned1h,1),NumberLines);
    LineStdevActivityBinned12h=zeros(size(ActivityBinned12h,1),NumberLines);
    LineStdevActivityBinned24h=zeros(size(SleepBinned24h,1),NumberLines);
    
    LineSEMSleepBinned1h=zeros(size(SleepBinned1h,1),NumberLines);
    LineSEMSleepBinned12h=zeros(size(ActivityBinned12h,1),NumberLines);
    LineSEMSleepBinned24h=zeros(size(SleepBinned24h,1),NumberLines);
    LineSEMActivityBinned1h=zeros(size(ActivityBinned1h,1),NumberLines);
    LineSEMActivityBinned12h=zeros(size(ActivityBinned12h,1),NumberLines);
    LineSEMActivityBinned24h=zeros(size(SleepBinned24h,1),NumberLines);
    
    SleepBinned1hByLine=cell(1,NumberLines);
    SleepBinned12hByLine=cell(1,NumberLines);
    SleepBinned24hByLine=cell(1,NumberLines);
    
    % average sleep for lines and average sleep split into days
    if TrimDataEven=='y'
        LineAvgSleepByDay=reshape(LineAvgSleep,1440,NumberDays(1),[]);
        LineAvgSleepAvgDay=squeeze(mean(LineAvgSleepByDay,2,'omitnan'));
    end
            
    % bin and average variables for alive animals in each line
    for y=1:NumberLines
        
        % 1h bins
        for vv=1:size(SleepBinned1h,1)
            LineAvgSleepBinned1h(vv,y)=mean(SleepBinned1h(vv,AliveIndicesByLine{y}));
            LineStdevSleepBinned1h(vv,y)=std(SleepBinned1h(vv,AliveIndicesByLine{y}));
            LineSEMSleepBinned1h(vv,y)=LineStdevSleepBinned1h(vv,y)/sqrt(AliveNumberInLine(y));
            LineAvgActivityBinned1h(vv,y)=mean(ActivityBinned1h(vv,AliveIndicesByLine{y}));
            LineStdevActivityBinned1h(vv,y)=std(ActivityBinned1h(vv,AliveIndicesByLine{y}));
            LineSEMActivityBinned1h(vv,y)=LineStdevActivityBinned1h(vv,y)/sqrt(AliveNumberInLine(y));
        end
        
        % 12h bins
        for vv=1:size(SleepBinned12h,1)
            LineAvgSleepBinned12h(vv,y)=mean(SleepBinned12h(vv,AliveIndicesByLine{y}));
            LineStdevSleepBinned12h(vv,y)=std(SleepBinned12h(vv,AliveIndicesByLine{y}));
            LineSEMSleepBinned12h(vv,y)=LineStdevSleepBinned12h(vv,y)/sqrt(AliveNumberInLine(y));
            LineAvgActivityBinned12h(vv,y)=mean(ActivityBinned12h(vv,AliveIndicesByLine{y}));
            LineStdevActivityBinned12h(vv,y)=std(ActivityBinned12h(vv,AliveIndicesByLine{y}));
            LineSEMActivityBinned12h(vv,y)=LineStdevActivityBinned12h(vv,y)/sqrt(AliveNumberInLine(y));
        end
        
        % 24h bins
        for vv=1:size(SleepBinned24h,1)
            LineAvgSleepBinned24h(vv,y)=mean(SleepBinned24h(vv,AliveIndicesByLine{y}));
            LineStdevSleepBinned24h(vv,y)=std(SleepBinned24h(vv,AliveIndicesByLine{y}));
            LineSEMSleepBinned24h(vv,y)=LineStdevSleepBinned24h(vv,y)/sqrt(AliveNumberInLine(y));
            LineAvgActivityBinned24h(vv,y)=mean(ActivityBinned24h(vv,AliveIndicesByLine{y}));
            LineStdevActivityBinned24h(vv,y)=std(ActivityBinned24h(vv,AliveIndicesByLine{y}));
            LineSEMActivityBinned24h(vv,y)=LineStdevActivityBinned24h(vv,y)/sqrt(AliveNumberInLine(y));
        end
        
        % average day across exp for each line for 1h bins for activity and sleep
        LineAvgSleepBinned1hAvgDay(:,y)=mean(AvgSleepBinned1hAvgDay(:,AliveIndicesByLine{y}),2);
        LineSEMSleepBinned1hAvgDay(:,y)=std(AvgSleepBinned1hAvgDay(:,AliveIndicesByLine{y}),0,2)/sqrt(size(AliveIndicesByLine{y},2));
        LineAvgActivityBinned1hAvgDay(:,y)=mean(AvgActivityBinned1hAvgDay(:,AliveIndicesByLine{y}),2);
        LineSEMActivityBinned1hAvgDay(:,y)=std(AvgActivityBinned1hAvgDay(:,AliveIndicesByLine{y}),0,2)/sqrt(size(AliveIndicesByLine{y},2));
        
        % sort animals from each line into arrays
        SleepBinned1hByLine{y}=SleepBinned1h(:,AliveIndicesByLine{y});
        SleepBinned12hByLine{y}=SleepBinned12h(:,AliveIndicesByLine{y});
        SleepBinned24hByLine{y}=SleepBinned24h(:,AliveIndicesByLine{y});
    end
end
 
%% set up output matrix and write to file

% transpose and concatenate variables
output=horzcat(AvgSleepPerDay',...
    AvgSleepPerDayDark',...
    AvgSleepPerDayLight',...
    AvgSleepBoutLength',...
    AvgSleepBoutsPerDay',...
    AvgSleepLatency',...
    AvgActivityPerAwakeMin',...
    AvgAwakePerDay',...
    AvgAwakePerDayDark',...
    AvgAwakePerDayLight',...
    AvgAwakeBoutLength');

% transpose and concatenate line variables
if strcmp(AveragingMode,'none')<1
    if isempty(NumberLines)<1
        OutputLines=horzcat(AliveNumberInLine',...
            LineAvgSleepPerDay',...
            LineSEMSleepPerDay',...
            LineAvgSleepPerDayDark',...
            LineSEMSleepPerDayDark',...
            LineAvgSleepPerDayLight',...
            LineSEMSleepPerDayLight',...
            LineAvgSleepBoutLength',...
            LineSEMSleepBoutLength',...
            LineAvgSleepBoutsPerDay',...
            LineSEMSleepBoutsPerDay',...
            LineAvgSleepLatency',...
            LineSEMSleepLatency',...
            LineAvgActivityPerAwakeMin',...
            LineSEMActivityPerAwakeMin',...
            LineAvgAwakePerDay',...
            LineSEMAwakePerDay',...
            LineAvgAwakePerDayDark',...
            LineSEMAwakePerDayDark',...
            LineAvgAwakePerDayLight',...
            LineSEMAwakePerDayLight',...
            LineAvgAwakeBoutLength',...
            LineSEMAwakeBoutLength',...
            LineSleepPercentile05',...
            LineSleepPercentile25',...
            LineSleepPercentile50',...
            LineSleepPercentile75',...
            LineSleepPercentile95');
    end
end

% directory
cd (pathname)

% excel output, avoiding overwrite
if exist(['analysis_HYPNOS11_' date '.xls'],'file')<1
    outputfile=fopen(['analysis_HYPNOS11_' date '.xls'], 'w');
else
    x=2;
    while x>0
        if exist(['analysis_HYPNOS11_' date '_' num2str(x) '.xls'], 'file')<1
            outputfile=fopen(['analysis_HYPNOS11_' date '_' num2str(x) '.xls'], 'w');
            break
        else
            x=x+1;
        end
    end
end

% write single animal data
fprintf(outputfile, '%s \t', '***INDIVIDUAL ANIMAL DATA AVERAGED ACROSS EXPERIMENT***');
fprintf(outputfile, '\n');
fprintf(outputfile, '%s \t', '');
fprintf(outputfile, '%s \t', 'SLEEP');
fprintf(outputfile, '\t \t \t \t \t');
fprintf(outputfile, '%s \t', 'AWAKE');
fprintf(outputfile, '\n');
fprintf(outputfile, '\t');
fprintf(outputfile, '%s \t', 'TOTAL');
fprintf(outputfile, '%s \t', 'NIGHTTIME');
fprintf(outputfile, '%s \t', 'DAYTIME');
fprintf(outputfile, '%s \t', 'BOUT LENGTH');
fprintf(outputfile, '%s \t', 'BOUT NUMBER');
fprintf(outputfile, '%s \t', 'LATENCY');
fprintf(outputfile, '%s \t', 'ACTIVITY');
fprintf(outputfile, '%s \t', 'TOTAL');
fprintf(outputfile, '%s \t', 'NIGHTTIME');
fprintf(outputfile, '%s \t', 'DAYTIME');
fprintf(outputfile, '%s \t', 'BOUT LENGTH');
fprintf(outputfile, '\n');
fprintf(outputfile, '%s \t', 'animal');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min');
fprintf(outputfile, '%s \t', 'Num/day');
fprintf(outputfile, '%s \t', 'Min');
fprintf(outputfile, '%s \t', 'Counts/Min');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min/day');
fprintf(outputfile, '%s \t', 'Min');
fprintf(outputfile, '\n');

for x=1:size(output,1)
    fprintf(outputfile, '%s \t', filenamesDAMs{x});
    fprintf(outputfile, '%6.2f \t', output(x,:));
    fprintf(outputfile, '\n');
end

fprintf(outputfile, '\n \n');

% write line data
if ~strcmp(AveragingMode,'none')    
    fprintf(outputfile, '%s \t', '***AVERAGE DATA FOR LINES***');
    fprintf(outputfile, '\n');
    
    fprintf(outputfile, '\t \t \t \t');
    fprintf(outputfile, '%s \t', 'SLEEP');
    fprintf(outputfile, '\t \t \t \t \t \t \t \t \t \t \t');
    fprintf(outputfile, '%s \t', 'AWAKE');
    fprintf(outputfile, '\t \t \t \t \t \t \t \t \t');
    fprintf(outputfile, '%s \t', 'TOTAL SLEEP PER DAY'); 
    fprintf(outputfile, '\n');
    
    fprintf(outputfile, '\t \t \t \t');
    fprintf(outputfile, '%s \t', 'TOTAL');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'NIGHTTIME');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'DAYTIME');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'BOUT LENGTH');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'BOUT NUMBER');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'LATENCY');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'ACTIVITY');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'TOTAL');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'NIGHTTIME');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'DAYTIME');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'BOUT LENGTH');
    fprintf(outputfile, '\t');
    fprintf(outputfile, '%s \t', 'PERCENTILES');
    fprintf(outputfile, '\n');
    
    fprintf(outputfile, '%s \t', 'Sample');
    fprintf(outputfile, '%s \t', 'first animal');
    fprintf(outputfile, '%s \t', 'last animal');
    fprintf(outputfile, '%s \t', 'n');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Num/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Counts/min');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min/day');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', 'Min');
    fprintf(outputfile, '%s \t', 'SEM');
    fprintf(outputfile, '%s \t', '5th pct');
    fprintf(outputfile, '%s \t', '25th pct');
    fprintf(outputfile, '%s \t', '50th pct');
    fprintf(outputfile, '%s \t', '75th pct');
    fprintf(outputfile, '%s \t', '95th pct');
    fprintf(outputfile, '\n');
    
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%s \t', filenamesDAMs{LineIndicesFirst(y)});
        fprintf(outputfile, '%s \t', filenamesDAMs{LineIndicesLast(y)});
        fprintf(outputfile, '%6.2f \t', OutputLines(y,:));
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');   
    fprintf(outputfile, '%s \t', '***INDIVDUAL ANIMAL AVGERAGE DATA, BY LINE***');

    fprintf(outputfile, '\n');
    fprintf(outputfile, '%s \t', '***TOTAL SLEEP (MIN/DAY)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepPerDayByLine{y});
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***NIGHTTIME SLEEP (MIN/DAY)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepPerDayDarkByLine{y});
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***DAYTIME SLEEP (MIN/DAY)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepPerDayLightByLine{y});
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***SLEEP BOUT LENGTH (MIN)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepBoutLengthByLine{y});
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***SLEEP BOUT NUMBER (PER DAY)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepBoutsPerDayByLine{y});
        fprintf(outputfile, '\n');
    end
    
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***SLEEP BOUT LATENCY (MIN)***');
    fprintf(outputfile, '\n');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '%6.2f \t', AvgSleepLatencyByLine{y});
        fprintf(outputfile, '\n');
    end
    
    % line sleep profile for avg day
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***LINE SLEEP PROFILES - AVERAGE DAY***');
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '\t');
    end
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', 'AVG');
        fprintf(outputfile, '%s \t', 'SEM');
    end
    fprintf(outputfile, '\n');
    for x=1:24
        fprintf(outputfile, '%s \t', horzcat('hr ',num2str(x-1),'-',num2str(x)));
        for y=1:NumberLines
            fprintf(outputfile, '%6.2f \t', LineAvgSleepBinned1hAvgDay(x,y));
            fprintf(outputfile, '%6.2f \t', LineSEMSleepBinned1hAvgDay(x,y));
        end
        fprintf(outputfile, '\n');
    end
    
    % line sleep profile for experiment
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***LINE SLEEP PROFILES - ENTIRE EXPERIMENT***');
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '\t');
    end
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', 'AVG');
        fprintf(outputfile, '%s \t', 'SEM');
    end
    fprintf(outputfile, '\n');
    for x=1:size(LineAvgSleepBinned1h,1)
        fprintf(outputfile, '%s \t', horzcat('hr ',num2str(x-1),'-',num2str(x)));
        for y=1:NumberLines
            fprintf(outputfile, '%6.2f \t', LineAvgSleepBinned1h(x,y));
            fprintf(outputfile, '%6.2f \t', LineSEMSleepBinned1h(x,y));
        end
        fprintf(outputfile, '\n');
    end
    
    % line activity profile for avg day
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***LINE ACTIVITY PROFILES - AVERAGE DAY***');
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '\t');
    end
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', 'AVG');
        fprintf(outputfile, '%s \t', 'SEM');
    end
    fprintf(outputfile, '\n');
    for x=1:24
        fprintf(outputfile, '%s \t', horzcat('hr ',num2str(x-1),'-',num2str(x)));
        for y=1:NumberLines
            fprintf(outputfile, '%6.2f \t', LineAvgActivityBinned1hAvgDay(x,y));
            fprintf(outputfile, '%6.2f \t', LineSEMActivityBinned1hAvgDay(x,y));
        end
        fprintf(outputfile, '\n');
    end
    
    % line activity profile for experiment
    fprintf(outputfile, '\n \n');
    fprintf(outputfile, '%s \t', '***LINE ACTIVITY PROFILES - ENTIRE EXPERIMENT***');
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', genotypes{y});
        fprintf(outputfile, '\t');
    end
    fprintf(outputfile, '\n');
    fprintf(outputfile, '\t');
    for y=1:NumberLines
        fprintf(outputfile, '%s \t', 'AVG');
        fprintf(outputfile, '%s \t', 'SEM');
    end
    fprintf(outputfile, '\n');
    for x=1:size(LineAvgSleepBinned1h,1)
        fprintf(outputfile, '%s \t', horzcat('hr ',num2str(x-1),'-',num2str(x)));
        for y=1:NumberLines
            fprintf(outputfile, '%6.2f \t', LineAvgActivityBinned1h(x,y));
            fprintf(outputfile, '%6.2f \t', LineSEMActivityBinned1h(x,y));
        end
        fprintf(outputfile, '\n');
    end
end

fclose(outputfile);

% save .mat file, avoiding overwrite
if exist(['analysis_HYPNOS11_' date '.mat'],'file')<1
    save (['analysis_HYPNOS11_' date '.mat'])
else
    x=2;
    while x>0
        if exist(['analysis_HYPNOS11_' date '_' num2str(x) '.mat'], 'file')<1
            save(['analysis_HYPNOS11_' date '_' num2str(x) '.mat']);
            break
        else
            x=x+1;
        end
    end
end

disp 'Analysis complete.'
