function network = NetEvents(S_bin,tsscn,running,netthresh) 
%% Procedure for the detection of synchronous Events using binarized Ca-imaging data as input
% Inputs: 
%   S_bin      Matrix of Events onsets (cellID X time)
%   tsscn      Timestamps for the matrix
%   running    Logical whether the animal was running
%   netthresh  Threshold for NEs (Default is 1% of cellnumber)
% Output:
%   network    cluster containing acumulated readouts as specified within
%               the script
  

%% Detect Network events 

Fs = 1000*length(tsscn)/tsscn(end);     % Sampling frequency
T = 1/Fs;                                       % Sampling period
win = .2;                                       % binning window in s
wind = 1;                                       % window in between two events to be considered as one
span = round(win/T);                            % Span of moving average window
cellnum = size(S_bin,1);
if ~isempty(netthresh)
    thresh = netthresh;
else
    thresh = ceil(0.01 * size(S_bin,1));        % minimal number of cells needed for event
    if thresh < 4;thresh = 4;end                % lower bound for threshold (original 4)
end

netev = smooth(sum(S_bin),span)*span;
% netev = sum(S_bin);
netval = netev;
netev(netev<=thresh) = 0;
netev(netev>thresh) = 1;
wind = round(wind/T);
netev(end-span:end) = 0;
netev(1:1+span) = 0;
%%
for j = 1:2
    %%   
    stepup = find(diff(netev)==1);
    stepdown = find(diff(netev)==-1); 
    if length(stepup) < length(stepdown)
        stepdown = stepdown(2:end);
    end
    netpos = zeros(1,length(stepup));
    for i = 1 : length(stepdown)-1
        if stepup(i+1)-stepdown(i) < wind
            netev(stepdown(i):stepup(i+1)) = 1;
        end
        netpos(i) = stepup(i)+find(netval(stepup(i):stepdown(i)) == max(netval(stepup(i):stepdown(i))),1);
    end   
end


netpos(i+1) = stepup(i+1)+find(netval(stepup(i+1):stepdown(i+1)) == max(netval(stepup(i+1):stepdown(i+1))),1);
if isempty(netpos) || netpos(1) == 0  
    network.thresh = [];    
    network.netev = netev;
    network.netpos = [];
    network.netID = [];
    network.netprob = [];
    network.netfreq = zeros(4,3);
    network.netraster = [];   
    network.netdist = []; 
    network.nethist = []; 
    network.netshufflehist = []; 
    network.celldist = []; 
    network.netnum = [];
    network.coact = [];
    network.intEvInt = zeros(1,200);
 
    
    return
end
netpos = netpos-1;

%% Cell ID per event

netID = cell(1,length(netpos));
netprob = zeros(1,size(S_bin,1));
netnum = zeros(1,length(netpos));
for i = 1 : length(netpos)
    a = zeros(1,size(S_bin,1));
    for j = 1:size(S_bin,1)
        if find(S_bin(j,netpos(i)-span:netpos(i)+span),1)
            a(j) = 1;
        end
    end
    if length(find(a)) > thresh
        netID{i} = find(a);
        netnum(i) = length(find(a));
        netprob = netprob +a;  
    end
end

netpos = netpos(netnum>0);
netID = netID(netnum>0);
netnum = netnum(netnum>0);

netev(:) = 0;
netev(netpos) = 1;

%% calculate frequencies in different stages

lges = tsscn(end)/1000/60;                     %total time in min
lrun = T*length(running(running==1))/60;   %time running
lstand = T*length(running(running==0))/60; %time standing

netfreq = zeros(2,3);

netfreq(1,1) = sum(netev);
netfreq(1,2) = sum(netev(running==1));
netfreq(1,3) = sum(netev(running==0));

netfreq(2,1) = netfreq(1,1)/lges;
netfreq(2,2) = netfreq(1,2)/lrun;
netfreq(2,3) = netfreq(1,3)/lstand;

if lrun < lstand
    a = find(running==1);
    b = find(running==0);
    b = b(1:length(a));
    netfreq(5,1) = netfreq(1,1);
    netfreq(5,2) = sum(netev(a));
    netfreq(5,3) = sum(netev(b));
    netfreq(6,1) = netfreq(1,1)/lges;
    netfreq(6,2) = sum(netev(a))/lrun;
    netfreq(6,3) = sum(netev(b))/lrun;
else
    a = find(running==1);
    b = find(running==0);
    a = a(1:length(b));
    netfreq(5,1) = netfreq(1,1);
    netfreq(5,2) = sum(netev(a));
    netfreq(5,3) = sum(netev(b));
    netfreq(6,1) = netfreq(1,1)/lges;
    netfreq(6,2) = sum(netev(a))/lstand;
    netfreq(6,3) = sum(netev(b))/lstand;
end
%% The raster
netraster = zeros(size(S_bin,1),length(netID));
for i = 1:length(netID)
    netraster(netID{i},i)=1;
end

%% filter all events that occured during running
standnet = running(netpos)==0;
netID = netID(standnet);
netpos = netpos(standnet);
netnum = netnum(standnet);
netraster = netraster(:,standnet);
netprob = sum(netraster,2)';
netev(:) = 0;
netev(netpos) = 1;


%% inter event interval
if ~isempty(find(running,1))
    start = find(diff(running)==1);
    stop = find(diff(running) == -1);
    start = reshape(start,1,length(start));
    stop = reshape(stop,1,length(stop));
    if stop(1)>start(1)
        stop = [1 stop];
    end
    if start(end)<stop(end)
        start = [start length(netev)];
    end

    c = [];
    for i = 1 :length(stop)
        a = netev(stop(i):start(i));
        time = tsscn(stop(i):start(i));
        b = time(a == 1);
        if length(b) > 2
            c = [c; diff(b)];
        end
    end
    intEvInt = histcounts(c,0:100:20000);
else
    intEvInt = NaN(1,200);
end
%%

%% check for co-activity between cells
coact = zeros(cellnum);
for i = 1:length(netID)
    a = netID{i};
    for j = 1:length(a)
        k = j+1;
        while k<=length(a)
            coact(a(j),a(k)) = coact(a(j),a(k))+1;
            coact(a(k),a(j)) = coact(a(k),a(j))+1;           
            k = k+1;
        end
    end
end

% coact = coact./length(netID);

%% cos-distance between network events

netdist = zeros(size(netraster,2));
for i = 1:length(netdist)
    for j = 1:length(netdist)
        netdist(i,j) = (netraster(:,j)'*netraster(:,i))./(sqrt(sum(netraster(:,j)))* sqrt(sum(netraster(:,i))) );
    end
end
netdist(1:length(netdist)+1:end) = NaN;

% Shuffle analysis
numit = 1000;        
shufdist = zeros(size(netdist,1),size(netdist,2),numit);
parfor ii = 1:numit 
    shufraster = zeros(size(netraster));
    for jj = 1:size(netraster,1)
        a = randperm(size(netraster,2));
        shufraster(jj,:) = netraster(jj,a);
    end
    tempdist = zeros(size(netdist,1),size(netdist,2));
    for i = 1:length(netdist)
        for j = 1:length(netdist)
            tempdist(i,j) = (shufraster(:,j)'*shufraster(:,i))./(sqrt(sum(shufraster(:,j)))* sqrt(sum(shufraster(:,i))) );
        end
    end
    tempdist(1:length(tempdist)+1:end) = NaN;
    shufdist(:,:,ii) = tempdist;
end

nethist = histcounts(netdist(~isnan(netdist)),0:.02:1,'normalization','probability');
netshufflehist = histcounts(shufdist(~isnan(shufdist)),0:.02:1,'normalization','probability');


%% cos-distance between cells

celldist = zeros(size(netraster,1));
for i = 1:length(celldist)
    for j = 1:length(celldist)
        celldist(i,j) = (netraster(j,:)*netraster(i,:)')./(sqrt(sum(netraster(j,:)))* sqrt(sum(netraster(i,:))) );
    end
end



%%
network.thresh = thresh;
network.netev = netev;
network.netpos = tsscn(netpos);
network.netID = netID;
network.netprob = netprob;
network.netfreq = netfreq;
network.netraster = netraster;
network.netdist = netdist;
network.nethist = nethist;
network.netshufflehist = netshufflehist;
network.celldist = celldist;
network.netnum = netnum;
network.coact = coact;
network.intEvInt = intEvInt;



end
