functions {
    vector malaria_fit(vector phi, vector theta, real[] x_r, int[] x_i) {
        /*########## (DECLARE) ##########*/
        /*########## (declare) fixed  parameters ##########*/
        real max_iRBC;
        real kappa;
        real muM;
        real R0;
        real I0;
        real ds[x_i[1]];

        /*########## (declare) fitted  parameters ##########*/
        real muR_dash;
        real muR_dash2;
        real rho;
        real psi_N1;
        real psi_N2;
        real beta;
        
        real sd_RBC;
        real sd_iRBC;

        real lp_total;

        /*########## (define) derived  parameters ##########*/
        real R_dev;
        real RBC_midn;
        real lambda;
        
        /*########## (declare) data storage  ##########*/
        // One for evaluation 9pm starting
        // The first position is 9h,
        real y_init[x_i[2]]; //                x_i[2] == n_Eq

        // 15h hours delayed with respect to y_init
        // 1st position is midnight following Day 0
        real y_midn[x_i[1],x_i[2]]; //x_i[1] == n_ds, x_i[2] == n_Eq plus 1 is for M

        /*########## (DEFINE) ##########*/
        /*########## (define) fixed  parameters ##########*/
        max_iRBC = x_r[1];
        kappa    = x_r[2];
        muM      = x_r[3];
        R0       = x_r[4];
        I0       = x_r[5];
        
        /*########## (define) estimated  parameters ##########*/
        sd_RBC  = phi[1];
        sd_iRBC = phi[2];
        
        muR_dash  = theta[1];
        muR_dash2 = theta[2];
        rho       = theta[3];
        psi_N1    = theta[4];
        psi_N2    = theta[5];
        beta      = theta[6];
                
        /*########## (define) initial conditions ##########*/
        y_init[1] = 0;      //N1
        y_init[2] = 0;      //N2
        y_init[3] = R0;     //R
        y_init[4] = I0;     //I
        y_init[5] = 0;      //M

        //Wind the clock forward by 15 hours, 0.625 day (from 9h to 12h)
        y_midn[1,1] = 0.625 * psi_N1*(y_init[4]/max_iRBC);
        y_midn[1,2] = 0.625 * psi_N2*(y_init[4]/max_iRBC)+y_init[2];

        /*########## Model phase: RBC turnover (instantaneous) ##########*/
        R_dev = R0 - sum(y_init[3:4]);
        if(R_dev<0) {R_dev =0;}

        y_midn[1,3] = y_init[3]*exp(-(muR_dash+y_midn[1,1])) + R0*(1-exp(-muR_dash2)) + rho*R_dev;
        y_midn[1,4] = y_init[4]*exp(-(muR_dash+y_midn[1,1]+y_midn[1,2]));

        /*########## Model phase: iRBC bursting ##########*/
        y_midn[1,5] = beta*y_midn[1,4];
        
        /*########## Model phase: Parasite invasion ##########*/
        lambda = y_midn[1,5]/(y_midn[1,3]+(muM/kappa));
        
        y_midn[1,3] = y_midn[1,3]*exp(-lambda);
        y_midn[1,4] = lambda*y_midn[1,3];
        
        /*########## (MODEL) ##########*/
        for (d in 1:(x_i[1]-1)){
            /*########## Model phase: Host response regulation ##########*/
            y_midn[d+1,1] = psi_N1*(y_midn[d,4]/max_iRBC);
            y_midn[d+1,2] = psi_N2*(y_midn[d,4]/max_iRBC)+y_midn[d,2];
            
            /*########## Model phase: RBC turnover (instantaneous) ##########*/
            if((d-2)<1){
                R_dev = R0 - sum(y_init[3:4]);
            } else{
                R_dev = R0 - sum(y_midn[(d-2),3:4]);
            }
            if(R_dev<0) {R_dev =0;}
            
            y_midn[d+1,3] = y_midn[d,3]*exp(-(muR_dash+y_midn[d+1,1])) + R0*(1-exp(-muR_dash2)) + rho*R_dev;
            y_midn[d+1,4] = y_midn[d,4]*exp(-(muR_dash+y_midn[d+1,1]+y_midn[d+1,2]));
                        
            /*########## Model phase: iRBC bursting ##########*/
            y_midn[d+1,5] = beta*y_midn[d+1,4];
            
            /*########## Model phase: Parasite invasion ##########*/
            lambda = y_midn[d+1,5]/(y_midn[d+1,3]+(muM/kappa));
            
            y_midn[d+1,3] = y_midn[d+1,3]*exp(-lambda);
            y_midn[d+1,4] = lambda*y_midn[d+1,3];

        }

        /*########## (LIKELIHOOD) ##########*/
        lp_total = 0;
        for (d in 1:x_i[1]){
            // x_i[1] = n_ds
            if (x_i[(2+0*x_i[1]+1):(2+1*x_i[1])][d]!=0){
                // Note: 0 is used as a magic number replacing NA
                lp_total += normal_lpdf(x_i[(2+0*x_i[1]+1):(2+1*x_i[1])][d] | sum(y_midn[d,3:4]),sd_RBC);
            }

            if (x_i[(2+1*x_i[1]+1):(2+2*x_i[1])][d]!=0){
                // Note: 0 is used as a magic number replacing NA
                lp_total += normal_lpdf(log10(x_i[(2+1*x_i[1]+1):(2+2*x_i[1])][d]) | log10(y_midn[d,4]+1),sd_iRBC);
            }
        }
    
        return [lp_total]';

    }
}

data {
    int<lower = 1> n_RndEffs; // Number of mouse random effects (correlated)
    int<lower = 1> n_Eq; // Number of differential equations in the system
    
    int<lower = 1> n_obs; // Number of data points (total)
    int<lower = 1> n_mice; // Number of mice (total)

    int<lower = 1> n_strains; // Number of strains fitted
    int<lower = 1> strain_id[n_mice];
    
    int day[n_obs];
    int y_RBC[n_obs];
    int y_iRBC[n_obs];

    int<lower = 1> n_ds; // Number of time steps

    real max_iRBC;
    real kappa;
    real muM;
    real R0[n_mice];
    real I0[n_mice];
}

transformed data {
    // int data
    int x_i[n_mice,2+(2*n_ds)];

    // real data
    real x_r[n_mice,5];
    
    for(j in 1:n_mice) {
        x_i[j, 1] = n_ds;
        x_i[j, 2] = n_Eq;
        x_i[j,(2+0*n_ds+1):(2+1*n_ds)] = y_RBC[((n_ds*(j-1))+1):((n_ds*(j-1))+n_ds)];
        x_i[j,(2+1*n_ds+1):(2+2*n_ds)] = y_iRBC[((n_ds*(j-1))+1):((n_ds*(j-1))+n_ds)];

        x_r[j,1] = max_iRBC;
        x_r[j,2] = kappa;
        x_r[j,3] = muM;
        x_r[j,4] = R0[j];
        x_r[j,5] = I0[j];
    }
}

parameters {
    real muR_dash;
    real muR_dash2;
    real rho;
    real psi_N1;
    real psi_N2;
    real beta;

    vector<lower = 0>[n_RndEffs] sd_s;
    vector<lower = 0>[n_RndEffs] sd_u;

    cholesky_factor_corr[n_RndEffs] L_s;
    cholesky_factor_corr[n_RndEffs] L_u;

    matrix[n_RndEffs,n_strains] z_s;
    matrix[n_RndEffs,n_mice] z_u;

    real sd_RBC;
    real sd_iRBC;
}

transformed parameters{
    vector[2] phi; // constant parameters between subjects: measurement errors for RBC and iRBC
    matrix[n_RndEffs,n_strains] s; //strain level random effects
    matrix[n_RndEffs,n_mice] u; //subject level random effects
    vector[n_RndEffs] theta[n_mice]; // boundary indexing at theta[n_mice][n_RndEffs+1]

    phi[1] = exp(sd_RBC)  * 5*10^5;
    phi[2] = exp(sd_iRBC) * 0.2;

    s = diag_pre_multiply(sd_s, L_s) * z_s;
    u = diag_pre_multiply(sd_u, L_u) * z_u;

    for(j in 1:n_mice) {
        theta[j][1] = exp(muR_dash+        s[1,strain_id[j]]  + u[1,j])*0.025;
        theta[j][2] = exp(muR_dash2+       s[2,strain_id[j]]  + u[2,j])*0.025;
        theta[j][3] = exp(rho+             s[3,strain_id[j]]  + u[3,j])*0.25;
        theta[j][4] = exp(psi_N1+          s[4,strain_id[j]]  + u[4,j])*1.0;
        theta[j][5] = exp(psi_N2+          s[5,strain_id[j]]  + u[5,j])*1.0;
        theta[j][6] = exp(beta+            s[6,strain_id[j]]  + u[6,j])*7.0;;
    }
}

model {
    muR_dash ~ normal(0,1); // Miller et al.
    muR_dash2 ~ normal(0,1); // Miller et al.
    rho ~ normal(0,1); // Miller et al.
    psi_N1 ~ normal(0,1);
    psi_N2 ~ normal(0,1);
    beta ~ normal(0,1); // Miller et al.
    
    sd_s ~ normal(0,1);
    sd_u ~ normal(0,1);

    to_vector(z_s) ~ normal(0,1);
    to_vector(z_u) ~ normal(0,1);

    L_s ~ lkj_corr_cholesky(5);
    L_u ~ lkj_corr_cholesky(5);

    sd_RBC ~ normal(0,1);
    sd_iRBC ~ normal(0,1);
    
    target += map_rect(malaria_fit, phi, theta, x_r, x_i);
}
