%% ripple_timefrequency.m (Figure 1)

% This script shows the time-frequency plot and ripple (80-120 Hz) band 
%   filtered signal in each trial for a representative intracranial EEG 
%   electrode in the medial temporal lobe (electrode TT1)

% last modified: March 2021



%% Review single trial time-frequency plot
% Visualizes each trial, one by one; click on plot to advance trials
% Close the figure to escape
clearvars; clc; close all

% Load the data
load('fig1_NIH025_s1_chanTT1_data.mat'); % contains 'TF','timevec','freqvec'
% TF: time-frequency data from wavelet transform
% timevec: (1 x time), -4 to 1 second around retrieval
% freqvec: (1 x frequency), 150 bins linearly spaced between 1 and 150 Hz

load('fig1_NIH025_s1_chanTT1_rippleband_trials.mat'); % contains 'ripple_band_TT1','ripple_band_trials_time'
% timeseries_trials_TT1: (trial x time), ripple band (80-120 Hz) filtered signal
% timeseries_trials_time: (1 x time), -4 to 1 second around retrieval

load('fig1_NIH025_s1_chanTT1_rippleraster.mat'); % contains 'ripple_raster_TT1','ripple_raster_time'
% ripple_raster_TT1: (trial x time), raster of iEEG ripples detected using
%       threshold previously described in Vaz et al., 2019, Science
% ripple_raster_time: (1 x time), -4 to 1 second around retrieval


% z-score the time-frequency data
TF_z = (TF - nanmean(TF,3)) ./ nanstd(TF,0,3);

hfig1      = figure('Position',[1700 1700 1100 900]);
patchcolor = [0.9686 0.7451 0.6510];

for trial = 1:size(TF,1)
    clf
    % time-frequency data of trial
    TF_trial = squeeze(TF_z(trial,:,:));
    
    % time-frequency plot
    subplot(2,1,1)
    imagesc(timevec,freqvec,TF_trial,[0 5])
    xlabel('Time (s)'); ylabel('Frequency (Hz)')
    axis([-3 0.5 1 150])
    colormap(jet); %colorbar
    set(gca,'Fontsize',35)
    set(gca,'YDir','normal')
    title(['trial ' num2str(trial)],'Fontsize',25)
    
    % ripple band (80-120 Hz) filtered signal
    subplot(2,1,2)
    plot(ripple_band_trials_time,ripple_band_TT1(trial,:),'k-'); set(gca,'ydir','normal')
    xlim([-3 0.5])
    ylim([-15 15])
    set(gca,'Fontsize',35)
    
    hold on; 
    grp = bwconncomp(squeeze(ripple_raster_TT1(trial,:)));
    id = grp.PixelIdxList; 
    if ~isempty(id) & length(id{1}) < 1000
        for i = 1:length(id)
            p=patch(ripple_band_trials_time([id{i}(1) id{i}(1) id{i}(end) id{i}(end)]),...
                [-15 15 15 -15],patchcolor,'EdgeColor','none'); 
            set(p,'FaceAlpha',0.3); 
        end
    end
    
    
    w = waitforbuttonpress;
end






%% Ripple rate example
% Plots the ripple rate plot in Figure 1
% Load data
load('fig1_NIH025_s1_chanTT1_ripplerate.mat')
% ripplerate_sm_corr: (trial x time), mean smoothed ripple rate of correct trials 
% ripplerate_sm_incorr: (trial x time), mean smoothed ripple rate of correct trials 
% ripplerate_sm_corr_se: (trial x time), standard error across correct trials 
% ripplerate_sm_incorr_se: (trial x time), standard error across incorrect trials 
% timevec_sm: (1 x time) smoothing with 600 ms window with 90% overlap for -4 to 1
%   second around retrieval

corrcolor   = [0.22,0.68,0.15];
incorrcolor = [0.81,0.15,0.15]; 

% correct versus incorrect
hfig1 = figure('Position',[1700 1700 1100 600]);
shadedErrorBar(timevec_sm,ripplerate_sm_corr,ripplerate_sm_corr_se,'lineprops','g-','patchsaturation',0.1)
hold on; plot([0 0],[-1 3],'k-','LineWidth',2)
hold on; plot([-4 1],[0 0],'k-','LineWidth',2)
hold on; pc = plot(timevec_sm,ripplerate_sm_corr,'Color',corrcolor,'LineWidth',6);
shadedErrorBar(timevec_sm,ripplerate_sm_incorr,ripplerate_sm_incorr_se,...
    'lineprops','r-','patchsaturation',0.1)
hold on; pi = plot(timevec_sm,ripplerate_sm_incorr,'Color',incorrcolor,'LineWidth',6);
box off
set(gca,'LineWidth',2); set(gca,'FontSize',55)

ax = gca; ax.XAxisLocation = 'origin'; ax.YAxisLocation = 'origin';
set(gca,'xtick',[-3 -2 -1 0]); set(gca,'ytick',[-1:1:2])

axis([-3 0.5 -0.2 2])
legend([pc pi],' Correct  ',' Incorrect  ','location','best','FontSize',45)
legend boxoff













