%% spikerate_LFP_iEEG.m (Figure 2)
% This script shows correlation between continuous spike rate estimate, LFP
% ripple amplitude and iEEG ripple amplitude. 

% last modified: March 2021


%% LOAD DATA
clearvars; clc; close all
load('fig2_contspk_LFP_iEEG_data.mat')
% CONTAINS:
%   subjnames: (1 x subjects) cell array of subject labels
%   subjsessions: (1 x subjects) cell array, each element contains the session
%       numbers for the subject
%   iEEGripple_sig_cont: (subject x session) cell array, each element
%       contains continous iEEG ripple amplitude (1 x time x trials), where
%       time is -4s before to retrieval, Fs = 1000 Hz 
%   iEEGripple_win: (subject x session) cell array, each element
%       contains smoothed iEEG ripple amplitude (windows x trials), using 100 ms 
%       sliding window without overlap, over -4s prior to retrieval, Fs = 1000 Hz 
%   lfpripple_sig_cont: (subject x session) cell array, each element
%       contains LFP ripple amplitude (1 x time x trials), where
%       time is -4s before to retrieval, Fs = 1000 Hz 
%   lfpripple_win: (subject x session) cell array, each element
%       contains smoothed LFP ripple amplitude (windows x trials), using 100 ms 
%       sliding window without overlap, over -4s prior to retrieval, Fs = 1000 Hz 
%   spkrt_Z_win: (subject x session) cell array, each element contains 
%       smoothed continuous estimate of average spiking for units in each 
%       microelectrode (windows x trials); continuous estimate computed by convolving
%       spike raster with Gaussian (sigma = 25), smoothing was done by
%       100 ms sliding window without overlap 


ripplecolor   = [0.0392 0.0392 0.6706]; 
microripcolor = [0.5922 0.1765 0.80];





%% Plot cross correlation between spike rate and micro-LFP ripple amplitude 
% Figure 2d, average over sessions for each subject
clc; clltr_SP = NaN(1,1); clltr_RE_z = NaN(1,1);
for sub = 1:numel(subjsessions)
    for session = 1:numel(subjsessions{sub})
        temp_spkrt_RE   = {};
        temp_rpplpwr_RE = {};
        if size(spkrt_Z_win{sub,session},2)
            % spike rate and ripple amplitude concatenated over trials
            for trial = 1:size(spkrt_Z_win{sub,session},2)
                spkrt_RE_trial   = squeeze(cat(1,spkrt_Z_win{sub,session}(:,trial)));
                rpplpwr_RE_trial = squeeze(cat(1,lfpripple_win{sub,session}(:,trial)));

                rpplpwr_RE_trial(isnan(spkrt_RE_trial))   = []; rpplpwr_RE_trial(isnan(spkrt_RE_trial))   = [];
                spkrt_RE_trial(isnan(rpplpwr_RE_trial))   = []; rpplpwr_RE_trial(isnan(rpplpwr_RE_trial)) = [];

                temp_spkrt_RE{trial}   = spkrt_RE_trial;
                temp_rpplpwr_RE{trial} = rpplpwr_RE_trial;
            end
            
            % spike rate vs micro-LFP ripple correlation
            [r,pv] = corr(cat(1,temp_spkrt_RE{:}),cat(1,temp_rpplpwr_RE{:}),'type','Spearman'); 
            clltr_RE_z(sub,session) = atanh(r); % z-transformed correlation
            clltr_RE_z(clltr_RE_z==0) = NaN;
            
        end
    end
end

% plot correlation by subject
hfig1 = figure('Position',[100 100 600 600]);
p1    = bar([1],nanmean(nanmean(clltr_RE_z,2)),0.7,'FaceColor',microripcolor,'FaceAlpha',0.8);
hold on; errorbar([1],nanmean(nanmean(clltr_RE_z,2)),NaN,nanstd(nanmean(clltr_RE_z,2))/...
    sqrt(numel(find(~isnan(nanmean(clltr_RE_z,2))))),'k','LineWidth',4,'CapSize',50)
[h,p,~,stats] = ttest(nanmean(clltr_RE_z,2));


subjcolors = {[0.5922 0.1765 0.80],[0.8392 0.4510 0.2863],...
   [0.2804 0.9549 0.4569],[0.9608 0.2314 0.4235],...
   [0.3020 0.7451 0.9333],[0.5686 0.4980 0.3255]};

for sub = 1:numel(subjsessions)
    hold on; scatter([1]+0.35,nanmean(clltr_RE_z(sub,:),2),800,'filled','MarkerFaceAlpha',0.6,...
        'MarkerFaceColor',subjcolors{sub},'MarkerEdgeColor','k','linewidth',2,'jitter','off')
end

axis([0 2 0 1.5])
ax = gca; ax.XAxisLocation = 'origin';
xticks([1]); xticklabels({'LFP ripple'})

box off; 
set(gca,'LineWidth',3); set(gca,'FontSize',40)
title(['p = ' num2str(p)],'fontsize',30)
set(gca,'ticklength',[0.02 0.01])






%% Plot cross correlation of spike rate and macro-iEEG ripple amplitude 
% for each recording session across six patients with microelectrode array
% and iEEG recordings
clc; cnt = 1; 
hfig1 = figure('Position',[100 2400 1300 1500]);
cllt_corrcoeff = NaN(size(spkrt_Z_win));

clltSpkRt = [];
for sub = 1:numel(subjsessions)
    for sess = 1:numel(subjsessions{sub})
        if ~isempty(subjsessions{sub}{sess})
            % spike rate and ripple amplitude concatenated over trials
            spkrt_RE_trial   = squeeze(cat(1,spkrt_Z_win{sub,sess}(:)));
            rpplpwr_RE_trial = squeeze(cat(1,iEEGripple_win{sub,sess}(:)));

            rpplpwr_RE_trial = rpplpwr_RE_trial(:); 
            spkrt_RE_trial   = spkrt_RE_trial(:);
            spkrt_RE_trial   = spkrt_RE_trial(:);
            rpplpwr_RE_trial = rpplpwr_RE_trial(:);

            rpplpwr_RE_trial(isnan(spkrt_RE_trial)) = []; spkrt_RE_trial(isnan(spkrt_RE_trial))   = [];
            spkrt_RE_trial(isnan(rpplpwr_RE_trial))   = []; rpplpwr_RE_trial(isnan(rpplpwr_RE_trial)) = []; 
            
            % plot correlation between each spike rate and macro-iEEG ripple amplitude
            % across all time windows across trials for each session
            subplot(4,3,cnt)
            plot(spkrt_RE_trial(:),rpplpwr_RE_trial(:),'.','MarkerSize',20,'Color',ripplecolor)
            
            l1 = lsline; set(l1,'Color','k','LineWidth',3)
            axis([min(spkrt_RE_trial(:))-2 max(spkrt_RE_trial(:))+4 ...
                min(rpplpwr_RE_trial(:))-2 max(rpplpwr_RE_trial(:))+4])

            [r,pval,rlo,rup] = corrcoef(spkrt_RE_trial(:),rpplpwr_RE_trial(:));
            cllt_corrcoeff(sub,sess) = r(1,2);
            % plot correlation values in text
            if pval(1,2) < 0.001
                str=sprintf(' r = %1.2f (%1.2f,%1.2f)\n p < 0.001',r(1,2),rlo(1,2),rup(1,2));
            elseif pval(1,2) >= 0.001
                str=sprintf(' r = %1.2f (%1.2f,%1.2f)\n p = %1.3f',r(1,2),rlo(1,2),rup(1,2),pval(1,2));
            end
            T = text(min(get(gca, 'xlim')), max(get(gca, 'ylim')), str); 
            set(T, 'fontsize', 20, 'verticalalignment', 'top', 'horizontalalignment', 'left');
            title([subjnames{sub} ', s' num2str(sess)])
            
            set(gca,'FontSize',24)
            box on

            cnt = cnt+1; 
        end
    end
end


xlabel('Spike Rate (Z)','FontSize',20)
ylabel('iEEG Ripple Amplitude (Z)','FontSize',20)





