%% spike_LFP_ppc.m (Figures 4)
% Shows computation of pairwise phase consistency (PPC) of spikes to LFP
% reference: Vinck et al. 2010 NeuroImage for PPC measure

% last modified: March 2021


%% LOAD DATA
clearvars; clc; close all
fprintf('Loading data \n')
load('fig4_spike_LFP_NIH037_s1_data.mat')
% CONTAINS:
%  LFP_timeseries: (electrodes x trials x time), matrix of unfiltered local field potential (LFP) 
%  el_cnt: (1 x electrodes), [1:number of microelectrodes]
%  freqs: (1 x freqs), array, 60 bins logarithmically spaced between 2 and 400 Hz 
%  trials: (1 x trials), [1:number of trials]
%  response: (1 x trials), binary vector, 0 = incorrect memory retrieval, 
%       1 = means correct memory retrieval
% 
%  spike_raster: (units x time x trials)
%  lfp_channels: (1 x units), electrode label for each unit, use to find local
%       electrode for each unit, spike_raster and lfp_channels can be used
%       to remove spikes and interpolate the LFP_timeseries
% 
%  spike_id: (electrodes x trials), cell array containing all spike indices
%       from -4 sec before retrieval, sampling frequency of 1000 Hz
%  spike_ph: (electrodes x trials), cell array containing spike to LFP phases for
%       each frequency in freqs for each spike, instantaneous phase obtained 
%       using wavelet convolution, used to compute PPC
%
%  spike_id_lfprip: (electrodes x trials), cell array containing all spike indices
%       from -4 sec before retrieval for spikes within detected LFP ripples
%  spike_ph_lfprip: (electrodes x trials), cell array containing spike to LFP phases for
%       each frequency in freqs for each spike within detected LFP ripples, 
%       instantaneous phase obtained using wavelet convolution, used to compute PPC
%  NOTE: change spike_ph to spike_ph_lfprip in lines 55 to 57 for PPC of spikes within ripples
% 
%  lfpripple_logic: (trial x time x electrodes), raster of detected LFP
%       ripples using ripple detection as previously described in Vaz et
%       al., 2019, Science



%% Compute pairwise phase consistent using spike phases
% for all spikes, inside and outside detected LFP ripples

% specify number of samples for fair comparison across trial conditions
lim_cnt = 10; 

% specify number of surrogates (number of times to compute PPC for each condition: all trials, correct trials, incorrect trials)
numsurr = 100; 

[ppc_all,ppc_corr,ppc_incorr] = deal(NaN(numsurr,numel(el_cnt),numel(freqs)));
[input_tr, input_corr, input_incorr] = deal(cell(cell(numel(el_cnt),1)));

for surr = 1:numsurr
    fprintf('-- PPC for subsample %d \n',surr)
    for lfp_el = 1:numel(el_cnt)
        [input_tr{lfp_el}, input_corr{lfp_el}, input_incorr{lfp_el}] = deal([]);

        input_tr{lfp_el} = [input_tr{lfp_el}; (cell2mat({spike_ph{lfp_el,:}}))']; % all trials
        input_corr{lfp_el} = [input_corr{lfp_el}; (cell2mat({spike_ph{lfp_el,find(response == 1)}}))']; % correct trials
        input_incorr{lfp_el} = [input_incorr{lfp_el}; (cell2mat({spike_ph{lfp_el,find(response == 0)}}))']; % incorrect trials

        if size(input_tr{lfp_el},1) > lim_cnt & size(input_corr{lfp_el},1) > lim_cnt & size(input_incorr{lfp_el},1) > lim_cnt
            input_tr_sub = input_tr{lfp_el}(randperm(size(input_tr{lfp_el},1),lim_cnt),:);
            for ev = 1:size(input_tr_sub,2)
                outsum = input_tr_sub(:,ev); 
                
                % pairwise phase consistent (PPC) computation, reference Vinck et al. 2010 NeuroImage
                theta_x = real(exp(1i*outsum));
                theta_y = imag(exp(1i*outsum));
                ppctemp = (theta_x * theta_x') + (theta_y * theta_y'); 
                for ii = 1:size(outsum,1)
                    ppctemp(ii,ii) = NaN; % remove self comparisons
                end
                ppc_all(surr,lfp_el,ev) = nanmean(ppctemp(:)); % average over symmetric matrix of dot products of all pairs of spikes
            end


            input_corr_sub = input_corr{lfp_el}(randperm(size(input_corr{lfp_el},1),lim_cnt),:);
            for ev = 1:size(input_corr_sub,2)
                outsum = input_corr_sub(:,ev); 
                
                % pairwise phase consistent (PPC) computation, reference Vinck et al. 2010 NeuroImage
                theta_x = real(exp(1i*outsum));
                theta_y = imag(exp(1i*outsum));
                ppctemp = (theta_x * theta_x') + (theta_y * theta_y'); %
                for ii = 1:size(outsum,1)
                    ppctemp(ii,ii) = NaN; % remove self comparisons
                end
                ppc_corr(surr,lfp_el,ev) = nanmean(ppctemp(:)); % average over symmetric matrix of dot products of all pairs of spikes
            end


            input_incorr_sub = input_incorr{lfp_el}(randperm(size(input_incorr{lfp_el},1),lim_cnt),:);
            for ev = 1:size(input_incorr_sub,2)
                outsum  = input_incorr_sub(:,ev); %
                
                % pairwise phase consistent (PPC) computation, reference Vinck et al. 2010 NeuroImage
                theta_x = real(exp(1i*outsum));
                theta_y = imag(exp(1i*outsum));
                ppctemp = (theta_x * theta_x') + (theta_y * theta_y'); %
                for ii  = 1:size(outsum,1)
                    ppctemp(ii,ii) = NaN; % remove self comparisons
                end
                ppc_incorr(surr,lfp_el,ev) = nanmean(ppctemp(:));
            end

        end
    end
end





%% Plot PPC for all trials and for correct vs incorrect trials

figure; plot(freqs,squeeze(nanmean(nanmean(ppc_all,1),2)),'b-'); set(gca,'xscale','log')
xlim([2 200])
box off
set(gca,'fontsize',35)
title('all trials','fontsize',25)
xticks([2 10 30 100 200])
xticklabels([2 10 30 100 200])

figure; plot(freqs,squeeze(nanmean(nanmean(ppc_corr,1),2)),'b-'); set(gca,'xscale','log')
hold on; plot(freqs,squeeze(nanmean(nanmean(ppc_incorr,1),2)),'k-'); set(gca,'xscale','log')
xlim([2 200])
box off
set(gca,'fontsize',35)
title('correct vs incorrect trials','fontsize',25)
legend({' correct',' incorrect'})
legend boxoff
xticks([2 10 30 100 200])
xticklabels([2 10 30 100 200])






%% Method for spike interpolation

lfp_channels_u        = unique(lfp_channels);
LFP_timeseries_spkrmv = NaN(size(LFP_timeseries));

for trial = 1:numel(trials)
    fprintf('.. removing spikes for trial %d \n',trial)
    for lfp_el = 1:numel(el_cnt)        
        spkChId = find(lfp_channels==lfp_channels_u(lfp_el)); 

        %--- INTERPOLATION
        lfp_tr_el  = squeeze(LFP_timeseries(lfp_el,trial,:));

        %---- remove waveform and interprolate
        rippleID = conv(squeeze(sum(spike_raster(spkChId,:,trial),1)),ones(4,1)); %,'same'); %
        rippleID = rippleID(2:end-2); % -1 ms to +2 ms
                    
        lfp_tr_el(find(rippleID)) = NaN;
        lfp_tr_el(find(rippleID)) = interp1(find(~isnan(lfp_tr_el)),lfp_tr_el(~isnan(lfp_tr_el)),find(rippleID),'pchip');

        LFP_timeseries_spkrmv(lfp_el,trial,:) = lfp_tr_el;
        %----
    end
end

el = 1; tr = 1;
figure; plot(squeeze(LFP_timeseries(el,tr,:)))
hold on; plot(squeeze(LFP_timeseries_spkrmv(el,tr,:)))
legend({' orig', ' spike removed'},'location','northoutside'); legend boxoff
axis tight; box off
set(gca,'fontsize',25)







