%% LFPripple_iEEGripple_PPC.m (Figures 3)
% Shows computation of pairwise phase consistency (PPC) of LFP ripple band
% to iEEG ripple band. Each microelectrode array (MEA) electrode is compared to
% iEEG ripples detected in four nearby iEEG electrodes. For detection of iEEG 
% ripples, refer to Methods. Note, this takes a long time to run. Each iEEG
% electrode takes ~25 mins as PPC is measured for each detected iEEG
% ripple. Refer to Vinck et al., 2010, NeuroImage for PPC measure

% last modified: March 2021


%% LOAD DATA
% % save('fig3_LFP_iEEG_ripple_NIH037_data.mat','LFP_signal','iEEGripple_envZ',...
% %     'iEEGripple_logic','LFPripple_sig')

clear all; close all; clc
fprintf('Loading data \n')
load('fig3_LFP_iEEG_ripple_NIH037_data.mat')
% CONTAINS:
%   iEEGripple_envZ: (trial x time x iEEG electrode), ripple band (80-120 Hz)
%       iEEG signal across trials -4 s prior to retrieval, Fs = 1000 Hz
%   iEEGripple_logic: (trial x time x iEEG electrode), raster of detected
%       discrete ripples across trials -4 s prior to retrieval, Fs = 1000 Hz
%   LFP_signal: (trial x MEA electrode x time), unfiltered LFP signal 
%       across trials -4 s prior to retrieval, Fs = 1000 Hz
%   LFPripple_sig: (trial x MEA electrode x time), ripple band (80-120 Hz)
%       LFP signal across trials -4 s prior to retrieval, Fs = 1000 Hz



%% Compute LFP ripple - iEEG ripple PPC

% All unique electrode pairs for all 96 MEA electrodes 
chanpair = nchoosek([1:size(LFP_signal,2)],2); 

iEEG_elcnt = size(iEEGripple_envZ,3);
trcnt      = size(iEEGripple_envZ,1);

% Initialize variables to collect measures
iEEGripple_max    = cell(iEEG_elcnt,1);
LFPripple_ppc_max = cell(iEEG_elcnt,1);

nandiag = zeros(size(chanpair,1));
for nd = 1:size(nandiag,1)
    nandiag(nd,nd) = NaN; % Make matrix with NaN in the diagonal  
end

% Loop through each iEEG electrode near MEA
for iEEG_el = 1:iEEG_elcnt
    tic 
    iEEGripple_max{iEEG_el}    = [];
    LFPripple_ppc_max{iEEG_el} = [];
    
    ripplemacro_env_z = squeeze(iEEGripple_envZ(:,:,iEEG_el)); % trial x time
    
    for trial = 1:trcnt
        fprintf('.. trial %d \n',trial)
        
        % Find iEEG ripples
        grp = bwconncomp(squeeze(iEEGripple_logic(trial,:,iEEG_el)));
        id  = grp.PixelIdxList; 
        
        ripplemicro_sig = squeeze(LFPripple_sig(trial,:,:));
        
        % Compute PPC for each iEEG ripple
        for iiRip = 1:length(id)
            fprintf('..... ripple %d \n',iiRip)

            [ripmacro_max, maxid] = max(ripplemacro_env_z(trial,id{iiRip}));
            [~,ripmacro_max_id]   = find(ripplemacro_env_z(trial,:)==ripmacro_max);

            ripplemicro_phase     = angle(hilbert(ripplemicro_sig'))';


            % Collect ripple phase difference across all MEA electrode
            % pairs during iEEG ripple
            LFPripple_ph  = [ripplemicro_phase(:,id{iiRip})];%
            findpairdiff  = @(x) (LFPripple_ph(x,:));
            pairedphdiff  = findpairdiff(chanpair);
            pairedphdiff  = reshape(pairedphdiff,size(chanpair,1),2,numel(id{iiRip}));
            pairedphdiff  = squeeze(pairedphdiff(:,1,:) - pairedphdiff(:,2,:));
            
            
            % Collect iEEG ripple amplitude peak
            iEEGripple_max{iEEG_el}(end+1) = ripplemacro_env_z(trial,ripmacro_max_id);


            % Compute pairwise phase consistency (PPC)
            ppc_ripple = NaN(numel(id{iiRip}),1);
            for tstep  = 1:numel(id{iiRip})
                outsum = pairedphdiff(:,tstep)'; %
                
                theta_x = real(exp(1i*outsum))';
                theta_y = imag(exp(1i*outsum))';
                ppctemp = (theta_x * theta_x') + (theta_y * theta_y'); %
                                
                ppctemp = ppctemp + nandiag; % remove self comparisons
                
                ppc_ripple(tstep) = nanmean(ppctemp(:));
            end

            [maxppc,maxppcid] = max(ppc_ripple); % get maximum PPC

            % Collect LFP ripple PPC peak
            LFPripple_ppc_max{iEEG_el}(end+1) = maxppc;

        end
    end
    toc
end





%% Plot LFP ripple PPC vs peak iEEG ripple amplitude (z)
% each data point represents an iEEG ripple

figure('Position',[0 0 900 800]); 
for iEEG_el = 1:iEEG_elcnt
    % Plot for iEEG ripples detected in each iEEG electrode
    subplot(2,2,iEEG_el)
    plot(LFPripple_ppc_max{iEEG_el}, iEEGripple_max{iEEG_el},'b.','Markersize',40)
    
    % Pearson correlation
    [rv,pv] = corrcoef(LFPripple_ppc_max{iEEG_el},iEEGripple_max{iEEG_el});

    xlabel('Peak PPC'); ylabel('Peak iEEG Ripple Amplitude (Z)')
    title(['el: ' num2str(iEEG_el) ', r = ' num2str(round(rv(1,2),3)) ', p = ' num2str(round(pv,3))])
    box off
    set(gca,'fontsize',30)

end








