# -*- coding: utf-8 -*-
#
# This source file is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This source file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Date: 2020
#
# Authors: Paulina Dąbrowska (p.dabrowska@fz-juelich.de) [1,2]
#          Moritz Layer(m.layer@fz-juelich.de)[1, 2]
#
# Affiliations:
# [1] Institute of Neuroscience and Medicine (INM-6) and Institute for Advanced
#     Simulation (IAS-6) and JARA Institute Brain Structure-Function
#     Relationships (INM-10), Jülich Research Centre, Jülich, Germany
# [2] RWTH Aachen University, Aachen, Germany

"""
Partially creates fig. 1 of the manuscript, leaving panels A and B empty for
manual inclusion of the graphics stored in the `sketches/` main directory.

By modifying the parameters defined in the preamble, the user can alter the
plot appearance and file format, and choose the recording session, for which
a figure shall be generated. The meaning of each parameter is detailed in the
form of python comments in the relevant parts of the code.
"""

from __future__ import division, print_function

import numpy as np
import h5py_wrapper as h5w
import matplotlib.pyplot as plt
from matplotlib import gridspec
from string import ascii_uppercase

import helpers as hel
import salt_and_pepper
from helpers import coldict

# parameters
monkey = 'E2'  # E1, E2, N1 or N2, in the manuscript E2 is presented
formats = ['pdf']  # ['pdf', 'svg', 'eps', 'png'] for saving figures
ifSave = True
fonts = 11  # font sizes
dpi = 300
# if correlation coefficients are to be plotted instead of covariances
use_corr_coef = True

###############################################################################
RScovs_path = 'preprocessed/RScovs_' + monkey + '.h5'
cons_path = 'preprocessed/consistency_{}.txt'.format(monkey)

# load the preprocessed data
measures = h5w.load(RScovs_path)
# pairwise covariance matrix N_units x N_units
covm = measures['covm']
if use_corr_coef:
    avar = np.diagonal(covm)
    norm = np.sqrt(np.outer(avar, avar))
    covm = covm / norm
# physical positions of electrodes on the Utah array per SU
positions = measures['positions']  # N_units x 2
# threshold for consistency to separate putative excitatory and inhibitory SUs
eiThres = measures['eiThres']
# lists of putative excitatory/inhibitory SUs IDs
eIds, iIds = hel.ei_lists(cons_path, eiThres)

###############################################################################
# composite figure 1
# A. brain with Utah array
# B. binning schematics
# C. salt-and-pepper plot
# D. covariance distribution

# dims of figs given in mm -> convert to inches
width = hel.mm2inch(180)
height = hel.mm2inch(150)  # 185

fig = plt.figure(figsize=[width, height], dpi=dpi)
gs_top = gridspec.GridSpec(1, 5, figure=fig, top=0.95, bottom=0.7, left=0.05,
                           right=0.95)
gs_bottom = gridspec.GridSpec(1, 7, figure=fig, top=0.55, bottom=0.1,
                              left=0.05, right=0.95)
ax_brain = plt.subplot(gs_top[:, :2])
ax_covdist = plt.subplot(gs_top[:, 2:])
ax_binning = plt.subplot(gs_bottom[:, :3])
ax_spices = plt.subplot(gs_bottom[:, 3:])

# add letters to name panels
axs = [ax_brain, ax_binning, ax_covdist, ax_spices]
xs = [-0.1, -0.1, -0.15, -0.2]
for n, ax in enumerate(axs):
    ax.text(xs[n], 1.05, ascii_uppercase[n], transform=ax.transAxes,
            size=fonts + 1, weight='bold')

# add an inscription and an arrow to panel A
ax_brain.text(0.3, 0.8, 'Utah array\n(motor cortex)',
              transform=ax_brain.transAxes, size=fonts,
              horizontalalignment='center')
ax_brain.arrow(0.3, 0.75, 0.15, -0.25, length_includes_head=True,
               width=0.005, head_width=0.03, head_length=0.05, fc='k')

# remove axes from panels A and B
for ax in [ax_brain, ax_binning]:
    ax.set_frame_on(False)
    ax.set_xticks([])
    ax.set_yticks([])

# salt-and-pepper plot # # # # # # # # # # # # # # # # # # # # #
ref_idx = 96  # for E2: 96 (54) (invisible on a plot = unclassified)

salt_and_pepper.sap_plt(ax_spices,
                        covm,
                        eIds,
                        iIds,
                        positions,
                        reference_neuron=ref_idx,
                        plt_cb=True,
                        cbarlabel='correlation coefficient',
                        # Define Plot-Appearance
                        plt_SensID=False,
                        markersize=49,
                        min_msize=9,
                        fontsize=fonts
                        )

hel.labels(ax_spices, xlabel=r'$x [mm]$', ylabel=r'$y [mm]$', legend=False,
           fontsize=fonts)

# colorbar axis
cax = fig.get_axes()[-1]
yticks = [-0.2, -0.1, 0, 0.1, 0.2]
cax.set_yticks(yticks)
cax.set_yticklabels([-0.2, '', 0, '', 0.2], fontsize=fonts - 1)
cax.set_ylabel('correlation coefficient', fontsize=fonts, rotation=270,
               labelpad=5)
ax_spices.set_aspect('equal', adjustable='box')

# correlation distribution # # # # # # # # # # # # # # # # # # #
Nbins = 100
bins = np.linspace(-2, 2, Nbins)
step = bins[1] - bins[0]
nlist = []

to_plot = covm[eIds, :][:, eIds]
n, _ = hel.distribution(ax_covdist, to_plot, bins=bins, color=coldict['exc'],
                        lw=2.5, label='E-E', histtype='step', normed=True)
nlist.append(max(n))

to_plot = covm[iIds, :][:, iIds]
n, _ = hel.distribution(ax_covdist, to_plot, bins=bins, color=coldict['inh'],
                        lw=2.5, label='I-I', histtype='step', normed=True)
nlist.append(max(n))

to_plot = covm[eIds, :][:, iIds]
n, _ = hel.distribution(ax_covdist, to_plot, bins=bins, color=coldict['excinh'],
                        lw=1.5, label='E-I', histtype='step', normed=True)
nlist.append(max(n))

ax_covdist.set_xlim([-0.75, 0.75])
ax_covdist.set_ylim([0, max(nlist) * 1.05])
title = 'monkey {}, 1.0 s bins'.format(monkey)
hel.labels(ax_covdist, xlabel='correlation coefficient', ylabel='pdf', legend=True,
           fontsize=fonts)  #, title=title)

if not ifSave:
    plt.show()
# saving # # # # # # # # # # # #
namef = 'plots/fig1_' + monkey
hel.savef(fig, namef, formats, ifSave, transparent=True, dpi=dpi)
