%Analysis of Inter burst intervals (IBI) in the human seizure data.
%Burst detection uses LFP signal and is based on signal amplitude threshold.
% INPUT file:
% Human_seizure.mat
%      
% OUTPUT:
%   figure showing LFP signal and IBI fitted with linear (lin), exponential (exp), square root
%   (sqrt), logarithmic (log) functions with RMSE for each fit.
% VERSION:
%  Piotr Suffczynski, 29.06.2022

clear
close all
load Human_seizure

on = 4.5e4;
off = 9.6e4;
range = (on:off);  %signal range

%Time samples corresponding to spikes detected based on signal threshold
on1 = on;
off1 = 8.0e4;
on2 = off1+1;
off2 = off;
normal = LFP(on1:off1);
inverted = -LFP(on2:off2);  %inversion of LFP signal for burst detection
combined_LFP = [normal' inverted'];

%Time samples corresponding to spikes detected based on signal threshold
threshold  = 0.4;
Ind = find((combined_LFP) > threshold);

%Interspike intervals
ISI  = (t(Ind(2:end))-t(Ind(1:end-1)));

%Only interspike intervals > 150 ms, otherwise they are within a single burst 
ISI_th= 0.15;
Index = find(ISI > ISI_th);
 
%----------------LFP plot----------
fontsize = 10;

subplot(5,1,1);
plot(t, LFP,'k','LineWidth', 0.5)
hold on
%Ploting detected bursts
Index_range = 9:length(Index);
plot([t(Ind(Index(Index_range)))' t(Ind(Index(Index_range)))'] + t(on), [2.5 3], 'k','LineWidth', 0.25)
plot([t(Ind(Index(end)))' t(Ind(Index(end)))']+t(on)+ISI(Index(end)), [2.5 3],'k','LineWidth', 0.25) %last interval

%Interburst intervals - distance between bursts
IBI(1:length(Index_range)-1) = t(Ind(Index(Index_range(2:end))))-t(Ind(Index(Index_range(1:end-1))));
IBI(length(Index_range)) = t(Ind(end))-t(Ind(Index(Index_range(end))));

axis([0 100 -2.5 3])
set(gca, 'Fontsize', fontsize)
box off
ylabel('LFP (mV)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom') 
title ('TLE patient')

%--------------Human linear IBI plot----------------------------

subplot(5,1,2);
x = t(Ind(Index(Index_range)))+t(on);
y = IBI;
plot(x, y,'k.')
ylabel('IBI (s)')
set(gca, 'Fontsize', fontsize)
box off
grid on
axis([45 100 0 1.8])

hold on
c = polyfit(x,y,1);
y_est = polyval(c,x);
plot(x,y_est,'r', 'LineWidth',1)

%RMSE
ErrorVector_linear = y_est - y;
sse_linear = sum(ErrorVector_linear .^ 2);
RMS_linear = sqrt(sse_linear/length(x));

text(88,0.3, ['lin, RMSE ' num2str(RMS_linear)],'Fontsize',fontsize)


%--------------Exponential IBI plot----------------------------
subplot(5,1,3);
semilogy(x, y,'k.')
xe = x; %orignal x values stored in xe
x = x - min(x);
xval = x;
[xval, yval, est] = curve_fitting_exponential_1_optimized(x,y,xval);
semilogy(x+min(xe), y - est(1),'k.')
hold on
plot(xval + min(xe),yval ,'r','LineWidth',1);
grid on 
set(gcf,'color','w');
set(gca, 'Fontsize', fontsize)
box off
xlabel('Time (s)')
ylabel('IBI (s), log')
axis([45 100 0.01 1.2])

%RMSE
ErrorVector_exp = yval - y + est(1);
sse_exp = sum(ErrorVector_exp.^ 2);
RMS_exp = sqrt(sse_exp/length(x));
text(87,0.023, ['exp, RMSE ' num2str(RMS_exp)],'Fontsize',fontsize)

%--------------Square root IBI plot----------------------------
subplot(5,1,4);
xval = max(x) - x + 1; %time until end of SLE
plot(xval, y,'k.')
grid
hold on
c = polyfit(1./sqrt(xval),y,1);
y_est = polyval(c,1./sqrt(xval));
plot(xval,y_est,'r','LineWidth',1)
set(gca, 'XDir','reverse')
set(gca, 'Fontsize', fontsize)
box off
xlabel('Time until end of SLE (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'top')
ylabel('IBI (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')
axis([-5 50 0 2])

%RMSE
ErrorVector_sqrt = y_est - y;
sse_sqrt = sum(ErrorVector_sqrt.^ 2);
RMS_sqrt = sqrt(sse_sqrt/length(x));
text(7, 0.5, ['sqrt, RMSE ' num2str(RMS_sqrt)],'Fontsize',fontsize)

%--------------Logarithmic IBI plot----------------------------
subplot(5,1,5);
semilogx(xval, y,'k.')
grid
c = polyfit(log(xval),y,1);
y_est = polyval(c,log(xval));
hold on
semilogx(xval,y_est,'r','LineWidth',1)
set(gca, 'XDir','reverse')
box off
xlabel('Time until end of SLE (s), log','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'top')
ylabel('IBI (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')

%RMSE
ErrorVector_log = y_est - y;
sse_log = sum(ErrorVector_log.^ 2);
RMS_log = sqrt(sse_log/length(x));
text(1.8,0.5, ['log, RMSE ' num2str(RMS_log)],'Fontsize',fontsize)
axis([0.7 70 0 2])

%Exponential Curve Fitting based on 
%https://stackoverflow.com/questions/29749235/matlab-exponential-curve-fitting-without-toolbox

function [xval, yval,est] = curve_fitting_exponential_1_optimized(x,y,xval)

x = x(:);  % bring the data into the standard, more convenient format of column vectors
y = y(:);

Aguess = min(y) - (max(y)-min(y)) / 2;
guess = [ones(size(x)), -x] \ log(y - Aguess);
Bguess = exp(guess(1));
Cguess = guess(2);

start_point = [Aguess, Bguess, Cguess];
est = fminsearch(@expfun, start_point);

    function [sse, FittedCurve] = expfun(params)
        A = params(1);
        B = params(2);
        C = params(3);
        FittedCurve = A + B .* exp(-C * x);
        ErrorVector_exp = FittedCurve - y;
        sse = sum(ErrorVector_exp .^ 2);
      end

%yval = est(1)+est(2) * exp(-est(3) * xval);

yval = est(2) * exp(-est(3) * xval);
end




% 