%Analysis of Inter burst intervals (IBI) in the model data.
%Burst detection is based on voltage membrane potential signal (VPY)
%and uses signal derrivative threshold.
% INPUT file:
% Model_SLE.mat
%      
% OUTPUT:
%   figure showing LFP signal and IBI fitted with linear (lin), exponential (exp), square root
%   (sqrt), logarithmic (log) functions with RMSE for each fit.
% VERSION:
%  Piotr Suffczynski, 29.06.2022

load Model_SLE
close all

dt = 2e-4;
t1 = (3.8e5); %signal range
t2 = (7e5);
V = VP(t1:t2); %voltage membrane potential for detection

%Time samples corresponding to spikes detected based on voltage threshold
Ind = find((V > -7) > 0);
%Interspike intervals
ISI  = t(Ind(2:end))-t(Ind(1:end-1));

%Only interspike intervals > 300*sampling interval, otherwise they are within a single burst
Index = find(ISI>300*dt);

%--------------LFP plot----------------------------
fontsize = 10; %font size
toffset = 55;
subplot(5,1,1)
plot(t-toffset, LFP,'k', 'LineWidth', 0.5)
hold on

%Ploting detected bursts
Index_range = 50:length(Index);
plot([t(Ind(Index(Index_range))) t(Ind(Index(Index_range)))]+t(t1)-toffset, [0.05 0.08], 'k','LineWidth', 0.25)
plot([0 0]+t(t1)-toffset+t(Ind(end)), [0.05 0.08], 'k','LineWidth', 0.25)% last burst

%Interburst intervals - distance between bursts
IBI(1:length(Index_range)-1) = t(Ind(Index(Index_range(2:end))))-t(Ind(Index(Index_range(1:end-1))));
IBI(length(Index_range)) = t(Ind(end))-t(Ind(Index(Index_range(end))));

axis([0 65 -0.2 0.1])
set(gca, 'Fontsize', fontsize)
box off
ylabel('LFP (mV)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom') 
text(30,0.23, 'Model','Fontsize',fontsize,'FontWeight', 'Bold')


%--------------Linear IBI plot----------------------------
subplot(5,1,2)
x = (t(Ind(Index(Index_range)))+t(t1)-toffset);%
y = IBI';
plot(x, y,'k.')
ylabel('IBI (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')
axis([30 60 0 1.2])
grid
set(gca, 'Fontsize', fontsize)
box off
%line fitting
hold on
c = polyfit(x,y,1);
y_est = polyval(c,x);
plot(x,y_est,'r','LineWidth',1)


%RMSE
ErrorVector_linear = y_est - y;
sse_linear = sum(ErrorVector_linear .^ 2);
RMS_linear = sqrt(sse_linear/length(x));

text(53,0.23, ['lin, RMSE ' num2str(RMS_linear)],'Fontsize',fontsize)


%--------------Exponential IBI plot----------------------------
subplot(5,1,3)
xe = x; %orignal x values stored in xe
x = x - min(x);
xval = x;
[xval, yval, est] = curve_fitting_exponential_1_optimized(x,y,xval);
semilogy(x+min(xe), y - est(1),'k.',xval + min(xe),yval ,'r');
hold on
grid on 
xlabel('Time (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'top')
ylabel('IBI (s), log','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')
axis([30 60 0.01 1])
set(gcf,'color','w');
set(gca, 'Fontsize', fontsize)
box off

%RMSE
ErrorVector_exp = yval + est(1) - y;
sse_exp = sum(ErrorVector_exp.^ 2);
RMS_exp = sqrt(sse_exp/length(x));

text(53,0.025, ['exp, RMSE ' num2str(RMS_exp)],'Fontsize',fontsize)

%--------------Square root IBI plot----------------------------
subplot(5,1,4);
xval = max(x) - x + 1; %time until end of SLE
plot(xval, y,'k.')
grid
hold on
c = polyfit(1./sqrt(xval),y,1);
y_est = polyval(c,1./sqrt(xval));
plot(xval,y_est,'r','LineWidth',1)
set(gca, 'XDir','reverse')
set(gca, 'Fontsize', fontsize)
box off
xlabel('Time until end of SLE (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'top')
ylabel('IBI (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')
axis([0 30 0 1.2])

%RMSE
ErrorVector_sqrt = y_est - y;
sse_sqrt = sum(ErrorVector_sqrt.^ 2);
RMS_sqrt = sqrt(sse_sqrt/length(x));
text(7,0.3, ['sqrt, RMSE ' num2str(RMS_sqrt)],'Fontsize',fontsize)

%--------------Logarithmic IBI plot----------------------------
subplot(5,1,5);
semilogx(xval, y,'k.')
grid
c = polyfit(log(xval),y,1);
y_est = polyval(c,log(xval));
hold on
semilogx(xval,y_est,'r','LineWidth',1)
set(gca, 'XDir','reverse')
box off
xlabel('Time until end of SLE (s), log','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'top')
ylabel('IBI (s)','FontSize',fontsize, 'FontWeight', 'Normal', 'VerticalAlignment', 'bottom')

%RMSE
ErrorVector_log = y_est - y;
sse_log = sum(ErrorVector_log.^ 2);
RMS_log = sqrt(sse_log/length(x));
text(2,0.36, ['log, RMSE ' num2str(RMS_log)],'Fontsize',fontsize)
axis([0.7 60 0 1.2])


%Exponential Curve Fitting based on 
%https://stackoverflow.com/questions/29749235/matlab-exponential-curve-fitting-without-toolbox

function [xval, yval,est] = curve_fitting_exponential_1_optimized(x,y,xval)

x = x(:);  % bring the data into the standard, more convenient format of column vectors
y = y(:);

Aguess = min(y) - (max(y)-min(y)) / 2;
guess = [ones(size(x)), -x] \ log(y - Aguess);
Bguess = exp(guess(1));
Cguess = guess(2);

start_point = [Aguess, Bguess, Cguess];
est = fminsearch(@expfun, start_point);

    function [sse, FittedCurve] = expfun(params)
        A = params(1);
        B = params(2);
        C = params(3);
        FittedCurve = A + B .* exp(-C * x);
        ErrorVector_exp = FittedCurve - y;
        sse = sum(ErrorVector_exp .^ 2);
      end

%yval = est(1)+est(2) * exp(-est(3) * xval);

yval = est(2) * exp(-est(3) * xval);
end